/*****************************************************************************
 * Copyright (c) 1997-2007, Intel Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *
 * $Id$ 
 *****************************************************************************/

#include <string.h>

#include "JVMPIInterface.h"
#include "LogAssert.h"
#include "OSA.h"

using namespace Martini::JPI;
using namespace Martini::MPI;
using namespace Martini::Infrastructure;
using namespace Martini::OSA;

static CEventManager *gs_pEventManager;
static CJVMPIInterface *gs_pThis;

static char* CreateCopy(const char* szSource)
{
    if (NULL == szSource)
    {
        return NULL;
    }
    char *szDest = new char[strlen(szSource) + 1];
    strcpy(szDest, szSource);
    return szDest;
}

extern "C" void NotifyEvent(JVMPI_Event *pJVMPIEvent)
{
    static CThreadInfoManager *pThreadInfoManager = 
        gs_pEventManager->GetDM()->GetThreadInfoManager();

    // reset the JVMPI_REQUESTED_EVENT bit (if exists) so we can translate the actual
    // JVMPI event to an MPI event.
    // (The JVMPI_REQUESTED_EVENT bit is set by the JVM when the event is sent in response
    // to a RequestEvent call)
    jint jvmpiEvent = pJVMPIEvent->event_type & ~JVMPI_REQUESTED_EVENT;

    TEventType mpiEvent = gs_pThis->GetMpiEvent(jvmpiEvent);
    SJVMData data;
    SEmData mpiData;
    SEmData *pMpiData = &mpiData;
    mpiData.u.pJVMData = &data;
    mpiData.dataType = MPI_VM;
    mpiData.pJniEnv = pJVMPIEvent->env_id;
    bool bShouldDeleteClassLoadData = false;
    bool bSendEvent = true;

    switch (jvmpiEvent) 
    {
    case JVMPI_EVENT_METHOD_ENTRY:
        data.dataType = DT_METHOD_ENTER;
        data.u.method.methodId = pJVMPIEvent->u.method.method_id;
    	break;

    case JVMPI_EVENT_METHOD_EXIT:
        data.dataType = DT_METHOD_LEAVE;
        data.u.method.methodId = pJVMPIEvent->u.method.method_id;
        break;

    case JVMPI_EVENT_COMPILED_METHOD_LOAD:
        MARTINI_INFORMATIVE("CJVMPIInterface", 5, false, "NotifyEvent: COMPILED_METHOD_LOAD");
        data.dataType = DT_METHOD_LOAD;
        data.u.methodLoad.methodId = pJVMPIEvent->u.compiled_method_load.method_id;
        data.u.methodLoad.uiCodeSize = pJVMPIEvent->u.compiled_method_load.code_size;
        data.u.methodLoad.pCodeAddr = pJVMPIEvent->u.compiled_method_load.code_addr;

        // Populate the "native-to-source-line" line map
        data.u.methodLoad.nativeToSrcLineMap.uiSize = 
            pJVMPIEvent->u.compiled_method_load.lineno_table_size;
        data.u.methodLoad.nativeToSrcLineMap.uiActualSize = 
            data.u.methodLoad.nativeToSrcLineMap.uiSize;
        data.u.methodLoad.nativeToSrcLineMap.entries = 
            (SLineNumberTableEntry *)pJVMPIEvent->u.compiled_method_load.lineno_table;

        // Reset all other line tables (information is not available)
        data.u.methodLoad.nativeToManagedLineMap.uiSize = 0;
        data.u.methodLoad.nativeToManagedLineMap.uiActualSize = 0;
        data.u.methodLoad.nativeToManagedLineMap.entries = NULL;

        data.u.methodLoad.managedToSrcLineMap.uiSize = 0;
        data.u.methodLoad.managedToSrcLineMap.uiActualSize = 0;
        data.u.methodLoad.managedToSrcLineMap.entries = NULL;
        break;

    case JVMPI_EVENT_CLASS_LOAD:
        MARTINI_INFORMATIVE("CJVMPIInterface", 5, false, "NotifyEvent: CLASS_LOAD");
        data.dataType = DT_CLASS_PREPARE;
        data.u.classPrepare.szClassName = pJVMPIEvent->u.class_load.class_name;
        data.u.classPrepare.szSourceName = pJVMPIEvent->u.class_load.source_name;
        data.u.classPrepare.uiNumMethods = pJVMPIEvent->u.class_load.num_methods;
        data.u.classPrepare.szGenericSig = NULL;
        data.u.classPrepare.jniClassLocalRef = NULL;  // member not needed since class 
                                                      // redefinition is not supported under 
                                                      // JVMPI
        data.u.classPrepare.jniLoaderLocalRef = NULL; // not supported by JVMPI
        data.u.classPrepare.pMethodInfo = new SJVMMethodInfo[data.u.classPrepare.uiNumMethods];
        if (data.u.classPrepare.pMethodInfo == NULL)
        {
            MARTINI_ERROR("JVMPIInterface", "Failed to allocate memory");
            data.u.classPrepare.uiNumMethods = 0;
        }
        else
        {
            bShouldDeleteClassLoadData = true;
        }
        unsigned int i;
        for (i = 0; i < data.u.classPrepare.uiNumMethods; ++i)
        {
            data.u.classPrepare.pMethodInfo[i].methodId = 
                pJVMPIEvent->u.class_load.methods[i].method_id;   
            data.u.classPrepare.pMethodInfo[i].szMethodName = 
                pJVMPIEvent->u.class_load.methods[i].method_name;
            data.u.classPrepare.pMethodInfo[i].szMethodSignature = 
                pJVMPIEvent->u.class_load.methods[i].method_signature;
            data.u.classPrepare.pMethodInfo[i].lineNumbers.uiStart =
                pJVMPIEvent->u.class_load.methods[i].start_lineno;
            data.u.classPrepare.pMethodInfo[i].lineNumbers.uiEnd =
                pJVMPIEvent->u.class_load.methods[i].end_lineno;  
            data.u.classPrepare.pMethodInfo[i].szMethodGeneric = NULL;
            data.u.classPrepare.pMethodInfo[i].uiAccessFlags = 0;
        }
        break;

    case JVMPI_EVENT_CLASS_LOAD_HOOK:
        MARTINI_INFORMATIVE("CJVMPIInterface", 5, false, "NotifyEvent: CLASS_LOAD_HOOK");
        data.dataType = DT_CLASS_LOAD_HOOK;
        data.u.classLoadHook.szClassName = NULL; // not available in JVMPI
        data.u.classLoadHook.classDataLen = pJVMPIEvent->u.class_load_hook.class_data_len;
        data.u.classLoadHook.pucClassData = pJVMPIEvent->u.class_load_hook.class_data;
        data.u.classLoadHook.pNewClassDataLen = 
            &pJVMPIEvent->u.class_load_hook.new_class_data_len;
        data.u.classLoadHook.ppucNewClassData = &pJVMPIEvent->u.class_load_hook.new_class_data;
        data.u.classLoadHook.pfnAllocate = pJVMPIEvent->u.class_load_hook.malloc_f;
        data.u.classLoadHook.classBeingRedefined = NULL; // not available in JVMPI
        data.u.classLoadHook.loader = NULL; // not available in JVMPI
        break;

    case JVMPI_EVENT_THREAD_START:
        MARTINI_INFORMATIVE("CJVMPIInterface", 5, false, "NotifyEvent: THREAD_START");
        if (0 != pThreadInfoManager->GetThreadId(pJVMPIEvent->u.thread_start.thread_env_id))
        {
            // EV_THREAD_START was already reported for this thread
            bSendEvent = false;
        }
        else
        {
            data.dataType = DT_THREAD_START;
            data.u.threadStart.pThreadEnv = pJVMPIEvent->u.thread_start.thread_env_id;
            data.u.threadStart.szName = CreateCopy(pJVMPIEvent->u.thread_start.thread_name);
            data.u.threadStart.szGroupName = 
                CreateCopy(pJVMPIEvent->u.thread_start.group_name);
            data.u.threadStart.szParentGroupName = 
                CreateCopy(pJVMPIEvent->u.thread_start.parent_name);
            data.u.threadStart.threadLocalRef = NULL;
        }
        break;
    case JVMPI_EVENT_THREAD_END: 
        MARTINI_INFORMATIVE("CJVMPIInterface", 5, false, "NotifyEvent: THREAD_END");
        data.dataType = DT_THREAD_END;
        data.u.threadEnd.pThreadEnv = pJVMPIEvent->env_id;
        data.u.threadEnd.threadLocalRef = NULL;
        break;

    case JVMPI_EVENT_COMPILED_METHOD_UNLOAD:
        MARTINI_INFORMATIVE("CJVMPIInterface", 5, false, "NotifyEvent: COMPILED_METHOD_UNLOAD");
        data.dataType = DT_METHOD_UNLOAD;
        data.u.method.methodId = pJVMPIEvent->u.compiled_method_unload.method_id;
        break;

    case JVMPI_EVENT_JVM_INIT_DONE:
        MARTINI_INFORMATIVE("CJVMPIInterface", 5, false, 
            "NotifyEvent: JVM_INIT_DONE/JVM_SHUT_DOWN");
        // fall through
    case JVMPI_EVENT_JVM_SHUT_DOWN:
        data.dataType = DT_VM;
        data.u.vmEvent.pJniEnv = pJVMPIEvent->env_id;
        break;
    case JVMPI_EVENT_GC_START:
    case JVMPI_EVENT_GC_FINISH:
        // events with no data
        pMpiData = NULL;
        break;
    default:
        MARTINI_INFORMATIVE1("CJVMPIInterface", 5, false, 
            "NotifyEvent: unhandled event %d", jvmpiEvent);
    }
    
    if (bSendEvent)
    {
        gs_pEventManager->NotifyMpiEvent(mpiEvent, pMpiData);
    }

    if (bShouldDeleteClassLoadData)
    {
        delete data.u.classPrepare.pMethodInfo;
    }
}


//////////////////////////////////////////////////////////////////////////
// CJVMPIInterface implementation

CJVMPIInterface::CJVMPIInterface()
{
    MARTINI_INFORMATIVE("CJVMPIInterface", 0, false, "Using JVMPI");
    gs_pThis = this;
}

CJVMPIInterface::~CJVMPIInterface()
{

}

void CJVMPIInterface::InitEventTranslationArrays()
{
    // initialize the event translation arrays
    memset(m_vMpi2JVMPIEvents, 0, EM_EVENT_LAST * sizeof(jint));
    m_vMpi2JVMPIEvents[EV_JITTED_METHOD_LOADED] = JVMPI_EVENT_COMPILED_METHOD_LOAD;
    m_vMpi2JVMPIEvents[EV_JITTED_METHOD_UNLOADED] = JVMPI_EVENT_COMPILED_METHOD_UNLOAD;
    m_vMpi2JVMPIEvents[EV_THREAD_START] = JVMPI_EVENT_THREAD_START;
    m_vMpi2JVMPIEvents[EV_THREAD_END] = JVMPI_EVENT_THREAD_END;
    m_vMpi2JVMPIEvents[EV_GC_START] = JVMPI_EVENT_GC_START;
    m_vMpi2JVMPIEvents[EV_GC_END] = JVMPI_EVENT_GC_FINISH;
    m_vMpi2JVMPIEvents[EM_EVENT_INTERNAL_CLASS_PREPARE] = JVMPI_EVENT_CLASS_LOAD;
    m_vMpi2JVMPIEvents[EV_JAVA_CLASS_FILE_LOAD_HOOK] = JVMPI_EVENT_CLASS_LOAD_HOOK;
    m_vMpi2JVMPIEvents[EV_VM_INIT] = JVMPI_EVENT_JVM_INIT_DONE;
    m_vMpi2JVMPIEvents[EV_VM_SHUTDOWN] = JVMPI_EVENT_JVM_SHUT_DOWN;
    m_vMpi2JVMPIEvents[EV_METHOD_ENTER] = JVMPI_EVENT_METHOD_ENTRY;
    m_vMpi2JVMPIEvents[EV_METHOD_LEAVE] = JVMPI_EVENT_METHOD_EXIT;

    memset(m_vJVMPI2MpiEvents, 0, JVMPI_MAX_EVENT_TYPE_VAL * sizeof(TEventType));
    m_vJVMPI2MpiEvents[JVMPI_EVENT_COMPILED_METHOD_LOAD] = EV_JITTED_METHOD_LOADED;
    m_vJVMPI2MpiEvents[JVMPI_EVENT_COMPILED_METHOD_UNLOAD] = EV_JITTED_METHOD_UNLOADED;    
    m_vJVMPI2MpiEvents[JVMPI_EVENT_THREAD_START] =  EV_THREAD_START;
    m_vJVMPI2MpiEvents[JVMPI_EVENT_THREAD_END] = EV_THREAD_END;
    m_vJVMPI2MpiEvents[JVMPI_EVENT_GC_START] = EV_GC_START;
    m_vJVMPI2MpiEvents[JVMPI_EVENT_GC_FINISH] = EV_GC_END;
    m_vJVMPI2MpiEvents[JVMPI_EVENT_CLASS_LOAD] = (TEventType)EM_EVENT_INTERNAL_CLASS_PREPARE;
    m_vJVMPI2MpiEvents[JVMPI_EVENT_CLASS_LOAD_HOOK] = EV_JAVA_CLASS_FILE_LOAD_HOOK;
    m_vJVMPI2MpiEvents[JVMPI_EVENT_JVM_INIT_DONE] = EV_VM_INIT;
    m_vJVMPI2MpiEvents[JVMPI_EVENT_JVM_SHUT_DOWN] = EV_VM_SHUTDOWN;
    m_vJVMPI2MpiEvents[JVMPI_EVENT_METHOD_ENTRY] = EV_METHOD_ENTER;
    m_vJVMPI2MpiEvents[JVMPI_EVENT_METHOD_EXIT] = EV_METHOD_LEAVE;
}


TResult CJVMPIInterface::Initialize(TId moduleId, JavaVM *pJVM)
{
    m_pJVM = pJVM;

    // Initialize the Capabilities Manager
    ICapabilityFactory *pCapFactory = new CPICapabilityFactory();
    if (NULL == pCapFactory)
    {
        MARTINI_ERROR("CJVMPIInterface", "failed to initialize the capability manager");
        return MRTE_ERROR_FAIL;
    }
    m_pCapManager->Initialize(pCapFactory);
    delete pCapFactory;

    // Get JVMPI interface pointer
    m_pJVMPIInterface = NULL;
    jint ret = pJVM->GetEnv((void **)&m_pJVMPIInterface, JVMPI_VERSION_1);
    if (ret != JNI_OK)
    {
        return MRTE_ERROR_FAIL;
    }
    
    // Get a handle of the event manager
    m_pEventManager = gs_pEventManager = CEventManager::GetInstance();
    if (m_pEventManager == NULL)
    {
        return MRTE_ERROR_FAIL;
    }
    
    // Init internal data structures
    InitEventTranslationArrays();

    InformAvailableEvents();


    // set callback pointers
    m_pJVMPIInterface->NotifyEvent = NotifyEvent;

    return MRTE_RESULT_OK;
}

void CJVMPIInterface::ProfilerExit()
{

}

TResult CJVMPIInterface::RegisterEvent(TEventType mpiEvent)
{
    jint jvmpiEvent = m_vMpi2JVMPIEvents[mpiEvent];
    if (!jvmpiEvent)
    {
        // event is not supported
        return MRTE_ERROR_FAIL;
    }
    
    int iRes = m_pJVMPIInterface->EnableEvent(jvmpiEvent, NULL);

    if (iRes == JVMPI_SUCCESS)
    {
        return MRTE_RESULT_OK;
    }
    else 
    {
        return MRTE_ERROR_FAIL;
    }
}

TResult CJVMPIInterface::DisableEvent(TEventType mpiEvent)
{
    jint jvmpiEvent = m_vMpi2JVMPIEvents[mpiEvent];
    
    int iRes = m_pJVMPIInterface->DisableEvent(jvmpiEvent, NULL);

    if (iRes == JVMPI_SUCCESS)
    {
        return MRTE_RESULT_OK;
    }
    else 
    {
        return MRTE_ERROR_FAIL;
    }
}

TResult CJVMPIInterface::EnableEvent(TEventType mpiEvent)
{
    jint jvmpiEvent = m_vMpi2JVMPIEvents[mpiEvent];
    
    int iRes = m_pJVMPIInterface->EnableEvent(jvmpiEvent, NULL);

    if (iRes == JVMPI_SUCCESS)
    {
        return MRTE_RESULT_OK;
    }
    else 
    {
        return MRTE_ERROR_FAIL;
    }
}

TResult CJVMPIInterface::RequestClassLoadEvent(UIOP methodId)
{
    m_pJVMPIInterface->DisableGC();
    jobjectID jvmpiClassId = m_pJVMPIInterface->GetMethodClass((jmethodID)methodId);
    jint res = m_pJVMPIInterface->RequestEvent(JVMPI_EVENT_CLASS_LOAD, (void *)jvmpiClassId);
    m_pJVMPIInterface->EnableGC();
    if (res == JVMPI_SUCCESS )
    {
        return MRTE_RESULT_OK;
    }
    return MRTE_ERROR_FAIL;
}


bool CJVMPIInterface::CanSupplyClassLoadHookToAllClasses()
{
    // this should be according to vendor. IBM can supply. BEA and Sun can't supply
    return false;
}

void CJVMPIInterface::InformAvailableEvents()
{
    int event;
    for (event = 0; event < EM_EVENT_LAST; ++event)
    {
        if (m_vMpi2JVMPIEvents[event] != 0)
        {
            // this is a supported event
            m_pEventManager->SetEventImplementor((TEventType)event, MPI_VM);
        }
    }
}

TResult CJVMPIInterface::GetJNIEnv(JNIEnv **ppJNIEnv)
{
    if (!ppJNIEnv)
    {
        return MRTE_ERROR_ILLEGAL_ARGUMENT;
    }
    
    JNIEnv *pJNIEnv = NULL;
    
    jint ret = m_pJVM->GetEnv((void**)&pJNIEnv, JNI_VERSION_1_4);
    
    if (ret != JNI_OK)
    {
        ret = m_pJVM->GetEnv((void**)&pJNIEnv, JNI_VERSION_1_2);
        if (ret != JNI_OK)
        {
            return MRTE_ERROR_FAIL;
        }
    }
    
    *ppJNIEnv = pJNIEnv;
    return MRTE_RESULT_OK;
}

EInterfaceType CJVMPIInterface::GetInterfaceType()
{
    return IT_JVMPI;
}

TResult CJVMPIInterface::GetCurrentThreadCpuTime(TTimeStamp *pCpu)
{
    *pCpu = m_pJVMPIInterface->GetCurrentThreadCpuTime();
    return MRTE_RESULT_OK;
}

TResult CJVMPIInterface::RunGC()
{
    m_pJVMPIInterface->RunGC();
    return MRTE_RESULT_OK;
}
//////////////////////////////////////////////////////////////////////////
// CPICapabilityFactory implementation

ICapability* CPICapabilityFactory::CreateCapability(ECapabilityType cap)
{
    // JVMPI does not support the notion of capabilities, so just indicate
    // which JPI capability is supported in this environment and which is not.
    ICapability *pCap = NULL;

    switch (cap)
    {
    case CT_CAN_GET_THREAD_TIMES:
    case CT_CAN_REGISTER_MULTIPLE_CALLGRAPH_EVENT_CLIENTS:
        pCap = new CGenericSupportedCapability();
    	break;
    default:
        pCap = new CGenericUnsupportedCapability();
        break;
    }

    return pCap;
}
