/*****************************************************************************
 * Copyright (c) 1997, 2010 Intel Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *
 * $Id$
 *****************************************************************************/

#include <string.h>
#include <stdlib.h>

#include "JVMTIInterface.h"
#include "LibraryLoader.h"
#include "LogAssert.h"
#include "JpiGlobals.h"

using namespace Martini::JPI;
using namespace Martini::MPI;
using namespace Martini::Infrastructure;
using namespace Martini::OSA;
using namespace Martini::RTUtil;

struct SObjectRefIterationControl
{
    SObjectRefIterationControl(): pObjRefInfo(NULL), uiNextArrayElement(0),
        result(MRTE_RESULT_OK), pTagVector(NULL), opsIterHeap(ITER_HEAP_NO_OPS) {}
    SObjectReferenceInfo *pObjRefInfo;
    TTagVector *pTagVector;
    U64 uiNextArrayElement;
    IterHeapOps opsIterHeap;
    TResult result;
};

//
// A map between JVMTI heap root kinds and MPI object reference types
//
static EObjectReferenceType JvmtiRootKind2MpiRefType[] = {
    OR_OTHER,                               //!< Unknown value
    OR_JAVA_JNI_GLOBAL,                     //!< JVMTI_HEAP_ROOT_JNI_GLOBAL
    OR_SYSTEM_CLASS,                        //!< JVMTI_HEAP_ROOT_SYSTEM_CLASS
    OR_MONITOR,                             //!< JVMTI_HEAP_ROOT_MONITOR
    OR_STACK_LOCAL,                         //!< JVMTI_HEAP_ROOT_STACK_LOCAL
    OR_JAVA_JNI_LOCAL,                      //!< JVMTI_HEAP_ROOT_JNI_LOCAL
    OR_THREAD,                              //!< JVMTI_HEAP_ROOT_THREAD
    OR_OTHER                                //!< JVMTI_HEAP_ROOT_OTHER
};

//
// A map between JVMTI object reference kinds and MPI object reference types
//
static EObjectReferenceType JvmtiRefKind2MpiRefType[] = {
    OR_OTHER,                               //!< Unknown value
    OR_CLASS,                               //!< JVMTI_REFERENCE_CLASS
    OR_FIELD,                               //!< JVMTI_REFERENCE_FIELD
    OR_ARRAY_ELEMENT,                       //!< JVMTI_REFERENCE_ARRAY_ELEMENT
    OR_JAVA_CLASS_LOADER,                   //!< JVMTI_REFERENCE_CLASS_LOADER
    OR_JAVA_REFERENCE_SIGNERS,              //!< JVMTI_REFERENCE_SIGNERS
    OR_JAVA_PROTECTION_DOMAIN,              //!< JVMTI_REFERENCE_PROTECTION_DOMAIN
    OR_INTERFACE,                           //!< JVMTI_REFERENCE_INTERFACE
    OR_STATIC_FIELD,                        //!< JVMTI_REFERENCE_STATIC_FIELD
    OR_CONSTANT_POOL                        //!< JVMTI_REFERENCE_CONSTANT_POOL
};

static CEventManager *gs_pEventManager;
static CJVMTIInterface *gs_pThis;

extern "C" void *AllocateBuffer(unsigned int uiSize)
{
    unsigned char *pBuffer;
    jvmtiEnv *pJvmtiEnv = gs_pThis->GetJVMTIInterface();
    jvmtiError err = pJvmtiEnv->Allocate(uiSize, &pBuffer);
    if (err == JVMTI_ERROR_NONE)
    {
        return (void *)pBuffer;
    }
    return NULL;
}

void ExtractFirstAndLastLineNumbers(jint *pFirstLine,
                                    jint *pLastLine,
                                    jvmtiLineNumberEntry *pLineNumberTable,
                                    jint tableSize)
{
    // The line number table returned from JVMTI is not sorted by line number.
    // Iterate through the table and find the minimum (first) and the maximun (last)
    // line numbers.
    if (pLineNumberTable != NULL && tableSize > 0)
    {
        jint first = pLineNumberTable[0].line_number;
        jint last = pLineNumberTable[0].line_number;
        jint i;
        for (i = 0; i < tableSize; ++i)
        {
            if (pLineNumberTable[i].line_number < first)
            {
                first = pLineNumberTable[i].line_number;
            }
            if (pLineNumberTable[i].line_number > last)
            {
                last = pLineNumberTable[i].line_number;
            }
        }
        *pFirstLine = first;
        *pLastLine = last;

    }
    else
    {
        *pFirstLine = 0;
        *pLastLine = 0;
    }
}

//////////////////////////////////////////////////////////////////////////
// JVMTI event handlers

extern "C" void JNICALL MonitorContendedEnterHandler(jvmtiEnv *pJvmtiEnv,
                                                     JNIEnv* pJniEnv,
                                                     jthread thread,
                                                     jobject object)
{
    SJVMData data;
    SEmData mpiData;
    mpiData.pJniEnv = pJniEnv;
    mpiData.u.pJVMData = &data;
    mpiData.dataType = MPI_VM;
    data.dataType = DT_CONTENDED_MONITOR;
    memset(&data.u.contendedMonitor, 0, sizeof(data.u.contendedMonitor));

    data.u.contendedMonitor.threadLocalRef = thread;
    data.u.contendedMonitor.monitorLocalRef = object;

    MARTINI_INFORMATIVE3("CJVMTIInterface", 5, false,
        "[MonitorContendedEnterHandler] pJniEnv = %p, thread = %p, monitor = %p",
        pJniEnv, thread, object);

    // Send the event
    gs_pEventManager->NotifyMpiEvent(EV_CONTENDED_MONITOR_ENTER, &mpiData);
}

extern "C" void JNICALL MonitorContendedEnteredHandler(jvmtiEnv *pJvmtiEnv,
                                                       JNIEnv* pJniEnv,
                                                       jthread thread,
                                                       jobject object)
{
    SJVMData data;
    SEmData mpiData;
    mpiData.pJniEnv = pJniEnv;
    mpiData.u.pJVMData = &data;
    mpiData.dataType = MPI_VM;
    data.dataType = DT_CONTENDED_MONITOR;
    memset(&data.u.contendedMonitor, 0, sizeof(data.u.contendedMonitor));

    data.u.contendedMonitor.threadLocalRef = thread;
    data.u.contendedMonitor.monitorLocalRef = object;

    MARTINI_INFORMATIVE3("CJVMTIInterface", 5, false,
        "[MonitorContendedEnteredHandler] pJniEnv = %p, thread = %p, monitor = %p",
        pJniEnv, thread, data.u.contendedMonitor.monitorLocalRef);

    // Send the event
    gs_pEventManager->NotifyMpiEvent(EV_CONTENDED_MONITOR_ENTERED, &mpiData);
}

extern "C" void JNICALL MonitorWaitHandler(jvmtiEnv *pJvmtiEnv,
                                           JNIEnv* pJniEnv,
                                           jthread thread,
                                           jobject object,
                                           jlong timeout)
{
    SJVMData data;
    SEmData mpiData;
    mpiData.pJniEnv = pJniEnv;
    mpiData.u.pJVMData = &data;
    mpiData.dataType = MPI_VM;
    data.dataType = DT_MONITOR_WAIT;
    memset(&data.u.monitorWait, 0, sizeof(data.u.monitorWait));

    data.u.monitorWait.threadLocalRef = thread;
    data.u.monitorWait.monitorLocalRef = object;
    data.u.monitorWait.timeout = timeout;

    MARTINI_INFORMATIVE3("CJVMTIInterface", 5, false,
        "[MonitorWaitHandler] pJniEnv = %p, thread = %p, monitor = %p",
        pJniEnv, thread, object);

    // Send the event
    gs_pEventManager->NotifyMpiEvent(EV_MONITOR_WAIT, &mpiData);
}

extern "C" void JNICALL MonitorWaitedHandler(jvmtiEnv *pJvmtiEnv,
                                             JNIEnv* pJniEnv,
                                             jthread thread,
                                             jobject object,
                                             jboolean bTimedOut)
{
    SJVMData data;
    SEmData mpiData;
    mpiData.pJniEnv = pJniEnv;
    mpiData.u.pJVMData = &data;
    mpiData.dataType = MPI_VM;
    data.dataType = DT_MONITOR_WAITED;
    memset(&data.u.monitorWaited, 0, sizeof(data.u.monitorWaited));

    data.u.monitorWaited.threadLocalRef = thread;
    data.u.monitorWaited.monitorLocalRef = object;
    data.u.monitorWaited.bTimedOut = bTimedOut ? true : false;

    MARTINI_INFORMATIVE3("CJVMTIInterface", 5, false,
        "[MonitorWaitedHandler] pJniEnv = %p, thread = %p, monitor = %p",
        pJniEnv, thread, object);

    // Send the event
    gs_pEventManager->NotifyMpiEvent(EV_MONITOR_WAITED, &mpiData);
}

extern "C" void JNICALL VMObjectAllocHandler(jvmtiEnv *pJvmtiEnv,
                                             JNIEnv* pJniEnv,
                                             jthread thread,
                                             jobject obj,
                                             jclass cls,
                                             jlong size)
{
    // NOTE: Most object allocations are captured using BCI and handled with a JNI call
    //       to the ObjectAllocHandler or ArrayAllocHandler functions in EventManager.cpp.
    //       This event is used for tracking object allocations which can't be tracked
    //       using BCI, such as allocations from native code, reflection and so on.

    static CDataManager *s_pDataManager = gs_pEventManager->GetDM();
    static CObjectInfoManager *s_pObjectInfoManager = s_pDataManager->GetObjectInfoManager();
    static IJVM * s_pJVMInterface = gs_pEventManager->GetJVMInterface();

    // Invoke the selectivity filter (if defined) to determine whether to report an
    // event for this object. This is not optimal, and can be improved by applying the
    // selectivity filter during instrumentation (not supported in this slice).
    bool bNotify = true;
    jlong classTag;
    TResult res = s_pJVMInterface->GetObjectTag(&classTag, cls);
    SJvmtiObjectTag *pClsTagInfo = s_pObjectInfoManager->GetTagInfo(classTag);
    const char *szClassName = NULL;
    Martini::RTUtil::MCString className;
    Martini::RTUtil::MCString classFriendlyName;
    if(pClsTagInfo != NULL && pClsTagInfo->jclassInfo.szClassName != NULL){
        szClassName = pClsTagInfo->jclassInfo.szClassName;
    }else{
        s_pJVMInterface->GetObjectClassName(className, obj);
        s_pDataManager->GetClassName(className.Get(), classFriendlyName);
        szClassName = classFriendlyName.Get();
    }
    
    // printf("VMObjectAllocHandler, object alloced %s\n", szClassName);

    if (!MRTE_FAILED(res))
    {
        IHeapFilter *pHeapFilter = (IHeapFilter*)gs_pEventManager->GetEventGroupFilter(EG_HEAP);
        if (pHeapFilter != NULL && pClsTagInfo->jclassInfo.szClassName != NULL)
        {
            SHeapFilterData heapFilterData;
            heapFilterData.szClassName = szClassName;
            bNotify = pHeapFilter->ShouldNotify(heapFilterData);
        }
    }

    if (bNotify)
    {
        // Tag the object
        jlong tag;
        TId objectId;
        TResult res = s_pObjectInfoManager->CreateTag(&objectId, &tag, obj,
            pJniEnv, false /* bSaveInDb */);
        if (MRTE_FAILED(res))
        {
            // Cannot tag the object. Do not report the event
            MARTINI_INFORMATIVE("CJVMTIInterface", 0, false,
                "[VMObjectAllocHandler] failed to allocate object id");
            return;
        }

        if(pClsTagInfo != NULL){
            SJvmtiObjectTag *pTagInfo = s_pObjectInfoManager->GetTagInfo(tag);
            SObjectInfo *pObjectInfo = new SObjectInfo();
            MARTINI_ASSERT("JVMTI ObjectReferenceCallback", pObjectInfo, "out of memory");
            memset(pObjectInfo, 0, sizeof(SObjectInfo));
            pObjectInfo->uiSize = size;
            pObjectInfo->classId = pClsTagInfo->jclassInfo.classId;
            pObjectInfo->arrayType = pClsTagInfo->jclassInfo.arrayType;
            pTagInfo->jobjectInfo.pObjectInfo = pObjectInfo;
            pTagInfo->jobjectInfo.szClassName = pClsTagInfo->jclassInfo.szClassName;
            gs_pEventManager->GenObjectAllocEvent(NULL, NULL, tag, -1, -1);
        }else{
            gs_pEventManager->GenObjectAllocEvent(pJniEnv, obj, tag, -1, -1);
        }
    }
}

extern "C" void JNICALL ObjectFreeHandler(jvmtiEnv *jvmti_env, jlong tag)
{
    static CObjectInfoManager *pObjectInfoManager =
        gs_pEventManager->GetDM()->GetObjectInfoManager();

    // Some objects are being tagged internally by JPI and should not be reported.
    // Make sure to report only the objects which pass the filter
    SJvmtiObjectTag *pTag = pObjectInfoManager->GetTagInfo(tag);
    if (pTag->bNotify)
    {
        JNIEnv *pJniEnv = (JNIEnv*)gs_pThis->m_ptlsJniEnv->GetValue();
        SJVMData data;
        SEmData mpiData;
        mpiData.u.pJVMData = &data;
        mpiData.dataType = MPI_VM;
        mpiData.pJniEnv = pJniEnv;
        data.dataType = DT_OBJECT_FREE;
        data.u.objectFree.tag = tag;

        // Disable all "MPI Unsafe" APIs in the context of the event-handler callback
        CJpiGlobals::Instance()->SetThreadMpiUnsafe();

        gs_pEventManager->NotifyMpiEvent(EV_OBJECT_FREE, &mpiData);

        // Enable all MPI APIs
        CJpiGlobals::Instance()->SetThreadMpiSafe();

    }
    // Free memory associated with the object

    // TODO: for memory intensive applications (e.g. SPECjbb2005), this causes
    //       huge overhead (observed on Windows, not tested on Linux).
    //       need to investigate and maybe implement a more efficient memory
    //       allocator for tags.
    pObjectInfoManager->FreeObject(tag);
}

extern "C" void JNICALL CompiledMethodLoadHandler(jvmtiEnv *pJvmtiEnv,
                              jmethodID method,
                              jint codeSize,
                              const void* pCodeAddr,
                              jint mapLength,
                              const jvmtiAddrLocationMap* map,
                              const void* compile_info)
{
    SJVMData data;
    SEmData mpiData;
    bool bIsLineInfoAvailable = false;
    jvmtiLineNumberEntry *pLineNumberTable = NULL;
    jint i;

    mpiData.u.pJVMData = &data;
    mpiData.dataType = MPI_VM;
    mpiData.pJniEnv = 0;
    data.dataType = DT_METHOD_LOAD;
    data.u.methodLoad.methodId = method;
#ifdef IPF_ARCH
	// BEA JRockit 1.5.0_04+ does not return the code starting address. To work around this,
	// if pCodeAddr is not aligned on a bundle boundary, round it to the nearest address and
	// update the codeSize attribute
	if ((UIOP)pCodeAddr & (UIOP)0xF)
	{
		unsigned int uiCodeOffset = 0x10 - ((UIOP)pCodeAddr % 0x10);
		data.u.methodLoad.uiCodeSize = codeSize - uiCodeOffset;
		data.u.methodLoad.pCodeAddr = (const void *)((UIOP)pCodeAddr + uiCodeOffset);
	}
	else
	{
		data.u.methodLoad.uiCodeSize = codeSize;
		data.u.methodLoad.pCodeAddr = pCodeAddr;
	}
#else
    data.u.methodLoad.uiCodeSize = codeSize;
    data.u.methodLoad.pCodeAddr = pCodeAddr;
#endif

    memset(&data.u.methodLoad.nativeToSrcLineMap, 0, sizeof(SLineNumberTable));
    memset(&data.u.methodLoad.nativeToManagedLineMap, 0, sizeof(SLineNumberTable));
    memset(&data.u.methodLoad.managedToSrcLineMap, 0, sizeof(SLineNumberTable));

    jvmtiJlocationFormat format;
    pJvmtiEnv->GetJLocationFormat(&format);
    if (format == JVMTI_JLOCATION_JVMBCI)
    {
        bIsLineInfoAvailable = true;

        // Populate the "managed-to-source-line" line map
        jint tableSize;
        jvmtiError err = pJvmtiEnv->GetLineNumberTable(method, &tableSize, &pLineNumberTable);
        if (err == JVMTI_ERROR_NONE)
        {
            data.u.methodLoad.managedToSrcLineMap.uiSize = tableSize;
            data.u.methodLoad.managedToSrcLineMap.uiActualSize = tableSize;
            data.u.methodLoad.managedToSrcLineMap.entries = new SLineNumberTableEntry[tableSize];
            if (NULL == data.u.methodLoad.managedToSrcLineMap.entries)
            {
                // Not enough memory for the table
                bIsLineInfoAvailable = false;
            }
            else
            {
                for (i = 0; i < tableSize; ++i)
                {
                    data.u.methodLoad.managedToSrcLineMap.entries[i].uiLineNumber =
                        pLineNumberTable[i].line_number;
                    data.u.methodLoad.managedToSrcLineMap.entries[i].uiOffset =
                        (unsigned int)pLineNumberTable[i].start_location;
                }
            }
        }
        else
        {
            bIsLineInfoAvailable = false;
        }


        // Populate the "native-to-managed" line map. This is done by using the
        // JVMTI jvmtiAddrLocationMap data structure which contains this information
        data.u.methodLoad.nativeToManagedLineMap.uiSize = mapLength;
        data.u.methodLoad.nativeToManagedLineMap.uiActualSize = mapLength;
        data.u.methodLoad.nativeToManagedLineMap.entries = new SLineNumberTableEntry[mapLength];
        if (NULL == data.u.methodLoad.nativeToManagedLineMap.entries)
        {
            // Not enough memory for the table
            bIsLineInfoAvailable = false;
        }
        else
        {
            SLineNumberTableEntry *pNative2BytecodeTable =
                data.u.methodLoad.nativeToManagedLineMap.entries;
            for (i = 0; i < mapLength; ++i)
            {
                pNative2BytecodeTable[i].uiLineNumber = (unsigned int)(map[i].location);
                // Sun returns offset instead of jit address. Work around until they fix
                if ((UIOP)map[i].start_address >= (UIOP)pCodeAddr)
                {
                    pNative2BytecodeTable[i].uiOffset =
                        (unsigned int)((UIOP)map[i].start_address - (UIOP)pCodeAddr);
                }
                else
                {

#if (defined(__linux__) && defined(__s390x__)) || ( (defined(_SOLARIS)|| defined(_SOLARISX86)) && defined(_LP64)) || (defined(_AIX) && defined(_LP64))
                	// problems when explicitly casting to unsigned int, so do this and have it
                	// implicitly casted
                    pNative2BytecodeTable[i].uiOffset = (unsigned long int)(map[i].start_address);
#else

	#if defined(MVS) && defined(_LP64)
        		    pNative2BytecodeTable[i].uiOffset = (unsigned int)((UIOP)map[i].start_address);
    #else
                    pNative2BytecodeTable[i].uiOffset = (unsigned int)(map[i].start_address);
	#endif

#endif
                }
            }
        }

    }

    if (bIsLineInfoAvailable == false)
    {
        // no line number info :(
        if (data.u.methodLoad.managedToSrcLineMap.entries)
        {
            delete [] data.u.methodLoad.managedToSrcLineMap.entries;
        }
        data.u.methodLoad.managedToSrcLineMap.entries = NULL;
        data.u.methodLoad.managedToSrcLineMap.uiSize = 0;
        data.u.methodLoad.managedToSrcLineMap.uiActualSize = 0;

        if (data.u.methodLoad.nativeToManagedLineMap.entries)
        {
            delete [] data.u.methodLoad.nativeToManagedLineMap.entries;
        }
        data.u.methodLoad.nativeToManagedLineMap.entries = NULL;
        data.u.methodLoad.nativeToManagedLineMap.uiSize = 0;
        data.u.methodLoad.nativeToManagedLineMap.uiActualSize = 0;
    }

//    char *pMethodName, *pMethodSignature;
//    pJvmtiEnv->GetMethodName(method, &pMethodName, &pMethodSignature, NULL);
    gs_pEventManager->NotifyMpiEvent(EV_JITTED_METHOD_LOADED, &mpiData);

    // now delete all allocated structs:
    if (pLineNumberTable)
    {
        pJvmtiEnv->Deallocate((unsigned char*)pLineNumberTable);
    }
    if (data.u.methodLoad.managedToSrcLineMap.entries)
    {
        delete [] data.u.methodLoad.managedToSrcLineMap.entries;
    }
    if (data.u.methodLoad.nativeToManagedLineMap.entries)
    {
        delete [] data.u.methodLoad.nativeToManagedLineMap.entries;
    }
}

extern "C"  void JNICALL CompiledMethodUnloadHandler(jvmtiEnv *pJvmtiEnv, jmethodID method,
                                                     const void* code_addr)
{
//    fprintf(stderr, "CompiledMethodUnLoadHandler\n");
    SJVMData data;
    SEmData mpiData;
    mpiData.u.pJVMData = &data;
    mpiData.dataType = MPI_VM;
    mpiData.pJniEnv = 0;
    data.dataType = DT_METHOD_UNLOAD;

    data.u.method.methodId = method;
    gs_pEventManager->NotifyMpiEvent(EV_JITTED_METHOD_UNLOADED, &mpiData);
}


extern "C" void JNICALL ThreadEndHandler(jvmtiEnv *pJvmtiEnv, JNIEnv* pJniEnv, jthread thread)
{
    SJVMData data;
    SEmData mpiData;
    mpiData.pJniEnv = pJniEnv;
    mpiData.u.pJVMData = &data;
    mpiData.dataType = MPI_VM;
    data.dataType = DT_THREAD_END;
    data.u.threadEnd.pThreadEnv = pJniEnv;
    data.u.threadEnd.threadLocalRef = thread;
    MARTINI_INFORMATIVE2("CJVMTIInterface", 5, false,
        "[ThreadEndHandler] pJniEnv = %p, thread = %p", pJniEnv, thread);
    gs_pEventManager->NotifyMpiEvent(EV_THREAD_END, &mpiData);
}

extern "C" void JNICALL ThreadStartHandler(jvmtiEnv *pJvmtiEnv, JNIEnv* pJniEnv, jthread thread)
{
    SJVMData data;
    SEmData mpiData;
    mpiData.pJniEnv = pJniEnv;
    mpiData.u.pJVMData = &data;
    mpiData.dataType = MPI_VM;
    data.dataType = DT_THREAD_START;
    memset(&data.u.threadStart, 0, sizeof(data.u.threadStart));
    data.u.threadStart.pThreadEnv = pJniEnv;

    data.u.threadStart.threadLocalRef = thread;
    MARTINI_INFORMATIVE2("CJVMTIInterface", 5, false,
        "[ThreadStartHandler] pJniEnv = %p, thread = %p",
        pJniEnv, thread);

    // Get thread information
    SThreadInfo threadInfo;
    TResult iRes = gs_pThis->GetThreadInfo(&threadInfo, thread);
    if (MRTE_SUCCEEDED(iRes))
    {
        data.u.threadStart.szName = threadInfo.szName;
        data.u.threadStart.szGroupName = threadInfo.szGroupName;
        data.u.threadStart.szParentGroupName = threadInfo.szParentGroupName;
    }

    // Send the event
    gs_pEventManager->NotifyMpiEvent(EV_THREAD_START, &mpiData);
}

extern "C" void JNICALL VMDeathHandler(jvmtiEnv *pJvmtiEnv, JNIEnv* pJniEnv)
{
    MARTINI_INFORMATIVE1("CJVMTIInterface", 5, false, "[VMDeathHandler] pJniEnv = %p",
        pJniEnv);
    SJVMData data;
    SEmData mpiData;
    mpiData.u.pJVMData = &data;
    mpiData.dataType = MPI_VM;
    mpiData.pJniEnv = pJniEnv;
    data.dataType = DT_VM;
    data.u.vmEvent.pJniEnv = pJniEnv;
    gs_pEventManager->NotifyMpiEvent(EV_VM_SHUTDOWN, &mpiData);
}


extern "C" void JNICALL ClassFileLoadHookHandler
    (jvmtiEnv *pjvmtiEnv,
     JNIEnv* pJniEnv,
     jclass classBeingRedefined,
     jobject loader,
     const char* szClassName,
     jobject protectionDomain,
     jint classDataLen,
     const unsigned char* pucClassData,
     jint* pNewClassDataLen,
     unsigned char** ppucNewClassData)
{
    if (szClassName)
    {
        MARTINI_INFORMATIVE2("CJVMTIInterface", 5, false,
            "[ClassFileLoadHookHandler] %s, class loader = %p", szClassName, loader);
    }
    SJVMData data;
    data.dataType = DT_CLASS_LOAD_HOOK;
    SEmData mpiData;
    mpiData.u.pJVMData = &data;
    mpiData.pJniEnv = pJniEnv;
    mpiData.dataType = MPI_VM;

    data.u.classLoadHook.szClassName = szClassName;
    data.u.classLoadHook.classDataLen = classDataLen;
    data.u.classLoadHook.pucClassData = pucClassData;
    data.u.classLoadHook.pNewClassDataLen = pNewClassDataLen;
    data.u.classLoadHook.ppucNewClassData = ppucNewClassData;
    data.u.classLoadHook.pfnAllocate = AllocateBuffer;
    data.u.classLoadHook.classBeingRedefined = classBeingRedefined;
    data.u.classLoadHook.loader = loader;
    gs_pEventManager->NotifyMpiEvent(EV_JAVA_CLASS_FILE_LOAD_HOOK, &mpiData);
    if (pucClassData == *ppucNewClassData)
    {
        //TODO: this may break chained agents waiting for ClassFileLoadHookEvent. Check this
        *ppucNewClassData = 0;
        *pNewClassDataLen = 0;
    }
}

extern "C" void JNICALL DynamicCodeGeneratedHandler(jvmtiEnv *pJvmtiEnv, const char* name,
                                                    const void* address, jint length)
{
    SJVMData data;
    data.dataType = DT_DYNAMIC_CODE;
    SEmData mpiData;
    mpiData.u.pJVMData = &data;
    mpiData.dataType = MPI_VM;
    mpiData.pJniEnv = 0;

    data.u.dynamicCode.address = address;
    data.u.dynamicCode.length = length;
    data.u.dynamicCode.name = name;

    gs_pEventManager->NotifyMpiEvent(EV_JAVA_DYNAMIC_CODE_GENERATED, &mpiData);
}

extern "C" void JNICALL ClassLoadHandler(jvmtiEnv *pJvmtiEnv,
                                         JNIEnv *pJniEnv,
                                         jthread thread,
                                         jclass klass)
{
    SJVMData data;
    data.dataType = DT_CLASS_LOAD;
    SEmData mpiData;
    mpiData.u.pJVMData = &data;
    mpiData.dataType = MPI_VM;
    mpiData.pJniEnv = pJniEnv;
    jvmtiError err;

    data.u.classLoad.jniClassLocalRef = klass;

    // get the class name
    char *pSignature = NULL;
    char *pClassGeneric = NULL;
    err = pJvmtiEnv->GetClassSignature(klass, &pSignature, &pClassGeneric);
    if (err != JVMTI_ERROR_NONE)
    {
        MARTINI_INFORMATIVE1("CJVMTIInterface", 0, false,
            "Failed to get class name for class %p", klass);
        data.u.classLoad.szClassName = NULL;
    }
    else
    {
        data.u.classLoad.szClassName = pSignature;
    }

    MARTINI_INFORMATIVE2("CJVMTIInterface", 5, false,
        "ClassLoad event received: class = '%s', generic = '%s'", pSignature, pClassGeneric);

    // dispatch the event
    gs_pEventManager->NotifyMpiEvent((TEventType)EM_EVENT_INTERNAL_CLASS_LOAD, &mpiData);

    // deallocate all the allocated buffers
    if (pSignature)
    {
        pJvmtiEnv->Deallocate((unsigned char*)pSignature);
    }
}

/**
 * The sole purpose of this function seems to be to create an SEmData for the event,
 * and then to pass it to along through a NotifyMpiEvent call.
 *
 * This function is registered as a listener in InitEventCallbacks(), and called
 * by the JVM.
 * */
extern "C" void JNICALL ClassPrepareHandler(jvmtiEnv *pJvmtiEnv, JNIEnv* pJniEnv,
                                            jthread thread, jclass klass)
{
    //fprintf(stderr, "ClassPrepareHandler [thread: %d]\n", GetCurrentThreadId());
    SJVMData data;
    data.dataType = DT_CLASS_PREPARE;
    SEmData mpiData;
    mpiData.u.pJVMData = &data;
    mpiData.dataType = MPI_VM;
    mpiData.pJniEnv = pJniEnv;
    jvmtiError err;

    data.u.classPrepare.jniClassLocalRef = klass;

    char *pSignature = NULL;
    char *pSourceFileName = NULL;
    char *pClassGeneric = NULL;
    err = pJvmtiEnv->GetClassSignature(klass, &pSignature, &pClassGeneric);

    if (err != JVMTI_ERROR_NONE)
    {
        MARTINI_INFORMATIVE1("CJVMTIInterface", 0, false,
            "Failed to get class name for klass==0x%08X", klass);
        data.u.classPrepare.szClassName = NULL;
        data.u.classPrepare.szGenericSig = NULL;
    }
    else
    {
        data.u.classPrepare.szClassName = pSignature;
        data.u.classPrepare.szGenericSig = pClassGeneric;
    }

	// Where either of these to values are NULL, they should not be passed to printf. (Many platforms, such as Solaris, will segfault on NULLs when %s is expected.)
	if(pSignature != NULL && pClassGeneric != NULL) {
		MARTINI_INFORMATIVE2("CJVMTIInterface", 5, false,
			"ClassPrepare event received: class = '%s', generic = '%s'", pSignature, pClassGeneric);
	}

    err = pJvmtiEnv->GetSourceFileName(klass, &pSourceFileName);
    if (err != JVMTI_ERROR_NONE)
    {
        data.u.classPrepare.szSourceName = NULL;
    }
    else
    {
        data.u.classPrepare.szSourceName = pSourceFileName;
    }

    jobject loader = NULL;
    err = pJvmtiEnv->GetClassLoader(klass, &loader);
    if (err != JVMTI_ERROR_NONE)
    {
        data.u.classPrepare.jniLoaderLocalRef = NULL;
    }
    else
    {
        data.u.classPrepare.jniLoaderLocalRef = loader;
    }

    jint numMethods = 0;
    jmethodID *pMethods = NULL;
    err = pJvmtiEnv->GetClassMethods(klass, &numMethods, &pMethods);
    if (err != JVMTI_ERROR_NONE)
    {
        MARTINI_INFORMATIVE1("CJVMTIInterface", 0, false,
            "Failed to get class methods for klass:0x%08X", klass);
        data.u.classPrepare.uiNumMethods = 0;
    }
    else
    {
        data.u.classPrepare.uiNumMethods = numMethods;
    }

    data.u.classPrepare.pMethodInfo = new SJVMMethodInfo[data.u.classPrepare.uiNumMethods];
    if (data.u.classPrepare.pMethodInfo == NULL)
    {
        MARTINI_INFORMATIVE("CJVMTIInterface", 0, false, "Failed to allocate memory");
        data.u.classPrepare.uiNumMethods = 0;
    }
    unsigned int i;
    char *pMethodName = NULL, *pMethodSig = NULL, *pMethodGeneric = NULL;
    jvmtiLineNumberEntry *pLineNumberTable = NULL;
    for (i = 0; i < data.u.classPrepare.uiNumMethods; ++i)
    {
        data.u.classPrepare.pMethodInfo[i].methodId = pMethods[i];
        err = pJvmtiEnv->GetMethodName(pMethods[i], &pMethodName, &pMethodSig, &pMethodGeneric);
        if (err != JVMTI_ERROR_NONE)
        {
            data.u.classPrepare.pMethodInfo[i].szMethodName = NULL;
            data.u.classPrepare.pMethodInfo[i].szMethodSignature = NULL;
            data.u.classPrepare.pMethodInfo[i].szMethodGeneric = NULL;
            MARTINI_INFORMATIVE1("CJVMTIInterface", 0, false,
                "Failed to get method name for method:%p", pMethods[i]);
        }
        else
        {
            data.u.classPrepare.pMethodInfo[i].szMethodName = pMethodName;
            data.u.classPrepare.pMethodInfo[i].szMethodSignature = pMethodSig;
            data.u.classPrepare.pMethodInfo[i].szMethodGeneric = pMethodGeneric;
        }

        jint accessFlags;
        err = pJvmtiEnv->GetMethodModifiers(pMethods[i], &accessFlags);
        if (err != JVMTI_ERROR_NONE)
        {
            data.u.classPrepare.pMethodInfo[i].uiAccessFlags = 0;
        }
        else
        {
            data.u.classPrepare.pMethodInfo[i].uiAccessFlags = (unsigned short)accessFlags;
        }

        jint tableSize = 0;
        err = pJvmtiEnv->GetLineNumberTable(pMethods[i], &tableSize, &pLineNumberTable);
        if (err != JVMTI_ERROR_NONE)
        {
            // information unavailable
            data.u.classPrepare.pMethodInfo[i].lineNumbers.uiStart = 0;
            data.u.classPrepare.pMethodInfo[i].lineNumbers.uiEnd = 0;
        }
        else
        {
            jint firstLine;
            jint lastLine;
            ExtractFirstAndLastLineNumbers(&firstLine, &lastLine, pLineNumberTable, tableSize);
            data.u.classPrepare.pMethodInfo[i].lineNumbers.uiStart = firstLine;
            data.u.classPrepare.pMethodInfo[i].lineNumbers.uiEnd = lastLine;

            // deallocate the line number table
            pJvmtiEnv->Deallocate((unsigned char*)pLineNumberTable);
        }
    }
    gs_pEventManager->NotifyMpiEvent(EM_EVENT_INTERNAL_CLASS_PREPARE, &mpiData);

    //now deallocate all the allocated buffers
    if (pSignature)
    {
        pJvmtiEnv->Deallocate((unsigned char*)pSignature);
    }
    if (pSourceFileName)
    {
        pJvmtiEnv->Deallocate((unsigned char*)pSourceFileName);
    }
    if (pClassGeneric)
    {
        pJvmtiEnv->Deallocate((unsigned char*)pClassGeneric);
    }
    if (pMethods)
    {
        pJvmtiEnv->Deallocate((unsigned char*)pMethods);
    }
    // deallocate methods' name and signature
    for (i = 0; i < data.u.classPrepare.uiNumMethods; ++i)
    {
        if (data.u.classPrepare.pMethodInfo[i].szMethodName)
        {
            pJvmtiEnv->Deallocate((unsigned char*)data.u.classPrepare.pMethodInfo[i].szMethodName);
        }
        if (data.u.classPrepare.pMethodInfo[i].szMethodSignature)
        {
            pJvmtiEnv->Deallocate((unsigned char*)data.u.classPrepare.pMethodInfo[i].szMethodSignature);
        }
        if (data.u.classPrepare.pMethodInfo[i].szMethodGeneric)
        {
            pJvmtiEnv->Deallocate((unsigned char*)data.u.classPrepare.pMethodInfo[i].szMethodGeneric);
        }
    }

    if (data.u.classPrepare.pMethodInfo)
    {
        delete [] data.u.classPrepare.pMethodInfo;
    }

    if (data.u.classPrepare.jniLoaderLocalRef)
    {
        pJniEnv->DeleteLocalRef(data.u.classPrepare.jniLoaderLocalRef);
    }
}

extern "C" void JNICALL VMInitHandler(jvmtiEnv *pJvmtiEnv, JNIEnv* pJniEnv, jthread thread)
{
    MARTINI_INFORMATIVE1("CJVMTIInterface", 5, false, "[VMInitHandler] pJniEnv = %p",
        pJniEnv);
    jint numLoadedClasses = 0;
    jclass *pClasses;

    // Simulate Class Load and Class Prepare events for all classes that were loaded prior to
    // the VM Init event. This is required for Sun 1.5.0 VMs which do not adhere to the JVMTI
    // standard and do not generate these events for classes loaded before VM_INIT.
    // Event handlers should be aware that this patch may cause the Class Load and
    // Class Prepare events to be sent multiple times for the same class.
    jvmtiError err = pJvmtiEnv->GetLoadedClasses(&numLoadedClasses, &pClasses);
    if (err == JVMTI_ERROR_NONE)
    {
        int i;
        for (i = 0; i < numLoadedClasses; ++i)
        {
            jint status = 0;
            jclass klass = pClasses[i];
            pJvmtiEnv->GetClassStatus(klass, &status);
            if (status & JVMTI_CLASS_STATUS_PREPARED)
            {
                ClassPrepareHandler(pJvmtiEnv, pJniEnv, thread, pClasses[i]);
            }
            pJniEnv->DeleteLocalRef(klass);
        }
        pJvmtiEnv->Deallocate((unsigned char *)pClasses);
    }

    // Generate Compiled Method Load events for all methods that were compiled (Jitted)
    // prior to the VM_INIT event
    pJvmtiEnv->GenerateEvents(JVMTI_EVENT_COMPILED_METHOD_LOAD);

    // Initialization completed. Generate VM_INIT event
    SJVMData data;
    SEmData mpiData;
    mpiData.u.pJVMData = &data;
    mpiData.dataType = MPI_VM;
    mpiData.pJniEnv = pJniEnv;
    data.dataType = DT_VM;
    data.u.vmEvent.pJniEnv = pJniEnv;

    gs_pEventManager->NotifyMpiEvent(EV_VM_INIT, &mpiData);
}


extern "C" void JNICALL GarbageCollectionFinishHandler(jvmtiEnv *pJvmtiEnv)
{
    SJVMData data;
    SEmData mpiData;
    mpiData.u.pJVMData = &data;
    mpiData.dataType = MPI_VM;

    // Disable all "MPI Unsafe" APIs in the context of the event-handler callback
    CJpiGlobals::Instance()->SetThreadMpiUnsafe();

    gs_pEventManager->NotifyMpiEvent(EV_GC_END, &mpiData);

    CObjectInfoManager *pObjectInfoManager = gs_pEventManager->GetDM()->GetObjectInfoManager();
	// If heap instance data is being collected, remove the objects from the map data store 
	// that have been garbage collected.
	if(pObjectInfoManager->isHeapObjDataCollectionEnabled()) {
		pObjectInfoManager->RemoveUnusedObjectsFromInstDataMap();
	}

    // Enable all MPI APIs
    CJpiGlobals::Instance()->SetThreadMpiSafe();
}

extern "C" void JNICALL GarbageCollectionStartHandler(jvmtiEnv *pJvmtiEnv)
{
    SJVMData data;
    SEmData mpiData;
    mpiData.u.pJVMData = &data;
    mpiData.dataType = MPI_VM;

    // Disable all "MPI Unsafe" APIs in the context of the event-handler callback
    CJpiGlobals::Instance()->SetThreadMpiUnsafe();

    gs_pEventManager->NotifyMpiEvent(EV_GC_START, &mpiData);

    // Enable all MPI APIs
    CJpiGlobals::Instance()->SetThreadMpiSafe();
}

jvmtiIterationControl
HeapRootCallbackImpl(jvmtiHeapRootKind rootKind,
                     jlong classTag,
                     jlong size,
                     jlong *pTag,
                     void *pData)
{
    MARTINI_ASSERT("JVMTI HeapRootCallback", pTag != NULL, "Internal JVMTI error. "
        "Object tag pointer is NULL");

    SObjectRefIterationControl *pIterData = (SObjectRefIterationControl*)pData;

    static CObjectInfoManager *s_pObjectInfoManager =
        gs_pEventManager->GetDM()->GetObjectInfoManager();

    if (pIterData->pObjRefInfo != NULL && pIterData->pObjRefInfo->uiSize <= pIterData->uiNextArrayElement)
    {
        // No more space in the allocated references array. The iteration will be continued
        // in order to know the required size
        pIterData->result = MRTE_ERROR_BUFFER_TOO_SHORT;
        pIterData->uiNextArrayElement++;
        return JVMTI_ITERATION_CONTINUE;
    }

    // Find the object id of the target object
    TId objectId = 0;

    if(pIterData->opsIterHeap == ITER_HEAP_BUFF_OBJ){
            if (*pTag != 0)
            {
                objectId = s_pObjectInfoManager->GetOrUpdateIdFromTag(pTag);
                if (0 == objectId)
                {
                    MARTINI_INFORMATIVE("JVMTI HeapRootCallback", 0, false,
                        "failed to allocate an id for a tagged object");
                    pIterData->result = MRTE_ERROR_FAIL;
                    return JVMTI_ITERATION_ABORT;
                }
            }
            else
            {
                // The current object is not tagged. Need to allocate a new id and tag it
                if (MRTE_FAILED(s_pObjectInfoManager->NewObjectAndTag(&objectId, pTag)))
                {
                    MARTINI_INFORMATIVE("JVMTI HeapRootCallback", 0, false,
                        "failed to allocate id and tag for a non-tagged object");
                    pIterData->result = MRTE_ERROR_FAIL;
                    return JVMTI_ITERATION_ABORT;
                }
            }

        SObjectReference *arrRefs = pIterData->pObjRefInfo->pReferences;
        arrRefs[pIterData->uiNextArrayElement].srcObjectId = 0; // Heap root
        arrRefs[pIterData->uiNextArrayElement].type = JvmtiRootKind2MpiRefType[rootKind];

        pIterData->pTagVector->Push(*pTag);
        arrRefs[pIterData->uiNextArrayElement].targetObjectId = objectId;

        pIterData->uiNextArrayElement++;
    }else if(pIterData->opsIterHeap == ITER_HEAP_GEN_ALLOC_EVENT){
        bool bNotify;
        SJvmtiObjectTag *pClsTagInfo = s_pObjectInfoManager->GetTagInfo(classTag);
        if(pClsTagInfo == NULL) return JVMTI_ITERATION_CONTINUE;
        IHeapFilter *pHeapFilter = (IHeapFilter*)gs_pEventManager->GetEventGroupFilter(EG_HEAP);
        if (pHeapFilter != NULL && pClsTagInfo->jclassInfo.szClassName != NULL){
            SHeapFilterData heapFilterData;
            heapFilterData.szClassName = pClsTagInfo->jclassInfo.szClassName;
            bNotify = pHeapFilter->ShouldNotify(heapFilterData);
        }

        if(*pTag == 0 && bNotify){
            if (MRTE_FAILED(s_pObjectInfoManager->NewObjectAndTag(&objectId, pTag)))
            {
                MARTINI_INFORMATIVE("JVMTI HeapRootCallback", 0, false,
                "failed to allocate id and tag for a non-tagged object");
                pIterData->result = MRTE_ERROR_FAIL;
                return JVMTI_ITERATION_ABORT;
            }

            if(pClsTagInfo != NULL){
                SJvmtiObjectTag *pTagInfo = s_pObjectInfoManager->GetTagInfo(*pTag);
                SObjectInfo *pObjectInfo = new SObjectInfo();
                MARTINI_ASSERT("JVMTI HeapRootCallback", pObjectInfo, "out of memory");
                memset(pObjectInfo, 0, sizeof(SObjectInfo));
                pObjectInfo->uiSize = size;
                pObjectInfo->classId = pClsTagInfo->jclassInfo.classId;
                pObjectInfo->arrayType = pClsTagInfo->jclassInfo.arrayType;
                pTagInfo->jobjectInfo.pObjectInfo = pObjectInfo;
                pTagInfo->jobjectInfo.szClassName = pClsTagInfo->jclassInfo.szClassName;
            }
            gs_pEventManager->GenObjectAllocEvent(NULL, NULL, *pTag, -1, -1);
        }
    }
    return JVMTI_ITERATION_CONTINUE;
}

extern "C" jvmtiIterationControl JNICALL
HeapRootCallback(jvmtiHeapRootKind rootKind,
                 jlong classTag,
                 jlong size,
                 jlong *pTag,
                 void *pData)
{
    return HeapRootCallbackImpl(rootKind, classTag, size, pTag, pData);
}

extern "C" jvmtiIterationControl JNICALL
StackReferenceCallback(jvmtiHeapRootKind rootKind,
                       jlong classTag,
                       jlong size,
                       jlong *pTag,
                       jlong threadTag,
                       jint frameDepth,
                       jmethodID method,
                       jint slot,
                       void *pData)
{
    return HeapRootCallbackImpl(rootKind, classTag, size, pTag, pData);
}

extern "C" jvmtiIterationControl JNICALL
ObjectReferenceCallback(jvmtiObjectReferenceKind referenceKind,
                        jlong classTag,
                        jlong targetSize,
                        jlong *pTargetTag,
                        jlong srcTag,
                        jint srcIndex,
                        void *pData)
{
    MARTINI_ASSERT("JVMTI ObjectReferenceCallback", pTargetTag != NULL,
        "Internal JVMTI error. Object tag pointer is NULL");

    static CObjectInfoManager *s_pObjectInfoManager =
        gs_pEventManager->GetDM()->GetObjectInfoManager();

    SObjectRefIterationControl *pIterData = (SObjectRefIterationControl*)pData;
    if (pIterData->pObjRefInfo != NULL && pIterData->pObjRefInfo->uiSize <= pIterData->uiNextArrayElement)
    {
        // No more space in the allocated references array. The iteration will be continued
        // in order to know the required size
        pIterData->result = MRTE_ERROR_BUFFER_TOO_SHORT;
        pIterData->uiNextArrayElement++;
        return JVMTI_ITERATION_CONTINUE;
    }


// Find the object id of the target object
    TId objectId = 0;

    if(pIterData->opsIterHeap == ITER_HEAP_BUFF_OBJ){

        if (*pTargetTag != 0)
        {
        objectId = s_pObjectInfoManager->GetOrUpdateIdFromTag(pTargetTag);
        if (0 == objectId)
        {
            MARTINI_INFORMATIVE("JVMTI ObjectReferenceCallback", 0, false,
                "failed to allocate an id for a tagged object");
            pIterData->result = MRTE_ERROR_FAIL;
            return JVMTI_ITERATION_ABORT;
        }
    }
    else
    {
        // The current object is not tagged. Need to allocate a new id and tag it
        if (MRTE_FAILED(s_pObjectInfoManager->NewObjectAndTag(&objectId, pTargetTag)))
        {
                MARTINI_INFORMATIVE("JVMTI ObjectReferenceCallback", 0, false,
                    "failed to allocate id and tag for a non-tagged object");
                pIterData->result = MRTE_ERROR_FAIL;
                return JVMTI_ITERATION_ABORT;
            }
        }

        SObjectReference *arrRefs = pIterData->pObjRefInfo->pReferences;
        arrRefs[pIterData->uiNextArrayElement].type = JvmtiRefKind2MpiRefType[referenceKind];

        arrRefs[pIterData->uiNextArrayElement].targetObjectId = objectId;
        pIterData->pTagVector->Push(*pTargetTag);

        // The source object must be already tagged by previous iterations
        SJvmtiObjectTag *pSrcTagInfo = s_pObjectInfoManager->GetTagInfo(srcTag);
        if (0 == pSrcTagInfo->jobjectInfo.objectId)
        {
            MARTINI_INFORMATIVE("JVMTI ObjectReferenceCallback", 0, false,
                "source object's tag does not contain an object id");
            pIterData->result = MRTE_ERROR_FAIL;
            return JVMTI_ITERATION_ABORT;
        }
        arrRefs[pIterData->uiNextArrayElement].srcObjectId = pSrcTagInfo->jobjectInfo.objectId;

        pIterData->uiNextArrayElement++;
    }
    else if(pIterData->opsIterHeap == ITER_HEAP_GEN_ALLOC_EVENT){
        bool bNotify;
        SJvmtiObjectTag *pClsTagInfo = s_pObjectInfoManager->GetTagInfo(classTag);
        if(pClsTagInfo == NULL) return JVMTI_ITERATION_CONTINUE;
        IHeapFilter *pHeapFilter = (IHeapFilter*)gs_pEventManager->GetEventGroupFilter(EG_HEAP);
        if (pHeapFilter != NULL && pClsTagInfo->jclassInfo.szClassName != NULL){
            SHeapFilterData heapFilterData;
            heapFilterData.szClassName = pClsTagInfo->jclassInfo.szClassName;
            bNotify = pHeapFilter->ShouldNotify(heapFilterData);
        }

        if(*pTargetTag == 0 && bNotify){
            if (MRTE_FAILED(s_pObjectInfoManager->NewObjectAndTag(&objectId, pTargetTag)))
            {
                MARTINI_INFORMATIVE("JVMTI ObjectReferenceCallback", 0, false,
                "failed to allocate id and tag for a non-tagged object");
                pIterData->result = MRTE_ERROR_FAIL;
                return JVMTI_ITERATION_ABORT;
            }

            if(pClsTagInfo != NULL){
                SJvmtiObjectTag *pTagInfo = s_pObjectInfoManager->GetTagInfo(*pTargetTag);
                SObjectInfo *pObjectInfo = new SObjectInfo();
                MARTINI_ASSERT("JVMTI ObjectReferenceCallback", pObjectInfo, "out of memory");
                memset(pObjectInfo, 0, sizeof(SObjectInfo));
                pObjectInfo->uiSize = targetSize;
                pObjectInfo->classId = pClsTagInfo->jclassInfo.classId;
                pObjectInfo->arrayType = pClsTagInfo->jclassInfo.arrayType;
                pTagInfo->jobjectInfo.pObjectInfo = pObjectInfo;
                pTagInfo->jobjectInfo.szClassName = pClsTagInfo->jclassInfo.szClassName;
            }

            gs_pEventManager->GenObjectAllocEvent(NULL, NULL, *pTargetTag, -1, -1);
        }
    }

    return JVMTI_ITERATION_CONTINUE;
}

//////////////////////////////////////////////////////////////////////////
// CJVMTIInterface implementation

CJVMTIInterface::CJVMTIInterface()
{
    MARTINI_INFORMATIVE("CJVMTIInterface", 0, false, "Using JVMTI");
    gs_pThis = this;
    m_csJvmti = NULL;
}

CJVMTIInterface::~CJVMTIInterface()
{
}

void CJVMTIInterface::InitEventTranslationArrays()
{
    // initialize the event translation arrays
    memset(m_vMpi2JVMTIEvents, 0, EM_EVENT_LAST * sizeof(jvmtiEvent));
    m_vMpi2JVMTIEvents[EV_JITTED_METHOD_LOADED] = JVMTI_EVENT_COMPILED_METHOD_LOAD;
    m_vMpi2JVMTIEvents[EV_JITTED_METHOD_UNLOADED] = JVMTI_EVENT_COMPILED_METHOD_UNLOAD;
    m_vMpi2JVMTIEvents[EV_THREAD_START] = JVMTI_EVENT_THREAD_START;
    m_vMpi2JVMTIEvents[EV_THREAD_END] = JVMTI_EVENT_THREAD_END;
    m_vMpi2JVMTIEvents[EV_GC_START] = JVMTI_EVENT_GARBAGE_COLLECTION_START;
    m_vMpi2JVMTIEvents[EV_GC_END] = JVMTI_EVENT_GARBAGE_COLLECTION_FINISH;
    m_vMpi2JVMTIEvents[EM_EVENT_INTERNAL_CLASS_PREPARE] = JVMTI_EVENT_CLASS_PREPARE;
    m_vMpi2JVMTIEvents[EV_JAVA_CLASS_FILE_LOAD_HOOK] = JVMTI_EVENT_CLASS_FILE_LOAD_HOOK;
    m_vMpi2JVMTIEvents[EM_EVENT_INTERNAL_CLASS_LOAD] = JVMTI_EVENT_CLASS_LOAD;
    m_vMpi2JVMTIEvents[EV_VM_INIT] = JVMTI_EVENT_VM_INIT;
    m_vMpi2JVMTIEvents[EV_VM_SHUTDOWN] = JVMTI_EVENT_VM_DEATH;
    m_vMpi2JVMTIEvents[EV_JAVA_DYNAMIC_CODE_GENERATED] = JVMTI_EVENT_DYNAMIC_CODE_GENERATED;
    m_vMpi2JVMTIEvents[EV_OBJECT_FREE] = JVMTI_EVENT_OBJECT_FREE;
    m_vMpi2JVMTIEvents[EM_EVENT_INTERNAL_VM_OBJECT_ALLOC] = JVMTI_EVENT_VM_OBJECT_ALLOC;
    m_vMpi2JVMTIEvents[EV_MONITOR_WAIT] = JVMTI_EVENT_MONITOR_WAIT;
    m_vMpi2JVMTIEvents[EV_MONITOR_WAITED] = JVMTI_EVENT_MONITOR_WAITED;
    m_vMpi2JVMTIEvents[EV_CONTENDED_MONITOR_ENTER] = JVMTI_EVENT_MONITOR_CONTENDED_ENTER;
    m_vMpi2JVMTIEvents[EV_CONTENDED_MONITOR_ENTERED] = JVMTI_EVENT_MONITOR_CONTENDED_ENTERED;

    memset(m_vJVMTI2MpiEvents, 0, (JVMTI_MAX_EVENT_TYPE_VAL + 1) * sizeof(SJVMTIEventData));
    //m_vJVMTI2MpiEvents[JVMTI_EVENT_COMPILED_METHOD_LOAD].mpiEvent = EV_JITTED_METHOD_LOADED;
    m_vJVMTI2MpiEvents[JVMTI_EVENT_COMPILED_METHOD_LOAD].capability.can_generate_compiled_method_load_events = 1;
    m_vJVMTI2MpiEvents[JVMTI_EVENT_COMPILED_METHOD_LOAD].bCapabilityIsSet = true;

    m_vJVMTI2MpiEvents[JVMTI_EVENT_COMPILED_METHOD_UNLOAD].mpiEvent = EV_JITTED_METHOD_UNLOADED;
    m_vJVMTI2MpiEvents[JVMTI_EVENT_COMPILED_METHOD_UNLOAD].capability.can_generate_compiled_method_load_events = 1;
    m_vJVMTI2MpiEvents[JVMTI_EVENT_COMPILED_METHOD_UNLOAD].bCapabilityIsSet = true;

    m_vJVMTI2MpiEvents[JVMTI_EVENT_THREAD_START].mpiEvent =  EV_THREAD_START;
    m_vJVMTI2MpiEvents[JVMTI_EVENT_THREAD_START].capability.can_tag_objects = 1;
    m_vJVMTI2MpiEvents[JVMTI_EVENT_THREAD_START].bCapabilityIsSet = true;

    m_vJVMTI2MpiEvents[JVMTI_EVENT_THREAD_END].mpiEvent =  EV_THREAD_END;
    m_vJVMTI2MpiEvents[JVMTI_EVENT_THREAD_END].capability.can_tag_objects = 1;
    m_vJVMTI2MpiEvents[JVMTI_EVENT_THREAD_END].bCapabilityIsSet = true;

    m_vJVMTI2MpiEvents[JVMTI_EVENT_GARBAGE_COLLECTION_START].mpiEvent = EV_GC_START;
    m_vJVMTI2MpiEvents[JVMTI_EVENT_GARBAGE_COLLECTION_START].capability.can_generate_garbage_collection_events = 1;
    m_vJVMTI2MpiEvents[JVMTI_EVENT_GARBAGE_COLLECTION_START].bCapabilityIsSet = true;

    m_vJVMTI2MpiEvents[JVMTI_EVENT_GARBAGE_COLLECTION_FINISH].mpiEvent = EV_GC_END;
    m_vJVMTI2MpiEvents[JVMTI_EVENT_GARBAGE_COLLECTION_FINISH].capability.can_generate_garbage_collection_events = 1;
    m_vJVMTI2MpiEvents[JVMTI_EVENT_GARBAGE_COLLECTION_FINISH].bCapabilityIsSet = true;

    m_vJVMTI2MpiEvents[JVMTI_EVENT_CLASS_PREPARE].mpiEvent = EM_EVENT_INTERNAL_CLASS_PREPARE;
    // this is needed in order to get all required info we need in Class Prepare event
    m_vJVMTI2MpiEvents[JVMTI_EVENT_CLASS_PREPARE].capability.can_get_line_numbers = true;
    m_vJVMTI2MpiEvents[JVMTI_EVENT_CLASS_PREPARE].capability.can_get_source_file_name = true;

    m_vJVMTI2MpiEvents[JVMTI_EVENT_CLASS_PREPARE].bCapabilityIsSet = true;

    m_vJVMTI2MpiEvents[JVMTI_EVENT_CLASS_FILE_LOAD_HOOK].mpiEvent = EV_JAVA_CLASS_FILE_LOAD_HOOK;
    m_vJVMTI2MpiEvents[JVMTI_EVENT_CLASS_FILE_LOAD_HOOK].capability.can_generate_all_class_hook_events = 1;
    m_vJVMTI2MpiEvents[JVMTI_EVENT_CLASS_FILE_LOAD_HOOK].bCapabilityIsSet = true;

    m_vJVMTI2MpiEvents[JVMTI_EVENT_CLASS_LOAD].mpiEvent = EM_EVENT_INTERNAL_CLASS_LOAD;
    m_vJVMTI2MpiEvents[JVMTI_EVENT_CLASS_LOAD].bCapabilityIsSet = false;

    m_vJVMTI2MpiEvents[JVMTI_EVENT_VM_INIT].mpiEvent = EV_VM_INIT;
    m_vJVMTI2MpiEvents[JVMTI_EVENT_VM_INIT].bCapabilityIsSet = false;

    m_vJVMTI2MpiEvents[JVMTI_EVENT_VM_DEATH].mpiEvent = EV_VM_SHUTDOWN;
    m_vJVMTI2MpiEvents[JVMTI_EVENT_VM_DEATH].bCapabilityIsSet = false;

    m_vJVMTI2MpiEvents[JVMTI_EVENT_DYNAMIC_CODE_GENERATED].mpiEvent = EV_JAVA_DYNAMIC_CODE_GENERATED;
    m_vJVMTI2MpiEvents[JVMTI_EVENT_DYNAMIC_CODE_GENERATED].bCapabilityIsSet = false;

    m_vJVMTI2MpiEvents[JVMTI_EVENT_OBJECT_FREE].mpiEvent = EV_OBJECT_FREE;
    m_vJVMTI2MpiEvents[JVMTI_EVENT_OBJECT_FREE].capability.can_tag_objects = 1;
    m_vJVMTI2MpiEvents[JVMTI_EVENT_OBJECT_FREE].capability.can_generate_object_free_events = 1;
    m_vJVMTI2MpiEvents[JVMTI_EVENT_OBJECT_FREE].bCapabilityIsSet = true;

    m_vJVMTI2MpiEvents[JVMTI_EVENT_VM_OBJECT_ALLOC].mpiEvent = EM_EVENT_INTERNAL_VM_OBJECT_ALLOC;
    m_vJVMTI2MpiEvents[JVMTI_EVENT_VM_OBJECT_ALLOC].capability.can_generate_vm_object_alloc_events = 1;
    m_vJVMTI2MpiEvents[JVMTI_EVENT_VM_OBJECT_ALLOC].bCapabilityIsSet = true;

    m_vJVMTI2MpiEvents[JVMTI_EVENT_MONITOR_WAIT].mpiEvent = EV_MONITOR_WAIT;
    m_vJVMTI2MpiEvents[JVMTI_EVENT_MONITOR_WAIT].capability.can_generate_monitor_events = 1;
    m_vJVMTI2MpiEvents[JVMTI_EVENT_MONITOR_WAIT].capability.can_get_monitor_info = 1;
    m_vJVMTI2MpiEvents[JVMTI_EVENT_MONITOR_WAIT].capability.can_tag_objects = 1;
    m_vJVMTI2MpiEvents[JVMTI_EVENT_MONITOR_WAIT].capability.can_get_current_contended_monitor = 1;
    m_vJVMTI2MpiEvents[JVMTI_EVENT_MONITOR_WAIT].bCapabilityIsSet = true;

    m_vJVMTI2MpiEvents[JVMTI_EVENT_MONITOR_WAITED].mpiEvent = EV_MONITOR_WAITED;
    m_vJVMTI2MpiEvents[JVMTI_EVENT_MONITOR_WAITED].capability.can_generate_monitor_events = 1;
    m_vJVMTI2MpiEvents[JVMTI_EVENT_MONITOR_WAITED].capability.can_get_monitor_info = 1;
    m_vJVMTI2MpiEvents[JVMTI_EVENT_MONITOR_WAITED].capability.can_tag_objects = 1;
    m_vJVMTI2MpiEvents[JVMTI_EVENT_MONITOR_WAITED].capability.can_get_current_contended_monitor = 1;
    m_vJVMTI2MpiEvents[JVMTI_EVENT_MONITOR_WAITED].bCapabilityIsSet = true;

    m_vJVMTI2MpiEvents[JVMTI_EVENT_MONITOR_CONTENDED_ENTER].mpiEvent = EV_CONTENDED_MONITOR_ENTER;
    m_vJVMTI2MpiEvents[JVMTI_EVENT_MONITOR_CONTENDED_ENTER].capability.can_generate_monitor_events = 1;
    m_vJVMTI2MpiEvents[JVMTI_EVENT_MONITOR_CONTENDED_ENTER].capability.can_get_monitor_info = 1;
    m_vJVMTI2MpiEvents[JVMTI_EVENT_MONITOR_CONTENDED_ENTER].capability.can_tag_objects = 1;
    m_vJVMTI2MpiEvents[JVMTI_EVENT_MONITOR_CONTENDED_ENTER].capability.can_get_current_contended_monitor = 1;
    m_vJVMTI2MpiEvents[JVMTI_EVENT_MONITOR_CONTENDED_ENTER].bCapabilityIsSet = true;

    m_vJVMTI2MpiEvents[JVMTI_EVENT_MONITOR_CONTENDED_ENTERED].mpiEvent = EV_CONTENDED_MONITOR_ENTERED;
    m_vJVMTI2MpiEvents[JVMTI_EVENT_MONITOR_CONTENDED_ENTERED].capability.can_generate_monitor_events = 1;
    m_vJVMTI2MpiEvents[JVMTI_EVENT_MONITOR_CONTENDED_ENTERED].capability.can_get_monitor_info = 1;
    m_vJVMTI2MpiEvents[JVMTI_EVENT_MONITOR_CONTENDED_ENTERED].capability.can_tag_objects = 1;
    m_vJVMTI2MpiEvents[JVMTI_EVENT_MONITOR_CONTENDED_ENTERED].capability.can_get_current_contended_monitor = 1;
    m_vJVMTI2MpiEvents[JVMTI_EVENT_MONITOR_CONTENDED_ENTERED].bCapabilityIsSet = true;
}

void CJVMTIInterface::InitEventsCallback()
{
    memset(&m_EventsCallbacks, 0, sizeof(jvmtiEventCallbacks));
    m_EventsCallbacks.ClassFileLoadHook = ClassFileLoadHookHandler;
    m_EventsCallbacks.ClassPrepare = ClassPrepareHandler;
    m_EventsCallbacks.ClassLoad = ClassLoadHandler;
    m_EventsCallbacks.CompiledMethodLoad = CompiledMethodLoadHandler;
    m_EventsCallbacks.CompiledMethodUnload = CompiledMethodUnloadHandler;
    m_EventsCallbacks.GarbageCollectionStart = GarbageCollectionStartHandler;
    m_EventsCallbacks.GarbageCollectionFinish = GarbageCollectionFinishHandler;
    m_EventsCallbacks.ThreadStart = ThreadStartHandler;
    m_EventsCallbacks.ThreadEnd = ThreadEndHandler;
    m_EventsCallbacks.VMInit = VMInitHandler;
    m_EventsCallbacks.VMDeath = VMDeathHandler;
    m_EventsCallbacks.DynamicCodeGenerated = DynamicCodeGeneratedHandler;
    m_EventsCallbacks.VMObjectAlloc = VMObjectAllocHandler;
    m_EventsCallbacks.ObjectFree = ObjectFreeHandler;
    m_EventsCallbacks.MonitorContendedEnter = MonitorContendedEnterHandler;
    m_EventsCallbacks.MonitorContendedEntered = MonitorContendedEnteredHandler;
    m_EventsCallbacks.MonitorWait = MonitorWaitHandler;
    m_EventsCallbacks.MonitorWaited = MonitorWaitedHandler;
}

TResult CJVMTIInterface::SetBootClassPath()
{
    // Set boot class path according to BISTRO_INSTALL_DIR or other variables
    char *szInstallDir = new char[MAX_PATH];
    if (!szInstallDir)
    {
        return MRTE_ERROR_OUT_OF_MEMORY;
    }
    szInstallDir[0] = 0;
    unsigned int uiActualSize = 0;
    TResult res = GetLibraryPathFromEnv(szInstallDir, MAX_PATH, &uiActualSize);
    if (MRTE_ERROR_BUFFER_TOO_SHORT == res)
    {
        // Resize szInstallDir and try again
        delete [] szInstallDir;
        szInstallDir = new char[uiActualSize + 1];
        if (!szInstallDir)
        {
            return MRTE_ERROR_OUT_OF_MEMORY;
        }
        res = GetLibraryPathFromEnv(szInstallDir, uiActualSize + 1, &uiActualSize);
    }
    if (MRTE_FAILED(res) || MRTE_RESULT_FALSE == res)
    {
        MARTINI_ERROR2("CJVMTIInterface", "%s or %s environment variables are not set",
            BISTRO_INSTALL_DIR_ENV_VAR, JAVA_PROFILER_HOME_ENV_VAR);
        delete [] szInstallDir;
        return MRTE_ERROR_FAIL;
    }
    jvmtiError jvmtiRes = m_pJVMTIInterface->AddToBootstrapClassLoaderSearch(szInstallDir);

    delete [] szInstallDir;
    if (jvmtiRes != JVMTI_ERROR_NONE)
    {
        return MRTE_ERROR_FAIL;
    }
    return MRTE_RESULT_OK;
}

void CJVMTIInterface::InformAvailableEvents()
{
    int event;
    for (event = 0; event < EM_EVENT_LAST; ++event)
    {
        if (m_vMpi2JVMTIEvents[event] != 0)
        {
            // this is a supported event
            m_pEventManager->SetEventImplementor((TEventType)event, MPI_VM);
        }
    }
}

TResult CJVMTIInterface::Initialize(TId moduleId, JavaVM *pJVM)
{
    m_pJVM = pJVM;

    if (!m_csJvmti)
    {
        m_csJvmti = CreateThreadSync();
        if (!m_csJvmti)
        {
            MARTINI_ERROR("CJVMTIInterface", "failed to initialize critical section");
            return MRTE_ERROR_FAIL;
        }
    }

    // Get JVMTI interface pointer
    m_pJVMTIInterface = NULL;
    jint ret = pJVM->GetEnv((void **)&m_pJVMTIInterface, JVMTI_VERSION);
    if (ret != JNI_OK)
    {
        MARTINI_ERROR("CJVMTIInterface", "failed to obtain JVMTI interface");
        return MRTE_ERROR_FAIL;
    }

    // Initialize the Capabilities Manager
    ICapabilityFactory *pCapFactory = new CTICapabilityFactory(m_pJVMTIInterface);
    if (NULL == pCapFactory)
    {
        MARTINI_ERROR("CJVMTIInterface", "failed to initialize the capability manager");
        return MRTE_ERROR_FAIL;
    }
    m_pCapManager->Initialize(pCapFactory);
    delete pCapFactory;

    // Enable VM capabilities required for enabling/disabling call-graph and heap events
    //TODO: optimize: delay enabling until features which require these capabilities are used
    if (m_pCapManager->Supported(CT_CAN_ENABLE_OR_DISABLE_BCI_GENERATED_EVENTS))
    {
        m_pCapManager->Enable(CT_CAN_ENABLE_OR_DISABLE_BCI_GENERATED_EVENTS);
    }
    else
    {
        // Enabling/disabling call-graph and heap events is not supported by this VM.
        // Log this information and notify the External Controller
        MARTINI_INFORMATIVE("CJVMTIInterface", 0, true, "VM does not support class "
            "redefinition. Call-graph and heap events cannot be fully disabled");
    }

    // Enable VM capabilities required for getting thread CPU times
    if (m_pCapManager->Supported(CT_CAN_GET_THREAD_TIMES))
    {
        m_pCapManager->Enable(CT_CAN_GET_THREAD_TIMES);
    }
    else
    {
        // Thread CPU time information is not supported by this VM.
        MARTINI_INFORMATIVE("CJVMTIInterface", 0, true, "VM does not support thread CPU times");
    }

    // Enable VM capabilities required for the suspending/resuming the VM
    if (m_pCapManager->Supported(CT_CAN_SUSPEND_RESUME_VM))
    {
        m_pCapManager->Enable(CT_CAN_SUSPEND_RESUME_VM);
    }
    else
    {
        // Suspending and resuming threads is not supported by this VM.
        MARTINI_INFORMATIVE("CJVMTIInterface", 0, true, "VM does not support suspending and "
            "resuming VM threads");
    }

    // Get a handle of the event manager
    m_pEventManager = gs_pEventManager = CEventManager::GetInstance();
    if (m_pEventManager == NULL)
    {
        MARTINI_ERROR("CJVMTIInterface", "failed to retreive Event Manager instance");
        return MRTE_ERROR_FAIL;
    }

    // Init internal data structures
    InitEventTranslationArrays();
    InitEventsCallback();
    InformAvailableEvents();

    // Set callback pointers
    jvmtiError jvmtiRes;
    jvmtiRes = m_pJVMTIInterface->SetEventCallbacks(&m_EventsCallbacks,
        sizeof(jvmtiEventCallbacks));
    if (jvmtiRes != JVMTI_ERROR_NONE)
    {
        MARTINI_ERROR("CJVMTIInterface", "failed to register JVMTI callbacks");
        return MRTE_ERROR_FAIL;
    }

    return MRTE_RESULT_OK;
}

void CJVMTIInterface::ProfilerExit()
{
}

TResult CJVMTIInterface::RegisterEvent(TEventType mpiEvent)
{
    jvmtiError jvmtiRes;
    jvmtiEvent event = m_vMpi2JVMTIEvents[mpiEvent];
    if (0 == event)
    {
        // Event is not supported
        return MRTE_ERROR_FAIL;
    }

    if (m_vJVMTI2MpiEvents[event].bCapabilityIsSet)
    {
        // Add the capabilities required by the event
        jvmtiCapabilities capability = m_vJVMTI2MpiEvents[event].capability;
        jvmtiRes = m_pJVMTIInterface->AddCapabilities(&capability);
        if (jvmtiRes != JVMTI_ERROR_NONE)
        {
            return MRTE_ERROR_FAIL;
        }
    }

    // If instrumentation is needed, add the location of the recorder classes to the
    // JVM boot class path
    if (mpiEvent == EV_JAVA_CLASS_FILE_LOAD_HOOK)
    {
        TResult mrteRes = SetBootClassPath();
        if (MRTE_FAILED(mrteRes))
        {
            return mrteRes;
        }
    }

    // enable the JVMTI event associated with the given MPI event
    jvmtiRes = m_pJVMTIInterface->SetEventNotificationMode(JVMTI_ENABLE, event, NULL);
    if (jvmtiRes != JVMTI_ERROR_NONE)
    {
        return MRTE_ERROR_FAIL;
    }

    return MRTE_RESULT_OK;
}

TResult CJVMTIInterface::DisableEvent(TEventType mpiEvent)
{
    jvmtiError jvmtiRes;
    jvmtiEvent jvmtiEvent = m_vMpi2JVMTIEvents[mpiEvent];
    jvmtiRes = m_pJVMTIInterface->SetEventNotificationMode(JVMTI_DISABLE, jvmtiEvent, NULL);
    if (jvmtiRes != JVMTI_ERROR_NONE)
    {
        return MRTE_ERROR_FAIL;
    }

    return MRTE_RESULT_OK;
}

TResult CJVMTIInterface::EnableEvent(TEventType mpiEvent)
{
    jvmtiError jvmtiRes;
    jvmtiEvent jvmtiEvent = m_vMpi2JVMTIEvents[mpiEvent];
    jvmtiRes = m_pJVMTIInterface->SetEventNotificationMode(JVMTI_ENABLE, jvmtiEvent, NULL);
    if (jvmtiRes != JVMTI_ERROR_NONE)
    {
        return MRTE_ERROR_FAIL;
    }

    return MRTE_RESULT_OK;
}


bool CJVMTIInterface::CanSupplyClassLoadHookToAllClasses()
{
    return true;
}


TResult CJVMTIInterface::GetJNIEnv(JNIEnv **ppJNIEnv)
{
    if (!ppJNIEnv)
    {
        return MRTE_ERROR_ILLEGAL_ARGUMENT;
    }

    JNIEnv *pJNIEnv = NULL;

    jint ret = m_pJVM->GetEnv((void**)&pJNIEnv, JNI_VERSION_1_4);

    if (ret != JNI_OK)
    {
        ret = m_pJVM->GetEnv((void**)&pJNIEnv, JNI_VERSION_1_2);
        if (ret != JNI_OK)
        {
            return MRTE_ERROR_FAIL;
        }
    }

    *ppJNIEnv = pJNIEnv;
    return MRTE_RESULT_OK;
}

EInterfaceType CJVMTIInterface::GetInterfaceType()
{
    return IT_JVMTI;
}

//
// Checks whether the current (calling) thread is attached to the VM
//
bool CJVMTIInterface::IsCurrentThreadAttached()
{
    JNIEnv *pNewJNIEnv;
    jint iVMErr = m_pJVM->GetEnv((void **)&pNewJNIEnv, JNI_VERSION_1_4);
    if (JNI_OK == iVMErr)
    {
        m_ptlsJniEnv->SetValue(pNewJNIEnv);
        return true;
    }
    return false;
}

TResult CJVMTIInterface::RedefineClasses(jint classCount,
                                         const jvmtiClassDefinition* classDefinitions)
{
    if (!IsCurrentThreadAttached())
    {
        return MRTE_ERROR_FAIL;
    }

    // redefine classes
    jvmtiError err = m_pJVMTIInterface->RedefineClasses(classCount, classDefinitions);

    if (JVMTI_ERROR_NONE == err)
    {
        return MRTE_RESULT_OK;
    }
    else
    {
        MARTINI_INFORMATIVE2("CJVMTIInterface", 0, false,
            "RedefineClasses: failed to redefine %d classes. JVMTI Error: %d\n",
            classCount, err);
        return MRTE_ERROR_FAIL;
    }
}

TMemoryAllocatorFunc CJVMTIInterface::GetMemoryAllocator()
{
    return AllocateBuffer;
}

jclass CJVMTIInterface::FindClass(const char *szClassName)
{
    if (!IsCurrentThreadAttached())
    {
        MARTINI_INFORMATIVE1("CJVMTIInterface", 0, false,
            "FindClass('%s'): current thread is not attached", szClassName);
        return NULL;
    }
    JNIEnv *pNewJNIEnv = (JNIEnv*)m_ptlsJniEnv->GetValue();
    jclass cls = pNewJNIEnv->FindClass(szClassName);

    return cls;
}

static char* CreateCopy(const char* szSource)
{
	if (NULL == szSource)
	{
        return NULL;
    }
    size_t len = strlen(szSource) + 1;
    char *szDest = new char[len];
    memset(szDest, 0, len);
    strcpy(szDest, szSource);
    return szDest;
}

//
// Retrieves thread information from JVMTI
//
// Parameters:
//      pThreadInfo [out]   : a placeholder for writing thread information.
//                            Internal strings are allocated from the heap using 'new'
//                            and should be deallocated by the caller when not needed
//      thread      [in]    : the thread
//
// Returns:
//      MRTE_RESULT_OK          : success
//      MRTE_ERROR_NULL_PTR     : pThreadInfo is null
//      MRTE_ERROR_PHASE_FAILURE: the VM has not been initialized yet
//      MRTE_ERROR_FAIL         : failure
//
TResult CJVMTIInterface::GetThreadInfo(SThreadInfo *pThreadInfo, jthread thread)
{
    if (NULL == pThreadInfo)
    {
        return MRTE_ERROR_NULL_PTR;
    }
    if (!IsCurrentThreadAttached())
    {
        return MRTE_ERROR_FAIL;
    }

    jvmtiError iErr;
    jvmtiPhase phase;
    m_csJvmti->Enter();
    iErr = m_pJVMTIInterface->GetPhase(&phase);
    m_csJvmti->Leave();
    CHECK_JVMTI_ERROR(iErr);
    if (JVMTI_PHASE_LIVE != phase)
    {
        MARTINI_INFORMATIVE1("CJVMTIInterface", 5, false,
            "[GetThreadInfo] called for thread %p when JVM not in LIVE phase", thread);
        return MRTE_ERROR_PHASE_FAILURE;
    }

    TResult iRes = IsValidObjectRef(thread);
    CHECK_TRESULT(iRes);

    jvmtiThreadInfo threadInfo;
    iErr = m_pJVMTIInterface->GetThreadInfo(thread, &threadInfo);
    CHECK_JVMTI_ERROR(iErr);
    pThreadInfo->szName = CreateCopy(threadInfo.name);
    m_pJVMTIInterface->Deallocate((unsigned char*)threadInfo.name);

    jvmtiThreadGroupInfo groupInfo;
    iErr = m_pJVMTIInterface->GetThreadGroupInfo(threadInfo.thread_group, &groupInfo);
    CHECK_JVMTI_ERROR(iErr);
    pThreadInfo->szGroupName = CreateCopy(groupInfo.name);
    m_pJVMTIInterface->Deallocate((unsigned char*)groupInfo.name);

    // Parent Group information may be unavailable if the thread belongs to the top-level
    // group. Therefore, do not return an error if this information is not available.
    if (NULL != groupInfo.parent)
    {
        jvmtiThreadGroupInfo parentGroupInfo;
        iErr = m_pJVMTIInterface->GetThreadGroupInfo(groupInfo.parent, &parentGroupInfo);
        if (JVMTI_ERROR_NONE == iErr)
        {
            pThreadInfo->szParentGroupName = CreateCopy(parentGroupInfo.name);
            m_pJVMTIInterface->Deallocate((unsigned char*)parentGroupInfo.name);
        }
        else
        {
            pThreadInfo->szParentGroupName = NULL;
        }
    }
    else
    {
        pThreadInfo->szParentGroupName = NULL;
    }
    return MRTE_RESULT_OK;

}

TResult CJVMTIInterface::IsValidObjectRef(jobject obj)
{
    JNIEnv *pJniEnv;
    TResult iRes = GetJNIEnv(&pJniEnv);
    CHECK_TRESULT(iRes);
    if (pJniEnv->IsSameObject(obj, NULL))
    {
        return MRTE_ERROR_FAIL;
    }
    return MRTE_RESULT_OK;
}

TResult CJVMTIInterface::GetCurrentThreadCpuTime(TTimeStamp *pCpu)
{
    if (NULL == pCpu)
    {
        return MRTE_ERROR_NULL_PTR;
    }
    if (!IsCurrentThreadAttached())
    {
        *pCpu = GetTimeStamp();
    }
    else
    {
        jlong threadCPUTime;
        jvmtiError iErr = m_pJVMTIInterface->GetCurrentThreadCpuTime(&threadCPUTime);
        if (JVMTI_ERROR_NONE == iErr)
        {
			// The GetCurrentThreadCpuTime argument is declared as a jlong (signed 64-bit value)
			// but returns an unsigned 64-bit value so this assignment is ok
            *pCpu = threadCPUTime;
        }
        else
        {
            *pCpu = GetTimeStamp();
        }
    }
    return MRTE_RESULT_OK;
}

TResult CJVMTIInterface::SuspendThread(jthread thread)
{

//	MARTINI_INFORMATIVE1("CJVMTIInterface", 5, false, "SuspendThread called on thread, ptr(%d)", thread);

    if (!IsCurrentThreadAttached())
    {
        return MRTE_ERROR_FAIL;
    }

    jvmtiError err = m_pJVMTIInterface->SuspendThread(thread);
    if (JVMTI_ERROR_NONE != err)
    {
        MARTINI_INFORMATIVE1("CJVMTIInterface", 0, false, "SuspendThread failed: JVMTI error %d", err);
        return MRTE_ERROR_FAIL;
    }
    return MRTE_RESULT_OK;
}

TResult CJVMTIInterface::ResumeThread(jthread thread)
{
	MARTINI_INFORMATIVE1("CJVMTIInterface", 5, false,
		"ResumeThread called on thread, ptr(%d)", thread);

    if (!IsCurrentThreadAttached())
    {
        return MRTE_ERROR_FAIL;
    }

    jvmtiError err = m_pJVMTIInterface->ResumeThread(thread);
    if (JVMTI_ERROR_NONE != err)
    {
        MARTINI_INFORMATIVE1("CJVMTIInterface", 0, false,
            "ResumeThread failed: JVMTI error %d", err);
        return MRTE_ERROR_FAIL;
    }
    return MRTE_RESULT_OK;
}

TResult CJVMTIInterface::SetClassTag(jclass cls)
{
    if (!IsCurrentThreadAttached())
    {
        return MRTE_ERROR_FAIL;
    }

    JNIEnv *pJNIEnv = (JNIEnv*)m_ptlsJniEnv->GetValue();
    static CDataManager *s_pDataManager = gs_pEventManager->GetDM();
    static CObjectInfoManager *s_pObjectInfoManager = s_pDataManager->GetObjectInfoManager();

    TId clsObjId;
    jlong clsTag;
    TResult res = s_pObjectInfoManager->GetOrCreateTag(&clsObjId, &clsTag, cls,
        pJNIEnv, false/* bSaveInDb */);
    if (MRTE_FAILED(res)){
        MARTINI_INFORMATIVE("CJVMTIInterface", 0, false,
            "[SetClassTag] GetOrCreateTag failed");
        return MRTE_ERROR_FAIL;
    }

    SJvmtiObjectTag *pTagInfo = s_pObjectInfoManager->GetTagInfo(clsTag);
    if(pTagInfo->jclassInfo.classId != 0){
        return MRTE_RESULT_OK;
    }

    /*Class info is empty. Fill tags with class info*/
    char *pSignature = NULL;
    char *pClassGeneric = NULL;
    TId classLoaderId = 0;
    jlong classLoaderTag = 0;
    jobject classLoader = NULL;

    jvmtiError err = m_pJVMTIInterface->GetClassLoader(cls, &classLoader);
    if(err != JVMTI_ERROR_NONE)
    {
        MARTINI_INFORMATIVE1("CJVMTIInterface", 3, false,
            "[SetClassTag]  GetClassLoader failed: JVMTI error", err);
    }

    if(classLoader != NULL){
        TResult res = s_pObjectInfoManager->GetOrCreateTag(&classLoaderId, &classLoaderTag,
                    classLoader, pJNIEnv, false /* bSaveInDb */);
        if (MRTE_FAILED(res))
        {
            MARTINI_INFORMATIVE("CJVMTIInterface", 0, false,
                "[SetClassTag] GetOrCreateTag failed for class loader. "
                "The bootstrap class loader is assumed");
        }
    }

    err = m_pJVMTIInterface->GetClassSignature(cls, &pSignature, &pClassGeneric);
    if (err != JVMTI_ERROR_NONE)
    {
        MARTINI_INFORMATIVE1("CJVMTIInterface", 0, false,
            "[SetClsTag] GetClassSignature failed: JVMTI error %d", err);
        return MRTE_ERROR_FAIL;
    }

    if (pSignature)
    {
        if ('[' == pSignature[0])
        {
            pTagInfo->jclassInfo.arrayType = AT_OTHER;
        }
    }
    pTagInfo->jclassInfo.classId = s_pDataManager->GetOrAllocateClassIdFromClassName(
        pSignature, classLoaderId);

    Martini::RTUtil::MCString classFriendlyName;
    Martini::RTUtil::MCString classNativeName;
    classNativeName.Set(pSignature);
    s_pDataManager->GetClassName(pSignature, classFriendlyName);
    pTagInfo->jclassInfo.szClassName = strdup(classFriendlyName.Get());

    SetObjectTag(cls,clsTag);

    // Clean up
    if (pSignature)
    {
        m_pJVMTIInterface->Deallocate((unsigned char *)pSignature);
    }
    if (pClassGeneric)
    {
        m_pJVMTIInterface->Deallocate((unsigned char *)pClassGeneric);
    }
    if (classLoader != NULL)
    {
        pJNIEnv->DeleteLocalRef(classLoader);
    }
    return MRTE_RESULT_OK;
}

TResult CJVMTIInterface::SetObjectTag(jobject obj, jlong tag)
{
    if (!IsCurrentThreadAttached())
    {
        return MRTE_ERROR_FAIL;
    }

    jvmtiError err = m_pJVMTIInterface->SetTag(obj, tag);
    if (JVMTI_ERROR_NONE != err)
    {
        MARTINI_INFORMATIVE1("CJVMTIInterface", 0, false,
            "SetObjectTag failed: JVMTI error %d", err);
        return MRTE_ERROR_FAIL;
    }
    return MRTE_RESULT_OK;
}

TResult CJVMTIInterface::GetObjectTag(jlong *pTag, jobject obj)
{
    if (!IsCurrentThreadAttached())
    {
        return MRTE_ERROR_FAIL;
    }

    jvmtiError err = m_pJVMTIInterface->GetTag(obj, pTag);
    if (JVMTI_ERROR_NONE != err)
    {
        *pTag = 0;
        MARTINI_INFORMATIVE2("CJVMTIInterface", 0, false,
            "GetObjectTag failed for object ref %p: JVMTI error %d", obj, err);
        return MRTE_ERROR_FAIL;
    }
    return MRTE_RESULT_OK;
}

TResult CJVMTIInterface::GetObjectInfoFromTags(TTagVector &inTagVector,
                                               TObjectInfoVector *pOutInfoVector,
                                               TTagVector *pOutTagVector)
{
    if (!IsCurrentThreadAttached())
    {
        return MRTE_ERROR_FAIL;
    }

    JNIEnv *pNewJNIEnv = (JNIEnv*)m_ptlsJniEnv->GetValue();
    if (!pNewJNIEnv)
    {
        MARTINI_INFORMATIVE("CJVMTIInterface", 0, false,
            "[GetObjectInfoFromTags] Failed to get JNI environment");
        return MRTE_ERROR_FAIL;
    }

    jint jniErr = pNewJNIEnv->PushLocalFrame(inTagVector.Size());
    if (jniErr != 0)
    {
        MARTINI_INFORMATIVE("CJVMTIInterface", 0, false,
            "[GetObjectInfoFromTags] Not enough memory (JNI local references) to complete "
            "the operation");
        return MRTE_ERROR_FAIL;
    }

    jint objCount = 0;
    jobject *objArray;
    jlong *tagArray;
    TResult res = MRTE_ERROR_FAIL;

    jvmtiError err = m_pJVMTIInterface->GetObjectsWithTags(inTagVector.Size(),
        inTagVector.GetVector(), &objCount, &objArray, &tagArray);
    if (JVMTI_ERROR_NONE == err)
    {
        jint i;
        for (i = 0; i < objCount; ++i)
        {
            SObjectInfo *pObjectInfo = new SObjectInfo();
            memset(pObjectInfo, 0, sizeof(SObjectInfo));
            if (MRTE_FAILED(GetObjectInfo(pObjectInfo, objArray[i])))
            {
                break;
            }
            pOutTagVector->Push(tagArray[i]);
            pOutInfoVector->Push(pObjectInfo);
        }
        if (i == objCount)
        {
            // Information for all objects successfully retrieved
            res = MRTE_RESULT_OK;
        }
        else
        {
            MARTINI_INFORMATIVE1("CJVMTIInterface", 0, false,
                "[GetObjectInfoFromTags] failed to get information for all objects", err);
        }
    }
    else
    {
        MARTINI_INFORMATIVE1("CJVMTIInterface", 0, false,
            "[GetObjectInfoFromTags] GetObjectsWithTags failed: JVMTI error %d", err);
    }


    // Clean up
    pNewJNIEnv->PopLocalFrame(NULL);

    if (objArray)
    {
        m_pJVMTIInterface->Deallocate((unsigned char *)objArray);
    }
    if (tagArray)
    {
        m_pJVMTIInterface->Deallocate((unsigned char *)tagArray);
    }

    return res;
}

TResult CJVMTIInterface::GetObjectInfo(SObjectInfo *pObjectInfo, jobject obj)
{
    static CDataManager *s_pDataManager = gs_pEventManager->GetDM();
    static CObjectInfoManager *s_pObjectInfoManager = s_pDataManager->GetObjectInfoManager();

    if (!IsCurrentThreadAttached())
    {
        return MRTE_ERROR_FAIL;
    }

    jlong objectSize;
    jvmtiError err = m_pJVMTIInterface->GetObjectSize(obj, &objectSize);
    if (JVMTI_ERROR_NONE != err)
    {
        MARTINI_INFORMATIVE1("CJVMTIInterface", 0, false,
            "[GetObjectInfo] GetObjectSize failed: JVMTI error %d", err);
        pObjectInfo->uiSize = 0;
    }
    else
    {
        pObjectInfo->uiSize = objectSize;
    }

    // Get object's class information
    JNIEnv *pNewJNIEnv = (JNIEnv*)m_ptlsJniEnv->GetValue();
    if (pNewJNIEnv)
    {
        jclass cls = pNewJNIEnv->GetObjectClass(obj);
        if (NULL == cls)
        {
            MARTINI_INFORMATIVE1("CJVMTIInterface", 0, false,
                "[GetObjectInfo] GetObjectClass failed", err);
            return MRTE_ERROR_FAIL;
        }

        TId objectId;
        jlong tag;
        TResult res = s_pObjectInfoManager->GetOrCreateTag(&objectId, &tag, cls,
            pNewJNIEnv, false /* bSaveInDb */);
        if (MRTE_FAILED(res))
        {
            MARTINI_INFORMATIVE("CJVMTIInterface", 0, false,
                "[GetObjectInfo] GetOrCreateTag failed");
            return MRTE_ERROR_FAIL;

        }
        SJvmtiObjectTag *pTagInfo = s_pObjectInfoManager->GetTagInfo(tag);
        if (tag != 0 && pTagInfo->jclassInfo.classId != 0)
        {
            // The class information is already cached in the class tag
            pObjectInfo->classId = pTagInfo->jclassInfo.classId;
            pObjectInfo->arrayType = pTagInfo->jclassInfo.arrayType;
            return MRTE_RESULT_OK;
        }

        // Find class information
        char *pSignature = NULL;
        char *pClassGeneric = NULL;
        TId classLoaderId = 0;
        jlong classLoaderTag = 0;
        jobject classLoader = NULL;
        err = m_pJVMTIInterface->GetClassLoader(cls, &classLoader);
        if (err != JVMTI_ERROR_NONE)
        {
            MARTINI_INFORMATIVE1("CJVMTIInterface", 3, false,
                "[GetObjectInfo] GetClassLoader failed: JVMTI error %d."
                "The bootstrap class loader is assumed", err);
        }
        else
        {
            if (classLoader != NULL)
            {
                res = s_pObjectInfoManager->GetOrCreateTag(&classLoaderId, &classLoaderTag,
                    classLoader, pNewJNIEnv, false /* bSaveInDb */);
                if (MRTE_FAILED(res))
                {
                    MARTINI_INFORMATIVE("CJVMTIInterface", 0, false,
                        "[GetObjectInfo] GetOrCreateTag failed for class loader. "
                        "The bootstrap class loader is assumed");
                    classLoaderId = NULL;
                }
            }
        }

        err = m_pJVMTIInterface->GetClassSignature(cls, &pSignature, &pClassGeneric);
        if (err != JVMTI_ERROR_NONE)
        {
            MARTINI_INFORMATIVE1("CJVMTIInterface", 0, false,
                "[GetObjectInfo] GetClassSignature failed: JVMTI error %d", err);
            pObjectInfo->classId = 0;
            pObjectInfo->arrayType = AT_NONE;
        }
        else
        {
            if (pSignature)
            {
                if ('[' == pSignature[0])
                {
                    pObjectInfo->arrayType = AT_OTHER;
                }
            }
            pObjectInfo->classId = s_pDataManager->GetOrAllocateClassIdFromClassName(
                pSignature, classLoaderId);

            // Cache the information in the class tag
            if (tag != 0)
            {
                pTagInfo->jclassInfo.classId = pObjectInfo->classId;
                pTagInfo->jclassInfo.arrayType = pObjectInfo->arrayType;
            }
        }

        // Clean up
        if (pSignature)
        {
            m_pJVMTIInterface->Deallocate((unsigned char *)pSignature);
        }
        if (pClassGeneric)
        {
            m_pJVMTIInterface->Deallocate((unsigned char *)pClassGeneric);
        }
        pNewJNIEnv->DeleteLocalRef(cls);
        if (classLoader != NULL)
        {
            pNewJNIEnv->DeleteLocalRef(classLoader);
        }

    }
    return MRTE_RESULT_OK;
}

TResult CJVMTIInterface::GetObjectClassName(MCString &className, jobject obj)
{
    if (!IsCurrentThreadAttached())
    {
        return MRTE_ERROR_FAIL;
    }

    JNIEnv *pJniEnv = (JNIEnv*)m_ptlsJniEnv->GetValue();
    if (!pJniEnv)
    {
        return MRTE_ERROR_FAIL;
    }

    TResult res = MRTE_RESULT_OK;

    // Get object's class information
    jclass cls = pJniEnv->GetObjectClass(obj);

    char *pSignature = NULL;
    char *pClassGeneric = NULL;
    jvmtiError err = m_pJVMTIInterface->GetClassSignature(cls, &pSignature, &pClassGeneric);

    if (err != JVMTI_ERROR_NONE)
    {
        MARTINI_INFORMATIVE1("CJVMTIInterface", 0, false,
            "[GetObjectClassNameInfo] GetClassSignature failed: JVMTI error %d", err);
        res = MRTE_ERROR_FAIL;
    }
    else
    {
        className.Set(pSignature);
    }

    // Clean up
    if (pSignature)
    {
        m_pJVMTIInterface->Deallocate((unsigned char *)pSignature);
    }
    if (pClassGeneric)
    {
        m_pJVMTIInterface->Deallocate((unsigned char *)pClassGeneric);
    }
    if (cls)
    {
        pJniEnv->DeleteLocalRef(cls);
    }

    return res;

}

TResult CJVMTIInterface::GetObjectReferences(SObjectReferenceInfo *pData,
                                             TTagVector *pTagVector)
{
    pData->uiActualSize = 0;

    jvmtiPhase phase;
    jvmtiError err = m_pJVMTIInterface->GetPhase(&phase);
    if (err != JVMTI_ERROR_NONE)
    {
        MARTINI_ERROR1("CJVMTIInterface", "[GetObjectReferences] GetPhase failed: "
            "JVMTI error %d", err);
        return MRTE_ERROR_FAIL;
    }
    if (phase != JVMTI_PHASE_LIVE)
    {
        return MRTE_ERROR_PHASE_FAILURE;
    }

    SObjectRefIterationControl iterationData;
    iterationData.pObjRefInfo = pData;
    iterationData.pTagVector = pTagVector;
    iterationData.opsIterHeap = ITER_HEAP_BUFF_OBJ;
    pData->uiActualSize = 0;

    MARTINI_INFORMATIVE("CJVMTIInterface", 0, false, "[GetObjectReferences] Heap walk started");

    err = m_pJVMTIInterface->IterateOverReachableObjects(
        HeapRootCallback, StackReferenceCallback, ObjectReferenceCallback,
        &iterationData);

    if (err != JVMTI_ERROR_NONE)
    {
        MARTINI_INFORMATIVE1("CJVMTIInterface", 0, false,
            "[GetObjectReferences] IterateOverReachableObjects failed: JVMTI error %d", err);
        return MRTE_ERROR_FAIL;
    }

    MARTINI_INFORMATIVE("CJVMTIInterface", 0, false, "[GetObjectReferences] Heap walk finished");
    pData->uiActualSize = iterationData.uiNextArrayElement;
    return iterationData.result;

}

TResult CJVMTIInterface::GetMonitorUsage(TId *pOwnerThreadId,
                                         TIdArray *pOwnWaiters,
                                         TIdArray *pNotifyWaiters,
                                         BitSet dataItems,
                                         jobject monitorJniRef)
{
    JNIEnv *pJniEnv;
    if (MRTE_FAILED(GetJNIEnv(&pJniEnv)))
    {
        MARTINI_INFORMATIVE("CJVMTIInterface", 0, false, "[GetMonitorUsage] failed to get "
            "JNI environment");
        return MRTE_ERROR_FAIL;
    }

    jvmtiPhase phase;
    jvmtiError err = m_pJVMTIInterface->GetPhase(&phase);
    if (err != JVMTI_ERROR_NONE)
    {
        MARTINI_INFORMATIVE1("CJVMTIInterface", 0, false, "[GetMonitorUsage] GetPhase failed: "
            "JVMTI error %d", err);
        return MRTE_ERROR_FAIL;
    }
    if (phase != JVMTI_PHASE_LIVE)
    {
        return MRTE_ERROR_PHASE_FAILURE;
    }

    jvmtiMonitorUsage monUsage;
    err = m_pJVMTIInterface->GetObjectMonitorUsage(monitorJniRef, &monUsage);
    if (err != JVMTI_ERROR_NONE &&
        err != JVMTI_ERROR_THREAD_NOT_ALIVE)   /* JRockit may return this code for unowned monitors.
                                                  We treat this as a valid return code */
    {
        MARTINI_INFORMATIVE2("CJVMTIInterface", 0, false,
            "[GetMonitorUsage] GetObjectMonitorUsage failed for monitor %p: "
            "JVMTI error %d", monitorJniRef, err);
        return MRTE_ERROR_FAIL;
    }

    if (DR_MONITOR_OWNER_THREAD_ID & dataItems)
    {
        if (JVMTI_ERROR_THREAD_NOT_ALIVE == err ||
            NULL == monUsage.owner)
        {
            // Monitor is not owned by any thread
            *pOwnerThreadId = 0;
        }
        else
        {
            *pOwnerThreadId = gs_pEventManager->GetDM()->GetThreadInfoManager()->GetThreadId(
                monUsage.owner);
            pJniEnv->DeleteLocalRef(monUsage.owner);
            if (0 == *pOwnerThreadId)
            {
                MARTINI_INFORMATIVE1("CJVMTIInterface", 0, false,
                    "[GetMonitorUsage] Failed to get the the monitor owner (%p) thread id",
                    monUsage.owner);
                return MRTE_ERROR_FAIL;
            }
        }
    }

    TResult res = MRTE_RESULT_OK;
    if ((DR_MONITOR_OWN_WAITERS & dataItems) && monUsage.waiter_count > 0)
    {
        res = ProcessThreadsArray(pOwnWaiters, monUsage.waiter_count, monUsage.waiters);
        if (MRTE_FAILED(res))
        {
            return res;
        }
        FreeJThreadArray(monUsage.waiter_count, monUsage.waiters);
    }

    if ((DR_MONITOR_NOTIFY_WAITERS & dataItems) && monUsage.notify_waiter_count > 0)
    {
        res = ProcessThreadsArray(pNotifyWaiters, monUsage.notify_waiter_count,
            monUsage.notify_waiters);
        FreeJThreadArray(monUsage.notify_waiter_count, monUsage.notify_waiters);
    }

    return res;
}

void CJVMTIInterface::FreeJThreadArray(jint count, jthread *elements)
{
    if (0 == count || NULL == elements)
    {
        return;
    }

    JNIEnv *pJniEnv;
    if (MRTE_FAILED(GetJNIEnv(&pJniEnv)))
    {
        return;
    }

    jint i;
    for (i = 0; i < count; ++i)
    {
        pJniEnv->DeleteLocalRef(elements[i]);
    }
    m_pJVMTIInterface->Deallocate((unsigned char *)elements);

}

//
// Converts an array of jthreads to an array of thread ids
//
TResult CJVMTIInterface::ProcessThreadsArray(TIdArray *pThreadIds,
                                             jint threadsCount,
                                             jthread *threads)
{
    // Validity checks
    if (NULL == pThreadIds)
    {
        return MRTE_ERROR_NULL_PTR;
    }
    if (NULL == pThreadIds->entries)
    {
        return MRTE_ERROR_NULL_PTR;
    }
    if ((unsigned int)threadsCount > pThreadIds->uiSize)
    {
        return MRTE_ERROR_BUFFER_TOO_SHORT;
    }

    CThreadInfoManager *pThreadInfoManager = gs_pEventManager->GetDM()->GetThreadInfoManager();
    jint i;
    unsigned int uiNextIdEntry = 0;
    for (i = 0; i < threadsCount; ++i)
    {
        TId threadId = pThreadInfoManager->GetThreadId(threads[i]);
        if (threadId != 0)
        {
            pThreadIds->entries[uiNextIdEntry] = threadId;
            uiNextIdEntry++;
        }
    }
    pThreadIds->uiActualSize = uiNextIdEntry;

    return MRTE_RESULT_OK;
}

//
// Returns the stack trace for the given thread
//
// Parameters
//      pFrameBuffer [out]      preallocated array for storing the trace information
//      pFrameCount [out]       number of frames written to pFrameBuffer
//      thread [in]             thread for which to return stack trace. NULL = current thread
//      startDepth [in]         depth from which to begin retrieving frames.
//                              0 = current/deepest frame
//      maxFrameCount[in]       the maximum number of frames to retrieve
//
// Returns
//      MRTE_RESUL_OK               success
//      MRTE_ERROR_NULL_PTR         any of the output variables in NULL
//      MRTE_ERROR_PHASE_FAILURE    wrong JVMTI phase. Function can be called only during
//                                  the live phase
//      MRTE_ERROR_FAIL             a JVMTI error occurred
//
//
TResult CJVMTIInterface::GetStackTrace(jvmtiFrameInfo *pFrameBuffer,
                                       jint *pFrameCount,
                                       jthread thread,
                                       jint startDepth,
                                       jint maxFrameCount)
{
    if (NULL == pFrameBuffer || NULL == pFrameCount)
    {
        return MRTE_ERROR_NULL_PTR;
    }

    jvmtiPhase phase;
    jvmtiError err = m_pJVMTIInterface->GetPhase(&phase);
    if (err != JVMTI_ERROR_NONE)
    {
        MARTINI_INFORMATIVE1("CJVMTIInterface", 0, false, "[GetStackTrace] GetPhase failed: "
            "JVMTI error %d", err);
        return MRTE_ERROR_FAIL;
    }
    if (phase != JVMTI_PHASE_LIVE)
    {
        return MRTE_ERROR_PHASE_FAILURE;
    }

    err = m_pJVMTIInterface->GetStackTrace(thread, startDepth, maxFrameCount, pFrameBuffer,
        pFrameCount);

    if (err != JVMTI_ERROR_NONE)
    {
        MARTINI_INFORMATIVE1("CJVMTIInterface", 0, false, "[GetStackTrace] GetStackTrace "
            "failed: JVMTI error %d", err);
        return MRTE_ERROR_FAIL;
    }

    return MRTE_RESULT_OK;
}

TResult CJVMTIInterface::GetMethodInfo(MCString &name,
                                       MCString &sig,
                                       MCString &generic,
                                       jint *pAccessFlags,
                                       SMethodLineNumbers *pLineNumbers,
                                       jmethodID jvmMethodId)
{
    // Get method name and signature
    char *szName;
    char *szSig;
    char *szGeneric;
    jvmtiError err = m_pJVMTIInterface->GetMethodName(jvmMethodId, &szName, &szSig,
        &szGeneric);
    if (err != JVMTI_ERROR_NONE)
    {
        MARTINI_INFORMATIVE2("CJVMTIInterface", 0, false,
            "[GetMethodInfo] failed to get method name for method:%p. JVMTI error %d",
            jvmMethodId, err);
        return MRTE_ERROR_FAIL;
    }
    else
    {
        name.Set(szName);
        sig.Set(szSig);
        generic.Set(szGeneric);
        if (szName)
        {
            m_pJVMTIInterface->Deallocate((unsigned char*)szName);
        }
        if (szSig)
        {
            m_pJVMTIInterface->Deallocate((unsigned char*)szSig);
        }
        if (szGeneric)
        {
            m_pJVMTIInterface->Deallocate((unsigned char*)szGeneric);
        }
    }

    // Get method modifiers
    err = m_pJVMTIInterface->GetMethodModifiers(jvmMethodId, pAccessFlags);
    if (err != JVMTI_ERROR_NONE)
    {
        MARTINI_INFORMATIVE2("CJVMTIInterface", 0, false,
            "[GetMethodInfo] failed to get modifiers for method:%p. JVMTI error %d",
            jvmMethodId, err);
        return MRTE_ERROR_FAIL;
    }

    // Get method start and end lines
    jint tableSize;
    jvmtiLineNumberEntry *pLineNumberTable;
    err = m_pJVMTIInterface->GetLineNumberTable(jvmMethodId, &tableSize, &pLineNumberTable);
    if (err != JVMTI_ERROR_NONE)
    {
        // Line number table information is not available. Not a critical error
        pLineNumbers->uiStart = 0;
        pLineNumbers->uiEnd = 0;
        if (err != JVMTI_ERROR_ABSENT_INFORMATION
            && err != JVMTI_ERROR_NATIVE_METHOD)
        {
            MARTINI_INFORMATIVE2("CJVMTIInterface", 5, false,
                "[GetMethodInfo] failed to get line number table for method:%p. "
                "JVMTI error %d", jvmMethodId, err);
        }
    }
    else
    {
            pLineNumbers->uiStart = pLineNumberTable[0].line_number;
            pLineNumbers->uiEnd = pLineNumberTable[tableSize-1].line_number;
            m_pJVMTIInterface->Deallocate((unsigned char*)pLineNumberTable);
    }
    return MRTE_RESULT_OK;
}

TResult CJVMTIInterface::GetMethodDeclaringClassName(MCString &className,
                                                     TId *pClassLoaderObjectId,
                                                     jmethodID jvmMethodId)
{
    if (NULL == pClassLoaderObjectId)
    {
        return MRTE_ERROR_NULL_PTR;
    }

    JNIEnv *pJniEnv;
    if (MRTE_FAILED(GetJNIEnv(&pJniEnv)))
    {
        MARTINI_INFORMATIVE("CJVMTIInterface", 0, false,
            "[GetMethodDeclaringClassName] failed to get JNI environment");
        return MRTE_ERROR_FAIL;
    }

    static CObjectInfoManager *s_pObjectInfoManager =
        gs_pEventManager->GetDM()->GetObjectInfoManager();

    jclass cls;
    jvmtiError err = m_pJVMTIInterface->GetMethodDeclaringClass(jvmMethodId, &cls);
    if (err != JVMTI_ERROR_NONE)
    {
        MARTINI_INFORMATIVE2("CJVMTIInterface", 0, false,
            "[GetMethodDeclaringClassName] failed to get declaring class for method %p. "
            "JVMTI error %d", jvmMethodId, err);
        return MRTE_ERROR_FAIL;
    }

    jobject classLoader = NULL;
    err = m_pJVMTIInterface->GetClassLoader(cls, &classLoader);
    if (err != JVMTI_ERROR_NONE)
    {
        MARTINI_INFORMATIVE2("CJVMTIInterface", 0, false,
            "[GetMethodDeclaringClassName] failed to get loader of declaring class "
            "for method %p. JVMTI error %d. Bootstrap loader is assumed",
            jvmMethodId, err);
        *pClassLoaderObjectId = 0;
    }
    else if (NULL == classLoader)
    {
        *pClassLoaderObjectId = 0;
    }
    else
    {
        TId classLoaderId = 0;
        jlong classLoaderTag = 0;
        TResult res = s_pObjectInfoManager->GetOrCreateTag(&classLoaderId, &classLoaderTag,
            classLoader, pJniEnv, false /* bSaveInDb */);
        if (MRTE_FAILED(res))
        {
            MARTINI_INFORMATIVE1("CJVMTIInterface", 3, false,
                "[GetMethodDeclaringClassName] failed to get object id of class loader "
                "for method %p. Bootstrap loader is assumed", jvmMethodId);
            *pClassLoaderObjectId = 0;
        }
        else
        {
            *pClassLoaderObjectId = classLoaderId;
        }
    }

    char *szClassSig = NULL;
    char *szClassGeneric = NULL;
    err = m_pJVMTIInterface->GetClassSignature(cls, &szClassSig, &szClassGeneric);
    if (err != JVMTI_ERROR_NONE)
    {
        MARTINI_INFORMATIVE2("CJVMTIInterface", 0, false,
            "[GetMethodDeclaringClassName] failed to get name for class %p. "
            "JVMTI error %d", cls, err);
        return MRTE_ERROR_FAIL;
    }

    className.Set(szClassSig);

    // Cleanup
    if (szClassSig)
    {
        m_pJVMTIInterface->Deallocate((unsigned char *)szClassSig);
    }
    if (szClassGeneric)
    {
        m_pJVMTIInterface->Deallocate((unsigned char *)szClassGeneric);
    }
    pJniEnv->DeleteLocalRef(cls);
    if (classLoader)
    {
        pJniEnv->DeleteLocalRef(classLoader);
    }

    return MRTE_RESULT_OK;

}

TResult CJVMTIInterface::GetLineNumberTable(SLineNumberTable *pLineNumberTable,
                                            jmethodID jvmMethodId)
{
    if (NULL == pLineNumberTable)
    {
        return MRTE_ERROR_NULL_PTR;
    }
    if (NULL == pLineNumberTable->entries)
    {
        return MRTE_ERROR_NULL_PTR;
    }
    if (0 == pLineNumberTable->uiSize)
    {
        pLineNumberTable->uiActualSize = 0;
        return MRTE_RESULT_OK;
    }

    jvmtiJlocationFormat format;
    m_pJVMTIInterface->GetJLocationFormat(&format);
    if (format != JVMTI_JLOCATION_JVMBCI)
    {
        // The VM does not support the expected line number table format
        pLineNumberTable->uiActualSize = 0;
        return MRTE_RESULT_OK;
    }

    TResult res = MRTE_RESULT_OK;

    // Populate the "managed-to-source-line" line map
    jint tableSize;
    jvmtiLineNumberEntry *jvmtiLineNumberTable = NULL;
    jvmtiError err = m_pJVMTIInterface->GetLineNumberTable(jvmMethodId, &tableSize,
        &jvmtiLineNumberTable);
    if (JVMTI_ERROR_NONE == err)
    {
        // Check the user allocated memory whether meet the requirement
        if (tableSize > pLineNumberTable->uiSize)
        {
            pLineNumberTable->uiActualSize = tableSize;
            if (jvmtiLineNumberTable) m_pJVMTIInterface->Deallocate((unsigned char *)jvmtiLineNumberTable);
            return MRTE_ERROR_BUFFER_TOO_SHORT;
        }

        pLineNumberTable->uiActualSize = tableSize;
        
        size_t entriesToProcess = tableSize < (jint)pLineNumberTable->uiSize ?
            tableSize : pLineNumberTable->uiSize;
        
        for (size_t i = 0; i < entriesToProcess; ++i)
        {
            pLineNumberTable->entries[i].uiLineNumber = jvmtiLineNumberTable[i].line_number;
            pLineNumberTable->entries[i].uiOffset =
                (unsigned int)jvmtiLineNumberTable[i].start_location;
        }
        if (jvmtiLineNumberTable)
        {
            m_pJVMTIInterface->Deallocate((unsigned char *)jvmtiLineNumberTable);
        }
    }
    else if (JVMTI_ERROR_NATIVE_METHOD == err ||
             JVMTI_ERROR_ABSENT_INFORMATION == err)
    {
        // Line number information is not available. This is not an error
        pLineNumberTable->uiActualSize = 0;
    }
    else
    {
        res = MRTE_ERROR_FAIL;
        MARTINI_INFORMATIVE2("CJVMTIInterface", 0, false,
            "[GetLineNumberTable] failed to get line number table for method %p. "
            "JVMTI error %d", jvmMethodId, err);
    }
    return res;
}

TResult CJVMTIInterface::GetThreadState(jint *pState, jthread thread)
{
    if (NULL == pState)
    {
        return MRTE_ERROR_NULL_PTR;
    }
    jvmtiError err = m_pJVMTIInterface->GetThreadState(thread, pState);
    if (err != JVMTI_ERROR_NONE)
    {
        MARTINI_INFORMATIVE2("CJVMTIInterface", 0, false,
            "[GetThreadState] failed to get state for thread %p. "
            "JVMTI error %d", thread, err);
        return MRTE_ERROR_FAIL;
    }
    return MRTE_RESULT_OK;
}

TResult CJVMTIInterface::GetCurrentContendedMonitor(jobject *pMonObj, jthread thread)
{
    if (NULL == pMonObj)
    {
        return MRTE_ERROR_NULL_PTR;
    }
    jvmtiError err = m_pJVMTIInterface->GetCurrentContendedMonitor(thread, pMonObj);
    if (err != JVMTI_ERROR_NONE)
    {
        MARTINI_INFORMATIVE2("CJVMTIInterface", 0, false,
            "[GetCurrentContendedMonitor] failed to get current monitor for thread %p. "
            "JVMTI error %d", thread, err);
        return MRTE_ERROR_FAIL;
    }
    return MRTE_RESULT_OK;
}

TResult CJVMTIInterface::GetAllStackTraces(jvmtiStackInfo **pStackInfoArray,
                                           jint *pThreadCount,
                                           jint maxFrameCount)
{
    if (NULL == pStackInfoArray || NULL == pThreadCount)
    {
        return MRTE_ERROR_NULL_PTR;
    }
    jvmtiError err = m_pJVMTIInterface->GetAllStackTraces(maxFrameCount, pStackInfoArray,
        pThreadCount);
    if (err != JVMTI_ERROR_NONE)
    {
        MARTINI_INFORMATIVE1("CJVMTIInterface", 0, false,
            "[GetAllStackTraces] failed to get stack traces. JVMTI error %d", err);
        return MRTE_ERROR_FAIL;
    }
    return MRTE_RESULT_OK;
}

EThreadState CJVMTIInterface::VmThreadStateToMpiThreadState(jint vmThreadState)
{
    EThreadState mpiState = TS_UNKNOWN;
    if (vmThreadState & JVMTI_THREAD_STATE_SLEEPING)
    {
        mpiState = TS_SLEEPING;
    }
    else
    {
        switch (vmThreadState & JVMTI_JAVA_LANG_THREAD_STATE_MASK)
        {
        case JVMTI_JAVA_LANG_THREAD_STATE_NEW:
            mpiState = TS_UNSTARTED;
            break;
        case JVMTI_JAVA_LANG_THREAD_STATE_TERMINATED:
            mpiState = TS_TERMINATED;
            break;
        case JVMTI_JAVA_LANG_THREAD_STATE_RUNNABLE:
            mpiState = TS_RUNNING;
            break;
        case JVMTI_JAVA_LANG_THREAD_STATE_BLOCKED:
            mpiState = TS_BLOCKED;
            break;
        case JVMTI_JAVA_LANG_THREAD_STATE_WAITING:
        case JVMTI_JAVA_LANG_THREAD_STATE_TIMED_WAITING:
            mpiState = TS_WAITING;
            break;
        default:
            mpiState = TS_UNKNOWN;
        }
    }
    return mpiState;
}

TResult CJVMTIInterface::DeallocateVmBuffer(unsigned char *pBuffer)
{
    jvmtiError err = m_pJVMTIInterface->Deallocate(pBuffer);
    return (JVMTI_ERROR_NONE == err) ? MRTE_RESULT_OK : MRTE_ERROR_FAIL;
}

TResult CJVMTIInterface::RunGC()
{
    jvmtiError err = m_pJVMTIInterface->ForceGarbageCollection();
    return (JVMTI_ERROR_NONE == err) ? MRTE_RESULT_OK : MRTE_ERROR_FAIL;
}

TResult CJVMTIInterface::IterateLiveHeap(IterHeapOps iterops)
{
    jvmtiPhase phase;
    jvmtiError err = m_pJVMTIInterface->GetPhase(&phase);
    if (err != JVMTI_ERROR_NONE)
    {
        MARTINI_ERROR1("CJVMTIInterface", "[IterateLiveHeap] GetPhase failed: "
            "JVMTI error %d", err);
        return MRTE_ERROR_FAIL;
    }
    if (phase != JVMTI_PHASE_LIVE)
    {
        return MRTE_ERROR_PHASE_FAILURE;
    }

    SObjectRefIterationControl iterationData;
    iterationData.opsIterHeap = iterops;

    MARTINI_INFORMATIVE("CJVMTIInterface", 0, false, "[IterateLiveHeap] Heap walk started");

    err = m_pJVMTIInterface->IterateOverReachableObjects(
        HeapRootCallback, StackReferenceCallback,
        ObjectReferenceCallback, &iterationData);
    if(err != JVMTI_ERROR_NONE)
    {
        MARTINI_INFORMATIVE1("CJVMTIInterface", 0, false,
            "[IterateLiveHeap] Iterating heap failed: JVMTI error %d", err);
        return MRTE_ERROR_FAIL;
    }

    MARTINI_INFORMATIVE("CJVMTIInterface", 0, false, "[IterateLiveHeap] Heap walk finished");

    return MRTE_RESULT_OK;
}

bool CJVMTIInterface::CanRedefineClass(jclass cls)
{
    bool isModifiable = false;
    // Array and primitive types cannot be redefined.

    // TODO: JVMTI 1.1 on Java 6: use IsModifiableClass instead
    jint status = 0;
    jvmtiError err = m_pJVMTIInterface->GetClassStatus(cls, &status);
    if (JVMTI_ERROR_NONE == err)
    {
        isModifiable = !((status & JVMTI_CLASS_STATUS_ERROR)
            | (status & JVMTI_CLASS_STATUS_ARRAY)
            | (status & JVMTI_CLASS_STATUS_PRIMITIVE));
    }
    else
    {
        MARTINI_INFORMATIVE1("CJVMTIInterface", 0, false,
            "[GetClassStatus] failed to get class status. JVMTI error %d", err);
    }

    return isModifiable;
}

TResult CJVMTIInterface::CreateRawMonitor(const char *szName, CRawMonitor **pNewMonitor)
{
    if (NULL == pNewMonitor)
    {
        return MRTE_ERROR_NULL_PTR;
    }

    jrawMonitorID jvmtiMonitor;
    jvmtiError err = m_pJVMTIInterface->CreateRawMonitor(szName, &jvmtiMonitor);
    if (err != JVMTI_ERROR_NONE)
    {
        MARTINI_INFORMATIVE1("CJVMTIInterface", 0, false,
            "[CreateRawMonitor] failed to create raw monitor. JVMTI error %d", err);
        return MRTE_ERROR_FAIL;
    }

    *pNewMonitor = new CJVMTIRawMonitor(m_pJVMTIInterface, szName, jvmtiMonitor);
    return MRTE_RESULT_OK;
}

// Added for bug 291400
JVMVendor CJVMTIInterface::GetJVMVendor()
{
    char* value_ptr = NULL;
    jvmtiError err = m_pJVMTIInterface->GetSystemProperty("java.vm.vendor", &value_ptr);
    if (JVMTI_ERROR_NONE != err)
    {
        MARTINI_INFORMATIVE1("CJVMTIInterface", 0, false,
            "[GetJVMVendor] failed to get", err);
    }

    JVMVendor jvmVendor = (NULL != strstr(value_ptr, "IBM")) ? IBM_JVM : OTHER_JVM;
    m_pJVMTIInterface->Deallocate(reinterpret_cast<unsigned char*>(value_ptr));
    return jvmVendor;
}

void JNICALL JvmtiAgentThreadStartFunction(jvmtiEnv *pJvmtiEnv,
                                           JNIEnv *pJniEnv,
                                           void *pArgs)
{
    TAgentThreadStartFunctionArgs *pFuncArgs = (TAgentThreadStartFunctionArgs*)pArgs;
    pFuncArgs->entryPoint(pJniEnv, pFuncArgs->pArgs);
}

TResult CJVMTIInterface::RunAgentThread(jobject thread,
                                        TAgentThreadStartFunction proc,
                                        void *arg)
{
    TAgentThreadStartFunctionArgs *pArgs = new TAgentThreadStartFunctionArgs(proc, arg);
    jvmtiError err = m_pJVMTIInterface->RunAgentThread(thread, JvmtiAgentThreadStartFunction,
        pArgs, JVMTI_THREAD_NORM_PRIORITY);
    if (err != JVMTI_ERROR_NONE)
    {
        MARTINI_INFORMATIVE1("CJVMTIInterface", 0, false,
            "[RunAgentThread] RunAgentThread failed. JVMTI error %d", err);
        return MRTE_ERROR_FAIL;
    }
    return MRTE_RESULT_OK;
}

//////////////////////////////////////////////////////////////////////////
// CTICapabilityFactory implementation

ICapability* CTICapabilityFactory::CreateCapability(ECapabilityType cap)
{
    ICapability *pCap = NULL;

    switch (cap)
    {
    case CT_CAN_ENABLE_OR_DISABLE_BCI_GENERATED_EVENTS:
        pCap = new CTIEnableDisableBciGeneratedEventsCapability(m_pJvmtiEnv);
    	break;
    case CT_CAN_GET_THREAD_TIMES:
        pCap = new CTIGetThreadTimesCapability(m_pJvmtiEnv);
        break;
    case CT_CAN_SUSPEND_RESUME_VM:
        pCap = new CTISuspendResumeVmCapability(m_pJvmtiEnv);
        break;
    case CT_CAN_GENERATE_OBJECT_ALLOC_EVENTS:
        pCap = new CTIGenerateObjectAllocEventsCapability(m_pJvmtiEnv);
        break;
    default:
        pCap = new CGenericUnsupportedCapability();
        break;
    }

    return pCap;
}

//////////////////////////////////////////////////////////////////////////
// CTIEnableDisableCGEventsCapability

TResult CTIEnableDisableBciGeneratedEventsCapability::Enable()
{
    jvmtiCapabilities cap;
    memset(&cap, 0, sizeof(jvmtiCapabilities));
    cap.can_redefine_any_class = 1;
    cap.can_redefine_classes = 1;
    cap.can_suspend = 1;
    jvmtiError err;
    err = m_pJvmtiEnv->AddCapabilities(&cap);
    if (JVMTI_ERROR_NONE != err)
    {
        // Try again without can_redefine_any_class (this capability is desired but not required)
        cap.can_redefine_any_class = 0;
        err = m_pJvmtiEnv->AddCapabilities(&cap);
        if (JVMTI_ERROR_NONE != err)
        {
            MARTINI_INFORMATIVE("CJVMTIInterface", 0, false,
                "[CTIEnableDisableBciGeneratedEventsCapability] Required JVMTI capability 'can_redefine_classes' is "
                "not supported by the JVM");
            return MRTE_ERROR_NOT_SUPPORTED;
        }
        else
        {
            MARTINI_INFORMATIVE("CJVMTIInterface", 0, false,
                "[CTIEnableDisableBciGeneratedEventsCapability] JVM does not support the JVMTI capability "
                "'can_redefine_any_class'. This may cause problems with attach/detach operations");
        }
    }
    m_bEnabled = true;
    return MRTE_RESULT_OK;
}

bool CTIEnableDisableBciGeneratedEventsCapability::Supported()
{
    jvmtiCapabilities cap;
    memset(&cap, 0, sizeof(jvmtiCapabilities));
    jvmtiError err;
    err = m_pJvmtiEnv->GetPotentialCapabilities(&cap);
    if (JVMTI_ERROR_NONE != err)
    {
        return false;
    }
    return (cap.can_redefine_classes
        && cap.can_suspend);
}

TResult CTIEnableDisableBciGeneratedEventsCapability::Disable()
{
    jvmtiCapabilities cap;
    memset(&cap, 0, sizeof(jvmtiCapabilities));
    cap.can_redefine_any_class = 1;     // Although the environment may not posses this capability,
                                        // relinquishing it is OK according to the JVMTI specification
    cap.can_redefine_classes = 1;
    cap.can_suspend = 1;
    jvmtiError err;
    err = m_pJvmtiEnv->RelinquishCapabilities(&cap);
    if (JVMTI_ERROR_NONE != err)
    {
        return MRTE_ERROR_NOT_SUPPORTED;
    }
    m_bEnabled = false;
    return MRTE_RESULT_OK;
}

//////////////////////////////////////////////////////////////////////////
// CTIGetThreadTimesCapability

TResult CTIGetThreadTimesCapability::Enable()
{
    jvmtiCapabilities cap;
    memset(&cap, 0, sizeof(jvmtiCapabilities));
    cap.can_get_current_thread_cpu_time = 1;
    jvmtiError err;
    err = m_pJvmtiEnv->AddCapabilities(&cap);
    if (JVMTI_ERROR_NONE != err)
    {
        return MRTE_ERROR_NOT_SUPPORTED;
    }
    return MRTE_RESULT_OK;
}

bool CTIGetThreadTimesCapability::Enabled()
{
    jvmtiCapabilities cap;
    memset(&cap, 0, sizeof(jvmtiCapabilities));
    jvmtiError err;
    err = m_pJvmtiEnv->GetCapabilities(&cap);
    if (JVMTI_ERROR_NONE != err)
    {
        return false;
    }
    return (cap.can_get_current_thread_cpu_time);
}

bool CTIGetThreadTimesCapability::Supported()
{
    jvmtiCapabilities cap;
    memset(&cap, 0, sizeof(jvmtiCapabilities));
    jvmtiError err;
    err = m_pJvmtiEnv->GetPotentialCapabilities(&cap);
    if (JVMTI_ERROR_NONE != err)
    {
        return false;
    }
    return (cap.can_get_current_thread_cpu_time);
}

TResult CTIGetThreadTimesCapability::Disable()
{
    jvmtiCapabilities cap;
    memset(&cap, 0, sizeof(jvmtiCapabilities));
    cap.can_get_current_thread_cpu_time = 1;
    jvmtiError err;
    err = m_pJvmtiEnv->RelinquishCapabilities(&cap);
    if (JVMTI_ERROR_NONE != err)
    {
        return MRTE_ERROR_NOT_SUPPORTED;
    }
    return MRTE_RESULT_OK;
}

//////////////////////////////////////////////////////////////////////////
// CTISuspendResumeVmCapability

TResult CTISuspendResumeVmCapability::Enable()
{
    jvmtiCapabilities cap;
    memset(&cap, 0, sizeof(jvmtiCapabilities));
    cap.can_suspend = 1;
    jvmtiError err;
    err = m_pJvmtiEnv->AddCapabilities(&cap);
    if (JVMTI_ERROR_NONE != err)
    {
        return MRTE_ERROR_NOT_SUPPORTED;
    }
    return MRTE_RESULT_OK;
}

bool CTISuspendResumeVmCapability::Enabled()
{
    jvmtiCapabilities cap;
    memset(&cap, 0, sizeof(jvmtiCapabilities));
    jvmtiError err;
    err = m_pJvmtiEnv->GetCapabilities(&cap);
    if (JVMTI_ERROR_NONE != err)
    {
        return false;
    }
    return (cap.can_suspend);
}

bool CTISuspendResumeVmCapability::Supported()
{
    jvmtiCapabilities cap;
    memset(&cap, 0, sizeof(jvmtiCapabilities));
    jvmtiError err;
    err = m_pJvmtiEnv->GetPotentialCapabilities(&cap);
    if (JVMTI_ERROR_NONE != err)
    {
        return false;
    }
    return (cap.can_suspend);
}

TResult CTISuspendResumeVmCapability::Disable()
{
    jvmtiCapabilities cap;
    memset(&cap, 0, sizeof(jvmtiCapabilities));
    cap.can_suspend = 1;
    jvmtiError err;
    err = m_pJvmtiEnv->RelinquishCapabilities(&cap);
    if (JVMTI_ERROR_NONE != err)
    {
        return MRTE_ERROR_NOT_SUPPORTED;
    }
    return MRTE_RESULT_OK;
}

//////////////////////////////////////////////////////////////////////////
// CTIGenerateObjectAllocEventsCapability

TResult CTIGenerateObjectAllocEventsCapability::Enable()
{
    jvmtiCapabilities cap;
    memset(&cap, 0, sizeof(jvmtiCapabilities));
    cap.can_tag_objects = 1;
    jvmtiError err;
    err = m_pJvmtiEnv->AddCapabilities(&cap);
    if (JVMTI_ERROR_NONE != err)
    {
        return MRTE_ERROR_NOT_SUPPORTED;
    }
    return MRTE_RESULT_OK;
}

bool CTIGenerateObjectAllocEventsCapability::Enabled()
{
    jvmtiCapabilities cap;
    memset(&cap, 0, sizeof(jvmtiCapabilities));
    jvmtiError err;
    err = m_pJvmtiEnv->GetCapabilities(&cap);
    if (JVMTI_ERROR_NONE != err)
    {
        return false;
    }
    return (cap.can_tag_objects);
}

bool CTIGenerateObjectAllocEventsCapability::Supported()
{
    jvmtiCapabilities cap;
    memset(&cap, 0, sizeof(jvmtiCapabilities));
    jvmtiError err;
    err = m_pJvmtiEnv->GetPotentialCapabilities(&cap);
    if (JVMTI_ERROR_NONE != err)
    {
        return false;
    }
    return (cap.can_tag_objects);
}

TResult CTIGenerateObjectAllocEventsCapability::Disable()
{
    jvmtiCapabilities cap;
    memset(&cap, 0, sizeof(jvmtiCapabilities));
    cap.can_tag_objects = 1;
    jvmtiError err;
    err = m_pJvmtiEnv->RelinquishCapabilities(&cap);
    if (JVMTI_ERROR_NONE != err)
    {
        return MRTE_ERROR_NOT_SUPPORTED;
    }
    return MRTE_RESULT_OK;
}

//////////////////////////////////////////////////////////////////////////
// CJVMTIRawMonitor implementation

TResult CJVMTIRawMonitor::Enter()
{
    TResult res = MRTE_RESULT_OK;
    jvmtiError err = m_pJvmtiEnv->RawMonitorEnter(m_jvmtiMonitor);
    if (err != JVMTI_ERROR_NONE)
    {
        res = MRTE_ERROR_FAIL;
        MARTINI_ERROR1("CJVMTIRawMonitor", "RawMonitorEnter failed. JVMTI Error: %d", err);
    }
    return res;
}

TResult
CJVMTIRawMonitor::Exit()
{
    TResult res = MRTE_RESULT_OK;
    jvmtiError err = m_pJvmtiEnv->RawMonitorExit(m_jvmtiMonitor);
    if (err != JVMTI_ERROR_NONE)
    {
        res = MRTE_ERROR_FAIL;
        MARTINI_ERROR1("CJVMTIRawMonitor", "RawMonitorExit failed. JVMTI Error: %d", err);
    }
    return res;
}

TResult
CJVMTIRawMonitor::Wait(jlong timeoutMillis)
{
    TResult res = MRTE_RESULT_OK;
    if (!m_bSignaled)
    {
        jvmtiError err;
        while ((err = m_pJvmtiEnv->RawMonitorWait(m_jvmtiMonitor, 0)) ==
            JVMTI_ERROR_INTERRUPT);
        if (err != JVMTI_ERROR_NONE)
        {
            MARTINI_ERROR1("CJVMTIRawMonitor",
                "RawMonitorWait failed. JVMTI Error: %d", err);
            res = MRTE_ERROR_FAIL;
        }
    }
    else
    {
        res = MRTE_RESULT_OK;
    }

    m_bSignaled = false;
    return res;
}

TResult
CJVMTIRawMonitor::Notify()
{
    m_bSignaled = true;
    TResult res = MRTE_RESULT_OK;
    jvmtiError err = m_pJvmtiEnv->RawMonitorNotify(m_jvmtiMonitor);
    if (err != JVMTI_ERROR_NONE)
    {
        res = MRTE_ERROR_FAIL;
        MARTINI_ERROR1("CJVMTIRawMonitor", "RawMonitorNotify failed. JVMTI Error: %d", err);
    }
    return res;
}

TResult
CJVMTIRawMonitor::NotifyAll()
{
    TResult res = MRTE_RESULT_OK;
    jvmtiError err = m_pJvmtiEnv->RawMonitorNotifyAll(m_jvmtiMonitor);
    if (err != JVMTI_ERROR_NONE)
    {
        res = MRTE_ERROR_FAIL;
        MARTINI_ERROR1("CJVMTIRawMonitor", "RawMonitorNotifyAll failed. JVMTI Error: %d", err);
    }
    return res;
}

CJVMTIRawMonitor::~CJVMTIRawMonitor()
{
    m_pJvmtiEnv->DestroyRawMonitor(m_jvmtiMonitor);
}
