/*****************************************************************************
 * Copyright (c) 1997, 2010 Intel Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *
 * $Id$
 *****************************************************************************/

#ifndef MRTE_JVMTI_INTERFACE_H
#define MRTE_JVMTI_INTERFACE_H

#include "IJVM.h"
#include "jvmti.h"
#include "MRTEInfrastructureDefinitions.h"
#include "EventManager.h"

#define CHECK_JVMTI_ERROR(x) if (JVMTI_ERROR_NONE != x) return MRTE_ERROR_FAIL;
#define CHECK_TRESULT(x) if (MRTE_FAILED(x)) return MRTE_ERROR_FAIL;

namespace Martini { namespace JPI
{
    struct SJVMTIEventData
    {
        MPI::TEventType mpiEvent;
        jvmtiCapabilities capability;
        bool bCapabilityIsSet;
    };
    
    //////////////////////////////////////////////////////////////////////////
    // Capability-related classes

    /**
     * @brief Factory for JVMTI capabilities
     */
    class CTICapabilityFactory : public ICapabilityFactory
    {
    public:
        CTICapabilityFactory(jvmtiEnv *pJvmtiEnv): m_pJvmtiEnv(pJvmtiEnv) {}
        virtual ICapability* CreateCapability(ECapabilityType cap);

    private:
        jvmtiEnv *m_pJvmtiEnv;
    };

    /**
     * @brief Implementation of the CT_CAN_ENABLE_OR_DISABLE_CALLGRAPH_EVENTS capability
     * for JVMTI
     */
    class CTIEnableDisableBciGeneratedEventsCapability : public ICapability
    {
    public:
        CTIEnableDisableBciGeneratedEventsCapability(jvmtiEnv *pJvmtiEnv) 
            : m_pJvmtiEnv(pJvmtiEnv), m_bEnabled(false) {}
        virtual TResult Enable();
        virtual bool Enabled() { return m_bEnabled; }
        virtual bool Supported();
        virtual TResult Disable();

    private:
        jvmtiEnv *m_pJvmtiEnv;
        bool m_bEnabled;
    };

    /**
     * @brief Implementation of the CT_CAN_GET_THREAD_TIMES capability
     * for JVMTI
     */
    class CTIGetThreadTimesCapability : public ICapability
    {
    public:
        CTIGetThreadTimesCapability(jvmtiEnv *pJvmtiEnv) : m_pJvmtiEnv(pJvmtiEnv) {}
        virtual TResult Enable();
        virtual bool Enabled();
        virtual bool Supported();
        virtual TResult Disable();

    private:
        jvmtiEnv *m_pJvmtiEnv;
    };

    /**
     * @brief Implementation of the CT_CAN_SUSPEND_RESUME_VM capability
     * for JVMTI
     */
    class CTISuspendResumeVmCapability : public ICapability
    {
    public:
        CTISuspendResumeVmCapability(jvmtiEnv *pJvmtiEnv) : m_pJvmtiEnv(pJvmtiEnv) {}
        virtual TResult Enable();
        virtual bool Enabled();
        virtual bool Supported();
        virtual TResult Disable();

    private:
        jvmtiEnv *m_pJvmtiEnv;
    };

    /**
     * @brief Implementation of the CT_CAN_GENERATE_OBJECT_ALLOC_EVENTS capability
     * for JVMTI
     */
    class CTIGenerateObjectAllocEventsCapability : public ICapability
    {
    public:
        CTIGenerateObjectAllocEventsCapability(jvmtiEnv *pJvmtiEnv) : m_pJvmtiEnv(pJvmtiEnv) {}
        virtual TResult Enable();
        virtual bool Enabled();
        virtual bool Supported();
        virtual TResult Disable();

    private:
        jvmtiEnv *m_pJvmtiEnv;
    };

    //////////////////////////////////////////////////////////////////////////
    // CJVMTIRawMonitor

    /**
     * @brief A JVMTI-based implementation of a Java Raw Monitor
     */
    class CJVMTIRawMonitor : public CRawMonitor
    {
    public:
        CJVMTIRawMonitor(jvmtiEnv *pJvmtiEnv,
                         const char *szName, 
                         jrawMonitorID jvmtiMonitor)
            : CRawMonitor(szName), m_jvmtiMonitor(jvmtiMonitor), m_bSignaled(false),
              m_pJvmtiEnv(pJvmtiEnv)
        {
        }
        virtual ~CJVMTIRawMonitor();

        virtual TResult Enter();
        virtual TResult Exit();
        virtual TResult Wait(jlong timeoutMillis);
        virtual TResult Notify();
        virtual TResult NotifyAll();

    private:
        jvmtiEnv * m_pJvmtiEnv;
        jrawMonitorID m_jvmtiMonitor;
        bool m_bSignaled;
    };

    //////////////////////////////////////////////////////////////////////////
    // CJVMTIInterface 

    #if ( (defined(__linux__) && defined(__s390__))  || defined(_AIX) || defined(MVS) ) && !defined(JVMTI_MAX_EVENT_TYPE_VAL)
		// This ENUM value is not defined in IBM JDK 1.5.0, and thus must be defined here.
		#define JVMTI_MAX_EVENT_TYPE_VAL 84
	#endif

    /**
     * @brief JVM Interface implementation for JVMTI (J2SE 5.0+)
     */
    class CJVMTIInterface : public IJVM  
    {
    public:
        // IVM Methods
        virtual TResult Initialize(MPI::TId moduleId, JavaVM *pJVM);
        virtual void ProfilerExit();
        virtual TResult RegisterEvent(MPI::TEventType mpiEvent);
        virtual TResult RequestClassLoadEvent(UIOP classId) {return MRTE_ERROR_NOT_SUPPORTED;};
        virtual bool CanSupplyClassLoadHookToAllClasses();
        virtual TResult DisableEvent(MPI::TEventType mpiEvent);
        virtual TResult EnableEvent(MPI::TEventType mpiEvent);
        virtual TResult GetJNIEnv(JNIEnv **ppJNIEnv);
        virtual EInterfaceType GetInterfaceType();
        virtual TResult RedefineClasses(jint classCount, 
                                        const jvmtiClassDefinition* classDefinitions);
        virtual TMemoryAllocatorFunc GetMemoryAllocator();
        virtual jclass FindClass(const char *szClassName);
        virtual TResult GetThreadInfo(MPI::SThreadInfo *pThreadInfo, jthread thread);
        virtual TResult GetCurrentThreadCpuTime(MPI::TTimeStamp *pCpu);
        virtual TResult SuspendThread(jthread thread);
        virtual TResult ResumeThread(jthread thread);
        virtual TResult SetClassTag(jclass cls);
        virtual TResult SetObjectTag(jobject obj, jlong tag);
        virtual TResult GetObjectTag(jlong *pTag, jobject obj);
        virtual TResult GetObjectInfo(MPI::SObjectInfo *pObjectInfo, jobject obj);
        virtual TResult GetObjectInfoFromTags(TTagVector &inTagVector, 
                                              TObjectInfoVector *pOutInfoVector,
                                              TTagVector *pOutTagVector);
        virtual TResult GetObjectClassName(RTUtil::MCString &className, jobject obj);
        virtual TResult GetObjectReferences(MPI::SObjectReferenceInfo *pData,
                                            TTagVector *pTagVector);
        virtual TResult GetMonitorUsage(MPI::TId *pOwnerThreadId,
                                        MPI::TIdArray *pOwnWaiters,
                                        MPI::TIdArray *pNotifyWaiters,
                                        MPI::BitSet dataItems, 
                                        jobject monitorJniRef);
        virtual TResult GetStackTrace(jvmtiFrameInfo *pFrameBuffer, 
                                      jint *pFrameCount, 
                                      jthread thread, 
                                      jint startDepth, 
                                      jint maxFrameCount);
        virtual TResult GetMethodInfo(RTUtil::MCString &name,
                                      RTUtil::MCString &sig,
                                      RTUtil::MCString &generic,
                                      jint *pAccessFlags,
                                      MPI::SMethodLineNumbers *pLineNumbers,
                                      jmethodID jvmMethodId);
        virtual TResult GetMethodDeclaringClassName(RTUtil::MCString &className,
                                                    MPI::TId *pClassLoaderObjectId,
                                                    jmethodID jvmMethodId);
        virtual TResult GetLineNumberTable(MPI::SLineNumberTable *pLineNumberTable,
                                           jmethodID jvmMethodId);
        virtual TResult GetThreadState(jint *pState, jthread thread);
        virtual TResult GetCurrentContendedMonitor(jobject *pMonObj, jthread thread);
        virtual TResult GetAllStackTraces(jvmtiStackInfo **pStackInfoArray, 
                                          jint *pThreadCount,
                                          jint maxFrameCount);
        virtual MPI::EThreadState VmThreadStateToMpiThreadState(jint vmThreadState);
        virtual TResult DeallocateVmBuffer(unsigned char *pBuffer);
        virtual TResult RunGC();
        virtual TResult IterateLiveHeap(IterHeapOps iterops);
        virtual bool CanRedefineClass(jclass cls);
        virtual TResult CreateRawMonitor(const char *szName, CRawMonitor **pNewMonitor);
        virtual TResult RunAgentThread(jobject thread, 
                                       TAgentThreadStartFunction proc, 
                                       void *arg);
        JVMVendor GetJVMVendor();
        // Public methods
        CJVMTIInterface();
        virtual ~CJVMTIInterface();
        jvmtiEnv* GetJVMTIInterface() { return m_pJVMTIInterface;};

    private:
        // Private member functions
        void InitEventTranslationArrays();
        void InitEventsCallback();
        TResult SetBootClassPath();
        void InformAvailableEvents();
        bool IsCurrentThreadAttached();
        TResult IsValidObjectRef(jobject obj);
        TResult ProcessThreadsArray(MPI::TIdArray *pThreadIds, 
                                    jint threadsCount, 
                                    jthread *threads);
        void FreeJThreadArray(jint count, jthread *elements);

        // private member variables

        // interface to JVMTI
        jvmtiEnv *m_pJVMTIInterface;

        jvmtiEventCallbacks m_EventsCallbacks;

        // an array holding the translation from JVMTI events to MPI events
        SJVMTIEventData m_vJVMTI2MpiEvents[JVMTI_MAX_EVENT_TYPE_VAL+1];

        // an array holding the translation from MPI events to JVMTI events
        jvmtiEvent m_vMpi2JVMTIEvents[Infrastructure::EM_EVENT_LAST];

        // pointer to event manager
        CEventManager *m_pEventManager;

        // critical section for synchronizing calls to JVMTI functions
        OSA::IThreadSync *m_csJvmti;
    };

}};
#endif // #ifndef MRTE_JVMTI_INTERFACE_H
