/*****************************************************************************
 * Copyright (c) 1997, 2010 Intel Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *
 * $Id$ 
 *****************************************************************************/
#ifdef EM64T_ARCH
#pragma runtime_checks( "", off )
#endif

#include "JavaInstrumentorManager.h"
#include "LibraryLoader.h"
#include "OSA.h"
#include "LogAssert.h"
#include <string.h>
#include "CGProxy.h"
#include "HeapProxy.h"
#include "ThreadProxy.h"
#include "MString.h"
#include "MPIUtils.h"
#include "JpiGlobals.h"

#include <stdlib.h>

#include <assert.h>

using namespace Martini;
using namespace Martini::JPI;
using namespace Martini::Infrastructure;
using namespace Martini::OSA;
using namespace Martini::RTUtil;

//////////////////////////////////////////////////////////////////////////
// Global variables

CClassRedefThread *g_pClassRedefThread = NULL;

//////////////////////////////////////////////////////////////////////
// Global functions

// A global function to allocate memory from the local heap (using malloc).
// Needed for compatibility between Windows and Linux (malloc on Linux has a different
// signature)
void* LocalMemAlloc(unsigned int uiSize)
{
    return malloc(uiSize);
}

// Class Redefinition thread procedure
static void JNICALL
ClassRedefThreadProc(JNIEnv *pJniEnv, void *pArgs)
{
    if (g_pClassRedefThread)
    {
        g_pClassRedefThread->Run(pJniEnv, pArgs);
    }
}

//////////////////////////////////////////////////////////////////////////
// CClassRedefThread

CClassRedefThread::CClassRedefThread()
: m_monClassesAvailableForRedef(NULL), m_monClassRedefDone(NULL), 
m_bNeedRedefDoneNotification(false), m_JVMVendor(UNDEFINED_JVM)
{
    IJVM *pJVM = CJpiGlobals::Instance()->pJvmInterface;

    TResult res = pJVM->CreateRawMonitor("Class Redefinition monitor", 
        &m_monClassesAvailableForRedef);
    if (MRTE_FAILED(res))
    {
        MARTINI_ERROR("CJavaInstrumentorManager", "failed to create raw monitor");
    }
    res = pJVM->CreateRawMonitor("Class Redefinition Done", &m_monClassRedefDone);
    if (MRTE_FAILED(res))
    {
        MARTINI_ERROR("CJavaInstrumentorManager", "failed to create raw monitor");
    }
}

CClassRedefThread::~CClassRedefThread()
{
    if (m_monClassRedefDone)
    {
        delete m_monClassRedefDone;
    }
    if (m_monClassesAvailableForRedef)
    {
        delete m_monClassesAvailableForRedef;
    }
}

TResult CClassRedefThread::Start()
{
    IJVM *pJVM = CJpiGlobals::Instance()->pJvmInterface;

    jthread th = pJVM->AllocateJavaThread();
    if (NULL == th)
    {
        MARTINI_ERROR("CJavaInstrumentorManager", 
            "failed to allocate the Class Redefinition thread");
    }
    TResult res = pJVM->RunAgentThread(th, ClassRedefThreadProc, NULL);
    if (MRTE_FAILED(res))
    {
        MARTINI_ERROR("CJavaInstrumentorManager", 
            "failed to execute the Class Redefinition thread");

    }
    return res;
}

void CClassRedefThread::GetJvmVendor()
{
    if (UNDEFINED_JVM == m_JVMVendor)
        m_JVMVendor = CJpiGlobals::Instance()->pJvmInterface->GetJVMVendor();
}

void CClassRedefThread::Run(JNIEnv *pJniEnv, void *pArgs)
{
    CJpiGlobals::Instance()->pDataManager->GetThreadInfoManager()->SetThreadType(
        pJniEnv, STlsThreadInfo::EXTERNAL);

    // Infinitely wait and process class redefinition requests. The thread is a daemon thread
    // and will die automatically when the JVM shuts down
    while (true)
    {
        m_monClassesAvailableForRedef->Enter();      
        m_monClassesAvailableForRedef->Wait(0);

        // Classes are available for redefinition. When redefining classes, we must not hold
        // any lock, as this may cause deadlocks. Therefore, we copy the list of classes
        // to a temporary, thread-local vector, release the lock on the global queue
        // g_vecClassesToRedef and only then redefine classes

        unsigned int classInd;
        TJvmtiClassDefVector tempClassDefs;
        for (classInd = 0; classInd < m_vecClassesToRedefine.Size(); classInd++)
        {
            tempClassDefs.Push(m_vecClassesToRedefine.GetAt(classInd));
        }

        m_vecClassesToRedefine.Clear();

        bool bNeedRedefDoneNotification = m_bNeedRedefDoneNotification;
        m_bNeedRedefDoneNotification = false;

        m_monClassesAvailableForRedef->Exit();

        TResult iRes = MRTE_RESULT_OK;
        if (tempClassDefs.Size() > 0)
        {
            unsigned int batchRedefineClassesNum = 1;

            GetJvmVendor();
            if (m_JVMVendor == IBM_JVM)
            {// Add for bug 291400
                batchRedefineClassesNum = tempClassDefs.Size();
            }
            
            // RedefineClasses one class at a time (Sun and Bea cannot redefine classes
            // in batches efficiently)
            for (classInd = 0; classInd < tempClassDefs.Size(); classInd+=batchRedefineClassesNum)
            {
                iRes = CJpiGlobals::Instance()->pJvmInterface ->RedefineClasses(
                    batchRedefineClassesNum, tempClassDefs.GetVector() + classInd);

                // RedefineClasses should never fail. If it does, then classes which were loaded
                // after the VM was resumed may have been instrumented with the new 'enabled' 
                // status and are now in inconsistent state. This is a fatal error.
                if (MRTE_FAILED(iRes))
                {
                    MARTINI_ERROR("CJavaInstrumentorManager",
                        "RedefineClasses failed. "
                        "Instrumented classes may be in inconsistent state");
                }
            }
            
            MARTINI_INFORMATIVE1("CJavaInstrumentorManager", 3, false, 
                "[ClassRedefThreadProc] %u classes successfully redefined\n",
                tempClassDefs.Size());
        }
        
        if (bNeedRedefDoneNotification)
        {
            // Notify that class redefinition completed
            m_monClassRedefDone->Enter();
            m_monClassRedefDone->NotifyAll();
            m_monClassRedefDone->Exit();
        }

    } // while (true)
}

TResult CClassRedefThread::RedefineClasses(TJvmtiClassDefVector &vecClassesToRedefine, 
                                           bool isBlocking)
{
    MARTINI_INFORMATIVE2("CClassRedefThread", 3, false, 
        "Received request to to redefine %u classes. isBlocking = %u\n",
        vecClassesToRedefine.Size(), (U32)isBlocking);

    if(vecClassesToRedefine.Size() == 0){
        return MRTE_RESULT_OK;
    }

    m_monClassesAvailableForRedef->Enter();

    // Queue classes for redefinition
    unsigned int i;
    for (i = 0; i < vecClassesToRedefine.Size(); ++i)
    {
        m_vecClassesToRedefine.Push(vecClassesToRedefine.GetAt(i));
    }
    if (vecClassesToRedefine.Size() > 0 && isBlocking)
    {
        m_bNeedRedefDoneNotification = true;
    }

    if (isBlocking)
    {
        // Wait for Class Redefinition thread to complete
        m_monClassRedefDone->Enter();
    }
    
    // Notify the Class Redefinition thread
    m_monClassesAvailableForRedef->Notify();
    m_monClassesAvailableForRedef->Exit();

    if (isBlocking)
    {
        // Wait for Class Redefinition thread to complete
        m_monClassRedefDone->Wait(0);
        m_monClassRedefDone->Exit();
    }

    // Assume the redefinition process succeeded
    return MRTE_RESULT_OK;
}

//////////////////////////////////////////////////////////////////////////
// CAdaptorFactory

CAdaptorFactory* CAdaptorFactory::m_pInstance = NULL;

IInstrumentationAdaptor* CAdaptorFactory::CreateAdaptor(MPI::EEventGroup group)
{
    IInstrumentationAdaptor *pAdaptor = NULL;
    switch (group)
    {
    case MPI::EG_CALL_GRAPH:
        pAdaptor = new CCgAdaptorDelegate();
        break;
    case MPI::EG_HEAP:       
        pAdaptor = new CHeapAdaptorDelegate();
        break;
    case MPI::EG_THREAD_INTERACTION:       
        pAdaptor = new CThreadAdaptorDelegate();
        break;
    }
    return pAdaptor;
}

//////////////////////////////////////////////////////////////////////////
// CBaseAdaptorDelegate
TResult CBaseAdaptorDelegate::SetEventsStatus(bool enabled)
{
    if (!CJpiGlobals::Instance()->bJvmInitDone)
    {
        return MRTE_ERROR_PHASE_FAILURE;
    }
    
    JNIEnv *pJNIEnv = NULL;
    bool attached = false;

    TResult iRes = CJpiGlobals::Instance()->pJvmInterface->AttachCurrentThread(&pJNIEnv, 
        &attached);
    if (MRTE_FAILED(iRes))
    {
        return iRes;
    }

    iRes = MRTE_RESULT_OK;
    if (pJNIEnv->IsSameObject(m_grProxyClass, NULL))
    {
        MARTINI_INFORMATIVE("CBaseAdaptorDelegate",0, false, 
            "[SetEventsStatus] Invalid global reference for proxy Java class");
        iRes = MRTE_ERROR_FAIL;
    }

    // Invoke the SetEventsStatus method of the adaptor's proxy Java class
    pJNIEnv->CallStaticVoidMethod(m_grProxyClass, m_hSetEventsStatus, enabled ? 1 : 0);
    if (pJNIEnv->ExceptionOccurred() != NULL)
    {
        MARTINI_INFORMATIVE("CBaseAdaptorDelegate",0, false, 
            "[SetEventsStatus] Error when invoking SetEventsStatus method of proxy Java class");
        pJNIEnv->ExceptionClear();
        iRes = MRTE_ERROR_FAIL;
    }

    if (attached)
    {
        CJpiGlobals::Instance()->pJvmInterface->DetachCurrentThread();
    }
    return iRes;
}

//////////////////////////////////////////////////////////////////////////
// CCgAdaptorDelegate

TResult CCgAdaptorDelegate::Init(ILogAssert *pLogger, 
                                 MPI::IEventFilter *pFilter,
                                 MPI::BitSet configFlags)
{
    MPI::ICallGraphFilter *pCgFilter = (MPI::ICallGraphFilter*)pFilter;
    m_CGFilterProxy.Init(pCgFilter);

    if (!m_pAdaptor)
    {
        // Load the Adaptor library
        ILibraryLoader *pLibrary = LoadBistroLibrary(CGADAPTOR_DLL_NAME);
        if (NULL == pLibrary)
        {
            return MRTE_ERROR_FAIL;
        }
        
        TGetInstrumentationAdaptor pfnGetAdaptor = 
            (TGetInstrumentationAdaptor)pLibrary->GetEntry(GET_INSTRUMENTATION_ADAPTOR_NAME);
        if (pfnGetAdaptor)
        {
            // Obtain the Adaptor interface
            m_pAdaptor = pfnGetAdaptor();
        }
        
        pLibrary->Destroy();
    
        if (m_pAdaptor == NULL)
        {
            return MRTE_ERROR_FAIL;
        }
    }

    // Initialize the Adaptor
    TResult res = m_pAdaptor->Init(CLogAssert::GetInstance(), &m_CGFilterProxy, configFlags);
    return res;    
}

TResult CCgAdaptorDelegate::ModifyClass(MPI::TId classId, 
                                        const SClassFile &classToInstrument,
                                        TMemoryAllocatorFunc funcAllocator,
                                        SClassFile *pInstrumentedClass,
                                        IInstrumentationContext *pContext)
{
    CContext cgContext;
    TResult iRetVal = m_pAdaptor->ModifyClass(classId, classToInstrument, funcAllocator, 
        pInstrumentedClass, &cgContext);

    if (iRetVal == MRTE_RESULT_OK 
        || iRetVal == MRTE_ERROR_INSTRUMENTATION_NOT_NEEDED
        || iRetVal == MRTE_ERROR_UNABLE_TO_INSTRUMENT)
    {
        // Store the class information in the Data Manager.
        // This is done for all valid classes, and not just for instrumented classes.
        // The reason is that when using instrumentation, method ids are assigned by the 
        // CG Adaptor and not by the Data Manager, and these ids are needed for all valid 
        // classes, and not just for instrumented classes.
        iRetVal = CJpiGlobals::Instance()->pDataManager->SetClassInstrumentationInfo(
            &(cgContext.classInfo));
    }
    DeallocateInstrClassInfo(&cgContext.classInfo);
    return iRetVal;
}

TResult CCgAdaptorDelegate::ModifyByteCodes(MPI::TId classId, 
                                            const SClassFile &classToInstrument,
                                            TMemoryAllocatorFunc funcAllocator,
                                            SClassFile *pInstrumentedClass,
                                            IInstrumentationContext *pContext)
{
    CContext cgContext;
    CDataManager *pDM = CJpiGlobals::Instance()->pDataManager;
    TResult res = pDM->GetClassInstrContext(&cgContext, classId);
    if (MRTE_SUCCEEDED(res))
    {
        res = m_pAdaptor->ModifyByteCodes(classId, classToInstrument, funcAllocator, 
            pInstrumentedClass, &cgContext);
        if (MRTE_SUCCEEDED(res))
        {
            pDM->SetClassInstrumentationStatus(classId, true /* bEventsEnabled */);
        }
        delete [] cgContext.classInfo.pMethods;
    }
    return res;
}

void CCgAdaptorDelegate::JvmInitDone()
{
    // Register native callback methods in the wrapper class
    TResult iRetVal = RegisterNativeCallbacksInWrapper();
    if (MRTE_SUCCEEDED(iRetVal))
    {
        m_pAdaptor->JvmInitDone();
    }
    else
    {
        MARTINI_ERROR("CJavaInstrumentorManager", "failed to register native callbacks "
            "for the call-graph events recorder class");
        MARTINI_ASSERT("CJavaInstrumentorManager", MRTE_SUCCEEDED(iRetVal), "");
    }
}

TResult CCgAdaptorDelegate::RegisterNativeCallbacksInWrapper()
{
    CJpiGlobals *pJpiGlobals = CJpiGlobals::Instance();

    if (NULL == pJpiGlobals->pfnMethodEnterHandler 
        || NULL == pJpiGlobals->pfnMethodLeaveHandler 
        || NULL == pJpiGlobals->pfnMethodLeaveWithExcetionHandler)
    {
        return MRTE_ERROR_FAIL;
    }

    JNIEnv *pJNIEnv = NULL;

    TResult iRes = CJpiGlobals::Instance()->pJvmInterface->GetJNIEnv(&pJNIEnv);
    if (MRTE_FAILED(iRes))
    {
        return iRes;
    }
    
    jclass cls = pJNIEnv->FindClass(CGADAPTOR_CALLBACK_CLASS_NAME);
    if (cls == 0)
    {
        MARTINI_INFORMATIVE1("CJavaInstrumentorManager", 0, false, 
            "failed to load class '%s'", CGADAPTOR_CALLBACK_CLASS_NAME);
        return MRTE_ERROR_FAIL;
    }
    
    JNINativeMethod nativeMethods[CGADAPTOR_NUM_EVENT_HANDLERS];
    nativeMethods[0].name =      CGADAPTOR_CALLBACK_PROLOG_NAME;
    nativeMethods[0].signature = CGADAPTOR_CALLBACK_PROLOG_SIG;
    nativeMethods[0].fnPtr =     (void*)pJpiGlobals->pfnMethodEnterHandler;
    nativeMethods[1].name =      CGADAPTOR_CALLBACK_EPILOG_NAME;
    nativeMethods[1].signature = CGADAPTOR_CALLBACK_EPILOG_SIG;
    nativeMethods[1].fnPtr =     (void*)pJpiGlobals->pfnMethodLeaveHandler;
    nativeMethods[2].name =      CGADAPTOR_CALLBACK_EXCEPTION_NAME;
    nativeMethods[2].signature = CGADAPTOR_CALLBACK_EXCEPTION_SIG;
    nativeMethods[2].fnPtr =     (void*)pJpiGlobals->pfnMethodLeaveWithExcetionHandler;

    jint ret = pJNIEnv->RegisterNatives(cls, (JNINativeMethod *)nativeMethods, 
        CGADAPTOR_NUM_EVENT_HANDLERS);
    if (ret != 0)
    {
        return MRTE_ERROR_FAIL;
    }

    // Inform the Wrapper class that the native methods were instrumented
    jmethodID methodId = pJNIEnv->GetStaticMethodID(cls, "JVMInit", "()V");
    if (methodId == 0)
    {
        MARTINI_ERROR("CJavaInstrumentorManager", 
            "Failed to get static method 'JVMInit' in CGProxy wrapper class");
        return MRTE_ERROR_FAIL;
    }
    pJNIEnv->CallStaticVoidMethod(cls, methodId);
    MARTINI_INFORMATIVE("CJavaInstrumentorManager", 0, false, 
        "Using Martini Call-Graph Instrumentation Technology");

    // Store a handle to the SetEventsStatus method of the proxy class
    m_grProxyClass = (jclass)pJNIEnv->NewWeakGlobalRef(cls);
    if (NULL == m_grProxyClass)
    {
        MARTINI_ERROR("CJavaInstrumentorManager", 
            "Failed to create a weak global reference for the CGProxy wrapper class");
        return MRTE_ERROR_FAIL;
    }
    m_hSetEventsStatus = pJNIEnv->GetStaticMethodID(cls, "SetEventsStatus", "(I)V");
    if (0 == m_hSetEventsStatus)
    {
        MARTINI_ERROR("CJavaInstrumentorManager", 
            "Failed to get static method 'SetEventsStatus' in CGProxy wrapper class");
        return MRTE_ERROR_FAIL;
    }

    pJNIEnv->DeleteLocalRef(cls);

    return MRTE_RESULT_OK;
}




//////////////////////////////////////////////////////////////////////////
// CThreadAdaptorDelegate



TResult CThreadAdaptorDelegate::Init(ILogAssert *pLogger, 
                                     MPI::IEventFilter *pFilter,
                                     MPI::BitSet configFlags)
{
    //Add filtering support
    MPI::IThreadInteractionFilter *pThreadInteractionFilter = (MPI::IThreadInteractionFilter*)pFilter;
    m_ThreadInteractionFilterProxy.Init(pThreadInteractionFilter);

    if (!m_pAdaptor)
    {
        // Load the Adaptor library
        ILibraryLoader *pLibrary = LoadBistroLibrary(THREADADAPTOR_DLL_NAME);
        if (NULL == pLibrary)
        {
            return MRTE_ERROR_FAIL;
        }
        
        TGetInstrumentationAdaptor pfnGetAdaptor = 
            (TGetInstrumentationAdaptor)pLibrary->GetEntry(GET_INSTRUMENTATION_ADAPTOR_NAME);
        if (pfnGetAdaptor)
        {
            // Obtain the Adaptor interface
            m_pAdaptor = pfnGetAdaptor();
        }
        
        pLibrary->Destroy();
    
        if (m_pAdaptor == NULL)
        {
            return MRTE_ERROR_FAIL;
        }
    }

    // Initialize the Adaptor
    TResult res = m_pAdaptor->Init(CLogAssert::GetInstance(), &m_ThreadInteractionFilterProxy,
        configFlags); 
    return MRTE_RESULT_OK;
}

TResult CThreadAdaptorDelegate::ModifyClass(MPI::TId classId, 
                                        const SClassFile &classToInstrument,
                                        TMemoryAllocatorFunc funcAllocator,
                                        SClassFile *pInstrumentedClass,
                                        IInstrumentationContext *pContext)
{
    CContext context;
    TResult iRetVal = m_pAdaptor->ModifyClass(classId, classToInstrument, funcAllocator, 
        pInstrumentedClass, &context);
    DeallocateInstrClassInfo(&context.classInfo);
    return iRetVal;
}

TResult CThreadAdaptorDelegate::ModifyByteCodes(MPI::TId classId, 
                                            const SClassFile &classToInstrument,
                                            TMemoryAllocatorFunc funcAllocator,
                                            SClassFile *pInstrumentedClass,
                                            IInstrumentationContext *pContext)
{

    CContext context;
    CDataManager *pDM = CJpiGlobals::Instance()->pDataManager;
    TResult res = pDM->GetClassInstrContext(&context, classId);
    if (MRTE_SUCCEEDED(res))
    {
        res = m_pAdaptor->ModifyByteCodes(classId, classToInstrument, funcAllocator, 
            pInstrumentedClass, &context);
        if (MRTE_SUCCEEDED(res))
        {
            pDM->SetClassInstrumentationStatus(classId, true /* bEventsEnabled */);
        }
        delete [] context.classInfo.pMethods;
    }
    return res;
}

void CThreadAdaptorDelegate::JvmInitDone()
{
    // Register native callback methods in the wrapper class
    TResult iRetVal = RegisterNativeCallbacksInWrapper();
    if (MRTE_SUCCEEDED(iRetVal))
    {
        m_pAdaptor->JvmInitDone();
    }
    else
    {
        MARTINI_ERROR("CJavaInstrumentorManager", "failed to register native callbacks "
            "for the thread events recorder class");
        MARTINI_ASSERT("CJavaInstrumentorManager", MRTE_SUCCEEDED(iRetVal), "");
    }
}

TResult CThreadAdaptorDelegate::RegisterNativeCallbacksInWrapper()
{
    CJpiGlobals *pJpiGlobals = CJpiGlobals::Instance();

    if (NULL == pJpiGlobals->pfnThreadInteractionHandler)
    {
        return MRTE_ERROR_FAIL;
    }

    JNIEnv *pJNIEnv = NULL;

    TResult iRes = CJpiGlobals::Instance()->pJvmInterface->GetJNIEnv(&pJNIEnv);
    if (MRTE_FAILED(iRes))
    {
        return iRes;
    }
    
    jclass cls = pJNIEnv->FindClass(THREADADAPTOR_CALLBACK_CLASS_NAME);
    if (cls == 0)
    {
        MARTINI_INFORMATIVE1("CJavaInstrumentorManager", 0, false, 
            "failed to load class '%s'", THREADADAPTOR_CALLBACK_CLASS_NAME);
        return MRTE_ERROR_FAIL;
    }
    
    JNINativeMethod nativeMethods[THREADADAPTOR_NUM_EVENT_HANDLERS];
    nativeMethods[0].name =      THREADADAPTOR_CALLBACK_CALL_TI_NAME;
    nativeMethods[0].signature = THREADADAPTOR_CALLBACK_CALL_TI_SIG;
    nativeMethods[0].fnPtr =     (void*)pJpiGlobals->pfnThreadInteractionHandler;

    jint ret = pJNIEnv->RegisterNatives(cls, (JNINativeMethod *)nativeMethods, 
        THREADADAPTOR_NUM_EVENT_HANDLERS);
    if (ret != 0)
    {
        return MRTE_ERROR_FAIL;
    }

    // Inform the Wrapper class that the native methods were instrumented
    jmethodID methodId = pJNIEnv->GetStaticMethodID(cls, "JVMInit", "()V");
    if (methodId == 0)
    {
        MARTINI_ERROR("CJavaInstrumentorManager", 
            "Failed to get static method 'JVMInit' in ThreadProxy wrapper class");
        return MRTE_ERROR_FAIL;
    }
    pJNIEnv->CallStaticVoidMethod(cls, methodId);
    MARTINI_INFORMATIVE("CJavaInstrumentorManager", 0, false, 
        "Using Martini Thread Instrumentation Technology");

    // Store a handle to the SetEventsStatus method of the proxy class
    m_grProxyClass = (jclass)pJNIEnv->NewWeakGlobalRef(cls);
    if (NULL == m_grProxyClass)
    {
        MARTINI_ERROR("CJavaInstrumentorManager", 
            "Failed to create a weak global reference for the ThreadProxy wrapper class");
        return MRTE_ERROR_FAIL;
    }
    m_hSetEventsStatus = pJNIEnv->GetStaticMethodID(cls, "SetEventsStatus", "(I)V");
    if (0 == m_hSetEventsStatus)
    {
        MARTINI_ERROR("CJavaInstrumentorManager", 
            "Failed to get static method 'SetEventsStatus' in ThreadProxy wrapper class");
        return MRTE_ERROR_FAIL;
    }

    pJNIEnv->DeleteLocalRef(cls);

    return MRTE_RESULT_OK;
}


///////////////////////////////////////////////////////////////
// CJavaInstrumentorManager::CCGFilterProxy implementation

//TODO: Make one base class for FilterAdapters
bool 
CThreadAdaptorDelegate::CThreadInteractionFilterProxy::ShouldNotify(MPI::SThreadInteractionFilterData &filterData)
{
    if (NULL == m_pThreadInteractionFilter)
    {
        return true;
    }

    MCString strClassName;
    MCString strMethodName;
    
    // Translate class name from internal JVM representation to the Java syntax
    strClassName.Set(filterData.szClassName);
    strClassName.ReplaceAll("/", ".");

    // Translate special method names (such as <init>, <clinit>)
    CJpiGlobals::Instance()->pDataManager->GetUpdatedMethodName(strClassName.Get(), 
        filterData.szMethodName, &strMethodName);
    
    // Call the client's filter callback
    MPI::SThreadInteractionFilterData locFilterData;
    locFilterData.szClassName = strClassName.Get();
    locFilterData.szMethodName = strMethodName.Get();
    return m_pThreadInteractionFilter->ShouldNotify(locFilterData);
}


TResult CThreadAdaptorDelegate::CThreadInteractionFilterProxy::Init(MPI::IThreadInteractionFilter *pThreadInteractionFilter)
{
    m_pThreadInteractionFilter = pThreadInteractionFilter;
    return MRTE_RESULT_OK;
}



//
// Deallocates memory of pInstrClassInfo members
//
void CBaseAdaptorDelegate::DeallocateInstrClassInfo(SAdaptorClassInfo *pInstrClassInfo)
{
    if (!pInstrClassInfo)
    {
        return;
    }

    if (pInstrClassInfo->szClassName)
    {
        free(const_cast<char *>(pInstrClassInfo->szClassName));
    }
    if (pInstrClassInfo->szSourceFileName)
    {
        free(const_cast<char *>(pInstrClassInfo->szSourceFileName));
    }
    unsigned int i;
    for (i = 0; i < pInstrClassInfo->uiNumMethods; ++i)
    {
        DeallocateInstrMethodInfo(&(pInstrClassInfo->pMethods[i]));
    }
    if (pInstrClassInfo->pMethods)
    {
        free(pInstrClassInfo->pMethods);
    }
}

//
// Deallocates memory of pInstrMethodInfo members
//
void CBaseAdaptorDelegate::DeallocateInstrMethodInfo(SAdaptorMethodInfo *pInstrMethodInfo)
{
    if (!pInstrMethodInfo)
    {
        return;
    }

    if (pInstrMethodInfo->szName)
    {
        free(const_cast<char *>(pInstrMethodInfo->szName));
    }
    if (pInstrMethodInfo->szSignature)
    {
        free(const_cast<char *>(pInstrMethodInfo->szSignature));
    }
}


//////////////////////////////////////////////////////////////////////////
// CHeapAdaptorDelegate

TResult CHeapAdaptorDelegate::Init(ILogAssert *pLogger, 
                                   MPI::IEventFilter *pFilter,
                                   MPI::BitSet configFlags)
{
    if (!m_pAdaptor)
    {
        // Load the Adaptor library
        ILibraryLoader *pLibrary = LoadBistroLibrary(HEAPADAPTOR_DLL_NAME);
        if (NULL == pLibrary)
        {
            return MRTE_ERROR_FAIL;
        }
        
        TGetInstrumentationAdaptor pfnGetAdaptor = 
            (TGetInstrumentationAdaptor)pLibrary->GetEntry(GET_INSTRUMENTATION_ADAPTOR_NAME);
        if (pfnGetAdaptor)
        {
            // Obtain the Adaptor interface
            m_pAdaptor = pfnGetAdaptor();
        }
        
        pLibrary->Destroy();
    
        if (m_pAdaptor == NULL)
        {
            return MRTE_ERROR_FAIL;
        }
    }

    // Initialize the Adaptor
    TResult res = m_pAdaptor->Init(CLogAssert::GetInstance(), pFilter, configFlags);
    return res;    
}

TResult CHeapAdaptorDelegate::ModifyClass(MPI::TId classId, 
                                          const SClassFile &classToInstrument,
                                          TMemoryAllocatorFunc funcAllocator,
                                          SClassFile *pInstrumentedClass,
                                          IInstrumentationContext *pContext)
{
    CContext context;
    TResult res = m_pAdaptor->ModifyClass(classId, classToInstrument, funcAllocator, 
        pInstrumentedClass, &context);
    if (res == MRTE_RESULT_OK 
        || res == MRTE_ERROR_INSTRUMENTATION_NOT_NEEDED
        || res == MRTE_ERROR_UNABLE_TO_INSTRUMENT)
    {
        // Store the class information in the Data Manager.
        // This is done for all valid classes, and not just for instrumented classes.
        // The reason is that when using instrumentation, method ids are assigned by the 
        // Adaptor and not by the Data Manager, and these ids are needed for all valid 
        // classes, and not just for instrumented classes.
        res = CJpiGlobals::Instance()->pDataManager->SetClassInstrumentationInfo(
            &(context.classInfo));
    }
    DeallocateInstrClassInfo(&context.classInfo);
    return res;
}

TResult CHeapAdaptorDelegate::ModifyByteCodes(MPI::TId classId, 
                                              const SClassFile &classToInstrument,
                                              TMemoryAllocatorFunc funcAllocator,
                                              SClassFile *pInstrumentedClass,
                                              IInstrumentationContext *pContext)
{
    CContext context;
    CDataManager *pDM = CJpiGlobals::Instance()->pDataManager;
    TResult res = pDM->GetClassInstrContext(&context, 
        classId);
    if (MRTE_SUCCEEDED(res))
    {
        res = m_pAdaptor->ModifyByteCodes(classId, classToInstrument, funcAllocator, 
            pInstrumentedClass, &context);
        if (MRTE_SUCCEEDED(res))
        {
            pDM->SetClassInstrumentationStatus(classId, true /* bEventsEnabled */);
        }
        delete [] context.classInfo.pMethods;
    }
    return res;
}

void CHeapAdaptorDelegate::JvmInitDone()
{
    // Register native callback methods in the wrapper class
    TResult iRetVal = RegisterNativeCallbacksInWrapper();
    if (MRTE_SUCCEEDED(iRetVal))
    {
        m_pAdaptor->JvmInitDone();
    }
    else
    {
        MARTINI_ERROR("CJavaInstrumentorManager", "failed to register native callbacks "
            "for the heap events recorder class");
        MARTINI_ASSERT("CJavaInstrumentorManager", MRTE_SUCCEEDED(iRetVal), "");
    }
}

TResult CHeapAdaptorDelegate::RegisterNativeCallbacksInWrapper()
{
    CJpiGlobals *pJpiGlobals = CJpiGlobals::Instance();

    if (NULL == pJpiGlobals->pfnObjectAllocHandler 
        || NULL == pJpiGlobals->pfnArrayAllocHandler)
    {
        return MRTE_ERROR_FAIL;
    }

    JNIEnv *pJNIEnv = NULL;

    TResult iRes = CJpiGlobals::Instance()->pJvmInterface->GetJNIEnv(&pJNIEnv);
    if (MRTE_FAILED(iRes))
    {
        return iRes;
    }
    
    jclass cls = pJNIEnv->FindClass(HEAPADAPTOR_CALLBACK_CLASS_NAME);
    if (cls == 0)
    {
        MARTINI_INFORMATIVE1("CJavaInstrumentorManager", 0, false, 
            "failed to load class '%s'", HEAPADAPTOR_CALLBACK_CLASS_NAME);
        return MRTE_ERROR_FAIL;
    }
    
    JNINativeMethod nativeMethods[HEAPADAPTOR_NUM_EVENT_HANDLERS];
    nativeMethods[0].name =      HEAPADAPTOR_CALLBACK_OBJECT_ALLOC_NAME;
    nativeMethods[0].signature = HEAPADAPTOR_CALLBACK_OBJECT_ALLOC_SIG;
    nativeMethods[0].fnPtr =     (void*)pJpiGlobals->pfnObjectAllocHandler;
    nativeMethods[1].name =      HEAPADAPTOR_CALLBACK_ARRAY_ALLOC_NAME;
    nativeMethods[1].signature = HEAPADAPTOR_CALLBACK_ARRAY_ALLOC_SIG;
    nativeMethods[1].fnPtr =     (void*)pJpiGlobals->pfnArrayAllocHandler;

    jint ret = pJNIEnv->RegisterNatives(cls, (JNINativeMethod *)nativeMethods, 
        HEAPADAPTOR_NUM_EVENT_HANDLERS);
    if (ret != 0)
    {
        return MRTE_ERROR_FAIL;
    }

    // Inform the Wrapper class that the native methods were instrumented
    jmethodID methodId = pJNIEnv->GetStaticMethodID(cls, "JVMInit", "()V");
    if (methodId == 0)
    {
        MARTINI_ERROR("CJavaInstrumentorManager", 
            "Failed to get static method 'JVMInit' in HeapProxy class");
        return MRTE_ERROR_FAIL;
    }
    pJNIEnv->CallStaticVoidMethod(cls, methodId);
    MARTINI_INFORMATIVE("CJavaInstrumentorManager", 0, false, 
        "Using Martini Heap Instrumentation Technology");

    // Store a handle to the SetEventsStatus method of the proxy class
    m_grProxyClass = (jclass)pJNIEnv->NewWeakGlobalRef(cls);
    if (NULL == m_grProxyClass)
    {
        MARTINI_ERROR("CJavaInstrumentorManager", 
            "Failed to create a weak global reference for the HeapProxy wrapper class");
        return MRTE_ERROR_FAIL;
    }
    m_hSetEventsStatus = pJNIEnv->GetStaticMethodID(cls, "SetEventsStatus", "(I)V");
    if (0 == m_hSetEventsStatus)
    {
        MARTINI_ERROR("CJavaInstrumentorManager", 
            "Failed to get static method 'SetEventsStatus' in HeapProxy wrapper class");
        return MRTE_ERROR_FAIL;
    }

	// Register the HeapObjectData class native methods
	cls = pJNIEnv->FindClass(HEAPOBJDATA_CALLBACK_CLASS_NAME);
    if (cls == 0)
    {
        MARTINI_INFORMATIVE1("CJavaInstrumentorManager", 0, false, 
            "failed to load class '%s'", HEAPOBJDATA_CALLBACK_CLASS_NAME);
        return MRTE_ERROR_FAIL;
    }

	JNINativeMethod nativeHeapObjDataMethods[HEAPOBJDATA_NUM_EVENT_HANDLERS];
    nativeHeapObjDataMethods[0].name =      HEAPOBJDATA_CALLBACK_NAME;
    nativeHeapObjDataMethods[0].signature = HEAPOBJDATA_CALLBACK_SIG;
    nativeHeapObjDataMethods[0].fnPtr =     (void*)pJpiGlobals->pfnGetObjTIdHandler;
    
    ret = pJNIEnv->RegisterNatives(cls, (JNINativeMethod *)nativeHeapObjDataMethods, 
        HEAPOBJDATA_NUM_EVENT_HANDLERS);
    if (ret != 0)
    {
        return MRTE_ERROR_FAIL;
    }
    
    pJNIEnv->DeleteLocalRef(cls);

    return MRTE_RESULT_OK;
}

///////////////////////////////////////////////////////////////
// CJavaInstrumentorManager::CCGFilterProxy implementation

bool 
CCgAdaptorDelegate::CCgFilterProxy::ShouldNotify(MPI::SCallGraphFilterData &methodInfo)
{
    if (NULL == m_pCgFilter)
    {
        return true;
    }

    MCString strClassName;
    MCString strMethodName;
    
    // Translate class name from internal JVM representation to the Java syntax
    strClassName.Set(methodInfo.szClassName);
    strClassName.ReplaceAll("/", ".");

    // Translate special method names (such as <init>, <clinit>)
    CJpiGlobals::Instance()->pDataManager->GetUpdatedMethodName(strClassName.Get(), 
        methodInfo.szMethodName, &strMethodName);
    
    // Call the client's filter callback
    MPI::SCallGraphFilterData filterData;
    filterData.methodAccessType = methodInfo.methodAccessType;
    filterData.szClassName = strClassName.Get();
    filterData.szModuleName = NULL;
    filterData.szMethodName = strMethodName.Get();
    filterData.szMethodSignature = NULL;
    return m_pCgFilter->ShouldNotify(filterData);
}


TResult CCgAdaptorDelegate::CCgFilterProxy::Init(MPI::ICallGraphFilter *pCgFilter)
{
    m_pCgFilter = pCgFilter;
    return MRTE_RESULT_OK;
}

//////////////////////////////////////////////////////////////////////////
// CJavaInstrumentorManager

CJavaInstrumentorManager::CJavaInstrumentorManager(MPI::TId moduleId)
{
    m_ModuleId = moduleId;
    m_pDataManager = NULL;
    m_pEventManager = NULL;
    m_pParamChecker = NULL;
    m_pAdaptor = NULL;
    m_pClassLoadHookEvent = NULL;
    m_pJVMInitEvent = NULL;
    m_bJVMInit = false;
    m_bInstrGeneratedEventsEnabled = false;
    m_bHasFirstClassLoadHookArrived = false;
    m_bIsTentativeInstrumentation = false;
    m_InstrumentationEvent.eventGroup = MPI::EG_NONE;
    m_InstrumentationEvent.pEventFilter = NULL;
    m_csInstrumentation = NULL;
    m_pClassIdAllocator = NULL;
}

CJavaInstrumentorManager::~CJavaInstrumentorManager()
{
    if (m_pClassLoadHookEvent)
    {
        delete m_pClassLoadHookEvent;
    }

    if (m_pJVMInitEvent)
    {
        delete m_pJVMInitEvent;
    }
}


TResult 
CJavaInstrumentorManager::InitInstrumentation()
{
    TResult iRetVal = MRTE_RESULT_OK;
    static bool bWasInit = false;
    if (!bWasInit)
    {
        bWasInit = true;

        // Create the class id allocator. When using instrumentation, class ids are allocated
        // by the Instrumentation Manager
        m_pClassIdAllocator = new CIdAllocator();

        // Create the class_load_hook event observer
        m_pClassLoadHookEvent = new CClassLoadHookEventObserver(this);
        if (NULL == m_pClassLoadHookEvent)
        {
            return MRTE_ERROR_OUT_OF_MEMORY;
        }
        
        // Create the JVM init event observer
        m_pJVMInitEvent = new CJVMInitEventObserver(this);
        if (NULL == m_pJVMInitEvent)
        {
            return MRTE_ERROR_OUT_OF_MEMORY;
        }

        // Register for instrumentation event
        iRetVal = m_pEventManager->RegisterEventInternal(m_ModuleId, *m_pClassLoadHookEvent);
        if (MRTE_FAILED(iRetVal))
        {
            return iRetVal;
        }

        if (!m_bJVMInit)
        {
            // Register to JVM_init event. With BEA we can only instrument and use JNI
            // after JVM completed its initialization
            iRetVal = m_pEventManager->RegisterEventInternal(m_ModuleId, *m_pJVMInitEvent);
        }
        else
        {
            JVMInitEventHandler();
        }
    }

    return iRetVal;
}


//
// Loads the Call-graph Adaptor
//
TResult 
CJavaInstrumentorManager::InitInstrumentationAdaptor(MPI::EEventGroup group, 
                                                     MPI::IEventFilter *pEventFilter)
{
    TResult res = MRTE_ERROR_FAIL;
    m_pAdaptor = CAdaptorFactory::Instance()->CreateAdaptor(group);
    if (NULL != m_pAdaptor)
    {
        res = m_pAdaptor->Init(CLogAssert::GetInstance(), pEventFilter, 
            CJpiGlobals::Instance()->configFlags);
    }
    return res;
}


/**
 * Register to an event that is implemented by instrumentation          
 * 
 * Note: instrumentation event registration is currently supported for only one client.
 */
TResult CJavaInstrumentorManager::RegisterEvent(MPI::IEventObserver& event,
                                                bool bIsTentative,
                                                bool bIsEnabled)
{
    TResult iRetVal = m_pParamChecker->CheckEvent(event.Type());
    if (MRTE_FAILED(iRetVal))
    {
        return iRetVal;
    }

    iRetVal = InitInstrumentation();
    if (MRTE_FAILED(iRetVal))
    {
        return iRetVal;
    }

    // Set instrumentation information
    iRetVal = SetInstrumentationInfo(event);
    
    m_bIsTentativeInstrumentation = bIsTentative;
    m_bInstrGeneratedEventsEnabled = bIsEnabled;

    return iRetVal;
}


void CJavaInstrumentorManager::JVMInitEventHandler()
{
    m_bJVMInit = true;
    CJpiGlobals::Instance()->bJvmInitDone = true;

    if (!m_bHasFirstClassLoadHookArrived && m_bIsTentativeInstrumentation)
    {
        // this is a tentative instrumentation that did not succeed since no class-load-hook
        // arrived before JVM init
        return;
    }

    if (m_pAdaptor)
    {
        m_pAdaptor->JvmInitDone();
        SetEventStreamStatus(m_bInstrGeneratedEventsEnabled);
    }

    // Start the class redefinition thread
    if (m_pJVM->GetInterfaceType() == IT_JVMTI
        && m_pJVM->GetCapabilities()->Enabled(CT_CAN_ENABLE_OR_DISABLE_BCI_GENERATED_EVENTS))
    {
        g_pClassRedefThread = new CClassRedefThread();
        g_pClassRedefThread->Start();
    }
}

TResult CJavaInstrumentorManager::DisableInstrumentation()
{
    // disable class_load_hook and JVM_init events
    TResult iRes = m_pEventManager->DisableEventInternal(MPI::EV_JAVA_CLASS_FILE_LOAD_HOOK);
    if (MRTE_FAILED(iRes))
    {
        return iRes;
    }
    iRes = m_pEventManager->DisableEventInternal(MPI::EV_VM_INIT);
    return iRes;
}


TResult 
CJavaInstrumentorManager::SetInstrumentationInfo(MPI::IEventObserver& event)
{
    TResult iRetVal = MRTE_RESULT_OK;
    MPI::EEventGroup group = MPI::GetEventGroup(event.Type());
    if (MPI::EG_NONE == group)
    {
        // Event is not generated by instrumentation
        return MRTE_ERROR_FAIL;
    }
    

    if (m_InstrumentationEvent.eventGroup != MPI::EG_NONE)
    {
        if (m_InstrumentationEvent.eventGroup != group)
        {
            // Instrumentation needed for this event conflicts with an instrumentation
            // needed for a previously registered event
            //TODO: this limitation should be removed once JIM is enhanced to support
            //      multiple concurrent instrumentation adaptor.
            return MRTE_ERROR_CONFLICT;
        }
    }
    
    m_InstrumentationEvent.eventGroup = group;
    
    return MRTE_RESULT_OK;
}

TResult CJavaInstrumentorManager::Initialize(CDataManager *pDataManager, 
                                             CEventManager *pEventManager, 
                                             CParamChecker *pParamChecker)
{
    MARTINI_ASSERT("CJavaInstrumentorManager", pDataManager && pParamChecker && pEventManager, "");
    m_pDataManager = pDataManager;
    m_pEventManager = pEventManager;
    m_pParamChecker = pParamChecker;
    m_pJVM = CJpiGlobals::Instance()->pJvmInterface;

    // Inform Event Manager about supported events
    m_pEventManager->SetEventImplementor(MPI::EV_METHOD_ENTER, MPI_INSTRUMENTATION);
    m_pEventManager->SetEventImplementor(MPI::EV_METHOD_LEAVE, MPI_INSTRUMENTATION);
    m_pEventManager->SetEventImplementor(MPI::EV_OBJECT_ALLOC, MPI_INSTRUMENTATION);
    m_pEventManager->SetEventImplementor(MPI::EV_THREAD_INTERACTION, MPI_INSTRUMENTATION);

    if (!m_csInstrumentation)
    {
        m_csInstrumentation = CreateThreadSync();
        if (!m_csInstrumentation)
        {
            MARTINI_ERROR("CJavaInstrumentationManager", 
                "Unable to initialize critical section");
            return MRTE_ERROR_OSA_FAILURE;
        }
    }
    return MRTE_RESULT_OK;
}

TResult CJavaInstrumentorManager::InitializationPhaseCompleted()
{
    TResult res = MRTE_RESULT_OK;

    if (MPI::EG_NONE != m_InstrumentationEvent.eventGroup)
    {
        res = InitInstrumentationAdaptor(m_InstrumentationEvent.eventGroup, 
            m_InstrumentationEvent.pEventFilter);
        if (MRTE_FAILED(res))
        {
            MARTINI_ERROR("CJavaInstrumentorManager", 
                "Failed to init instrumentation adaptor");
            return res;
        }
    }
    return res;
}

void CJavaInstrumentorManager::SetClassFileNotInstrumented(SJVMData *pJVMEvent)
{
//    pJVMEvent->u.classLoadHook.pNewClassDataLen = 0;
//    pJVMEvent->u.classLoadHook.ppucNewClassData = 0;
    *pJVMEvent->u.classLoadHook.pNewClassDataLen = pJVMEvent->u.classLoadHook.classDataLen;
    *pJVMEvent->u.classLoadHook.ppucNewClassData = (unsigned char*)pJVMEvent->u.classLoadHook.pucClassData;
    MARTINI_ASSERT("CJavaInstrumentorManager", *pJVMEvent->u.classLoadHook.ppucNewClassData, "");
}
  
//
// "Class File Load Hook" event handler
//
// Call to this method originates in ClassFileLoadHookHandler in JVMTIInterface,
// and from there notifyMPIEvent is called, which calls NotifyObservers, which
// is passed to CClassLoadHookEventObserver::HandleEvent, which calls this. - JGW
//
void CJavaInstrumentorManager::InstrumentationEventHandler(SEmData *pEvent)
{
    SJVMData *pJVMEvent = pEvent->u.pJVMData;
    MARTINI_ASSERT("CJavaInstrumentorManager", pEvent->dataType == MPI_VM, "");

    // If the event was generated in response to a call to RedefineClasses, do nothing
    if (pJVMEvent->u.classLoadHook.classBeingRedefined != NULL)
    {
        SetClassFileNotInstrumented(pJVMEvent);
        return;
    }

    // If StackMapTable attribute calculation is enabled, then bootstrap classes
    // (i.e., classes loaded before VMInit) are not instrumented.
    if ((CJpiGlobals::Instance()->configFlags & MPI::CF_JAVA_ENABLE_STACKMAP_CALC)
        && !m_bJVMInit)
    {
        SetClassFileNotInstrumented(pJVMEvent);
        return;
    }

    TResult iRetVal = MRTE_RESULT_OK;
    if (m_bIsTentativeInstrumentation && !m_bHasFirstClassLoadHookArrived)
    {
        // have to find out if the first class-load-hook event arrived before or after
        // JVM completed initialization
        bool bIsCapableToInstrumentAllClasses = m_bJVMInit ? false : true;
        bool bShouldInstrument = m_pEventManager->UpdateTentativeInstrumentation(
            bIsCapableToInstrumentAllClasses);
        if (bShouldInstrument == false)
        {
            SetClassFileNotInstrumented(pJVMEvent);
            return;
        }
    }

    m_bHasFirstClassLoadHookArrived = true;

    SClassFile classToInstrument;
    classToInstrument.pClassFile = const_cast<unsigned char*>
        (pJVMEvent->u.classLoadHook.pucClassData);
    classToInstrument.uiSize = pJVMEvent->u.classLoadHook.classDataLen;

    // Allocate a new class id
    MPI::TId classId = m_pClassIdAllocator->AllocateId();

    if (pJVMEvent->u.classLoadHook.szClassName != NULL)
    {
        MPI::TId loaderId = 0;
        jlong loaderTag = 0;
        if (pJVMEvent->u.classLoadHook.loader != NULL)
        {
            iRetVal = m_pDataManager->GetObjectInfoManager()->GetOrCreateTag(&loaderId, 
                &loaderTag, pJVMEvent->u.classLoadHook.loader, pEvent->pJniEnv,
                false /* bSaveInDb */);
            if (MRTE_FAILED(iRetVal))
            {
                // Assume bootstrap class loader
                loaderId = 0;
            }
        }
        
        iRetVal = m_pDataManager->NewClass(classId, pJVMEvent->u.classLoadHook.szClassName,
            loaderId, true /* bFromInstrumentation */);
            
        if (MRTE_FAILED(iRetVal))
        {
            SetClassFileNotInstrumented(pJVMEvent);
            return;
        }
        
        if (MRTE_RESULT_KEY_ALREADY_EXISTS == iRetVal)
        {
            // Bugzilla 200815 (https://bugs.eclipse.org/bugs/show_bug.cgi?id=200815):
            // A class with an identical name was already loaded by a different class loader,
            // The current implementation identifies classes only by their fully qualified name
            // so instrumenting this class now may cause problems if the class needs to be 
            // redefined (wrong class buffer may be used). The current implementation
            // ignore this class to prevent such things from happening.
            MARTINI_INFORMATIVE1("CJavaInstrumentorManager", 3, false, 
                "[InstrumentationEventHandler] Class '%s' was previously loaded by another class loader. "
                "Class will not be intrumented", pJVMEvent->u.classLoadHook.szClassName);
                
            SetClassFileNotInstrumented(pJVMEvent);
            return;
        }
    }
    else
    {
        MARTINI_INFORMATIVE("CJavaInstrumentorManager", 3, false, 
                "[InstrumentationEventHandler] Class name is null.");
                
        SetClassFileNotInstrumented(pJVMEvent);
        return;
    }

    SClassFile *pClassWithBasicInstr = new SClassFile();
    SClassFile classToReturnToJVM;
    memset(&classToReturnToJVM, 0, sizeof(SClassFile));
    
    InstrumentationLock();

    // Apply basic instrumentation to the class file. The memory for the resulting class file
    // is allocated from the internal JPI heap (using malloc)
    // The result is placed into pClassWithBasicInstr.
    TResult modifyClassRes = m_pAdaptor->ModifyClass(classId, classToInstrument, LocalMemAlloc, 
        pClassWithBasicInstr);


    if (MRTE_SUCCEEDED(modifyClassRes))
    {
        // Store the instrumented class in the Data Manager. This image will be used
        // for redefining the class in the future (if needed)
        iRetVal = m_pDataManager->SetClassFileBuffer(classId, pClassWithBasicInstr);
        if (MRTE_SUCCEEDED(iRetVal))
        {
            // If the class was instrumented, check whether method-level BCI is also needed
            // and prepare a class buffer for the JVM
            TResult modifyByteCodesRes;
            if (m_bInstrGeneratedEventsEnabled)
            {
                modifyByteCodesRes = m_pAdaptor->ModifyByteCodes(classId, *pClassWithBasicInstr, 
                    pJVMEvent->u.classLoadHook.pfnAllocate, &classToReturnToJVM);
            }
            else
            {
                // Method-level BCI is not needed (events are not enabled)
                modifyByteCodesRes = MRTE_ERROR_INSTRUMENTATION_NOT_NEEDED;
            }

            // If method-level BCI failed or not needed, use the "basic instrumentation" class image
            if (MRTE_FAILED(modifyByteCodesRes))
            {
            	// Copies the class file byte array from 'pClassWithBasicInstr' to 'classToReturnToJVM'
                iRetVal = CopyClassFile(&classToReturnToJVM, *pClassWithBasicInstr, 
                    pJVMEvent->u.classLoadHook.pfnAllocate);
                MARTINI_INFORMATIVE2("CJavaInstrumentorManager", 5, false, 
                    "Using basic instrumentation (for instance, updates to constant pool only, for CGProf) for class %u '%s'",
                    (U32)classId, pJVMEvent->u.classLoadHook.szClassName);
            }
            else
            {
                MARTINI_INFORMATIVE2("CJavaInstrumentorManager", 5, false, 
                    "Using method-level instrumentation for class %u '%s'",
                    (U32)classId, pJVMEvent->u.classLoadHook.szClassName);

            }
        }
        else
        {
            MARTINI_INFORMATIVE2("CJavaInstrumentorManager", 0, false, 
                "Unable to store basic image of class '%s' in Data Manager. Class id %u is invalid", 
                pJVMEvent->u.classLoadHook.szClassName, (U32)classId);
        }


#ifdef DUMP_CLASS
        char szDumpFileName[1000];
        sprintf(szDumpFileName, "%s@%p", pJVMEvent->u.classLoadHook.szClassName, 
            pJVMEvent->u.classLoadHook.loader);
        if (MRTE_SUCCEEDED(iRetVal))
        {
            DumpClass(szDumpFileName, classToReturnToJVM);
        }
        else
        {
            DumpClass(szDumpFileName, classToInstrument);
        }

#endif
            
        if (MRTE_SUCCEEDED(iRetVal))
        {
            // Return the instrumented class file to the JVM
            *pJVMEvent->u.classLoadHook.ppucNewClassData = classToReturnToJVM.pClassFile;
            *pJVMEvent->u.classLoadHook.pNewClassDataLen = classToReturnToJVM.uiSize;
            MARTINI_INFORMATIVE2("CJavaInstrumentorManager", 5, false, 
                "Instrumented class %u '%s' will be returned to JVM", 
                (U32)classId, pJVMEvent->u.classLoadHook.szClassName);
        }
        else
        {
            delete pClassWithBasicInstr;
            SetClassFileNotInstrumented(pJVMEvent);
            if (MRTE_ERROR_FAIL == iRetVal)
            {
                MARTINI_INFORMATIVE2("CJavaInstrumentorManager", 0, false, 
                    "Class %u '%s' was not instrumented due to instrumentation errors",
                    (U32)classId, pJVMEvent->u.classLoadHook.szClassName);
            }
        }
    }
    else
    {
        delete pClassWithBasicInstr;
        SetClassFileNotInstrumented(pJVMEvent);
    }

    InstrumentationUnlock();
}

//
// Wrapper for ApplyInstrForLoadedClasses which verifies that the calling
// thread is attached to the VM
//
TResult CJavaInstrumentorManager::ApplyInstrForLoadedClasses(MPI::EEventGroup group, 
                                                             bool enabled)
{
    if (!m_bJVMInit)
    {
        return MRTE_ERROR_PHASE_FAILURE;
    }

    bool bAttached;
    JNIEnv *pJNIEnv;
    TResult iRes = m_pJVM->AttachCurrentThread(&pJNIEnv, &bAttached);
    if (MRTE_FAILED(iRes))
    {
        MARTINI_ERROR("CJavaInstrumentorManager", "failed to attach the External Control "
            "thread to the VM");
        return MRTE_ERROR_FAIL;
    }
    
    iRes = ApplyInstrForLoadedClassesImpl(group, enabled);
    if (bAttached)
    {
        m_pJVM->DetachCurrentThread();
    }
    return iRes;
}

//
// Enables or disables events for loaded classes
//
// Parameters:
//      group       : Event group to enable/disable
//      enabled     : Whether to enable (true) or disable (false) the event group
//
// Returns
//      MRTE_RESULT_OK  : success
//      MRTE_ERROR_FAIL : general failure
//
TResult CJavaInstrumentorManager::ApplyInstrForLoadedClassesImpl(MPI::EEventGroup group,
                                                                 bool enabled)
{
    // Obtain the instrumentation lock to make sure there are no pending instrumentation
    // operations
    InstrumentationLock();

    TResult iRes;
    TIdList *pClassIdList = NULL;
    TMRTEHandle hClassId;

    // Find the classes that need to be redefined    
    pClassIdList = new TIdList();
    iRes = m_pDataManager->GetInstrumentedClasses(pClassIdList);
    if (MRTE_FAILED(iRes))
    {
        // Error retrieving all instrumented classes. Cancel the operation and exit
        delete pClassIdList;
        InstrumentationUnlock();
        return iRes;
    }

    // Suspend all threads, while we instrument
    iRes = m_pDataManager->GetThreadInfoManager()->SuspendAllThreads();
    if (MRTE_FAILED(iRes))
    {
        MARTINI_INFORMATIVE("CJavaInstrumentorManager", 0, false, 
            "failed to suspend Java threads");
        InstrumentationUnlock();
        return iRes;
    }

    // Iterate the list of classes, instrument and queue for redefinition only classes
    // that have valid JNI reference and basic instrumentation image
    
    TJvmtiClassDefVector vecClassesToRedefine;
    vecClassesToRedefine.Reserve(pClassIdList->Count());

    hClassId = pClassIdList->GetFirst();
    while (NULL != hClassId)
    {
        MPI::TId classId = pClassIdList->GetData(hClassId);
        SClassFile classBytes;
        jclass classJniGlobalRef = NULL;
        MCString strNativeClassName;

        // Get the name, class bytes, and JNI Ref
        iRes = m_pDataManager->GetClassInstrumentationInfo(&strNativeClassName, 
            &classBytes, &classJniGlobalRef, classId);

        if (MRTE_SUCCEEDED(iRes) && classJniGlobalRef != NULL)
        {
			if (m_pJVM->CanRedefineClass(classJniGlobalRef)) 
			{
            	// Queues the classId for instrumentation, result of call is stored in vecClassesToRefine
                iRes = InstrumentAndQueueForRedef(&vecClassesToRedefine, classId, group, 
                    enabled);
                if (MRTE_SUCCEEDED(iRes))
                {
                    MARTINI_INFORMATIVE1("CJavaInstrumentorManager", 5, false, 
                        "Class '%s' added to redefinition list", 
                        strNativeClassName.Get());
                }
                else if (MRTE_ERROR_INSTRUMENTATION_NOT_NEEDED == iRes
                    || MRTE_ERROR_UNABLE_TO_INSTRUMENT == iRes)
                {
                    MARTINI_INFORMATIVE1("CJavaInstrumentorManager", 5, false, 
                        "Class '%s' will not be redefined due to user-defined filters or an internal reason", 
                        strNativeClassName.Get());
                }
                else
                {
                    // Instrumentation error. Cancel the operation and exit
                    delete pClassIdList;
                    TResult origIRes = iRes;

					// We had previously suspended all threads, so resume them
                    iRes = m_pDataManager->GetThreadInfoManager()->ResumeAllThreads();
                    if (MRTE_FAILED(iRes))
                    {
                        MARTINI_INFORMATIVE("CJavaInstrumentorManager", 0, false, "failed to resume Java threads");
                        InstrumentationUnlock();
						// Try again with instrumentation unlocked if it previously failed
                        iRes = m_pDataManager->GetThreadInfoManager()->ResumeAllThreads();

                        return origIRes;
                    }

                    InstrumentationUnlock();
                    return origIRes;
                }
            }
            else
            {
                MARTINI_INFORMATIVE1("CJavaInstrumentorManager", 5, false, 
                    "Class '%s' is not modifiable and cannot be redefined", 
                    strNativeClassName.Get());
            }
        }
        else
        {
            MARTINI_INFORMATIVE2("CJavaInstrumentorManager", 5, false,
                "Class '%s' with id = %d was not fully loaded and cannot be redefined",
                strNativeClassName.Get(), (U32)classId);
        }
        hClassId = pClassIdList->GetNext(hClassId);
    } // end while (more classes to instrument)

    // At this point, all loaded classes were instrumented as necessary.
    // We need to release the instrumentation lock (and resume all VM threads? - TODO: not sure...)
    // before calling the RedefineClasses API in order to prevent any deadlocks

    m_bInstrGeneratedEventsEnabled = enabled;
    delete pClassIdList;

    MARTINI_INFORMATIVE1("CJavaInstrumentorManager", 5, false, 
        "redefining %d classes", vecClassesToRedefine.Size());
    
    // Now that we've finished scanning through the classes, we resume all running threads
    iRes = m_pDataManager->GetThreadInfoManager()->ResumeAllThreads();
    if (MRTE_FAILED(iRes))
    {
        MARTINI_INFORMATIVE("CJavaInstrumentorManager", 0, false, 
            "failed to resume Java threads");
        InstrumentationUnlock();
        iRes = m_pDataManager->GetThreadInfoManager()->ResumeAllThreads();
        return iRes;
    }

    InstrumentationUnlock();
    
    iRes = RedefineMultipleClasses(vecClassesToRedefine, true /* isBlocking */);
    if (MRTE_FAILED(iRes))
    {
        MARTINI_ERROR1("CJavaInstrumentorManager", "failed to redefine %u classes",
            vecClassesToRedefine.Size());
    }

/*
    iRes = m_pDataManager->GetThreadInfoManager()->ResumeAllThreads();
*/
    return iRes;
}

//
// Applies instrumentation to the class with the specified id.
//
// Parameters:
//      pInstrumentedClass  [out]   : Instrumented class bytes
//      pClassJniGlobalRef  [out]   : JNI global reference of the class
//      classId             [in]    : The MPI ID of the class to instrument
//      group               [in]    : Event group to enable/disable
//      bEnabled            [in]    : Whether to apply instrumentation for enabling the event
//                                    group (true) or disabling it (false)
//
// Returns:
//      See IInstrumentationAdaptor::ModifyByteCodes for a list of return values.
//
TResult CJavaInstrumentorManager::ApplyInstrForClass(SClassFile *pInstrumentedClass,
                                                     jclass *pClassJniGlobalRef,
                                                     MPI::TId classId, 
                                                     MPI::EEventGroup group,
                                                     bool bEnabled)
{
    // NOTE: The group parameter is currently ignored. The active instrumentation
    //       adaptor is always used, regardless of the specified group.
    //       This is OK since this version supports only one adaptor (either Heap or
    //       Call Graph)

    if (m_pDataManager->AreEventsEnabledForClass(classId) == bEnabled)
    {
        // No need to instrument this class as its events status matches the desired status

        // NOTE: this implementation assumes that different filters cannot be applied without 
        //       disabling events and then enabling them again.
        return MRTE_ERROR_INSTRUMENTATION_NOT_NEEDED;
    }

    SClassFile classWithBasicInstr;
    MCString strNativeClassName;
    TMemoryAllocatorFunc funcAllocator = m_pJVM->GetMemoryAllocator();

    TResult iRes = m_pDataManager->GetClassInstrumentationInfo(&strNativeClassName, 
        &classWithBasicInstr, pClassJniGlobalRef, classId);
    if (MRTE_FAILED(iRes))
    {
        // Error retrieving class information. Cancel the operation and exit
        return iRes;
    }
    if (bEnabled)
    {
        // Apply method-level instrumentation to the class
        iRes = m_pAdaptor->ModifyByteCodes(classId, classWithBasicInstr, funcAllocator,
            pInstrumentedClass);
    }
    else
    {
        // Use the class with basic instrumentation
        iRes = CopyClassFile(pInstrumentedClass, classWithBasicInstr, funcAllocator);
        m_pDataManager->SetClassInstrumentationStatus(classId, false /* bEventsEnabled */);
    }
    return iRes;
}

//
// Copies a class file buffer
//
// Parameters:
//      pDest           [out]   : destination buffer
//      src             [in]    : source buffer
//      funcAllocator   [in]    : a function to use for allocating memory for pDest buffers
//
// Returns
//      MRTE_RESULT_OK              : success
//      MRTE_ERROR_OUT_OF_MEMORY    : not enough memory to complete the operation
//
TResult CJavaInstrumentorManager::CopyClassFile(SClassFile *pDest, 
                                                const SClassFile& src,
                                                TMemoryAllocatorFunc funcAllocator)
{
    pDest->uiSize = src.uiSize;
    pDest->pClassFile = (unsigned char *)funcAllocator(src.uiSize);
    if (NULL == pDest->pClassFile)
    {
        return MRTE_ERROR_OUT_OF_MEMORY;
    }
    memcpy(pDest->pClassFile, src.pClassFile, src.uiSize);
    return MRTE_RESULT_OK;
}

TResult CJavaInstrumentorManager::SetEventGroupFilter(MPI::EEventGroup group, 
                                                      MPI::IEventFilter &filter)
{
    // Check that the filter does not conflict with other registered events
    if (MPI::EG_NONE != m_InstrumentationEvent.eventGroup)
    {
        if (m_InstrumentationEvent.eventGroup != group)
        {
            // The client has already registered to an event which requires different
            // instrumentation
            return MRTE_ERROR_CONFLICT;
        }
    }
    else
    {
        m_InstrumentationEvent.eventGroup = group;
    }

    // Remember the filter.
    m_InstrumentationEvent.pEventFilter = &filter;

    // Selective events (i.e., events with filters) are always generated using instrumentation
    // so we can disable "tentative instrumentation"
    if (m_bIsTentativeInstrumentation)
    {
        m_bIsTentativeInstrumentation = false;
        m_pEventManager->UpdateTentativeInstrumentation(true);
    }

    return MRTE_RESULT_OK;
}

void CJavaInstrumentorManager::DumpClass(const char *szClassName, 
                                         SClassFile &classFile)
{
    if (NULL == szClassName)
    {
        return;
    }

    const char DUMP_DIR[] = "c:\\temp\\martini_class_dump\\";

    MCString className;
    className.Set(szClassName);
    className.ReplaceAll("/", ".");

    MCString dumpFileName;
    dumpFileName.Append(DUMP_DIR);
    dumpFileName.Append(className.Get());
    dumpFileName.Append(".class");
    
    FILE *pDumpFile = fopen(dumpFileName.Get(), "wb");
    if (!pDumpFile)
    {
        // Can't open file for writing
        return;
    }

    fwrite(classFile.pClassFile, 1, classFile.uiSize, pDumpFile);
    fclose(pDumpFile);
}

TResult CJavaInstrumentorManager::SetEventStreamStatus(bool enabled)
{
    if (NULL == m_pAdaptor)
    {
        MARTINI_INFORMATIVE("CJavaInstrumentorManager", 0, false, 
            "[SetEventStreamStatus] no instrumentation adaptor available. "
            "Event stream status cannot be changed");
        return MRTE_ERROR_NULL_PTR;
    }

    // SetEventsStatus is not an official member of the IInstrumentationAdaptor interface.
    // Therefore, a down-cast is used.
    CBaseAdaptorDelegate *pBaseDelegate = (CBaseAdaptorDelegate*)m_pAdaptor;
    return pBaseDelegate->SetEventsStatus(enabled);
}

TResult 
CJavaInstrumentorManager::RedefineMultipleClasses(TJvmtiClassDefVector &vecClassesToRedefine,
                                                  bool isBlocking)
{
    if (!g_pClassRedefThread)
    {
        return MRTE_ERROR_FAIL;
    }

    return g_pClassRedefThread->RedefineClasses(vecClassesToRedefine, isBlocking);
}

/**
 * The result of this function is placed into pClassRedefQueue, which contains 'jvmtiClassDefinition's
 * */
TResult
CJavaInstrumentorManager::InstrumentAndQueueForRedef(TJvmtiClassDefVector *pClassRedefQueue,
                                                     MPI::TId classId, 
                                                     MPI::EEventGroup group, 
                                                     bool enabled)
{
    SClassFile classToReturnToJVM;
    jclass classJniGlobalRef;
    TResult iRes = ApplyInstrForClass(&classToReturnToJVM, &classJniGlobalRef, classId, group, 
        enabled);
    if (MRTE_RESULT_OK == iRes)
    {
        jvmtiClassDefinition classDef;
        classDef.klass = classJniGlobalRef;
        classDef.class_byte_count = classToReturnToJVM.uiSize;
        classDef.class_bytes = classToReturnToJVM.pClassFile;
        assert(classDef.class_byte_count > 0);
        assert(classDef.class_bytes);
        pClassRedefQueue->Push(classDef);
    }
    return iRes;
}
