/*****************************************************************************
 * Copyright (c) 1997, 2009 Intel Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *
 * $Id$ 
 *****************************************************************************/

#ifndef _MRTE_JAVA_INSTRUMENTOR_MANAGER
#define _MRTE_JAVA_INSTRUMENTOR_MANAGER

#include "MpiAPI.h"
#include "EventManager.h"
#include "DataManager.h"
#include "ParamChecker.h"
#include "MRTEInfrastructureDefinitions.h"
#include "JIMEvents.h"
#include "MList.h"
#include "IdAllocator.h"

namespace Martini { namespace JPI
{
    typedef RTUtil::MVector<jvmtiClassDefinition> TJvmtiClassDefVector;

    struct SJIMEventInfo
    {
        MPI::IEventFilter *pEventFilter;
        MPI::EEventGroup eventGroup;
    };

    // Forward declarations
    class CDataManager;
    class CEventManager;
    class CJavaInstrumentorManager;  

    /**
     * @brief Adaptor Delegate base class
     */
    class CBaseAdaptorDelegate : public Infrastructure::IInstrumentationAdaptor
    {
    public:
        CBaseAdaptorDelegate() : m_pAdaptor(NULL), m_hSetEventsStatus(NULL), 
            m_grProxyClass(NULL) 
        {
        }

        virtual ~CBaseAdaptorDelegate()
        {
            if (m_pAdaptor)
            {
                delete m_pAdaptor;
            }
        }

        TResult SetEventsStatus(bool enabled);

    protected:

        void DeallocateInstrClassInfo(Infrastructure::SAdaptorClassInfo *pInstrClassInfo);
        void DeallocateInstrMethodInfo(Infrastructure::SAdaptorMethodInfo *pInstrMethodInfo);

        // The actual adaptor used by the proxy
        Infrastructure::IInstrumentationAdaptor *m_pAdaptor;

        // Global Reference to the adaptor's proxy Java class
        jclass m_grProxyClass;

        // Handle to the SetEventsStatus method of the adaptor's proxy Java class
        jmethodID m_hSetEventsStatus;
    };

    class CCgAdaptorDelegate : public CBaseAdaptorDelegate
    {
        // IInstrumentationAdaptor methods

        virtual TResult Init(Infrastructure::ILogAssert *pLogger = NULL,
                             MPI::IEventFilter* pFilter = NULL,
                             MPI::BitSet configFlags = MPI::CF_NONE);
        virtual TResult ModifyClass(MPI::TId classId, 
                                    const Infrastructure::SClassFile &classToInstrument,
                                    TMemoryAllocatorFunc funcAllocator,
                                    Infrastructure::SClassFile *pInstrumentedClass,
                                    Infrastructure::IInstrumentationContext *pContext = NULL);
        virtual TResult ModifyByteCodes(MPI::TId classId, 
                                        const Infrastructure::SClassFile &classToInstrument,
                                        TMemoryAllocatorFunc funcAllocator,
                                        Infrastructure::SClassFile *pInstrumentedClass,
                                        Infrastructure::IInstrumentationContext *pContext = NULL);
        virtual void JvmInitDone();

        /**
         * @brief Call-graph filter proxy for CGAdaptor
         *
         * A proxy for the call-graph filter of CGAdaptor. Used for translating method
         * and class names to friendly names, before sending them to the filter defined
         * by the MPI client
         */
        class CCgFilterProxy : public MPI::ICallGraphFilter
        {
        public:
            CCgFilterProxy() : m_pCgFilter(NULL) {}
            TResult Init(MPI::ICallGraphFilter *pCgFilter);
            virtual bool ShouldNotify(MPI::SCallGraphFilterData &methodInfo);

        private:
            MPI::ICallGraphFilter *m_pCgFilter;
        };

    private:
        TResult RegisterNativeCallbacksInWrapper();

        CCgFilterProxy m_CGFilterProxy;
    };

    class CHeapAdaptorDelegate : public CBaseAdaptorDelegate
    {
    public:
        // IInstrumentationAdaptor methods

        virtual TResult Init(Infrastructure::ILogAssert *pLogger = NULL,
                             MPI::IEventFilter* pFilter = NULL,
                             MPI::BitSet configFlags = MPI::CF_NONE);
        virtual TResult ModifyClass(MPI::TId classId, 
                                    const Infrastructure::SClassFile &classToInstrument,
                                    TMemoryAllocatorFunc funcAllocator,
                                    Infrastructure::SClassFile *pInstrumentedClass,
                                    Infrastructure::IInstrumentationContext *pContext = NULL);
        virtual TResult ModifyByteCodes(MPI::TId classId, 
                                        const Infrastructure::SClassFile &classToInstrument,
                                        TMemoryAllocatorFunc funcAllocator,
                                        Infrastructure::SClassFile *pInstrumentedClass,
                                        Infrastructure::IInstrumentationContext *pContext = NULL);
        virtual void JvmInitDone();

    private:
        TResult RegisterNativeCallbacksInWrapper();
    };


    class CThreadAdaptorDelegate : public CBaseAdaptorDelegate
    {
    public:
        // IInstrumentationAdaptor methods

        virtual TResult Init(Infrastructure::ILogAssert *pLogger = NULL,
                             MPI::IEventFilter* pFilter = NULL,
                             MPI::BitSet configFlags = MPI::CF_NONE);
        virtual TResult ModifyClass(MPI::TId classId, 
                                    const Infrastructure::SClassFile &classToInstrument,
                                    TMemoryAllocatorFunc funcAllocator,
                                    Infrastructure::SClassFile *pInstrumentedClass,
                                    Infrastructure::IInstrumentationContext *pContext = NULL);
        virtual TResult ModifyByteCodes(MPI::TId classId, 
                                        const Infrastructure::SClassFile &classToInstrument,
                                        TMemoryAllocatorFunc funcAllocator,
                                        Infrastructure::SClassFile *pInstrumentedClass,
                                        Infrastructure::IInstrumentationContext *pContext = NULL);
        virtual void JvmInitDone();

        /**
         * @brief Thread filter proxy for ThreadAdaptor
         *
         * A proxy for the thread filter of ThreadAdaptor. Used for translating method
         * and class names to friendly names, before sending them to the filter defined
         * by the MPI client
         */
        class CThreadInteractionFilterProxy : public MPI::IThreadInteractionFilter
        {
        public:
            CThreadInteractionFilterProxy() : m_pThreadInteractionFilter(NULL) {}
            TResult Init(MPI::IThreadInteractionFilter *pThreadFilter);
            virtual bool ShouldNotify(MPI::SThreadInteractionFilterData &filterData);

        private:
            MPI::IThreadInteractionFilter *m_pThreadInteractionFilter;
        };


    private:
        TResult RegisterNativeCallbacksInWrapper();

        CThreadInteractionFilterProxy m_ThreadInteractionFilterProxy;
    };


    /**
     * @brief Factory for Instrumentation Adaptors (Singleton)
     */
    class CAdaptorFactory
    {
    public:
        
        /**
         * @return Returns the singleton instance of the factory
         */
        static CAdaptorFactory* Instance() 
        {
            if (NULL == m_pInstance)
            {
                m_pInstance = new CAdaptorFactory();
            }
            return m_pInstance;
        }

        /**
         * @brief Creates an instrumentation adaptor that will apply BCI to generate
         *        events in the specified group
         *
         * @param[in]   group       Event Group
         *
         * @return      Instrumentation Adaptor interface, or NULL if the adaptor cannot
         *              be initialized.
         */
        Infrastructure::IInstrumentationAdaptor* CreateAdaptor(MPI::EEventGroup group);

    private:
        CAdaptorFactory() {}
        
        static CAdaptorFactory *m_pInstance;
    };

    /**
     * @brief Encapsulates a dedicated thread for redefining classes
     *
     * Encapsulates a dedicated thread for redefining classes and provides the means
     * for communicating with this thread.
     */
    class CClassRedefThread
    {
    public:
        
        CClassRedefThread();
        virtual ~CClassRedefThread();

        /**
         * @brief Allocates and starts the thread
         */
        TResult Start();
        
        /**
         * @brief Thread entry point.
         *
         * Thread entry point. Invoked by the thread created by the Start function.
         * This function is not meant to be invoked directly by clients.
         */
        void Run(JNIEnv *pJniEnv, void *pArgs);

        /**
         * @brief Queues classes for redefinition
         * 
         * Queues classes for redefinition, and then signals the dedicated thread to redefine
         * these classes.
         *
         * @param[in] vecClassesToRedefine  a vector of class definitions that needs to be
         *                                  replaced (redefined).
         * @param[in] isBlocking            when @c true, the function call blocks until all
         *                                  classes are redefined. When @c false, the function
         *                                  returns immediately and the redefinition process
         *                                  is done lazily.
         *
         * @retval MRTE_RESULT_OK           classes successfully redefined (or queued, if
         *                                  @c isBlocking is @c false.
         * @retval MRTE_ERROR_FAIL          operation failed.
         */
        TResult RedefineClasses(TJvmtiClassDefVector &vecClassesToRedefine,
                                bool isBlocking);
        
    private:
        // Monitor to notify thread when classes are available for redefinition
        CRawMonitor *m_monClassesAvailableForRedef;

        // Monitor to notify callers when class redefinition process is done
        CRawMonitor *m_monClassRedefDone;

        // Whether a notification is needed when redefinition process is done. Used for 
        // implementing the "synchronized" version of RedefineClasses.
        bool m_bNeedRedefDoneNotification;

        // Class redefinition queue
        TJvmtiClassDefVector m_vecClassesToRedefine;
        
        // Used to solve redefine performance problem, refer bug 291400
        JVMVendor m_JVMVendor;
 
        void GetJvmVendor();

    };

    /**
     * @brief Java Instrumentation Manager
     *
     * Manages all JPI instrumentation activities and communicates with instrumentation
     * adaptors.
     */
    class CJavaInstrumentorManager  
    {
    public:
        CJavaInstrumentorManager(MPI::TId moduleId);
        virtual ~CJavaInstrumentorManager();
        
        TResult RegisterEvent(MPI::IEventObserver& event, bool bIsTentative, bool bIsEnabled);

        TResult Initialize(CDataManager *pDataManager, 
                           CEventManager *pEventManager,
                           Infrastructure::CParamChecker *pParamChecker);

        TResult InitializationPhaseCompleted();

        void InstrumentationEventHandler(struct SEmData *pEvent);
        
        void JVMInitEventHandler();

        TResult DisableInstrumentation();

        TResult ApplyInstrForLoadedClasses(MPI::EEventGroup group, bool enabled);

        void InstrumentationLock()
        {
            m_csInstrumentation->Enter();
        }

        void InstrumentationUnlock()
        {
            m_csInstrumentation->Leave();
        }

        bool GetInstrGeneratedEventsStatus()
        {
            return m_bInstrGeneratedEventsEnabled;
        }

        void SetInstrGeneratedEventsStatus(bool bIsEnabled)
        {
            m_bInstrGeneratedEventsEnabled = bIsEnabled;
        }

        bool IsJVMInitDone()
        {
            return m_bJVMInit;
        }
        
        TResult SetEventGroupFilter(MPI::EEventGroup group, 
                                    MPI::IEventFilter &filter);

        TResult SetEventStreamStatus(bool enabled);

        TResult InstrumentAndQueueForRedef(TJvmtiClassDefVector *pClassRedefQueue,
                                           MPI::TId classId, 
                                           MPI::EEventGroup group, 
                                           bool enabled);
    
        TResult RedefineMultipleClasses(TJvmtiClassDefVector &vecClassesToRedefine,
                                        bool isBlocking);
        
    private:

        TResult ApplyInstrForClass(Infrastructure::SClassFile *pInstrumentedClass,
                                   jclass *pClassJniGlobalRef,
                                   MPI::TId classId, 
                                   MPI::EEventGroup group, 
                                   bool bEnabled);

        TResult ApplyInstrForLoadedClassesImpl(MPI::EEventGroup group, bool enabled);

        TResult SetInstrumentationInfo(MPI::IEventObserver& even);

        TResult RegisterNativeCallbacksInWrapper(CGAdaptor::TMethodEnterHandler methodEnterHandler, 
            CGAdaptor::TMethodLeaveHandler methodLeaveHandler,
            CGAdaptor::TMethodLeaveHandler methodLeaveWithExceptionHandler);
        void SetClassFileNotInstrumented(SJVMData *pStdEvent);
        TResult InitInstrumentation();
        TResult InitInstrumentationAdaptor(MPI::EEventGroup group, 
                                           MPI::IEventFilter *pEventFilter);

        TResult CopyClassFile(Infrastructure::SClassFile *pDest, 
                              const Infrastructure::SClassFile& src,
                              TMemoryAllocatorFunc funcAllocator);

        void DumpClass(const char *szClassName, Infrastructure::SClassFile &classFile);

        IJVM* m_pJVM;

        // module id
        MPI::TId m_ModuleId;

        // Class id allocator
        Infrastructure::CIdAllocator *m_pClassIdAllocator;

        // a pointer to the data manager
        CDataManager *m_pDataManager;

        // a pointer to the event manager
        CEventManager *m_pEventManager;

        // a pointer to the param checker
        Infrastructure::CParamChecker *m_pParamChecker;

        // a pointer to the active instrumentation adaptor. Only one is supported in this slice
        Infrastructure::IInstrumentationAdaptor *m_pAdaptor;

        // Information for the event JIM should generate. Only one is supported in this slice
        SJIMEventInfo m_InstrumentationEvent;

        // flag indicating that the JVM has completed initialization
        bool m_bJVMInit;

        // flag indicating that instrumentation-generated events are enabled for at least one
        // client
        bool m_bInstrGeneratedEventsEnabled;

        // instrumentation might considered tentative when we dont know if all classes can be
        // instrumented.
        bool m_bIsTentativeInstrumentation;

        bool m_bHasFirstClassLoadHookArrived;

        // event objects
        CClassLoadHookEventObserver     *m_pClassLoadHookEvent;
        CJVMInitEventObserver           *m_pJVMInitEvent;
        
        // critical section for synchronizing instrumentation operations
        OSA::IThreadSync *m_csInstrumentation;

    };
}}


#endif // _MRTE_JAVA_INSTRUMENTOR_MANAGER

