/*****************************************************************************
 * Copyright (c) 1997, 2010 Intel Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *
 * $Id$ 
 *****************************************************************************/

#include "ObjectInfoManager.h"
#include "LogAssert.h"
#include "JpiGlobals.h"

#if defined MVS
    #include <unistd.h>
#endif

// The name of the Java class containing the database implementation to store 
// the TPTP ID and analyze its corresponding Object for heap instance data collection.
#define HEAP_OBJ_DATA_CLASS_NAME_JAVA "org.eclipse.tptp.martini.analysis.HeapObjectData"
#define HEAP_OBJ_DATA_CLASS_NAME_NATIVE "org/eclipse/tptp/martini/analysis/HeapObjectData"

using namespace Martini;
using namespace JPI;
using namespace MPI;
using namespace OSA;
using namespace Infrastructure;

CObjectInfoManager::CObjectInfoManager() : m_pcsObjectManager(NULL), m_objIdAllocator(1),
    GC_THRESHOLD(1000000), m_pMapObjectIdToTag(NULL), m_uiCompactionCounter(0),
    m_pMapObjectIdToJniRef(NULL), m_gcCounter(0), m_cleanupThread(0), m_pCleanupMonitor(0), m_pJVM(NULL)
{
	// Default value is to not collect heap instance data.
	heap_obj_data_collection = false;
}

CObjectInfoManager::~CObjectInfoManager()
{
    if (m_pcsObjectManager)
    {
        m_pcsObjectManager->Destroy();
    }

    if (m_pMapObjectIdToTag)
    {
        delete m_pMapObjectIdToTag;
    }

    if (m_pMapObjectIdToJniRef)
    {
        delete m_pMapObjectIdToJniRef;
    }
}

// Find the class ID for the given class name, from jsseFindClassID
jclass objInfoMgrFindClassID(JNIEnv *env, char *clsname) {
    jclass clsid; /* the class ID for the class name */
    char *asciicls; /* the class name in ASCII format */

/* On OS/390, convert the class name at run-time into the ASCII format for the lookup. */
#if defined MVS
    asciicls = (char *) malloc(strlen(clsname) + 1);
    strcpy(asciicls, clsname);
    __etoa(asciicls);
#else
    asciicls = clsname;
#endif

    /* Look up the class ID, after appropriate transformation. */
    clsid = env->FindClass(asciicls);

/* On OS/390, free the ASCII class name before the class ID is returned. */
#if defined MVS
    free(asciicls);
#endif
    return clsid;
}

// Find the method ID given the class ID, method name and method signature. From
// jsseGetMethodID.
jmethodID objInfoMgrGetMethodID(JNIEnv *env, jclass clsid, char *mthd, char *sig) {
    jmethodID mthdid; /* the resulting method ID */
    char *asciimthd; /* the method name in ASCII format */
    char *asciisig; /* the method signature in ASCII format */

    /* On each platform, obtain the ASCII forms of the method name and signature. */
#if defined MVS
    asciimthd = (char *) malloc(strlen(mthd) + 1);
    asciisig = (char *) malloc(strlen(sig) + 1);
    strcpy(asciimthd, mthd);
    strcpy(asciisig, sig);
    __etoa(asciimthd);
    __etoa(asciisig);
#else
    asciimthd = mthd;
    asciisig = sig;
#endif

    /* Look up the method ID. */
    mthdid = env-> GetStaticMethodID(clsid, asciimthd, asciisig);

    /* free memory before the methid ID is returned. */
#if defined MVS
    free(asciimthd);
    free(asciisig);
#endif

    return mthdid;
}

TResult CObjectInfoManager::Init()
{
    if (NULL == m_pMapObjectIdToTag)
    {
        m_pMapObjectIdToTag = new TObjectIdToTagMap();
    }

    m_pMapObjectIdToTag->Init(false, 0, GC_THRESHOLD);

    if (NULL == m_pMapObjectIdToJniRef)
    {
        m_pMapObjectIdToJniRef = new TObjectIdToJniRefMap();
    }

    m_pMapObjectIdToJniRef->Init(false, 0, 1000);

    m_vecTagsToFree.Reserve(GC_THRESHOLD);

    // Initialize Critical Section
    if (!m_pcsObjectManager)
    {
        m_pcsObjectManager = CreateThreadSync();
        if (!m_pcsObjectManager)
        {
            MARTINI_ERROR("CObjectInfoManager", "Error creating critical section");
            return MRTE_ERROR_OSA_FAILURE;
        }
    }

    m_pJVM = CJpiGlobals::Instance()->pJvmInterface;
    
    return MRTE_RESULT_OK;    
}

// Assigns a TId for new Objects.
TResult CObjectInfoManager::NewObject(TId *pNewObjectId, jlong tag)
{
    if (0 == tag)
    {
        return MRTE_ERROR_FAIL;
    }
    *pNewObjectId = m_objIdAllocator.AllocateId();
    GetTagInfo(tag)->jobjectInfo.objectId = *pNewObjectId;
    return MRTE_RESULT_OK;
}

// Saves the Object in a database using the TId to reference it.
TResult CObjectInfoManager::SaveObjectInDb(TId objectId, 
                                           jlong tag, 
                                           jobject object, 
                                           const JNIEnv *pJniEnv)
{
    // NOTE: function is not synchronized. In most cases, a lock on the
    // Object Info Manager must be obtained before calling this function

    m_pMapObjectIdToTag->Set(objectId, tag);

    // Create a Weak Global Reference for the object to be used in subsequent
    // queries for object information

    // NOTE: the current implementation does not free this reference.
    // It is assumed that this reference will be created for only a small number
    // of objects (threads and monitors).
    jobject wgr = const_cast<JNIEnv*>(pJniEnv)->NewWeakGlobalRef(object);
    m_pMapObjectIdToJniRef->Set(objectId, wgr);
    
    return MRTE_RESULT_OK;
}

// Creates a unique tag and TId for the new Object and optionally stores it in the database.
TResult CObjectInfoManager::CreateTag(TId *pId, 
                                      jlong *pTag, 
                                      jobject obj,
                                      const JNIEnv *pJniEnv,
                                      bool bSaveInDb)
{
    IJVM *pJvmInterface = CJpiGlobals::Instance()->pJvmInterface;
    SJvmtiObjectTag *pTagInfo = new SJvmtiObjectTag(m_gcCounter);

    m_pcsObjectManager->Enter();

    jlong newTag = (jlong)pTagInfo;
    TId newId;
    
    TResult res = NewObject(&newId, newTag);
    m_pcsObjectManager->Leave();
    if (MRTE_FAILED(res))
    {
        return res;
    }

    res = pJvmInterface->SetObjectTag(obj, newTag);
    if (MRTE_FAILED(res))
    {
        return res;
    }

    // Stores the Object in a Map using the TPTP ID as the key
	// so it can be accessed when obtaining heap instance data information.
	if(heap_obj_data_collection) {
		AddToInstDataMap(newId, obj, (JNIEnv *)pJniEnv);
	}

    m_pcsObjectManager->Enter();
    if (bSaveInDb)
    {
        SaveObjectInDb(newId, newTag, obj, pJniEnv);
    }

    *pTag = newTag;
    *pId = newId;

    m_pcsObjectManager->Leave();
    return MRTE_RESULT_OK;
}

// Get the Tag for the Object or create it, if it does not exist, and optionally
// save in the database.
TResult CObjectInfoManager::GetOrCreateTag(TId *pId, 
                                           jlong *pTag, 
                                           jobject obj, 
                                           const JNIEnv *pJniEnv,
                                           bool bSaveInDb)
{
    IJVM *pJvmInterface = CJpiGlobals::Instance()->pJvmInterface;

    if (MRTE_FAILED(pJvmInterface->GetObjectTag(pTag, obj)))
    {
        return MRTE_ERROR_FAIL;
    }

    m_pcsObjectManager->Enter();

    if (*pTag != 0)
    {
        *pId = GetTagInfo(*pTag)->jobjectInfo.objectId;
		if (bSaveInDb)
		{
			// Make sure the object (id -> tag), (objRef -> id) mappings are stored in the 
            // database
			if (m_pMapObjectIdToTag->Get(*pId) == 0)
			{
                SaveObjectInDb(*pId, *pTag, obj, pJniEnv);
			}
		}
    }
    else
    {
        // Create a new tag for the object
        TResult res = CreateTag(pId, pTag, obj, pJniEnv, bSaveInDb);
        if (MRTE_FAILED(res))
        {
            m_pcsObjectManager->Leave();
            return res;
        }
    }

    m_pcsObjectManager->Leave();
    return MRTE_RESULT_OK;
}

TResult CObjectInfoManager::NewObjectAndTag(TId *pNewObjectId, jlong *pNewTag)
{
    SJvmtiObjectTag *pTagInfo = new SJvmtiObjectTag(m_gcCounter);
    *pNewTag = (jlong)pTagInfo;
    TResult res = NewObject(pNewObjectId, *pNewTag);
    return res;
}

TId CObjectInfoManager::GetOrUpdateIdFromTag(jlong *pTag)
{
    TId objectId = 0;
    SJvmtiObjectTag *pTagInfo = GetTagInfo(*pTag);
    if (0 == pTagInfo->jobjectInfo.objectId)
    {
        // The tag does not contain an object id. Allocate a new id and store it in the tag
        NewObject(&objectId, *pTag);
        pTagInfo->jobjectInfo.objectId = objectId;
    }
    else
    {
        // The tag contains the object id. Use it as-is
        objectId = pTagInfo->jobjectInfo.objectId;
    }
    return objectId;
}

SObjectInfo* CObjectInfoManager::GetInfo(TId objectId, const JNIEnv *pJniEnv)
{
    jlong tag = GetTagFromId(objectId);
    if (0 == tag)
    {
        // Object does not exists
        return NULL;
    }
    SJvmtiObjectTag *pTagInfo = GetTagInfo(tag);
    if (pTagInfo->jobjectInfo.pObjectInfo != NULL)
    {
        return pTagInfo->jobjectInfo.pObjectInfo;
    }

    // Get the object's information from the VM
    jobject jniRef = GetJniRefFromId(objectId, pJniEnv);
    if (0 == jniRef)
    {
        // Object does not exist
        return NULL;
    }

    SObjectInfo *pObjectInfo = new SObjectInfo();
    memset(pObjectInfo, 0, sizeof(SObjectInfo));

    TResult res = CJpiGlobals::Instance()->pJvmInterface->GetObjectInfo(pObjectInfo, jniRef);
    
    // Cache the information in the object's tag
    pTagInfo->jobjectInfo.pObjectInfo = pObjectInfo;

    return pObjectInfo;
}

// Given a TPTP ID return the Tag
jlong CObjectInfoManager::GetTagFromId(TId objectId)
{
    m_pcsObjectManager->Enter();
    jlong tag = m_pMapObjectIdToTag->Get(objectId);
    m_pcsObjectManager->Leave();

    return tag;
}

// Using the TPTP ID, obtain the Object from the database
jobject CObjectInfoManager::GetJniRefFromId(TId objectId, const JNIEnv *pJniEnv)
{
    m_pcsObjectManager->Enter();
    jobject jniRef = m_pMapObjectIdToJniRef->Get(objectId);
    m_pcsObjectManager->Leave();

    if (jniRef != 0)
    {
        // If the reference is not valid (points to non-existing object), clean up
        // the reference and the database entry
        if (const_cast<JNIEnv*>(pJniEnv)->IsSameObject(jniRef, NULL))
        {
            const_cast<JNIEnv*>(pJniEnv)->DeleteWeakGlobalRef((jweak)jniRef);
            m_pcsObjectManager->Enter();
            m_pMapObjectIdToJniRef->Del(objectId);
            m_pcsObjectManager->Leave();            
            jniRef = 0;
        }
    }

    return jniRef;
}

void CObjectInfoManager::FreeObject(jlong tag)
{
    SJvmtiObjectTag *pTagInfo = GetTagInfo(tag);
    TId objectId = pTagInfo->jobjectInfo.objectId;
    if (pTagInfo->jobjectInfo.pObjectInfo)
    {
        delete pTagInfo->jobjectInfo.pObjectInfo;
    }
    delete pTagInfo;

    m_pcsObjectManager->Enter();
    m_pMapObjectIdToJniRef->Del(objectId);
    m_pMapObjectIdToTag->Del(objectId);
    m_pcsObjectManager->Leave();
}

void CObjectInfoManager::CompactDatabase()
{
/*
    // Allow objects marked as "deleted" to survive 2 GC cycles before freeing their entries.
    // This enables the client a "grace period" of using IMpi::GetObjectInfo() even when the
    // actual object was already freed by the GC.
    if (m_uiCompactionCounter < 2)
    {
        m_uiCompactionCounter++;
        return;
    }
    m_uiCompactionCounter = 0;

    if (m_pMapObjectIdToTag->Count() < GC_THRESHOLD)
    {
        return;
    }

#ifdef _DEBUG
    m_pMapObjectIdToTag->PrintDebugStatistics(stdout);
#endif
    m_pcsObjectManager->Enter();

    // Copy all "living" object entries to a temporary vector
    TTagVector tagVector;
    unsigned int oldMapSize = m_pMapObjectIdToTag->Count();
    tagVector.Reserve(oldMapSize);
    m_pMapObjectIdToTag->Iterate(&ObjectMapIterator, &tagVector);

    // Clean-up the map and put back all living objects
    m_pMapObjectIdToTag->Empty();

    size_t i;
    for (i = 0; i < tagVector.Size(); ++i)
    {
        SJvmtiObjectTag *pTagInfo = (SJvmtiObjectTag*)tagVector.GetAt(i);
        if (pTagInfo->bDeleted)
        {
            if (pTagInfo->jobjectInfo.pObjectInfo)
            {
                delete pTagInfo->jobjectInfo.pObjectInfo;
            }
            delete pTagInfo;
        }
        else
        {
            m_pMapObjectIdToTag->Set(pTagInfo->jobjectInfo.objectId, (jlong)pTagInfo);
        }
    }
    m_pcsObjectManager->Leave();

#ifdef _DEBUG
    m_pMapObjectIdToTag->PrintDebugStatistics(stdout);
#endif
    MARTINI_INFORMATIVE2("CObjectInfoManager", 5, false, "database successfully compacted. "
        "Old size = %d, new size = %d", oldMapSize, m_pMapObjectIdToTag->Count());
*/
}

bool Martini::JPI::ObjectMapIterator(void *pParameter, const TId objectId, jlong tag)
{
    TTagVector *pTagVector = (TTagVector*)pParameter;
    pTagVector->Push(tag);
    return true;
}

// Store the Object in the map database referenced by the TPTP ID.
void CObjectInfoManager::AddToInstDataMap(TId objectId, jobject obj, JNIEnv *pJniEnv) {
	jmethodID mthAddObj = NULL;
	jclass clsBinding = objInfoMgrFindClassID(pJniEnv, HEAP_OBJ_DATA_CLASS_NAME_NATIVE);
	if(clsBinding) {
		mthAddObj = objInfoMgrGetMethodID(pJniEnv, clsBinding, "addToInstDataMap", "(JLjava/lang/Object;)V");
	}

	if(mthAddObj) {
		pJniEnv->CallStaticVoidMethod(clsBinding, mthAddObj, objectId, obj);

		jthrowable jException = pJniEnv->ExceptionOccurred();
		if(jException) {
			pJniEnv->ExceptionDescribe();
		}
	}
	else {
		MARTINI_INFORMATIVE("CObjectInfoManager::AddToInstDataMap", 0, false, "Cannot resolve Java method: addToInstDataMap().");
	}

	// If Garbage Collection is not already listening for cleanup, start a thread to clean up the maps periodically.
	if(m_cleanupThread == NULL)
		startObjectInstCollection();
}

// Using the TPTP ID, retrieve the object from the map database.
jobject CObjectInfoManager::GetObjectFromInstDataMap(TId objectId, JNIEnv *pJniEnv) {
	jobject obj = NULL;
	jmethodID mthGetObj = NULL;
	jclass clsBinding = objInfoMgrFindClassID(pJniEnv, HEAP_OBJ_DATA_CLASS_NAME_NATIVE);

	if(clsBinding) {
		mthGetObj = objInfoMgrGetMethodID(pJniEnv, clsBinding, "getObjectFromInstDataMap", "(J)Ljava/lang/Object;");
	}

	if(mthGetObj) {
		obj = pJniEnv->CallStaticObjectMethod(clsBinding, mthGetObj, objectId);
		jthrowable jException = pJniEnv->ExceptionOccurred();
		if(jException) {
			pJniEnv->ExceptionDescribe();
		}
	}
	else {
		MARTINI_INFORMATIVE("CObjectInfoManager::GetObjectFromInstDataMap", 0, false, "Cannot resolve Java method: getObjectFromInstDataMap().");
	}

	return obj;
}

// Using the TPTP ID, retrieve the object values from the map database.
jobject CObjectInfoManager::GetObjectValuesFromInstDataMap(TId objectId, JNIEnv *pJniEnv) {
	jobject obj = NULL;
	jmethodID mthGetObj = NULL;
	jclass clsBinding = objInfoMgrFindClassID(pJniEnv, HEAP_OBJ_DATA_CLASS_NAME_NATIVE);

	if(clsBinding) {
		mthGetObj = objInfoMgrGetMethodID(pJniEnv, clsBinding, "getObjectValuesFromInstDataMap", "(J)Ljava/lang/String;");
	}

	if(mthGetObj) {
		obj = pJniEnv->CallStaticObjectMethod(clsBinding, mthGetObj, objectId);
		jthrowable jException = pJniEnv->ExceptionOccurred();
		if(jException) {
			pJniEnv->ExceptionDescribe();
		}
	}
	else {
		MARTINI_INFORMATIVE("CObjectInfoManager::GetObjectValuesFromInstDataMap", 0, false, "Cannot resolve Java method: getObjectValuesFromInstDataMap().");
	}

	return obj;
}

// Using the TPTP ID, remove the entry from the map database.
/* Not in use.
void CObjectInfoManager::RemoveObjectFromInstDataMap(TId objectId, JNIEnv *pJniEnv) {
	// Check if heap object data collection is enabled
	if(!heap_obj_data_collection)
		return;

	jmethodID mthRemoveObj = NULL;
	jclass clsBinding = objInfoMgrFindClassID(pJniEnv, HEAP_OBJ_DATA_CLASS_NAME_NATIVE);
	if(clsBinding) {
		mthRemoveObj = objInfoMgrGetMethodID(pJniEnv, clsBinding, "removeObjectFromInstDataMap", "(J)V");
	}

	if(mthRemoveObj) {
		pJniEnv->CallStaticVoidMethod(clsBinding, mthRemoveObj, objectId);

		jthrowable jException = pJniEnv->ExceptionOccurred();
		if(jException) {
			pJniEnv->ExceptionDescribe();
		}
	}
	else {
		MARTINI_INFORMATIVE("CObjectInfoManager::removeObjectFromInstDataMap", 0, false, "Cannot resolve Java method: removeObjectFromInstDataMap().");
	}

}
*/

// Set the flag to determine if heap instance data should be collected.
void CObjectInfoManager::setHeapObjDataCollection(bool heapObjDataCollection) {
	heap_obj_data_collection = heapObjDataCollection;
}

// Check if heap instance data should be collected.
bool CObjectInfoManager::isHeapObjDataCollectionEnabled() {
	return heap_obj_data_collection;
}

// Calls the raw monitor thread to clean up the heap instance data map.
void CObjectInfoManager::RemoveUnusedObjectsFromInstDataMap() {

	if(m_cleanupThread == NULL)
		return;

	if( MRTE_FAILED(m_pCleanupMonitor->Enter()) ) {
		MARTINI_INFORMATIVE("CObjectInfoManager::RemoveUnusedObjectsFromInstDataMap", 0, false, "Could not obtain the monitor, cannot clean up");
		return;
	}
	m_pCleanupMonitor->Notify();
	m_pCleanupMonitor->Exit();

}

// Removes unused Object references from the heap instance data collection map.
void CObjectInfoManager::doRemoveUnusedObjectsFromInstDataMap(JNIEnv *pJniEnv) {
	// Check if the thread for cleanup has been created.
	if(m_cleanupThread == NULL)
		return;

	MARTINI_DEBUG("CObjectInfoManager::doRemoveUnusedObjectsFromInstDataMap()", 5, "enter");

	jmethodID mthRemoveUnusedObj = NULL;
	jclass clsBinding = objInfoMgrFindClassID(pJniEnv, HEAP_OBJ_DATA_CLASS_NAME_NATIVE);
	if(clsBinding) {
		mthRemoveUnusedObj = objInfoMgrGetMethodID(pJniEnv, clsBinding, "removeUnusedObjectsFromInstDataMap", "()V");
	}

	if(mthRemoveUnusedObj) {
		pJniEnv->CallStaticVoidMethod(clsBinding, mthRemoveUnusedObj);

		jthrowable jException = pJniEnv->ExceptionOccurred();
		if(jException) {
			pJniEnv->ExceptionDescribe();
		}
	}
	else {
		MARTINI_INFORMATIVE("CObjectInfoManager::doRemoveUnusedObjectsFromInstDataMap", 0, false, "Cannot resolve Java method: removeUnusedObjectsFromInstDataMap().");
	}

	MARTINI_DEBUG("CObjectInfoManager::doRemoveUnusedObjectsFromInstDataMap()", 5, "exit");
}

// The raw monitor to handle cleanup during garbage collection.
void CObjectInfoManager::runCleanupMonitor(JNIEnv* jni) {
	CObjectInfoManager* self = this;

	if( MRTE_FAILED(self->m_pCleanupMonitor->Enter()) ) {
		MARTINI_INFORMATIVE("CObjectInfoManager::objectInstCleanupThreadFunc", 0, false, "Could not obtain monitor; heap instance data collection cleanupwill be disabled.");
		return;
	}

	// We're ready to start collecting
	do {
		// Wait to be notified a cleanup is needed
		self->m_pCleanupMonitor->Wait(0);
		self->doRemoveUnusedObjectsFromInstDataMap(jni);
	} while( self->heap_obj_data_collection );

	self->m_pCleanupMonitor->Notify();
	self->m_pCleanupMonitor->Exit();
}

// The thread for running the garbage collector cleanup.
extern "C" void JNICALL objectInstCleanupThreadFunc(JNIEnv* jni, void* objectManager) {

	CObjectInfoManager* self = (CObjectInfoManager*)objectManager;

	self->runCleanupMonitor(jni);
}

// Create the raw monitor for cleanup of the heap instance data collection maps during garbage collection.
void CObjectInfoManager::startObjectInstCollection() {

	if(m_cleanupThread != NULL)
		return;
		
	MARTINI_DEBUG("CObjectInfoManager::startObjectInstCollection()", 5, "enter");

	m_cleanupThread = m_pJVM->AllocateJavaThread();
	if( NULL == m_cleanupThread ) {
		MARTINI_INFORMATIVE("CObjectInfoManager::startObjectInstCollection", 0, false, "Failed to create an agent thread; heap instance data collection cleanup will not be enabled.");
		return;
	}

	if( MRTE_FAILED(m_pJVM->CreateRawMonitor("ObjectInstCleanupMonitor", &m_pCleanupMonitor)) ) {
		MARTINI_INFORMATIVE("CObjectInfoManager::startObjectInstCollection", 0, false, "Cannot create raw monitor; heap instance data collection cleanup will not be enabled.");
		return;
	}

	m_pJVM->RunAgentThread( m_cleanupThread, objectInstCleanupThreadFunc, this);
	MARTINI_DEBUG("CObjectInfoManager::startObjectInstCollection()", 5, "exit");

}

// Disables the raw monitor for cleanup of the heap instance data collection maps during garbage collection.
void CObjectInfoManager::stopObjectInstCollection() {

	if(m_cleanupThread == NULL)
		return;

	MARTINI_DEBUG("CObjectInfoManager::stopObjectInstCollection()", 5, "enter");

	// Tell the thread to stop
	m_pCleanupMonitor->Enter();
	m_pCleanupMonitor->Notify();

	// Wait for it to exit
	m_pCleanupMonitor->Wait(0);
	m_pCleanupMonitor->Exit();

	// Cleanup
	delete m_pCleanupMonitor;
	m_pCleanupMonitor = NULL;
	m_cleanupThread = NULL;

	MARTINI_DEBUG("CObjectInfoManager::stopObjectInstCollection()", 5, "exit");
}

