/*****************************************************************************
 * Copyright (c) 1997, 2010 Intel Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *
 * $Id$ 
 *****************************************************************************/

#ifndef _MRTE_OBJECT_INFO_MANAGER_H
#define _MRTE_OBJECT_INFO_MANAGER_H

#include "jni.h"

#include "OSA.h"
#include "MRTEResults.h"
#include "MHash.h"
#include "MpiAPI.h"
#include "IdAllocator.h"
#include "IJVM.h"

namespace Martini { namespace JPI
{
    typedef RTUtil::MHashNoSync<MPI::TId, jlong> TObjectIdToTagMap;
    typedef RTUtil::MHashNoSync<MPI::TId, jobject> TObjectIdToJniRefMap;

    /**
     * @brief Manages object information
     */
    class CObjectInfoManager  
    {
    public:
        CObjectInfoManager();
	    virtual ~CObjectInfoManager();

        /**
         * @brief Initializes the Object Information Manager
         *
         * @retval MRTE_RESULT_OK           : success
         * @retval MRTE_ERROR_OSA_FAILURE   : error allocating required Operating System
         *                                    objects
         */
        TResult Init();

        /**
         * @brief Marks an object as "deleted". May also free the memory associated with the
         * object
         *
         * Marks the object as "deleted". The object id will still be valid for a short time
         * after, and may be deleted if there is a need to free memory.
         *
         * @param[in]   tag : the object's tag
         */
        void FreeObject(jlong tag);

        /**
         * @brief Create new tag and object id for the given object
         *
         * @param[out]  pId         : MPI id allocated for the object
         * @param[out]  pTag        : the new tag created for the object
         * @param[in]   obj         : local JNI reference of the object to tag. Must be valid
         * @param[in]   pJniEnv     : JNI environment of the calling thread
         * @param[in]   bSaveInDb   : whether to save the following mappings in the
         *                            Object Info Manager's internal databases:
         *                            id -> tag , id -> JNI ref
         *
         * @retval MRTE_RESULT_OK       : success
         * @retval MRTE_ERROR_FAIL      : failure. pId and pTag are not changed.
         */
        TResult CreateTag(MPI::TId *pId, 
                          jlong *pTag, 
                          jobject obj, 
                          const JNIEnv *pJniEnv,
                          bool bSaveInDb);

        /**
         * @brief Returns the object tag and MPI id. Create a new tag and id if the object is
         * not already tagged
         *
         * @param[out]  pId     : MPI id of the object
         * @param[out]  pTag    : the object's tag
         * @param[in]   obj     : local JNI reference of the object to tag. Must be valid
         * @param[in]   pJniEnv : JNI environment of the calling thread
         * @param[in]   bSaveInDb   : whether to save the following mappings in the
         *                            Object Info Manager's internal databases:
         *                            id -> tag , id -> JNI ref
         *
         * @retval MRTE_RESULT_OK       : success
         * @retval MRTE_ERROR_FAIL      : failure. pId and pTag are not changed.
         */
        TResult GetOrCreateTag(MPI::TId *pId, 
                               jlong *pTag, 
                               jobject obj, 
                               const JNIEnv *pJniEnv,
                               bool bSaveInDb);

        /**
         * @brief Returns the information associated with the given object's tag
         * 
         * @param[in]   tag : object's tag
         *
         * @return the information associated with the given tag
         */
        SJvmtiObjectTag* GetTagInfo(jlong tag) { return (SJvmtiObjectTag*)tag; }

        /**
         * @brief Returns the tag of the given object id
         * 
         * @param[in]   objectId : object id
         *
         * @return the tag of the object, or 0 if the object is not tagged or does not exist
         */
        jlong GetTagFromId(MPI::TId objectId);

        /**
         * @brief Returns the JNI reference of the given object id
         * 
         * @param[in]   objectId : object id
         * @param[in]   pJniEnv  : JNI environment of the calling thread
         *
         * @return the JNI reference of the object, or 0 if the object does not exist
         */
        jobject GetJniRefFromId(MPI::TId objectId, const JNIEnv *pJniEnv);

        /**
         * @brief Returns information for the specified object
         *
         * @param[in]   objectId : object id
         * @param[in]   pJniEnv  : JNI environment of the calling thread
         *
         * @return the object information, or null if the object does not exist
         */
        MPI::SObjectInfo* GetInfo(MPI::TId objectId, const JNIEnv *pJniEnv);

        /**
         * @brief Introduces a new object with a new tag to the Object Info Manager
         *
         * @param[out] pNewObjectId     : MPI id of the new object
         * @param[out] pNewTag          : The new JVMTI tag of the object
         *
         * @retval MRTE_RESULT_OK       : success
         * @retval MRTE_ERROR_FAIL      : failure
         */
        TResult NewObjectAndTag(MPI::TId *pNewObjectId, jlong *pNewTag);

        /**
         * @brief Returns the object id associated with the given tag, or creates a new
         * object id if the tag does not contain one
         *
         * @param[in,out] pTag  : pointer to a valid (non-zero) tag. Will be updated with the
         *                        object id in case no id is associated with the tag
         *
         * @return the object id, or 0 if the operation failed
         */
        MPI::TId GetOrUpdateIdFromTag(jlong *pTag);

        /**
         * @brief Frees memory for all objects marked as "deleted" and compacts the internal
         * hash table. Called in response to GC End event
         */
        void CompactDatabase();

        /**
         * @brief Increments the GC event counter
         */
        void IncrementGcCounter() { m_gcCounter++; }

        /**
         * @brief Returns the number of GC events
         *
         * @return the number of GC events
         */
        U32 GetGcCount() { return m_gcCounter; }
        
       /** 
        * Used for storing a new Object entry with TId into the data map.
        */
        void AddToInstDataMap(MPI::TId objectId, jobject obj, JNIEnv *pJniEnv);

		/**
		 * Retrieves the Object from the data map with TId.
		 */
        jobject GetObjectFromInstDataMap(MPI::TId objectId, JNIEnv *pJniEnv);

		/**
		 * Analyzes the Object identified by TId and returns the String result.
		 */
        jobject GetObjectValuesFromInstDataMap(MPI::TId objectId, JNIEnv *pJniEnv);

		/**
		 * Removes the Object identified by TId from the data map.
		 */
        //void RemoveObjectFromInstDataMap(MPI::TId objectId, JNIEnv *pJniEnv);

		/**
		 * Sets the flag to indicate instance data collection should be collected.
		 */
		void setHeapObjDataCollection(bool heapObjDataCollection);

		/**
		 * Checks the flag to indicate if instance data should be collected.
		 */
		bool isHeapObjDataCollectionEnabled();

		/**
		 * Cleans up the garbage collected objects from the heap instance data map.
		 */
		void RemoveUnusedObjectsFromInstDataMap();
		
		/**
		 * The raw montior that executes during garbage collection.
		 */
		void runCleanupMonitor(JNIEnv* jni);

    private:
        
        /**
         * @brief Introduces a new object to the Object Info Manager
         *
         * @param[out] pNewObjectId     : MPI id of the new object
         * @param[in]  tag              : JVMTI tag of the object (0 - object is not tagged)
         *
         * @retval MRTE_RESULT_OK       : success
         * @retval MRTE_ERROR_FAIL      : failure
         */
        TResult NewObject(MPI::TId *pNewObjectId, jlong tag);

        /**
         * @brief Saves an object in the Object Info Manager, allowing future
         * queries of object information by using its MPI object id
         *
         * This method stores the following mappings in the Object Info Manager:
         * - A mapping of the object id to its JVMTI tag
         * - a mapping of the object id to a JNI weak global refernce of the object.
         * Note that the method creates a JNI weak global reference for the object
         */
        TResult SaveObjectInDb(MPI::TId objectId, 
                               jlong tag,
                               jobject object,
                               const JNIEnv *pJniEnv);

		/**
		 * These functions manage the object instance data lifecycle and cleanup process.
		 *
		 *  startObjectInstCollection - starts the cleanup thread
		 *  stopObjectInstCollection  - disables instance data collection and stops the cleanup thread
		 *  objectInstCleanupThreadFunc - upon beginning execution, enables object instance data collection
		 *                                cleanup notifications until notified for shutdown
		 *  doRemoveUnusedObjectsFromInstDataMap - does the actual work of calling into HeapObjectData to clean up unused instances
		 */
        void startObjectInstCollection();
        void stopObjectInstCollection();
		void doRemoveUnusedObjectsFromInstDataMap(JNIEnv *pJniEnv);

		jobject m_cleanupThread;
		CRawMonitor* m_pCleanupMonitor;
		IJVM* m_pJVM;

        const U32 GC_THRESHOLD;
        
        // Critical section to guard internal databases
        OSA::IThreadSync *m_pcsObjectManager;
        
        // A map of object id to its JVMTI tag
        TObjectIdToTagMap *m_pMapObjectIdToTag;

        // A map of object id to its JNI global reference
        TObjectIdToJniRefMap *m_pMapObjectIdToJniRef;
        
        // Object id allocator
        Infrastructure::CIdAllocator m_objIdAllocator;

        // Counts the number of calls to CompactDatabase. Used for determining when
        // to do the actual compaction
        U32 m_uiCompactionCounter;

        // A vector of object tags associated with objects freed by the GC
        TTagVector m_vecTagsToFree;

        // A pointer to the next available slot in m_vecTagsToFree
        U32 m_uiNextFreeElement;

        // GC End event counter. Used for calculating object ages
        U32 m_gcCounter;
        
        // The flag to indicate heap instance data collection.
        bool heap_obj_data_collection;
    };

    bool ObjectMapIterator(void *pParameter, const MPI::TId objectId, jlong tag);


}}    

#endif // _MRTE_OBJECT_INFO_MANAGER_H
