/*****************************************************************************
 * Copyright (c) 1997, 2010 Intel Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *
 * $Id$ 
 *****************************************************************************/

#include "ThreadInfoManager.h"
#include "LogAssert.h"
#include "IJVM.h"
#include "EventManager.h"
#include "MVector.h"
#include "JpiGlobals.h"

using namespace Martini::JPI;
using namespace Martini::MPI;
using namespace Martini::Infrastructure;
using namespace Martini::OSA;
using namespace Martini::RTUtil;

typedef MVector<TId> TThreadsVector;
typedef MVector<STlsThreadInfo*> TThreadInfoVector;

//////////////////////////////////////////////////////////////////////
// CThreadInfoManager implementation

CThreadInfoManager::CThreadInfoManager()
: m_ptlsThreadInfo(NULL), m_pcsThreadManager(NULL), m_pEventManager(NULL),
    m_pThreadIdAllocator(NULL), m_pVMInterface(NULL)
{

}

CThreadInfoManager::~CThreadInfoManager()
{
    if (m_ptlsThreadInfo)
    {
        m_ptlsThreadInfo->Destroy();
    }
    if (m_pcsThreadManager)
    {
        m_pcsThreadManager->Destroy();
    }
    //TODO: deallocate all information from internal hash tables, and delete all
    //      global weak thread references stored in the database
}

TResult CThreadInfoManager::Init(CEventManager *pEventManager)
{
    m_pEventManager = pEventManager;
    m_pVMInterface = CJpiGlobals::Instance()->pJvmInterface;

    // Initialize Critical Section
    if (!m_pcsThreadManager)
    {
        m_pcsThreadManager = CreateThreadSync();
        if (!m_pcsThreadManager)
        {
            MARTINI_ERROR("CThreadInfoManager", "Error creating critical section");
            return MRTE_ERROR_OSA_FAILURE;
        }
    }

    // Initialize the Thread Local Storage index for storing thread information
    if (!m_ptlsThreadInfo)
    {
        m_ptlsThreadInfo = CreateThreadLocalStorage();
        if (!m_ptlsThreadInfo)
        {
            MARTINI_ERROR("CThreadInfoManager", "Error creating thread local storage");
            return MRTE_ERROR_OSA_FAILURE;
        }
    }

    // Initialize the Thread Id Allocator
    if (!m_pThreadIdAllocator)
    {
        m_pThreadIdAllocator = new CIdAllocator(1);
    }
    
    // Initialize hash tables
    m_JniEnv2MPIIdMap.Init(false, 0, 20);
    m_ThreadMPIId2InfoMap.Init(false, 0, 20);

    return MRTE_RESULT_OK;
    
}

TResult CThreadInfoManager::ThreadStart(TId *pNewThreadId, 
                                        const JNIEnv *pJniEnv, 
                                        jthread threadRef,
                                        bool bGenerateEvent,
                                        const char *szName, 
                                        const char *szGroupName, 
                                        const char *szParentGroupName)
{
    if (NULL == pJniEnv)
    {
        return MRTE_ERROR_NULL_PTR;
    }

    jthread threadWeakGlobalRef;
    if (NULL == threadRef)
    {
        threadWeakGlobalRef = NULL;
    }
    else
    {
        JNIEnv* pCurrentJniEnv;
        TResult res = m_pVMInterface->GetJNIEnv(&pCurrentJniEnv);
        if (MRTE_FAILED(res))
        {
            MARTINI_INFORMATIVE("CThreadInfoManager", 0, false, 
                "failed to obtain JNI environment of the calling thread");
            return MRTE_ERROR_FAIL;
        }
        threadWeakGlobalRef = pCurrentJniEnv->NewWeakGlobalRef(threadRef);
        if (NULL == threadWeakGlobalRef)
        {
            MARTINI_ERROR("CThreadInfoManager", 
                "failed to create a weak global reference for the thread");
            return MRTE_ERROR_FAIL;
        }
    }

    m_pcsThreadManager->Enter();

    TId threadId = m_JniEnv2MPIIdMap.Get((UIOP)pJniEnv);
    STlsThreadInfo *pTlsThreadInfo;
    bool bNewThreadInfo = false;

    if (0 == threadId)
    {
        // Create new database entries for this thread
        bNewThreadInfo = true;
        threadId = m_pThreadIdAllocator->AllocateId();
        pTlsThreadInfo = new STlsThreadInfo;
        memset(pTlsThreadInfo, 0, sizeof(STlsThreadInfo));
        pTlsThreadInfo->threadId = threadId;
        pTlsThreadInfo->threadInfo.szName = szName;
        pTlsThreadInfo->threadInfo.szGroupName = szGroupName;
        pTlsThreadInfo->threadInfo.szParentGroupName = szParentGroupName;
        pTlsThreadInfo->threadState = STlsThreadInfo::ALIVE;
        pTlsThreadInfo->threadType = STlsThreadInfo::JAVA;
        pTlsThreadInfo->pJniEnv = pJniEnv;
        pTlsThreadInfo->wgrThread = threadWeakGlobalRef;
        m_JniEnv2MPIIdMap.Set((UIOP)pJniEnv, threadId);
        m_ThreadMPIId2InfoMap.Set(threadId, pTlsThreadInfo);
        
        MARTINI_INFORMATIVE2("ThreadInfoManager", 5, false, 
            "JNI env %p is associated with new thread id %u", 
            pJniEnv, threadId);
    }
    else
    {
        // Fetch thread information
        pTlsThreadInfo = m_ThreadMPIId2InfoMap.Get(threadId);
        if (NULL == pTlsThreadInfo)
        {
            MARTINI_ERROR("CThreadInfoManager", "failed to retreive thread information");
            m_pcsThreadManager->Leave();
            return MRTE_ERROR_FAIL;
        }

        // Update thread JNI reference if needed
        if (0 == pTlsThreadInfo->wgrThread)
        {
            pTlsThreadInfo->wgrThread = threadWeakGlobalRef;
        }

        // Update thread information if needed
        if (NULL == pTlsThreadInfo->threadInfo.szName && szName != NULL)
        {
            bNewThreadInfo = true;
            pTlsThreadInfo->threadInfo.szName = szName;
            pTlsThreadInfo->threadInfo.szGroupName = szGroupName;
            pTlsThreadInfo->threadInfo.szParentGroupName = szParentGroupName;
        }
    }

    *pNewThreadId = pTlsThreadInfo->threadId;

    // Generate the Thread Start event if needed. The event will be generated only if it
    // was not generated before and if thread information is available.
    // Note that the internal lock is being released before notifying Event Manager
    m_pcsThreadManager->Leave();

    if (bGenerateEvent && bNewThreadInfo)
    {
        NotifyThreadStartEvent(pTlsThreadInfo);
    }
    
    return MRTE_RESULT_OK;

}


TResult CThreadInfoManager::UpdateCurrentThreadTls(TId threadId)
{
    STlsThreadInfo *pTlsThreadInfo = m_ThreadMPIId2InfoMap.Get(threadId);
    if (NULL == pTlsThreadInfo)
    {
        return MRTE_ERROR_FAIL;
    }
    
    m_ptlsThreadInfo->SetValue(pTlsThreadInfo);
    return MRTE_RESULT_OK;
}

TResult CThreadInfoManager::ThreadEnd(const JNIEnv *pJniEnv)
{
    if (NULL == pJniEnv)
    {
        return MRTE_ERROR_NULL_PTR;
    }

    TId threadId = m_JniEnv2MPIIdMap.Get((UIOP)pJniEnv);
    if (0 == threadId)
    {
        MARTINI_INFORMATIVE("CThreadInfoManager", 0, false, 
            "failed to convert an existing JNI pointer to thread id");
        return MRTE_ERROR_FAIL;
    }

    // Invalidate all information related to the given JNI pointer. This allows the caller
    // to reuse this pointer for new threads.
    m_pcsThreadManager->Enter();

    m_JniEnv2MPIIdMap.Del((UIOP)pJniEnv);
    STlsThreadInfo *pTlsThreadInfo = m_ThreadMPIId2InfoMap.Get(threadId);
    if (NULL != pTlsThreadInfo)
    {
        pTlsThreadInfo->threadState = STlsThreadInfo::TERMINATED;
    }

    m_pcsThreadManager->Leave();

    // Delete information from the TLS
    m_ptlsThreadInfo->SetValue(NULL);

    return MRTE_RESULT_OK;

}

TId CThreadInfoManager::GetThreadIdAndUpdateTls(const JNIEnv *pJniEnv, jthread thread)
{
    TResult iRes;
    TId threadId = GetThreadId(pJniEnv);
    if (0 == threadId)
    {
        // Get thread information (if possible)
        SThreadInfo threadInfo;
        memset(&threadInfo, 0, sizeof(SThreadInfo));
        if (thread != NULL)
        {
            m_pVMInterface->GetThreadInfo(&threadInfo, thread);
        }
        iRes = ThreadStart(&threadId, pJniEnv, thread, true, threadInfo.szName, 
            threadInfo.szGroupName, threadInfo.szParentGroupName);
        MARTINI_ASSERT("CEventManager", MRTE_SUCCEEDED(iRes), "");
    }
    iRes = UpdateCurrentThreadTls(threadId);
    MARTINI_ASSERT("CEventManager", MRTE_SUCCEEDED(iRes), "");
    return threadId;
}

TResult CThreadInfoManager::GetThreadInfo(SThreadInfo *pThreadInfo, TId threadId)
{
    if (NULL == pThreadInfo)
    {
        return MRTE_ERROR_NULL_PTR;
    }

    m_pcsThreadManager->Enter();

    STlsThreadInfo *pTlsThreadInfo = m_ThreadMPIId2InfoMap.Get(threadId);
    if (NULL == pTlsThreadInfo)
    {
        m_pcsThreadManager->Leave();
        return MRTE_ERROR_FAIL;
    }

    if (NULL == pTlsThreadInfo->threadInfo.szName)
    {
        m_pcsThreadManager->Leave();
        MARTINI_INFORMATIVE("ThreadInfoManager", 5, false, 
            "Information for a thread was requested before its Thread Start event was sent");
        return MRTE_ERROR_FAIL;
    }
    pThreadInfo->szName = pTlsThreadInfo->threadInfo.szName;
    pThreadInfo->szGroupName = pTlsThreadInfo->threadInfo.szGroupName;
    pThreadInfo->szParentGroupName = pTlsThreadInfo->threadInfo.szParentGroupName;

    m_pcsThreadManager->Leave();
    
    return MRTE_RESULT_OK;
}

void CThreadInfoManager::SetThreadType(const JNIEnv *pJniEnv, 
                                       STlsThreadInfo::EThreadType threadType)
{
    m_pcsThreadManager->Enter();
    TId threadId = GetThreadId(pJniEnv);
    if (0 != threadId)
    {
        STlsThreadInfo *pTlsThreadInfo = m_ThreadMPIId2InfoMap.Get(threadId);
        if (NULL != pTlsThreadInfo)
        {
            pTlsThreadInfo->threadType = threadType;
        }
        else
        {
            MARTINI_INFORMATIVE("ThreadInfoManager", 0, false, 
                "failed to set thread type. Thread information is not available");
        }
    }
    else
    {
        MARTINI_INFORMATIVE("ThreadInfoManager", 0, false, 
            "failed to set thread type. Unknown JNI environment");
    }
    m_pcsThreadManager->Leave();
}

bool ThreadInfoHashIterationCallback(void *pParameter, 
                                     const TId threadId,
                                     STlsThreadInfo *pTlsThreadInfo)
{
    TThreadsVector *pThreadsVector = (TThreadsVector*)pParameter;
    if (NULL != pTlsThreadInfo)
    {
        pThreadsVector->Push(threadId);
    }
    return true;
}

bool GetInfoForAllThreads(void *pParameter, 
                          const TId threadId,
                          STlsThreadInfo *pTlsThreadInfo)
{
    TThreadInfoVector *pThreadsVector = (TThreadInfoVector*)pParameter;
    if (NULL != pTlsThreadInfo)
    {
        pThreadsVector->Push(pTlsThreadInfo);
    }
    return true;
}

void CThreadInfoManager::UpdateInfo()
{
    TThreadsVector threadsVector;
    m_pcsThreadManager->Enter();
    m_ThreadMPIId2InfoMap.Iterate(ThreadInfoHashIterationCallback, &threadsVector);
    for (size_t i = 0; i < threadsVector.Size(); ++i)
    {
        TId threadId = threadsVector.GetAt(i);
        STlsThreadInfo *pTlsThreadInfo = m_ThreadMPIId2InfoMap.Get(threadId);
        if (NULL != pTlsThreadInfo && NULL == pTlsThreadInfo->threadInfo.szName)
        {
            SThreadInfo threadInfo;
            TResult iRes = m_pVMInterface->GetThreadInfo(&threadInfo, 
                pTlsThreadInfo->wgrThread);
            if (MRTE_SUCCEEDED(iRes))
            {
                pTlsThreadInfo->threadInfo.szName = threadInfo.szName;
                pTlsThreadInfo->threadInfo.szGroupName = threadInfo.szGroupName;
                pTlsThreadInfo->threadInfo.szParentGroupName = threadInfo.szParentGroupName;
                // Generate EV_THREAD_START event (again) so that the profiler can get
                // the thread information. The lock is released during the callback.
                m_pcsThreadManager->Leave();
                NotifyThreadStartEvent(pTlsThreadInfo);
                m_pcsThreadManager->Enter();
            }
        }
    }
    
    m_pcsThreadManager->Leave();
}

void CThreadInfoManager::VmShutDown()
{
    TThreadsVector threadsVector;
    m_pcsThreadManager->Enter();
    m_ThreadMPIId2InfoMap.Iterate(ThreadInfoHashIterationCallback, &threadsVector);
    for (size_t i = 0; i < threadsVector.Size(); ++i)
    {
        TId threadId = threadsVector.GetAt(i);
        STlsThreadInfo *pTlsThreadInfo = m_ThreadMPIId2InfoMap.Get(threadId);
        if (NULL != pTlsThreadInfo)
        {
            if (STlsThreadInfo::ALIVE == pTlsThreadInfo->threadState)
            {
                // Generate EV_THREAD_END event for this thread. 
                // The lock is released during the callback
                m_pcsThreadManager->Leave();
                NotifyThreadEndEvent(pTlsThreadInfo);
                m_pcsThreadManager->Enter();
            }
        }
    }
    
    m_pcsThreadManager->Leave();
}

void CThreadInfoManager::NotifyThreadStartEvent(const STlsThreadInfo *pTlsThreadInfo)
{
    SJVMData data;
    SEmData mpiData;
    mpiData.u.pJVMData = &data;
    mpiData.dataType = MPI_VM;
    data.dataType = DT_THREAD_START;
    data.u.threadStart.pThreadEnv = pTlsThreadInfo->pJniEnv;
    data.u.threadStart.szName = pTlsThreadInfo->threadInfo.szName;
    data.u.threadStart.szGroupName = pTlsThreadInfo->threadInfo.szGroupName;
    data.u.threadStart.szParentGroupName = pTlsThreadInfo->threadInfo.szParentGroupName;
    data.u.threadStart.threadLocalRef = pTlsThreadInfo->wgrThread;
    m_pEventManager->NotifyMpiEvent(EV_THREAD_START, &mpiData);
}

void CThreadInfoManager::NotifyThreadEndEvent(const STlsThreadInfo *pTlsThreadInfo)
{
    SJVMData data;
    SEmData mpiData;
    mpiData.u.pJVMData = &data;
    mpiData.dataType = MPI_VM;
    data.dataType = DT_THREAD_END;
    data.u.threadEnd.pThreadEnv = pTlsThreadInfo->pJniEnv;
    data.u.threadEnd.threadLocalRef = pTlsThreadInfo->wgrThread;
    m_pEventManager->NotifyMpiEvent(EV_THREAD_END, &mpiData);
}

TResult CThreadInfoManager::SuspendAllThreads()
{
    TThreadsVector threadsVector;

	MARTINI_INFORMATIVE("ThreadInfoManager", 5, false,  "[SuspendAllThreads] Suspending all threads");

    m_pcsThreadManager->Enter();

    m_ThreadMPIId2InfoMap.Iterate(ThreadInfoHashIterationCallback, &threadsVector);
    bool bError = false;
    for (size_t i = 0; i < threadsVector.Size(); ++i)
    {
        TId threadId = threadsVector.GetAt(i);
        STlsThreadInfo *pTlsThreadInfo = m_ThreadMPIId2InfoMap.Get(threadId);
        if (NULL != pTlsThreadInfo)
        {
            if (STlsThreadInfo::ALIVE == pTlsThreadInfo->threadState
                && STlsThreadInfo::JAVA == pTlsThreadInfo->threadType)
            {
                TResult iRes = m_pVMInterface->SuspendThread(pTlsThreadInfo->wgrThread);
                if (MRTE_SUCCEEDED(iRes))
                {
                    pTlsThreadInfo->threadState = STlsThreadInfo::SUSPENDED;
                }
                else
                {
                    bError = true;
                    break;
                }
            }
        }
    }
    
    m_pcsThreadManager->Leave();

    if (bError)
    {
        return MRTE_ERROR_FAIL;
    }
    return MRTE_RESULT_OK;
}

TResult CThreadInfoManager::ResumeAllThreads()
{
    TThreadsVector threadsVector;

	MARTINI_INFORMATIVE("ThreadInfoManager", 5, false,  "[ResumeAllThreads] Resuming all threads");

    m_pcsThreadManager->Enter();

    m_ThreadMPIId2InfoMap.Iterate(ThreadInfoHashIterationCallback, &threadsVector);
    bool bError = false;
    for (size_t i = 0; i < threadsVector.Size(); ++i)
    {
        TId threadId = threadsVector.GetAt(i);
        STlsThreadInfo *pTlsThreadInfo = m_ThreadMPIId2InfoMap.Get(threadId);
        if (NULL != pTlsThreadInfo)
        {
            if (STlsThreadInfo::SUSPENDED == pTlsThreadInfo->threadState)
            {
                TResult iRes = m_pVMInterface->ResumeThread(pTlsThreadInfo->wgrThread);
                if (MRTE_SUCCEEDED(iRes))
                {
                    pTlsThreadInfo->threadState = STlsThreadInfo::ALIVE;
                }
                else
                {
                    bError = true;
                    break;
                }
            }
        }
    }
    
    m_pcsThreadManager->Leave();

    if (bError)
    {
        return MRTE_ERROR_FAIL;
    }
    return MRTE_RESULT_OK;
}

STlsThreadInfo::EThreadType CThreadInfoManager::GetThreadType(const JNIEnv *pJniEnv)
{
    STlsThreadInfo::EThreadType res = STlsThreadInfo::UNKNOWN;

    TId threadId = GetThreadId(pJniEnv);
    if (0 != threadId)
    {
        STlsThreadInfo *pTlsThreadInfo = m_ThreadMPIId2InfoMap.Get(threadId);
        if (NULL != pTlsThreadInfo)
        {
            res = pTlsThreadInfo->threadType;
        }
    }

    return res;
}

STlsThreadInfo::EThreadType CThreadInfoManager::GetThreadType(TId threadId)
{
    STlsThreadInfo::EThreadType res = STlsThreadInfo::UNKNOWN;
    STlsThreadInfo *pTlsThreadInfo = m_ThreadMPIId2InfoMap.Get(threadId);
    if (NULL != pTlsThreadInfo)
    {
        res = pTlsThreadInfo->threadType;
    }
    return res;
}

jthread CThreadInfoManager::GetThreadObject(const JNIEnv *pJniEnv)
{
    jthread res = NULL;

    TId threadId = GetThreadId(pJniEnv);
    if (0 != threadId)
    {
        STlsThreadInfo *pTlsThreadInfo = m_ThreadMPIId2InfoMap.Get(threadId);
        if (NULL != pTlsThreadInfo)
        {
            res = pTlsThreadInfo->wgrThread;
        }
    }

    return res;
}

TId CThreadInfoManager::GetThreadId(jthread thread)
{
    TThreadInfoVector threadInfoVec;
    JNIEnv *pJniEnv;
    TResult res = m_pVMInterface->GetJNIEnv(&pJniEnv);
    if (MRTE_FAILED(res))
    {
        return 0;
    }
    m_ThreadMPIId2InfoMap.Iterate(GetInfoForAllThreads, &threadInfoVec);
    size_t i;
    TId threadId = 0;
    for (i = 0; i < threadInfoVec.Size(); ++i)
    {
        STlsThreadInfo *pInfo = threadInfoVec.GetAt(i);
        if (pJniEnv->IsSameObject(thread, pInfo->wgrThread))
        {
            threadId = pInfo->threadId;
            break;
        }
    }
    return threadId;
}

TResult CThreadInfoManager::GetStack(jvmtiFrameInfo *pFrameBuffer, 
                                     jint *pFrameCount, 
                                     TId threadId, 
                                     jint maxFrameCount)
{
    jthread thread = NULL;
    if (threadId != 0)
    {
        STlsThreadInfo *pTlsInfo = m_ThreadMPIId2InfoMap.Get(threadId);
        if (NULL == pTlsInfo)
        {
             MARTINI_INFORMATIVE1("ThreadInfoManager", 5, false, 
                "[GetStack] unknown thread id %u", threadId);
           return MRTE_ERROR_FAIL;
        }
        thread = pTlsInfo->wgrThread;
    }
    return m_pVMInterface->GetStackTrace(pFrameBuffer, pFrameCount, thread, 0, maxFrameCount);
}

TResult CThreadInfoManager::GetState(EThreadState *pState, TId threadId)
{
    jthread thread = NULL;
    if (threadId != 0)
    {
        STlsThreadInfo *pTlsInfo = m_ThreadMPIId2InfoMap.Get(threadId);
        if (NULL == pTlsInfo)
        {
             MARTINI_INFORMATIVE1("ThreadInfoManager", 5, false, 
                "[GetState] unknown thread id %u", threadId);
           return MRTE_ERROR_FAIL;
        }
        thread = pTlsInfo->wgrThread;
    }
    jint jvmState = 0;
    TResult res = m_pVMInterface->GetThreadState(&jvmState, thread);
    if (MRTE_FAILED(res))
    {
        return res;
    }

    *pState = m_pVMInterface->VmThreadStateToMpiThreadState(jvmState);
    
    return MRTE_RESULT_OK;
}

TResult CThreadInfoManager::GetThreadObject(jthread *pThread, TId threadId)
{
    if (NULL == pThread)
    {
        return MRTE_ERROR_NULL_PTR;
    }

    if (threadId != 0)
    {
        STlsThreadInfo *pTlsInfo = m_ThreadMPIId2InfoMap.Get(threadId);
        if (pTlsInfo != NULL)
        {
            *pThread = pTlsInfo->wgrThread;
        }
        else
        {
            return MRTE_ERROR_FAIL;
        }
    }
    else
    {
        *pThread = NULL; // NULL identifies the current thread in JVMTI
    }
    return MRTE_RESULT_OK;
}
