/*****************************************************************************
 * Copyright (c) 1997-2007, Intel Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *
 * $Id$ 
 *****************************************************************************/

#ifndef _MRTE_THREAD_INFO_MANAGER_H
#define _MRTE_THREAD_INFO_MANAGER_H

#include "jni.h"

#include "OSA.h"
#include "MRTEResults.h"
#include "MHash.h"
#include "MpiAPI.h"
#include "IdAllocator.h"
#include "IJVM.h"

namespace Martini { namespace JPI
{
    // forward declarations
    class CEventManager;

    /**
     * @struct STldThreadInfo
     * 
     * @brief Thread-specific information to keep in the Thread Local Storage
     */
    struct STlsThreadInfo
    {
        /**
         * @enum EThreadState
         * Thread state
         */
        enum EThreadState
        {
            ALIVE,                      //!< thread is alive
            SUSPENDED,                  //!< thread suspended by the Threads Manager
            TERMINATED                  //!< "Thread End" event received for this thread
        };

        /**
         * @enum EThreadType
         * Thread type
         */
        enum EThreadType
        {
            JAVA,                       //!< Java thread
            EXTERNAL,                   //!< External (non-Java) thread
            UNKNOWN
        };

        MPI::TId threadId;              //!< thread MPI id
        MPI::SThreadInfo threadInfo;    //!< thread information
        EThreadState threadState;       //!< thread state
        EThreadType threadType;         //!< thread type
        jthread wgrThread;              //!< thread object (Weak Global Reference)
        const JNIEnv *pJniEnv;          //!< thread JNI environment
    };

    /**
     * @typedef TJNIEnvToThreadIdMap
     *
     * @brief Map of a JNI environment pointer to a thread MPI id
     */
    typedef RTUtil::MHashSync<UIOP, MPI::TId> TJNIEnvToThreadIdMap;

    /**
     * @typedef TThreadInfoDB
     *
     * @brief Map of a thread MPI id to its information
     */
    typedef RTUtil::MHashSync<MPI::TId, STlsThreadInfo*> TThreadInfoDB;

    /**
     * @class CThreadInfoManager
     * 
     * Manages thread ids and dispatches Thread Start and Thread End JPI events.
     */
    class CThreadInfoManager  
    {
    public:
	    CThreadInfoManager();
	    virtual ~CThreadInfoManager();

        /**
         * @brief <b> Init </b> Initializes the Thread Manager
         *
         * @param pEventManager [in]: the singleton instance of the Event Manager.
         *                            Used by Thread Manager to notify the Thread Start
         *                            and Thread End events.
         *
         * @returns MRTE_RESULT_OK          : success
         * @returns MRTE_ERROR_OSA_FAILURE  : error allocating required Operating System
         *                                    objects (TLS, Critical Section)
         */
        TResult Init(CEventManager *pEventManager);

        /**
         * @brief <b> ThreadStart </b> Notifies Thread Manager about a new thread
         * 
         * This function notifies Thread Manager about a new thread. The Thread Manager
         * allocates a new MPI ID and optionally generates an EV_THREAD_START event.
         * This function does not modify the TLS of the current thread.
         * 
         * @param pNewThreadId     [out]: the MPI id allocated for the new thread
         * @param pJniEnv           [in]: JNI environment pointer of the started thread
         * @param threadRef         [in]: JNI reference (local or global) to the new thread.
         *                                Set to 0 if the thread object is not known.
         *                              : A weak global JNI reference will be created for the 
         *                                thread when it is stored in the Thread Info Manager
         * @param bGenerateEvent    [in]: whether to generate an EV_THREAD_START event or not.
         * @param szName            [in]: thread name
         * @param szGroupName       [in]: thread group name
         * @param szParentGroupName [in]: name of the thread's group parent
         * 
         * @returns MRTE_RESULT_OK      : success
         * @returns MRTE_ERROR_NULL_PTR : pJniEnv is null
         * @returns MRTE_ERROR_FAIL     : other failure
         */
        TResult ThreadStart(MPI::TId *pNewThreadId,
                            const JNIEnv *pJniEnv, 
                            jthread threadRef,
                            bool bGenerateEvent,
                            const char *szName, 
                            const char *szGroupName, 
                            const char *szParentGroupName);

        /**
         * @brief <b> UpdateCurrentThreadTls </b> Stores thread information in the current
         * thread's TLS.
         *
         * @param threadId  [in]: identifies the thread information to store in the TLS
         *
         * @returns MRTE_RESULT_OK      : success
         * @returns MRTE_ERROR_FAIL     : failure. Most likely threadId cannot be found
         *                                in the Thread Manager internal databases.
         */
        TResult UpdateCurrentThreadTls(MPI::TId threadId);

        /**
         * @brief <b> ThreadEnd </b> Notifies Thread Manager that a thread has ended
         * 
         * This function notifies Thread Manager that a thread has ended.
         * The Thread Manager generates an EV_THREAD_END event for the thread and then
         * invalidates all information associated with the thread's JNI environment, as well
         * as the information stored in the TLS of the current thread.
         * A subsequent call to ThreadStart with the same JNI environment will cause the
         * Thread Manager to allocate a new ID for the thread and to generate an 
         * EV_THREAD_START event. This behavior aligns with the JVM behavior of reusing 
         * Operating System threads for new Java threads.
         *
         * This function must be called from the dying thread.
         * 
         * @param pJniEnv           [in]: JNI environment pointer of the dying thread
         * 
         * @returns MRTE_RESULT_OK      : success
         * @returns MRTE_ERROR_NULL_PTR : pJniEnv argument is null
         * @returns MRTE_ERROR_FAIL     : other failure
         */
        TResult ThreadEnd(const JNIEnv *pJniEnv);

        /**
         * @brief <b> GetTlsInfo </b> Returns thread-specific information from the
         * Thread Local Storage (TLS).
         *
         * This function returns information for the current thread from the TLS.
         * For efficency, a pointer to the data is returned, and no validity checks or
         * copy operations are made.
         * 
         * @returns STlsThreadInfo*     : a pointer to the information stored in the TLS.
         *                                the client should use this pointer to query TLS
         *                                data and must never modify the data directly.
         * @returns NULL                : no information was previously stored in the TLS.
         */
        STlsThreadInfo* GetTlsInfo() { return (STlsThreadInfo*)m_ptlsThreadInfo->GetValue(); }

        /**
         * @brief <b> GetThreadId </b> Returns the MPI ID associated with the given JNI pointer
         *
         * @param pJniEnv   [in]: JNI environment pointer
         * 
         * @returns TId: when pJniEnv exists in the Thread Manager databases, returns the 
         *               MPI Id of the thread associated with pJniEnv.
         * @returns 0  : when pJniEnv does not exists in the Thread Manager databases.
         */
        MPI::TId GetThreadId(const JNIEnv *pJniEnv) 
        {
            return m_JniEnv2MPIIdMap.Get((UIOP)pJniEnv);
        }
        
        /**
         * @brief <b> GetThreadId </b> Returns the MPI ID associated with the given jthread
         *
         * @param thread    [in]: thread JNI reference (either local or global)
         * 
         * @returns TId: when jthread exists in the Thread Manager databases, returns the 
         *               MPI Id of the thread
         * @returns 0  : when jthread does not exists in the Thread Manager databases.
         */
        MPI::TId GetThreadId(jthread thread);
        
        /**
         * @brief <b> GetThreadIdAndUpdateTls </b> Returns the MPI Id associated with the
         * given JNI pointer and update the TLS.
         * If the JNI pointer is not known to the Threads Manager, a new thread id will be
         * allocated and the EV_THREAD_START event will be generated for the new thread.
         *
         * Call this function only when GetTlsInfo() returns NULL.
         *
         * @param pJniEnv   [in]: JNI environment pointer
         * @param thread    [in]: (optional) JNI reference of the thread object (if known).
         *                        used to generate thread information in EV_THREAD_START
         * 
         * @returns TId: the MPI Id assigned to the thread associated with pJniEnv.
         */
        MPI::TId GetThreadIdAndUpdateTls(const JNIEnv *pJniEnv, jthread thread = NULL);

        /**
         * @brief <b> GetThreadInfo <b> returns information for a given thread
         * thread.
         *
         * @param pThreadInfo  [out]: thread information. Must be allocated by the caller.
         * @param threadId      [in]: the thread
         *
         * @returns MRTE_RESULT_OK          : success
         * @returns MRTE_ERROR_NULL_PTR     : pThreadInfo is null
         * @returns MRTE_ERROR_FAIL         : failure
         */
        TResult GetThreadInfo(MPI::SThreadInfo *pThreadInfo, MPI::TId threadId);

        /**
         * @brief <b> GetThreadObject </b> returns the thread object of the given JNI 
         * environment.
         * 
         * @param pJniEnv   [in]: thread JNI environment.
         *
         * @return the thread object of the given JNI environment
         */
        jthread GetThreadObject(const JNIEnv *pJniEnv);

        /**
         * @brief Returns the thread object associated with the given thread id
         * 
         * @param pThread[out]  thread object
         * @param threadId[in]  thread id
         *
         * @retval MRTE_RESULT_OK           : success
         * @retval MRTE_ERROR_NULL_PTR      : pThread is null
         * @retval MRTE_ERROR_FAIL          : other (VM-specific) failure
         */
        TResult GetThreadObject(jthread *pThread, MPI::TId threadId);

        /**
         * @brief <b> SetThreadType </b> updates the thread type of the given JNI 
         * environment.
         * 
         * @param pJniEnv       [in]: thread JNI environment.
         * @param threadType    [in]: thread type
         */
        void SetThreadType(const JNIEnv *pJniEnv, STlsThreadInfo::EThreadType threadType);

        /**
         * @brief Returns the thread's type
         *
         * @param[in] pJniEnv   JNI environment to check
         *
         * @return the thread's type. UNKNOWN is returned if the thread does not exist
         */
        STlsThreadInfo::EThreadType GetThreadType(const JNIEnv *pJniEnv);

        /**
         * @brief Returns the thread's type
         *
         * @param[in] threadId  thread id
         *
         * @return the thread's type. UNKNOWN is returned if the thread does not exist
         */
        STlsThreadInfo::EThreadType GetThreadType(MPI::TId threadId);

        /**
         * @brief <b> UpdateInfo </b> updates information for all known threads
         */
        void UpdateInfo();

        /**
         * @brief Notifies the Thread Info Manager that the VM is shutting down
         */
        void VmShutDown();

        /**
         * @brief <b> SuspendAllThreads </b> suspends all Java threads
         *
         * @returns MRTE_RESULT_OK          : success
         * @returns MRTE_ERROR_FAIL         : failure
         */
        TResult SuspendAllThreads();

        /**
         * @brief <b> ResumeAllThreads </b> resumes all Java threads
         *
         * @returns MRTE_RESULT_OK          : success
         * @returns MRTE_ERROR_FAIL         : failure
         */
        TResult ResumeAllThreads();
        
        /**
         * @brief Returns the stack trace for the given thread
         *
         * @param[out] pFrameBuffer     a preallocated array for stack trace information
         * @param[out] pFrameCount      number of frames written to pFrameBuffer
         * @param[in]  threadId         thread for which to return stack trace. 
         *                              0 = current thread
         * @param[in]  maxFrameCount    maximum number of frames to retrieve
         * 
         * @retval MRTE_RESULT_OK       success
         * @retval MRTE_ERROR_FAIL      failure
         */
        TResult GetStack(jvmtiFrameInfo *pFrameBuffer,
                         jint *pFrameCount, 
                         MPI::TId threadId,
                         jint maxFrameCount);

        /**
         * @brief Returns the state of the given thread
         *
         * @param[out] pState           The thread's state
         * @param[in]  threadId         Thread for which to return the state. 
         *                              0 = current thread.
         *
         * @retval MRTE_RESUL_OK        success
         * @retval MRTE_ERROR_NULL_PTR  pState is NULL
         * @retval MRTE_ERROR_FAIL      other (VM-specific) failure
         */
        TResult GetState(MPI::EThreadState *pState,
                         MPI::TId threadId);

    private:
        // Generate EV_THREAD_START event
        void NotifyThreadStartEvent(const STlsThreadInfo *pTlsThreadInfo);

        // Generate EV_THREAD_END event
        void NotifyThreadEndEvent(const STlsThreadInfo *pTlsThreadInfo);

        // Pointer to the Event Manager
        CEventManager *m_pEventManager;

        // Pointer to the VM interface
        IJVM *m_pVMInterface;
        
        // Critical section to guard Thread Manager's internal databases
        OSA::IThreadSync *m_pcsThreadManager;

        // Thread Local Storage for maintaining thread-specific information
        OSA::IThreadLocalStorage *m_ptlsThreadInfo;

        // A synchronized hash table for mapping JNI environments to thread MPI ids
        TJNIEnvToThreadIdMap m_JniEnv2MPIIdMap;

        // A synchronized hash table for mapping thread MPI ids to its information
        TThreadInfoDB m_ThreadMPIId2InfoMap;
        
        // An allocator for thread MPI ids
        Infrastructure::CIdAllocator *m_pThreadIdAllocator;

    };

}}    

#endif // _MRTE_THREAD_INFO_MANAGER_H
