/*****************************************************************************
 * Copyright (c) 1997, 2010, Intel Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *
 * $Id$ 
 *****************************************************************************/
#ifdef EM64T_ARCH
#pragma runtime_checks( "", off )
#endif

#include "jvmpi.h"
#include "jvmti.h"
#include "LibraryLoader.h"
#include "OSA.h"
#include "MRTEResults.h"
#include "JPI.H"
#include <stdio.h>

#define JPI_INITIALIZE "JPI_Initialize"
#define JPI_LIBRARY  "JPI"

using namespace Martini::OSA;

/**
 * The purpose of this function is to parse the ac_home variable the JVMTI profiling options in the jvm parameters.
 * Returns false if not present.
 * szOptions is input (just the profiling options from the command line)
 * ac_home is output.
*/
// Add the function for solve agent controller dependency in bug 226572
static bool ExtractAcHome(char ac_home[], char* szOptions)
{
    const char* ac_home_token = "ac_home=";
    unsigned int ac_home_token_len = strlen(ac_home_token);
    char* str_ac_home_begin = strstr(szOptions, ac_home_token);

    if (NULL != str_ac_home_begin)
    {
        const char cmd_option_separator = ',';
        char* str_ac_home_end = strchr(str_ac_home_begin, cmd_option_separator);
        if (NULL != str_ac_home_end)
        {// extract ac_home path string
            unsigned int ac_home_len = str_ac_home_end - str_ac_home_begin - ac_home_token_len;
            if (ac_home_len+1 > MAX_PATH)
            {
                return false;
            }
            else
            {
                strncpy(ac_home, (str_ac_home_begin + ac_home_token_len), ac_home_len);
                ac_home[ac_home_len] = '\0';  // assure the string is end
            }

            // del the ac_home path info in szOptions
            char *p = str_ac_home_end + 1;
            unsigned int inteval = str_ac_home_end - str_ac_home_begin + 1; // +1 for cmd_option_separator
            while ('\0' != *p)
            {
                *(p-inteval) = *p;
                ++p;
            }
            *(p-inteval) = '\0';
        }
        else
        {// incorrect cmd option
            return false;
        }
    }

    return true;
}

#ifdef MVS

// Bug 317863 - Remove dependency on JAVA_PROFILER_HOME and other env vars for zOS
//  On zOS there is no reliable API to figure out the location of JPIBootLoader,
//  instead we rely on the client to specify a value for jpi_home= in the agent
//  options string. This function parses this option and returns a string with
//  the value.
//
// It is the responsibility of the caller to delete[] the return value.
static const char* ExtractJpiPath( const char* szOptions )
{
	const char* const token = "jpi_home=";
	const char* jpi_home = strstr(szOptions, token);
	if( !jpi_home )
		return NULL;

	const char* jpi_path = jpi_home + strlen(token);
	const char* jpi_options_end = strchr(jpi_path, ';');

	if( !jpi_options_end )
		// szOptions is malformed, but let it be discovered normally
		jpi_options_end = &szOptions[ strlen(szOptions) ];

	const char* jpi_path_end = strchr(jpi_path, ',');

	// Force the semicolon terminator in the following cases:
	// 1. jpi_path_end == NULL, no comma found in the input
	// 2. jpi_path_end lies beyond the semicolon, comma found 
	//    in profiler args, the semicolon is the correct terminator
	if( !jpi_path_end || jpi_path_end > jpi_options_end )
		jpi_path_end = jpi_options_end;

	const int jpi_path_length = jpi_path_end - jpi_path;
	char* path = new char[ jpi_path_length + 1 ];
	strncpy( path, jpi_path, jpi_path_length );
	path[ jpi_path_length ] = '\0';

	return path;
}

#endif

//<summary> This function is called directly by JVMPI/JVMTI as the first entry point to the 
// profiler agent. This function is responsible to load the correct JPI.dll and call its
// initialization function </summary>
//<param name='pVM'> java virtual machine instance </param>
//<param name='szOptions'> JPI and profiler agent options </param>
//<param name='reserved'> reserved </param>
//<returns> Success/Failure indication </returns>
//<remarks>
//<b>Return Values:</b><pre>
//<li>JNI_OK    initialization succeeded </li>
//<li>JNI_ERR   initialization failed </li></pre>
//</remarks>

TResult LoadJPI(JavaVM *pVm, char *szOptions, EJVMInterfaceType interfaceType)
{
    static bool bWasLoaded = false;
    if (bWasLoaded)
    {
        return MRTE_RESULT_OK;
    }
    bWasLoaded = true;

// Bug 317863
#ifdef MVS
    const char* jpi_path = ExtractJpiPath( szOptions );
    if( jpi_path )
    {
	    SetLibraryPath( jpi_path );
	    delete []jpi_path;
    }
#endif

    ILibraryLoader *pLibrary = LoadBistroLibrary(JPI_LIBRARY);

    if (pLibrary == NULL)
    {
//        std::cout << "Could not load " << JPI_LIBRARY << std::endl;
        return MRTE_ERROR_FAIL;
    }

   /* Added code for solving Agent Controller dependency in bug 226572: 
     * When server mode is enabled or controlled, preload agent controller libaries.
     * Otherwise, ACCollector can not be loaded if TPTP agent controller home is not
     * specified using ac_home, or not added ac path into PATH (On windows)|LD_LIBRARY_PATH (On Linux).
     */
    unsigned int option_len = strlen(szOptions);
    char* szNewOptions = new char[option_len + 1];
    if (NULL == szNewOptions)
    {
        return MRTE_ERROR_FAIL;
    }
    strncpy(szNewOptions, szOptions, option_len);
    szNewOptions[option_len] = '\0';

#if defined(__linux__) || defined (_WIN32) || defined(_AIX) || defined(MVS) || defined(__sparc) || defined(_SOLARISX86)
    if (NULL != strstr(szOptions, "server=enabled") || NULL != strstr(szOptions, "server=controlled"))
    {       
        char ac_home[MAX_PATH] = {'\0'};

        if (!ExtractAcHome(ac_home, szNewOptions))
        {
            delete []szNewOptions;
            return MRTE_ERROR_FAIL;
        }
        
        if (MRTE_FAILED(PreLoadAgentControllerLibrary(ac_home)))
        {
            delete []szNewOptions;
            return MRTE_ERROR_FAIL;
        }
    }
#endif

    //JPI_Initialize initFunc = (JPI_Initialize)GetProcAddress(handle, JPI_INITIALIZE);
    TJpiInitialize initFunc = (TJpiInitialize)pLibrary->GetEntry(JPI_INITIALIZE);

    if (initFunc)
    {
        int retVal = (*initFunc)(pVm, szNewOptions, interfaceType);
        delete []szNewOptions;
        if (retVal != 0)
        {
            return MRTE_ERROR_FAIL;
        }
    }
    else 
    {
        delete []szNewOptions;
        return MRTE_ERROR_FAIL;
    }
    pLibrary->Destroy();
    return MRTE_RESULT_OK;
}

extern "C" JNIEXPORT jint JNICALL JVM_OnLoad(JavaVM *pVm, char *szOptions, void *reserved)
{
    //fprintf(stderr, "--->JVM_OnLoad\n");
    TResult mrteRes = LoadJPI(pVm, szOptions, TYPE_JVMPI);
    if (MRTE_FAILED(mrteRes))
    {
        return JNI_ERR;
    }
    //fprintf(stderr, "<---JVM_OnLoad\n");
    return JNI_OK;
}

extern "C" JNIEXPORT jint JNICALL 
Agent_OnLoad(JavaVM *pVm, char *szOptions, void *reserved)
{
    TResult mrteRes = LoadJPI(pVm, szOptions, TYPE_JVMTI);
    if (MRTE_FAILED(mrteRes))
    {
        return JNI_ERR;
    }
    return JNI_OK;
}
