/*****************************************************************************
 * Copyright (c) 1997, 2010 Intel Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *
 * $Id$
 *****************************************************************************/
#ifdef EM64T_ARCH
#pragma runtime_checks( "", off )
#endif

#include "LibraryLoader.h"

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

// Added for bug 241984 and bug 226572
#ifdef __linux__
	#ifndef _GNU_SOURCE
		#define _GNU_SOURCE
	#endif

	#ifndef __USE_GNU
		#define __USE_GNU
	#endif

	#include <link.h>

#include <dirent.h>
#elif defined(_WIN32)
	#include <windows.h>
    #if _MSC_VER >= 1300 // for VC 7.0
    #ifndef _delayimp_h
    extern "C" IMAGE_DOS_HEADER __ImageBase;
    #endif
    #endif
#elif defined(_AIX)
    #include <dirent.h>
    #include <dlfcn.h>
    #include <errno.h>
    #include <sys/ldr.h>
#elif defined(MVS)
    #include <dirent.h>
    #include <dlfcn.h>
#elif defined(__sparc) || defined(_SOLARISX86)
	#include <dirent.h>
    #include <dlfcn.h>
    #include <link.h>
#endif
// Added for bug 241984 and bug 226572

using namespace Martini::OSA;

// Added for bugs 226572, 317444, 317863 to remove PATH depencency on Windows, AIX, MVS
#if defined(_WIN32) || defined(_AIX) || defined(MVS)
	typedef TResult (* LPFNDLLGETENVVAR)(char *, char *, unsigned int, unsigned int *);
	typedef TResult (* LPFNDLLSETENVVAR)(char *, char *);
	typedef ILibraryLoader* (* LPFNDLLCREATELIBLOADER)(const char *, const char *, bool);

	static LPFNDLLGETENVVAR pFuncGetEnvVar = NULL;
	static LPFNDLLSETENVVAR pFuncSetEnvVar = NULL;
	static LPFNDLLCREATELIBLOADER pFuncCreateLibLoader = NULL;
#endif
// Added for bug 226572 to remove PATH depencency on Windows

#ifdef __cplusplus
extern "C" {
#endif


// Added for bug 241984 and bug 226572. Record the default Path of library if JPI_PROFILER_HOME is not used.
// These two variables are presently used by the Linux and Windows implementations ONLY
static char defaultPath[MAX_PATH] = {'\0'};  // Default java profiler lib path
static unsigned int defaultPathSize = 0;  // Default java profiler path len without terminator '\0'

#if !defined(HAVE_RPATH)
// Bug 317444
// Dynamically load the libMartiniOSA.so library and bind the GetEnvironmentVar
// and CreateLibraryLoader entry points. Should really only be called from the defined(_AIX)
// version of GetDefaultLibPath or when global defaultPath has been correctly initialized.
static TResult BootstrapMartiniOSA()
{
#ifdef _WIN32
	const char* osaLibName = "\\MartiniOSA.dll";
#else
	const char* osaLibName = "/libMartiniOSA.so";
#endif
	char osaLibPath[MAX_PATH];
	
	if (MAX_PATH < defaultPathSize + strlen(osaLibName) + 1)
	{
		return MRTE_ERROR_FAIL;
	}
	
	strcpy(osaLibPath, defaultPath);
	if( 0 == defaultPathSize )
		// If we don't know the libpath we are relying on env variables; so 
		// remove the leading '/' and use an ambiguous path, and let the linker 
		// sort it out
		strcat(osaLibPath, osaLibName+1);
	else
		strcat(osaLibPath, osaLibName);
	
	// Load the lib
#ifdef _WIN32
    HMODULE module = ::LoadLibraryEx(osaLibPath, NULL, LOAD_WITH_ALTERED_SEARCH_PATH);
#else
	void* module = dlopen(osaLibPath, RTLD_NOW);
#endif
	if( NULL == module ) 
	{
		return MRTE_ERROR_LIBRARY_FAILURE;
	}

#ifdef _WIN32
	pFuncGetEnvVar = (LPFNDLLGETENVVAR)GetProcAddress(module, "GetEnvironmentVar");
	pFuncSetEnvVar = (LPFNDLLSETENVVAR)GetProcAddress(module, "SetEnvironmentVar");
    pFuncCreateLibLoader = (LPFNDLLCREATELIBLOADER)GetProcAddress(module, "CreateLibraryLoader");
#else
    pFuncGetEnvVar = (LPFNDLLGETENVVAR)dlsym(module, "GetEnvironmentVar");
    pFuncSetEnvVar = (LPFNDLLSETENVVAR)dlsym(module, "SetEnvironmentVar");
	pFuncCreateLibLoader = (LPFNDLLCREATELIBLOADER)dlsym(module, "CreateLibraryLoader");
#endif
	if (NULL != pFuncGetEnvVar && NULL !=pFuncCreateLibLoader && NULL != pFuncSetEnvVar)
	{
		return MRTE_RESULT_OK;
	}
	else
	{
		return MRTE_ERROR_LIBRARY_FAILURE;
	}
	
}
#endif

#if defined(__linux__)
// Get lib path info on Linux platform. This function will be called in GetLibraryPathFromEnv
/** Scans through the library env vars and sets defaultPath and defaultPathSize, and sets them to the directory that contains
 * the JVMTI profiler files.  */
static int GetDefaultLibPath(struct dl_phdr_info *info, size_t size, void *data)
{
    if ('\0' == defaultPath[0])
    {
	    // Because JPIBootLoader is loaded firstly, if other libraries lie in the same folder with it, we should find where it is
        const char *ptr = strstr(info->dlpi_name, "libJPIBootLoader.so");
        // Search libJPIBootLoader.so
        if ( NULL != ptr )
        {
            defaultPathSize = ptr - info->dlpi_name - 1; // move the pointer forward
            strncpy(defaultPath, info->dlpi_name, defaultPathSize);
        }
    }

    return 0;
}

#elif defined(_WIN32)
// Begin _WIN32 code section
// This local function is used to find the current Module info, it is called by function  GetDefaultLibPath
static HMODULE GetCurrentModule()
{
#if _MSC_VER < 1300  // earlier than .NET compiler (VC   6.0)
    MEMORY_BASIC_INFORMATION mbi;
    static int dummy;
    VirtualQuery(&dummy, &mbi, sizeof(mbi));
    return reinterpret_cast<HMODULE>(mbi.AllocationBase);
#else // VC   7.0
    return reinterpret_cast<HMODULE>(&__ImageBase);
#endif
}

// Get lib path info on Windows platform. This function will be called in GetLibraryPathFromEnv
// Only call this function when ('\0' == defaultPath[0]) is true
static int GetDefaultLibPath()
{
    if (0 == GetModuleFileName(GetCurrentModule(), (LPTSTR)defaultPath, MAX_PATH))
    {
        return MRTE_ERROR_FAIL;
    }

    char *pdest = strrchr(defaultPath, DIRECTORY_SEPARATOR);
    if (pdest != NULL)
    {
        *pdest = '\0';
    }
    defaultPathSize = strlen(defaultPath);

    // Resolve GetEnvironmentVar and CreateLibraryLoader entry points
    return BootstrapMartiniOSA();
}
// End of _WIN32 CODE SECTION

#elif defined(_AIX)

// Bug 317444: AIX implementation to determine the directory where libJPIBootLoader.so
// resides. This directory is used as the lib path to load further dependencies of 
// Martini removing the need to configure environment variables.
static int GetDefaultLibPath()
{
	int ret = -1, bufSize = 8192;
	char* loadedLibs = new char[bufSize];
	
	if( 0 == loadedLibs )
		return MRTE_ERROR_OUT_OF_MEMORY;

	// Call loadquery with an increasing buffer size until we succeed
	while( ret < 0 )
	{
		ret = loadquery(L_GETINFO, loadedLibs, bufSize);
		if( ret < 0 )
		{
			if( ENOMEM == errno ) 
			{
				// Double the buffer size and try again
				bufSize *= 2;
				delete []loadedLibs; loadedLibs = new char[bufSize];
				if( 0 == loadedLibs ) 
					return MRTE_ERROR_OUT_OF_MEMORY;
			} 
			else 
			{
				// Some error other than too small a buffer size
				delete[] loadedLibs;
				return MRTE_ERROR_LIBRARY_FAILURE;
			}
		}
	}
	
	// Iterate over each entry looking for JPIBootLoader
	struct ld_info* lib = (struct ld_info*)loadedLibs;
	while(1)
	{
		char* pathname = lib->ldinfo_filename;
		char* ptr = strstr(pathname, "libJPIBootLoader.so");
		if( 0 != ptr )
		{
			defaultPathSize = ptr - pathname - 1;
			strncpy(defaultPath, pathname, defaultPathSize);
			// Now that we've determined the libpath, try and load MartiniOSA
			delete []loadedLibs;
			return BootstrapMartiniOSA();
		}
		
		// Move on to next, or break out of loop if no more elements.
		if( 0 == lib->ldinfo_next )
			break;
		// ldinfo_next is offset to the next ld_info struct
		lib = (struct ld_info*)((char*)lib + lib->ldinfo_next);
	}
	
	delete []loadedLibs;
	return MRTE_ERROR_LIBRARY_FAILURE;
}

#elif defined(__sparc) || defined(_SOLARISX86)

// Bug 319616: Solaris implementation to determine the directory where libJPIBootLoader.so
// has been loaded from.
static int GetDefaultLibPath()
{
	Link_map* lm = NULL;
	int ret = dlinfo(RTLD_SELF, RTLD_DI_LINKMAP, &lm);

	if( ret < 0 )
	{
		return MRTE_ERROR_LIBRARY_FAILURE;
	}

	// Don't search explicitly for libJPIBootLoader.so, because the build
	// is setup such that each .so has its own copy of this function. The
	// result is that when this is called from libJPI.so for example, we 
	// will fail to find libJPIBootLoader.so and no one will be happy.
	//
	// Instead, find the last dir separator and use the string up until 
	// there as the lib path.
	char* endpath = strrchr(lm->l_name, '/');
	if( !endpath )
	{
		return MRTE_ERROR_FAIL;
	}

	defaultPathSize = endpath - lm->l_name;
	strncpy(defaultPath, lm->l_name, defaultPathSize);

	return MRTE_RESULT_OK;
}

#endif

// Added for bug 241984 and bug 226572. Record the default Path of library if JPI_PROFILER_HOME is not used.

/* OS specific wrapper around whether to call GetDefaultLibPath (Win) or dl_iterate_phdtr (Lin)
 * (The purpose is just to ensure that the global defaultPath is set). */
static TResult OsGetDefaultLibPath()
{
    if ('\0' == defaultPath[0])
    {
        #ifdef __linux__
	        dl_iterate_phdr(GetDefaultLibPath, NULL);
        #elif defined(_WIN32) || defined(_AIX) || defined(__sparc) || defined(_SOLARISX86)
	        return (TResult)GetDefaultLibPath();
        #endif
    }

    return MRTE_RESULT_OK;
}

// Wrap up the OS related get environment var code into local function
/** OS specific wrapper:
 * szVarName and uiBufferSize are input, szValueBuffer and puiBuffersize are output vars.
 * Just gets the environment variable of a given name. */
static TResult OsGetEnvironmentVar(char *szVarName, char *szValueBuffer,
        unsigned int uiBufferSize, unsigned int *puiBufferSize)
{
#ifdef MVS
	// MVS doesn't have GetDefaultLibPath; thus it is possible that the environment
	// CAN load MartiniOSA but it has not done so yet. So we try it if it hasn't
	if( NULL == pFuncCreateLibLoader)
		BootstrapMartiniOSA();
#endif

	#if defined(__linux__) || defined(__sparc) || defined(_SOLARISX86)
	    return GetEnvironmentVar(szVarName, szValueBuffer, uiBufferSize, puiBufferSize);
    #elif defined(_WIN32) || defined(_AIX) || defined(MVS)
	    // Added for bugs 226572, 317444, 317863 to remove PATH depencency on Windows, AIX, MVS
	    if (NULL == pFuncGetEnvVar)
	    {// MartiniOSA is not ready
	        return MRTE_ERROR_FAIL;
	    }
	    return (*pFuncGetEnvVar)(szVarName, szValueBuffer, uiBufferSize, puiBufferSize);
    #endif
}

/* Wrapper around the library loading.
 * libname is the library name to load, and libpath is the exact path of the library. */
static ILibraryLoader *OsCreateLibraryLoader(const char *szLibName, const char *szLibPath,
        bool bLoadOnce)
{
#ifdef MVS
	// MVS doesn't have GetDefaultLibPath; thus it is possible that the environment
	// CAN load MartiniOSA but it has not done so yet. So we try it if it hasn't
	if( NULL == pFuncCreateLibLoader) 
		BootstrapMartiniOSA();
#endif

    #if defined(__linux__) || defined(__sparc) || defined(_SOLARISX86)
	    return CreateLibraryLoader(szLibName, szLibPath, bLoadOnce);
    #elif defined(_WIN32) || defined(_AIX) || defined(MVS)
	    // Added for bugs 226572, 317444, 317863 to remove PATH depencency on Windows, AIX, MVS
	    if (NULL == pFuncCreateLibLoader)
	    {// MartiniOSA is not ready
	        return NULL;
	    }
	    return (*pFuncCreateLibLoader)(szLibName, szLibPath, bLoadOnce);
    #endif
}

/**
 * Set the default library load path to szLibPath. This is the alternative to 
 * GetDefaultLibPath, provided mainly for MVS which cannot reliably figure out
 * the location of JPIBootLoader.
 */
TResult SetLibraryPath(const char* szLibPath)
{
	int length = strlen(szLibPath);
	if( length-1 >= MAX_PATH )
		return MRTE_ERROR_FAIL;

	strcpy( defaultPath, szLibPath );
	defaultPathSize = strlen(defaultPath);

#ifndef HAVE_RPATH
	TResult res = BootstrapMartiniOSA();
	if( MRTE_FAILED(res) )
		return res;

	// Need to set JAVA_PROFILER_HOME here so that CJVMTIInterface::SetBootClassPath can 
	// find it later on in the initialization
	return (*pFuncSetEnvVar)(JAVA_PROFILER_HOME_ENV_VAR, defaultPath);
#else
	return SetEnvironmentVar(JAVA_PROFILER_HOME_ENV_VAR, defaultPath);
#endif
}

/** This function:
 * 1. First calls OsGetDefaultLibPath() to ensure that defaultPath is set.
 * 2. Scans through the environment variables looking for BISTRO_INSTALL_DIR or JAVA_PROFILER_HOME
 * 3. Stores the result in szLibPath, if those were found
 * 4. If they were not found, it just stores defaultPath. */
TResult GetLibraryPathFromEnv(char *szLibPath,
                              unsigned int uiLibPathSize,
                              unsigned int *puiActualSize)
{
    static char* EnvVarsToSearch[] = {
        BISTRO_INSTALL_DIR_ENV_VAR,
        JAVA_PROFILER_HOME_ENV_VAR
    };
    static size_t EnvVarsToSearchCount = 2;

#if defined(_WIN32) || defined(__linux__) || defined(_AIX) || defined(__sparc) || defined(_SOLARISX86)
    // Added for bug 241984 and bug 226572.
    if ( MRTE_RESULT_OK != OsGetDefaultLibPath() )
    {
        return MRTE_ERROR_FAIL;
    }
#endif

    TResult iRes = MRTE_RESULT_FALSE;
    for (size_t i = 0; i < EnvVarsToSearchCount; ++i)
    {
        iRes = OsGetEnvironmentVar(EnvVarsToSearch[i], szLibPath, uiLibPathSize, puiActualSize);
        if MRTE_FAILED(iRes)
        {
            return iRes;
        }
        if (MRTE_RESULT_OK == iRes)
        {
            // The environment variable exists. Use its value
            break;
        }
    }

#if defined(_WIN32) || defined(__linux__) || defined(_AIX) || defined(MVS) || defined(__sparc) || defined(_SOLARISX86)
    // Added for bug 241984 and bug 226572. Return default path as libraries loading directory.
    if ( (MRTE_RESULT_FALSE == iRes) && ('\0' != defaultPath[0]) )
    {
        if (uiLibPathSize <= defaultPathSize)
        {
            return MRTE_ERROR_BUFFER_TOO_SHORT;
	}

        *puiActualSize = defaultPathSize + 1;
        strcpy(szLibPath, defaultPath);
        iRes = MRTE_RESULT_OK;
    }
    // Added for bug 241984 and bug 226572. Return default path as libraries loading directory.
#endif

    return iRes;
}

ILibraryLoader *LoadBistroLibrary(const char* loadedDllName, bool bLoadOnce)
{
    if (NULL == loadedDllName)
    {
        return NULL;
    }

    char *szPath = new char[MAX_PATH];
    szPath[0] = 0;
    unsigned int uiPathSize = MAX_PATH;
    TResult res = GetLibraryPathFromEnv(szPath, MAX_PATH, &uiPathSize);
    if (MRTE_ERROR_BUFFER_TOO_SHORT == res)
    {// Resize szLoadedDllFullPath and try again
        delete []szPath;
        szPath = new char[uiPathSize + 1];
        res = GetLibraryPathFromEnv(szPath, uiPathSize + 1, &uiPathSize);
    }

    ILibraryLoader *pLibraryLoader = NULL;

    if (MRTE_SUCCEEDED(res))
    {
        if (MRTE_RESULT_OK == res)
        {// Try loading the library from the environmental setting path or profiler path
            pLibraryLoader = OsCreateLibraryLoader(loadedDllName, szPath, bLoadOnce);
        }

        if (NULL == pLibraryLoader)
        {// Try loading the library from the system path
            pLibraryLoader = OsCreateLibraryLoader(loadedDllName, NULL, bLoadOnce);
        }
    }
    delete []szPath;

    return pLibraryLoader;
}

//====Added below code for bug 226572 to remove agent controller dependency====

//============= Eclipse Plugin Agent Controller search begin =============

/**
 * Input: new_ac_home. Returns: the characters after the .v in the path
 * (e.g.  org.eclipse.tptp.platform.ac.win_ia32_4.4.100__.v200902101418__)
*/
char* GetEclipsePluginAcVersion(char new_ac_home[])
{
    const char *version_token = ".v";
    const unsigned int version_token_len = 2;
    char* strAcVersion = strstr(new_ac_home, version_token);

    if ( strAcVersion != NULL )
    {
        strAcVersion += version_token_len; // ignore ".v" at the beginning
    }

    return strAcVersion;
}

const unsigned int plugin_ac_version_info_len = 12;  // YYYYMMDDHHMM

// Description: When find a new ac home path new_ac_home, using function UpdateLatestAcHomeInfo
//                   to compare its version with latest_ac_version. The latest agent controller info will be
//                   saved in latest_ac_version.
//
/** Calls 'GetEclipsePluginAcVersion' with new_ac_home as the param.
 * Input: new_ac_home
 * Output: latest_ac_version, latest_ac_home.
 * If getEclipsePluginACVersion was able to extract the .v* portion then that is stored in
 * latest_ac_version, and new_ac_home is stored  in latest_ac_home. */
static void UpdateLatestAcHomeInfo(char new_ac_home[], char latest_ac_version[],
  char latest_ac_home[])
{
    char* strAcVersion = GetEclipsePluginAcVersion(new_ac_home);

    if ( strAcVersion != NULL &&
    	(strncmp(strAcVersion, latest_ac_version, plugin_ac_version_info_len) > 0) )
    {
        // save the latest version info for next comparing
        strncpy(latest_ac_version, strAcVersion, plugin_ac_version_info_len);

        // update the latest ac home info
        strncpy(latest_ac_home, new_ac_home, MAX_PATH-1);
    }
}

/** Appends / or \ to path, if dir name does not end with it */
unsigned int AppendDirSeparator(char path[])
{
    unsigned int path_len = strlen(path);

    if(path_len < MAX_PATH-2 // can hold another character DIRECTORY_SEPARATOR besides '\0'
        && path[path_len - 1] != DIRECTORY_SEPARATOR)
    {
        strcat(path, DIRECTORY_SEPARATOR_STR);
       ++path_len;
    }

    return path_len;
}

/**
 * Scans through the given 'plugins_path', and finds the IAC directory
 * Input: plugins_path
 * Output: Path that matches the IAC plugin prefix (for the relevant platform)
 *
 * Given a path to the plugins dir (plugins_path), it searches for the IAC directory  */
static void OsSearchLatestAcHome(char plugins_path[], char latest_ac_home[])
{
    // Eclipse plugins ac name pattern is "org.eclipse.tptp.platform.ac.OS_PLATFORM_X.Y.Z.vYYYYMMDDHHMM"
    char latest_ac_version_info[plugin_ac_version_info_len+1] = "000000000000";

    #if defined(__linux__) || defined(__sparc) || defined(_AIX) || defined(MVS) || defined(_SOLARISX86)

	    #if defined(IPF_ARCH)
		    const char* ac_home_prefix = "org.eclipse.tptp.platform.ac.linux_ipf_";
	    #elif defined(EM64T_ARCH)
		    const char* ac_home_prefix = "org.eclipse.tptp.platform.ac.linux_em64t_";
	    #else
		    const char* ac_home_prefix = "org.eclipse.tptp.platform.ac.linux_ia32_";
	    #endif

	    DIR* plugins_dir = opendir(plugins_path);
	    if (NULL == plugins_dir)
	    {
	        return;
	    }

	    struct dirent *dir_entry = readdir(plugins_dir);
	    while (NULL != dir_entry)
	    {
	        if (NULL != strstr(dir_entry->d_name, ac_home_prefix))
	        {// An agent controller plugin folder is founded
	            UpdateLatestAcHomeInfo(dir_entry->d_name, latest_ac_version_info, latest_ac_home);
	        }

	        dir_entry = readdir(plugins_dir);
	    }
	    (void) closedir(plugins_dir);

    #else

    #if defined(IPF_ARCH)
    const char* ac_home_prefix = "org.eclipse.tptp.platform.ac.win_ipf_";
    #elif defined(EM64T_ARCH)
    const char* ac_home_prefix = "org.eclipse.tptp.platform.ac.win_em64t_";
    #else
    const char* ac_home_prefix = "org.eclipse.tptp.platform.ac.win_ia32_";
    #endif

    WIN32_FIND_DATA data;

    char* ac_home_pattern = "org.eclipse.tptp.platform.ac.*";
    char search_file_info[MAX_PATH] = {'\0'};

    strncat(search_file_info, plugins_path, MAX_PATH-1);

    unsigned int search_file_info_len = AppendDirSeparator(search_file_info);

    if (search_file_info_len + strlen(ac_home_pattern) < MAX_PATH)
    {
        strcat(search_file_info, ac_home_pattern);
    }
    else
    {
        return;
    }

    HANDLE hDirectoryIterator = FindFirstFile(search_file_info, &data); // handle to directory iterator
    if (hDirectoryIterator == INVALID_HANDLE_VALUE)
    {
        return;
    }

    do
    {
        if ( ((data.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) != 0) // is directory
        	&& (NULL != strstr(data.cFileName, ac_home_prefix)) )
        {// An agent controller plugin folder is founded
            UpdateLatestAcHomeInfo(data.cFileName, latest_ac_version_info, latest_ac_home);
        }
    } while (0 != FindNextFile(hDirectoryIterator, &data) );

     if (hDirectoryIterator != INVALID_HANDLE_VALUE)
    {// close the iterator
        FindClose(hDirectoryIterator);
    }

    #endif
}

/** Returns true if plugins_path contains /plugins or \\plugins, false otherwise. */
static bool OsValidatePluginExist(char plugins_path[])
{
    #if defined(__linux__) || defined(__sparc) || defined(_AIX) || defined(MVS) || defined(_SOLARISX86)
	    const char* validateString= "/plugins";
    #else
	    const char* validateString= "\\plugins";
    #endif

    return (NULL != strstr(plugins_path, validateString));
}


// Search the latest version ac under <Eclipse_Home>\plugins
/**
 * Input path: (Potential) path to eclipse plugins directory (must end with  \plugins).
 * Ensures that path ends with \plugins, and that inside that directory, that that directory contains the IAC.
 * When it is determined that path contains IAC, it appends \agent_controller and stores that in path, then returns true.
 * else returns false. */
static bool GetPluginLatestAcPath(char path[])
{
    if (!OsValidatePluginExist(path))
    {
        return false;
    }

    char latest_ac_home[MAX_PATH] = {'\0'};

    OsSearchLatestAcHome(path, latest_ac_home);

    if ('\0' != latest_ac_home[0])
    {
        unsigned int path_len = AppendDirSeparator(path);
        strncat(path, latest_ac_home, MAX_PATH-path_len-1-strlen(latest_ac_home)); // -1: for '\0'
        path_len = strlen(path);

        const char* str_ac_home = (DIRECTORY_SEPARATOR == path[path_len-1]) ? "agent_controller" :
            #if defined(__linux__) || defined(__sparc) || defined(_SOLARISX86)
	            "/agent_controller" ;
            #else
	            "\\agent_controller" ;
            #endif

        strncat(path, str_ac_home, MAX_PATH-strlen(path)-1-strlen(str_ac_home));
        return true;
    }
    else
    {// Agent Controller plugin does not exist under eclipse plugins folder
        return false;
    }
}

//============= Eclipse Plugin Agent Controller search end =============

/**
 * Input: path - path to parse, ancestorRelativeLevel number of directories to go back.
 * Strips directories from the front of path
 * (e.g. \mydir\mydir2 becomes \mydir with an ancestorRelativeLevel of 1.)
 * Stores result in path, and return true or false. */
static bool GetAncestorPath(char path[], unsigned int ancestorRelativeLevel)
{
    char* pTail = NULL;

    while(ancestorRelativeLevel > 0)
    {
        pTail = strrchr(path, DIRECTORY_SEPARATOR);
        if (NULL == pTail)
        {
            return false;
        }

        *pTail = '\0';  // del the string after "plugins" on the path
        --ancestorRelativeLevel;
    }

    return true;
}

/** Appends either /lib or \bin to ac_home */
static void AppendLibraryStrToPath(char ac_home[])
{
    #if defined(__linux) || defined (__sparc) || defined(_AIX) || defined(MVS) || defined(_SOLARISX86)
	    const char* ac_relative_lib_path= "/lib";
    #else
	    const char* ac_relative_lib_path= "\\bin";
    #endif

    strncat(ac_home, ac_relative_lib_path, MAX_PATH - strlen(ac_home) -1);
}


/**
 * Called by PreLoadAgentControllerLibrary to get the path to the AC lib folder.
 * Uses the global defaultPath variable (which should point to the path of all of the JVMTI agent libraries), calls GetAncestorPath, and
 * then if true, calls GetPluginLatestAcPath.
 * output: ac_lib_path is output
 * returns true if able to get path, false otherwise. */
static bool GetAgentControllerLibPath(char ac_lib_path[])
{
    bool isGetAcSuccess = false;

    // When search ac on java profiler path, we need move to ancestor folder
    // relativeLevel will tell how many level we will move up referring to profiler path
    unsigned int relativeLevel = 0;

    // Try to find the latest version ac path in eclipse plugins referring to java profiler path
    strncpy(ac_lib_path, defaultPath, defaultPathSize);
    relativeLevel = 3;  // <Eclipse_Home>\plugins\org.eclipse.tptp.platform.jvmti.runtime_X.X.X.vXXX\agent_files\win_ia32(etc.)
    isGetAcSuccess = GetAncestorPath(ac_lib_path, relativeLevel) ?    // Get <Eclipse_Home>\plugins path
                                         GetPluginLatestAcPath(ac_lib_path) : false;

    if (!isGetAcSuccess)
    {// Not found ac in plugins, try to find standalone ac path referring to java profiler path
        memset(ac_lib_path, '\0', MAX_PATH);
        strncpy(ac_lib_path, defaultPath, defaultPathSize);
        relativeLevel = 2;  // <AC_HOME>\plugins\org.eclipse.tptp.javaprofiler
        isGetAcSuccess = GetAncestorPath(ac_lib_path, relativeLevel) ;     // Get <AC_HOME> path
    }

    if (isGetAcSuccess)
    {
        AppendLibraryStrToPath(ac_lib_path);
        return true;
    }
    else
    {
        return false;
    }
}

/**
 * Called by JPIBootLoader.cpp to load the agent controller libraries. str_ac_home is an output variable (and also an input variable).
 *
 * Follows a set of rules:
 * 1. Try loading the ac library in given ac_home path
 *   - calls 'AppendLibraryStrToPath'a
 * 2. Try loading the ac library in system path
 *   - calls 'OsCreateLibraryLoader' with null path
 * 3. Try loading ac in default eclipse 'plugins' (latest version ac) or standalone ac relative path to java profilers
 *   - calls GetAgentControllerLibPath
*/
TResult PreLoadAgentControllerLibrary(char str_ac_home[])
{
    ILibraryLoader *pAcBaseLibraryLoader = NULL;
    #if defined(__linux__) || defined(__sparc) || defined(_AIX) || defined(MVS) || defined(_SOLARISX86)
	    const char* str_acbase_lib = "tptpAgentBase";
	    const char* str_processcontrolUtil_lib = "processControlUtil";
	    ILibraryLoader *pProcessControlUtilLibraryLoader = NULL;
    #else
	    const char* str_acbase_lib = "AgentBase";
    #endif

    if ('\0' != str_ac_home[0])
    {// 1. Try loading the ac library in given ac_home path
        AppendLibraryStrToPath(str_ac_home);
        pAcBaseLibraryLoader = OsCreateLibraryLoader(str_acbase_lib, str_ac_home, false);

        #if defined(__linux__) || defined(__sparc) || defined(_SOLARISX86)
        if (NULL != pAcBaseLibraryLoader)
        { // To solve different dynamic lib path search problem casued by Linux
            pProcessControlUtilLibraryLoader = OsCreateLibraryLoader(str_processcontrolUtil_lib, str_ac_home, false);
        }
        #endif
    }

    if (NULL == pAcBaseLibraryLoader)
    {// 2. Try loading the ac library in system path
        pAcBaseLibraryLoader = OsCreateLibraryLoader(str_acbase_lib, NULL, false);

        #if defined(__linux__) || defined(__sparc) || defined(_SOLARISX86)
        if (NULL != pAcBaseLibraryLoader)
        {
            pProcessControlUtilLibraryLoader = OsCreateLibraryLoader(str_processcontrolUtil_lib, NULL, false);
        }
        #endif
    }

    if (NULL == pAcBaseLibraryLoader)
    {// 3. Try loading ac in default eclipse 'plugins' (latest version ac) or standalone ac relative path to java profilers
        char ac_lib_path[MAX_PATH] = {'\0'};

        if (GetAgentControllerLibPath(ac_lib_path))
        {
            pAcBaseLibraryLoader = OsCreateLibraryLoader(str_acbase_lib, ac_lib_path, false);

            #if defined(__linux__) || defined(__sparc) || defined(_SOLARISX86)
            if (NULL != pAcBaseLibraryLoader)
            {
                pProcessControlUtilLibraryLoader = OsCreateLibraryLoader(str_processcontrolUtil_lib, ac_lib_path, false);
            }
            #endif
        }
    }

    if (NULL != pAcBaseLibraryLoader)
    {// AC libraries are loaded successfully
        pAcBaseLibraryLoader->Destroy();
        pAcBaseLibraryLoader = NULL;

        #if defined(__linux__) || defined(__sparc) || defined(_SOLARISX86)
        if (NULL != pProcessControlUtilLibraryLoader)
        {
            pProcessControlUtilLibraryLoader->Destroy();
            pProcessControlUtilLibraryLoader = NULL;
        }
        else
        {
            return MRTE_ERROR_FAIL;
        }
        #endif

        return MRTE_RESULT_OK;
    }
    else
    {// Failed to load AC libraries in enabled or controlled mode
        return MRTE_ERROR_FAIL;
    }
}

//====Added above code for bug 226572 to remove agent controller dependency====

#ifdef __cplusplus
} //extern "C"
#endif
