/*****************************************************************************
 * Copyright (c) 1997, 2009 Intel Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *
 * $Id$ 
 *****************************************************************************/

#include "OSAL.h"
#include <sys/utsname.h>

namespace Martini { namespace OSA
{
    IThreadSync *CreateThreadSync()
    {
        CThreadSync* pThreadSync = new CThreadSync;
        if (!pThreadSync)
        {
            return NULL;
        }
        
        TResult res = pThreadSync->Create();
        if (MRTE_FAILED(res))
        {
            pThreadSync->Destroy();
            return NULL;
        }
        
        return pThreadSync;
    }
    
    
    ILibraryLoader *CreateLibraryLoader(const char *szLibName, const char *szLibPath, bool bLoadOnce)
    {
        CLibraryLoader *pLibraryLoader = new CLibraryLoader();
        if (!pLibraryLoader)
        {
            return NULL;
        }
        
        TResult res = pLibraryLoader->Create(szLibName, szLibPath, bLoadOnce);
        if (MRTE_FAILED(res))
        {
            pLibraryLoader->Destroy();
            return NULL;
        }
        
        return pLibraryLoader;
    }
    
    IReadOnlyFileMapping *CreateReadOnlyFileMapping(const char *szFileName)
    {
        CReadOnlyFileMapping *pROFileMapping = new CReadOnlyFileMapping();
        TResult retVal;
        
        if (! pROFileMapping)
        {
            return NULL;
        }
        
        retVal = pROFileMapping->Open(szFileName);
        
        if (MRTE_FAILED(retVal))
        {
            pROFileMapping->Destroy();
            return NULL;
        }
        return pROFileMapping;
    }
    
    IDirectoryHandle *CreateDirectoryHandle(const char* szDirectoryName)		
    {
        CDirectoryHandle *pDirectory = new CDirectoryHandle();
        TResult retVal;
        
        if (! pDirectory)
        {
            return NULL;
        }
        
        retVal = pDirectory->Open(szDirectoryName);
        
        if (MRTE_FAILED(retVal))
        {
            pDirectory->Destroy();
            return NULL;
        }
        return pDirectory;
    }
    
    IThreadLocalStorage *CreateThreadLocalStorage()		
    {
        CThreadLocalStorage *pTls = new CThreadLocalStorage();
        TResult retVal;
        
        if (! pTls)
        {
            return NULL;
        }
        
        retVal = pTls->Create();
        
        if (MRTE_FAILED(retVal))
        {
            pTls->Destroy();
            return NULL;
        }
        return pTls;
    }
    
#ifndef IPF_ARCH

#ifdef EM64T_ARCH
	#define rdtscll(val) __asm__ __volatile__ ("rdtsc ; shl $32, %%rdx ; add %%rdx, %%rax" : "=A" (val))
#else
	#if defined(__linux__) && defined(__s390__)
		#define rdtscll(val) __asm__ __volatile__ ("stck 0(%1)" : "=m" (val) : "a" (&val) : "cc");
	#elif defined(MVS)
		#define rdtscll(val) __asm __volatile__ ("stck 0(%1)" : "=m" (val) : "a" (&val) : "cc");
	#else
		#define rdtscll(val) __asm__ __volatile__ ("rdtsc" : "=A" (val))
	#endif
#endif

    U64 GetTimeStamp()
    {
        U64 res;
        rdtscll(res);
        return res;
    }
    
#else // #ifndef IPF_ARCH
    
    extern "C" U64 rdtscll();
    
    U64 GetTimeStamp()
    {
        return rdtscll();
    }
    
#endif // #ifndef IPF_ARCH
    
    // getting the env var from the file since some times the getenv function does 
    // not return a correct value
    TResult GetEnvFromSystemFile(char *szVarName, char *szValueBuffer, 
        unsigned int uiBufferSize, unsigned int *puiBufferSize)
    {
        FILE *process_environment_file = fopen("/proc/self/environ", "r");
        if (!process_environment_file)
            return MRTE_RESULT_FALSE;
        
        char *var_name ="";
        char *var_value ="";
        const int size_of_line  = MAX_PATH * 2 + 1;
        char one_line[size_of_line];
        
        bool found = false;
        unsigned int curr_file_position = 0;
        while (!found && fgets(one_line, size_of_line - 1, process_environment_file))
        {
            curr_file_position +=  strlen(one_line) + 1;
            char *delim_pos = strchr(one_line, '=');
            var_name = one_line;
            if (delim_pos)
            {
                delim_pos[0] = '\0';
                if (!strcmp(var_name, szVarName))
                {
                    found = true;
                    var_value = delim_pos + 1;
                }
            }
            fseek(process_environment_file, curr_file_position, SEEK_SET);
        }
        if (!found || !var_value) 
        {
            return MRTE_RESULT_FALSE;
        }
        
        *puiBufferSize = strlen(var_value);
        if (*puiBufferSize == 0)
        {
            return MRTE_RESULT_FALSE;
        }
        
        if (*puiBufferSize >= uiBufferSize)
        {
            (*puiBufferSize)++;
            return MRTE_ERROR_BUFFER_TOO_SHORT;
        }
        
        strncpy(szValueBuffer, var_value, uiBufferSize - 1);
        szValueBuffer[uiBufferSize - 1] = '\0';
        
        return MRTE_RESULT_OK;
    }
    
    
    TResult GetEnvironmentVar(char *szVarName, char *szValueBuffer, 
        unsigned int uiBufferSize, unsigned int *puiBufferSize)
    {
        if (szValueBuffer == NULL || szVarName == NULL || 
            szVarName[0] == '\0' || puiBufferSize == NULL)
        {
            return MRTE_ERROR_ILLEGAL_ARGUMENT;
        }
        
        if ((void *)__environ == NULL)
        {
            return GetEnvFromSystemFile(szVarName, szValueBuffer, uiBufferSize, 
                puiBufferSize);
        }
        
        char* pEnv = getenv(szVarName);
        if (pEnv == NULL)
        {
            return MRTE_RESULT_FALSE;
        }
        
        size_t len = strlen(pEnv);
        *puiBufferSize = len + 1;
        
        if (len == 0)
        {
            return MRTE_RESULT_FALSE;
        }
        
        if (len >= uiBufferSize)
        {
            return MRTE_ERROR_BUFFER_TOO_SHORT;
        }
        
        strncpy(szValueBuffer, pEnv, uiBufferSize - 1);
        szValueBuffer[uiBufferSize - 1] = '\0';
        
        return MRTE_RESULT_OK;
    }
    
    
    
    TResult SetEnvironmentVar(char *szVarName, char *szValue)
    {
        if (szVarName == NULL)
        {
            return MRTE_ERROR_ILLEGAL_ARGUMENT;
        }
        if (szValue == NULL || szValue[0] == '\0')
        {
            unsetenv(szVarName);
            return MRTE_RESULT_OK;
        }
        int iRes = setenv(szVarName, szValue, 1); 
        if (iRes == 0)
        {
            return MRTE_RESULT_OK;
        }
        return MRTE_ERROR_ILLEGAL_ARGUMENT;
    }
    
    TResult MakeDirectory(char *szDirPath) // add and optional flag for security
    {
        int res = mkdir(szDirPath,0777);
        if (res == -1)
        {
            return MRTE_ERROR_OSA_FAILURE;
        }
        res = chmod (szDirPath,0777);
        if (res == 0)
        {
            return MRTE_RESULT_OK;
        }
        else return MRTE_ERROR_OSA_FAILURE;
    }
    
    TResult GetExecutableFileName(char* szBuffer, unsigned int uiBufferSize,
        unsigned int *puiRequiredBufferSize)
    {
        if(szBuffer == 0)
        {
            return MRTE_ERROR_ILLEGAL_ARGUMENT;
        }
        
        char szLocalBuff[PATH_MAX];
        szLocalBuff[0] = '\0';
        
        int iWrittenBytes = readlink("/proc/self/exe", szLocalBuff, PATH_MAX);
        if(iWrittenBytes == -1)
        {
            return MRTE_ERROR_OSA_FAILURE;
        }
        if(puiRequiredBufferSize)
        {
            *puiRequiredBufferSize = iWrittenBytes + 1;
        }

        if(iWrittenBytes + 1 > uiBufferSize)
        {
            return MRTE_ERROR_BUFFER_TOO_SHORT;
        }
        szLocalBuff[iWrittenBytes] = '\0';
        strcpy(szBuffer, szLocalBuff);
        
        return MRTE_RESULT_OK;
    }
    
    unsigned int GetCurrProcessID()
    {
        return (unsigned int)getpid();
    }

    unsigned int GetCurrThreadID()
    {
        return (unsigned int)getpid();
    }

    TResult GetCurrentProcCommandLine(char* szBuffer, unsigned int uiBufferSize,
    unsigned int *puiRequiredBufferSize)
    {
        if(szBuffer == 0)
        {
            return MRTE_ERROR_ILLEGAL_ARGUMENT;
        }
                
        int iReadBytes = readlink("/proc/self/cmdline", szBuffer, uiBufferSize);
        if(iReadBytes == -1)
        {
            return MRTE_ERROR_OSA_FAILURE;
        }

        if(puiRequiredBufferSize)
        {
            *puiRequiredBufferSize = iReadBytes + 1;
        }
        
        if(iReadBytes + 1 > uiBufferSize)
        {
            return MRTE_ERROR_BUFFER_TOO_SHORT;
        }

        for (int i = 0; i < iReadBytes; ++i)
        {
            if (szBuffer[i] == 0)
            {
                szBuffer[i] = ' ';
            }
        }
        
        return MRTE_RESULT_OK;    
    }
    
    TResult GetThisMachineName(char* szBuffer, unsigned int uiBufferSize,
        unsigned int *puiRequiredBufferSize)
    {
        utsname buf;

        int iRes = uname(&buf);
        if (iRes < 0)
        {
            return MRTE_ERROR_OSA_FAILURE;
        }

        size_t len = strlen(buf.nodename);
        if (puiRequiredBufferSize)
        {
            *puiRequiredBufferSize = len + 1;
        }
        
        if (uiBufferSize < len + 1)
        {
            return MRTE_ERROR_BUFFER_TOO_SHORT;
        }

        strcpy(szBuffer, buf.nodename);
        
        return MRTE_RESULT_OK;
    }
    


}} // namespace 

