/*****************************************************************************
 * Copyright (c) 2009, Intel Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *
 * $Id: OSAL.cpp,v 1.1 2009/04/14 20:33:46 jwest Exp $
 *****************************************************************************/

#include "OSAL.h"
#include <sys/utsname.h>

#include <sys/time.h>
#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#define _STRUCTURED_PROC 1
#include <sys/fcntl.h>
#include <procfs.h>

//declaring static envstring. why? from man pages:
//  The string argument should not be an automatic variable. It should
//  be declared static if it is declared within a function because it
	//  cannot be automatically declared. A potential error is to call
	//  putenv() with a pointer to an automatic variable as the argument and
	//  to then exit the calling function while string is still part of the
	//  environment.
static char* envstring;

// PATH_MAX not defined anywhere in the headers
// the following command gives and indication of what it should be
// bash-2.05$ getconf -a | grep PATH_MAX
// PATH_MAX:                       1024

#ifndef PATH_MAX
	#define PATH_MAX 1024
#endif


namespace Martini { namespace OSA
{
    IThreadSync *CreateThreadSync()
    {
        CThreadSync* pThreadSync = new CThreadSync;
        if (!pThreadSync)
        {
            return NULL;
        }

        TResult res = pThreadSync->Create();
        if (MRTE_FAILED(res))
        {
            pThreadSync->Destroy();
            return NULL;
        }

        return pThreadSync;
    }


    ILibraryLoader *CreateLibraryLoader(const char *szLibName, const char *szLibPath, bool bLoadOnce)
    {
        CLibraryLoader *pLibraryLoader = new CLibraryLoader();
        if (!pLibraryLoader)
        {
            return NULL;
        }

        TResult res = pLibraryLoader->Create(szLibName, szLibPath, bLoadOnce);
        if (MRTE_FAILED(res))
        {
            pLibraryLoader->Destroy();
            return NULL;
        }

        return pLibraryLoader;
    }

    IReadOnlyFileMapping *CreateReadOnlyFileMapping(const char *szFileName)
    {
        CReadOnlyFileMapping *pROFileMapping = new CReadOnlyFileMapping();
        TResult retVal;

        if (! pROFileMapping)
        {
            return NULL;
        }

        retVal = pROFileMapping->Open(szFileName);

        if (MRTE_FAILED(retVal))
        {
            pROFileMapping->Destroy();
            return NULL;
        }
        return pROFileMapping;
    }

    IDirectoryHandle *CreateDirectoryHandle(const char* szDirectoryName)
    {
        CDirectoryHandle *pDirectory = new CDirectoryHandle();
        TResult retVal;

        if (! pDirectory)
        {
            return NULL;
        }

        retVal = pDirectory->Open(szDirectoryName);

        if (MRTE_FAILED(retVal))
        {
            pDirectory->Destroy();
            return NULL;
        }
        return pDirectory;
    }

    IThreadLocalStorage *CreateThreadLocalStorage()
    {
        CThreadLocalStorage *pTls = new CThreadLocalStorage();
        TResult retVal;

        if (! pTls)
        {
            return NULL;
        }

        retVal = pTls->Create();

        if (MRTE_FAILED(retVal))
        {
            pTls->Destroy();
            return NULL;
        }
        return pTls;
    }

    // no rdtscll on solaris
    // Solaris and HP-UX provide a function gethrtime returning a 64-bit time stamp with nano-
    //     second accuracy (using a very low overhead system call on Solaris UltraSPARC). */
    //The gethrtime() and gethrvtime() functions  both  return  an
    //     hrtime_t, which is a 64-bit (long long) signed integer.

    U64 GetTimeStamp()
    {
    	hrtime_t result;
        result = gethrtime();
        return result;
    }

    TResult GetEnvironmentVar(char *szVarName, char *szValueBuffer,
        unsigned int uiBufferSize, unsigned int *puiBufferSize)
    {
        if (szValueBuffer == NULL || szVarName == NULL ||
            szVarName[0] == '\0' || puiBufferSize == NULL)
        {
            return MRTE_ERROR_ILLEGAL_ARGUMENT;
        }

        char* pEnv = getenv(szVarName);
        if (pEnv == NULL)
        {
		    return MRTE_RESULT_FALSE;
        }

        size_t len = strlen(pEnv);
        *puiBufferSize = len + 1;

        if (len == 0)
        {
            return MRTE_RESULT_FALSE;
        }

        if (len >= uiBufferSize)
        {
            return MRTE_ERROR_BUFFER_TOO_SHORT;
        }

        strncpy(szValueBuffer, pEnv, uiBufferSize - 1);
        szValueBuffer[uiBufferSize - 1] = '\0';

        return MRTE_RESULT_OK;
    }



    TResult SetEnvironmentVar(char *szVarName, char *szValue)
    {
        if (szVarName == NULL)
        {
            return MRTE_ERROR_ILLEGAL_ARGUMENT;
        }
        if (szValue == NULL || szValue[0] == '\0')
        {
			// This is tricky ... no unsetenv on solaris.
			// could play around with putenv where name = nothing, but that is funky
			// removing unsetenv for now - better to have nothing than something that doesn't work right
			// TODO:
        	//unsetenv(szVarName);
            return MRTE_RESULT_OK;
        }

        // apparently, it needs to be in key=value notation on solaris
		sprintf( envstring, "%s=%s", szVarName, szValue );
		int iRes = putenv(envstring);

		if (iRes == 0)
        {
            return MRTE_RESULT_OK;
        }
        return MRTE_ERROR_ILLEGAL_ARGUMENT;
    }

    TResult MakeDirectory(char *szDirPath) // add and optional flag for security
    {
        int res = mkdir(szDirPath,0777);
        if (res == -1)
        {
            return MRTE_ERROR_OSA_FAILURE;
        }
        res = chmod (szDirPath,0777);
        if (res == 0)
        {
            return MRTE_RESULT_OK;
        }
        else return MRTE_ERROR_OSA_FAILURE;
    }

    TResult GetExecutableFileName(char* szBuffer, unsigned int uiBufferSize,
        unsigned int *puiRequiredBufferSize)
    {
        if(szBuffer == 0)
        {
            return MRTE_ERROR_ILLEGAL_ARGUMENT;
        }

	const char *szLocalBuff;
        szLocalBuff = getexecname();
        *puiRequiredBufferSize = strlen(szLocalBuff);

	if (uiBufferSize < strlen(szLocalBuff)) {
		return MRTE_ERROR_BUFFER_TOO_SHORT;
	}	

        if(szLocalBuff == 0)
        {
            return MRTE_ERROR_OSA_FAILURE;
        }
	strcpy( szBuffer, szLocalBuff );

        return MRTE_RESULT_OK;
    }

    unsigned int GetCurrProcessID()
    {
        return (unsigned int)getpid();
    }

    unsigned int GetCurrThreadID()
    {
        return (unsigned int)getpid();
    }

    TResult GetCurrentProcCommandLine(char* szBuffer, unsigned int uiBufferSize,
    unsigned int *puiRequiredBufferSize)
    {
        if(szBuffer == 0)
        {
            return MRTE_ERROR_ILLEGAL_ARGUMENT;
        }


        psinfo_t psi;
        FILE *psinfo = fopen("/proc/self/psinfo", "rb");
        char *origCmd = (char*)malloc(PATH_MAX);
        if (psinfo) {
           if (fread(&psi, sizeof(psi), 1, psinfo) == 1) {
              char **argv = (char**) psi.pr_argv;
              int argc = psi.pr_argc;
              char *offsetPtr = origCmd;
              int offset = 0;

              for ( int i = 0; i < argc; i++ ) {
                 int len = strlen(argv[i]) + 1;
                 memcpy( (void*)offsetPtr, argv[i], len );
                 offsetPtr += len;
                 offset += len;
                 origCmd[offset - 1] = ' ';
              }
           } else {
              return MRTE_ERROR_OSA_FAILURE;
           }
        } else {
                return MRTE_ERROR_OSA_FAILURE;
        }

        if(puiRequiredBufferSize)
        {
            *puiRequiredBufferSize = strlen(origCmd);
        }

        if(strlen(origCmd) > uiBufferSize)
        {
            return MRTE_ERROR_BUFFER_TOO_SHORT;
        }

        memcpy( szBuffer, origCmd, strlen(origCmd) );
        free( origCmd );

        return MRTE_RESULT_OK;
    }


    TResult GetThisMachineName(char* szBuffer, unsigned int uiBufferSize,
        unsigned int *puiRequiredBufferSize)
    {

    	struct utsname buf;
    	int iRes = uname(&buf);

        if (iRes < 0)
        {
            return MRTE_ERROR_OSA_FAILURE;
        }

        size_t len = strlen(buf.nodename);
        if (puiRequiredBufferSize)
        {
            *puiRequiredBufferSize = len + 1;
        }

        if (uiBufferSize < len + 1)
        {
            return MRTE_ERROR_BUFFER_TOO_SHORT;
        }

        strcpy(szBuffer, buf.nodename);

        return MRTE_RESULT_OK;
    }



}} // namespace
