/*****************************************************************************
 * Copyright (c) 1997, 2010, Intel Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *
 * $Id: OSAL.cpp,v 1.4 2010/09/08 15:20:46 mreid Exp $ 
 *****************************************************************************/

#include "OSAL.h"
#include <time.h>
#include <sys/utsname.h>

// For native/unicode conversions
#include <iconv.h>
#include <errno.h>
#include <langinfo.h>
#include <locale.h>

// For logging
#include <iostream>

/*
getconf -a | grep PATH_MAX
PATH_MAX       1023
*/

#ifndef PATH_MAX
	#define PATH_MAX 1024
#endif

namespace Martini { namespace OSA
{
    IThreadSync *CreateThreadSync()
    {
        CThreadSync* pThreadSync = new CThreadSync;
        if (!pThreadSync)
        {
            return NULL;
        }
        
        TResult res = pThreadSync->Create();
        if (MRTE_FAILED(res))
        {
            pThreadSync->Destroy();
            return NULL;
        }
        
        return pThreadSync;
    }
    
    
    ILibraryLoader *CreateLibraryLoader(const char *szLibName, const char *szLibPath, bool bLoadOnce)
    {
        CLibraryLoader *pLibraryLoader = new CLibraryLoader();
        if (!pLibraryLoader)
        {
            return NULL;
        }
        
        TResult res = pLibraryLoader->Create(szLibName, szLibPath, bLoadOnce);
        if (MRTE_FAILED(res))
        {
            pLibraryLoader->Destroy();
            return NULL;
        }
        
        return pLibraryLoader;
    }
    
    IReadOnlyFileMapping *CreateReadOnlyFileMapping(const char *szFileName)
    {
        CReadOnlyFileMapping *pROFileMapping = new CReadOnlyFileMapping();
        TResult retVal;
        
        if (! pROFileMapping)
        {
            return NULL;
        }
        
        retVal = pROFileMapping->Open(szFileName);
        
        if (MRTE_FAILED(retVal))
        {
            pROFileMapping->Destroy();
            return NULL;
        }
        return pROFileMapping;
    }
    
    IDirectoryHandle *CreateDirectoryHandle(const char* szDirectoryName)		
    {
        CDirectoryHandle *pDirectory = new CDirectoryHandle();
        TResult retVal;
        
        if (! pDirectory)
        {
            return NULL;
        }
        
        retVal = pDirectory->Open(szDirectoryName);
        
        if (MRTE_FAILED(retVal))
        {
            pDirectory->Destroy();
            return NULL;
        }
        return pDirectory;
    }
    
    IThreadLocalStorage *CreateThreadLocalStorage()		
    {
        CThreadLocalStorage *pTls = new CThreadLocalStorage();
        TResult retVal;
        
        if (! pTls)
        {
            return NULL;
        }
        
        retVal = pTls->Create();
        
        if (MRTE_FAILED(retVal))
        {
            pTls->Destroy();
            return NULL;
        }
        return pTls;
    }

    U64 GetTimeStamp()
    {
		U64 l;
		// The z/architecture store clock instruction is used to represent the timestamp as a 64-bit unsigned value.
		__stck(&l);
		return l;
    }
    
    // getting the env var from the file since some times the getenv function does 
    // not return a correct value
    TResult GetEnvFromSystemFile(char *szVarName, char *szValueBuffer, 
        unsigned int uiBufferSize, unsigned int *puiBufferSize)
    {
        FILE *process_environment_file = fopen("/proc/self/environ", "r");
        if (!process_environment_file)
            return MRTE_RESULT_FALSE;
        
        char *var_name ="";
        char *var_value ="";
        const int size_of_line  = MAX_PATH * 2 + 1;
        char one_line[size_of_line];
        
        bool found = false;
        unsigned int curr_file_position = 0;
        while (!found && fgets(one_line, size_of_line - 1, process_environment_file))
        {
            curr_file_position +=  strlen(one_line) + 1;
            char *delim_pos = strchr(one_line, '=');
            var_name = one_line;
            if (delim_pos)
            {
                delim_pos[0] = '\0';
                if (!strcmp(var_name, szVarName))
                {
                    found = true;
                    var_value = delim_pos + 1;
                }
            }
            fseek(process_environment_file, curr_file_position, SEEK_SET);
        }
        if (!found || !var_value) 
        {
            return MRTE_RESULT_FALSE;
        }
        
        *puiBufferSize = strlen(var_value);
        if (*puiBufferSize == 0)
        {
            return MRTE_RESULT_FALSE;
        }
        
        if (*puiBufferSize >= uiBufferSize)
        {
            (*puiBufferSize)++;
            return MRTE_ERROR_BUFFER_TOO_SHORT;
        }
        
        strncpy(szValueBuffer, var_value, uiBufferSize - 1);
        szValueBuffer[uiBufferSize - 1] = '\0';
        
        return MRTE_RESULT_OK;
    }
    
    
    TResult GetEnvironmentVar(char *szVarName, char *szValueBuffer, 
        unsigned int uiBufferSize, unsigned int *puiBufferSize)
    {
        if (szValueBuffer == NULL || szVarName == NULL || 
            szVarName[0] == '\0' || puiBufferSize == NULL)
        {
            return MRTE_ERROR_ILLEGAL_ARGUMENT;
        }
        
      //  if ((void *)environ == NULL)
      //  {
      //      return GetEnvFromSystemFile(szVarName, szValueBuffer, uiBufferSize, 
      //          puiBufferSize);
      //  }
        
        char* pEnv = getenv(szVarName);
        
        if (pEnv == NULL)
        {
            return MRTE_RESULT_FALSE;
        }
        
        size_t len = strlen(pEnv);
        *puiBufferSize = len + 1;
        
        if (len == 0)
        {
            return MRTE_RESULT_FALSE;
        }
        
        if (len >= uiBufferSize)
        {
            return MRTE_ERROR_BUFFER_TOO_SHORT;
        }
        
        strncpy(szValueBuffer, pEnv, uiBufferSize - 1);
        szValueBuffer[uiBufferSize - 1] = '\0';
        
        return MRTE_RESULT_OK;
    }
    
    
    
    TResult SetEnvironmentVar(char *szVarName, char *szValue)
    {
        if (szVarName == NULL)
        {
            return MRTE_ERROR_ILLEGAL_ARGUMENT;
        }
        
        if (szValue == NULL || szValue[0] == '\0')
        {
            unsetenv(szVarName);
            return MRTE_RESULT_OK;
        }

        int iRes = setenv(szVarName, szValue, 1);
        if (iRes == 0)
        {
	        return MRTE_RESULT_OK;
        }
        
        return MRTE_ERROR_ILLEGAL_ARGUMENT;
    }
    
    TResult MakeDirectory(char *szDirPath) // add and optional flag for security
    {
    	char* nativeBuffer = 0;
        unicode2native(&nativeBuffer, (char *)szDirPath, strlen(szDirPath));
        
        int res = mkdir(nativeBuffer,0777);
        if (res == -1)
        {
            return MRTE_ERROR_OSA_FAILURE;
        }
        
        res = chmod (nativeBuffer,0777);
        if(nativeBuffer != 0)
        	free(nativeBuffer);
        
        if (res == 0)
        {
            return MRTE_RESULT_OK;
        }
        else return MRTE_ERROR_OSA_FAILURE;
    }
    
    TResult GetExecutableFileName(char* szBuffer, unsigned int uiBufferSize,
        unsigned int *puiRequiredBufferSize)
    {
        if(szBuffer == 0)
        {
            return MRTE_ERROR_ILLEGAL_ARGUMENT;
        }
        
        char szLocalBuff[PATH_MAX];
        szLocalBuff[0] = '\0';
        
        int iWrittenBytes = readlink("/proc/self/exe", szLocalBuff, PATH_MAX);
        if(iWrittenBytes == -1)
        {
            return MRTE_ERROR_OSA_FAILURE;
        }
        if(puiRequiredBufferSize)
        {
            *puiRequiredBufferSize = iWrittenBytes + 1;
        }

        if(iWrittenBytes + 1 > uiBufferSize)
        {
            return MRTE_ERROR_BUFFER_TOO_SHORT;
        }
        szLocalBuff[iWrittenBytes] = '\0';
        strcpy(szBuffer, szLocalBuff);
        
        return MRTE_RESULT_OK;
    }
    
    unsigned int GetCurrProcessID()
    {
        return (unsigned int)getpid();
    }

    unsigned int GetCurrThreadID()
    {
        return (unsigned int)getpid();
    }

    TResult GetCurrentProcCommandLine(char* szBuffer, unsigned int uiBufferSize,
    unsigned int *puiRequiredBufferSize)
    {
        if(szBuffer == 0)
        {
            return MRTE_ERROR_ILLEGAL_ARGUMENT;
        }
                
        int iReadBytes = readlink("/proc/self/cmdline", szBuffer, uiBufferSize);
        if(iReadBytes == -1)
        {
            return MRTE_ERROR_OSA_FAILURE;
        }

        if(puiRequiredBufferSize)
        {
            *puiRequiredBufferSize = iReadBytes + 1;
        }
        
        if(iReadBytes + 1 > uiBufferSize)
        {
            return MRTE_ERROR_BUFFER_TOO_SHORT;
        }

        for (int i = 0; i < iReadBytes; ++i)
        {
            if (szBuffer[i] == 0)
            {
                szBuffer[i] = ' ';
            }
        }
        
        return MRTE_RESULT_OK;    
    }
    
    TResult GetThisMachineName(char* szBuffer, unsigned int uiBufferSize,
        unsigned int *puiRequiredBufferSize)
    {
        utsname buf;

        int iRes = uname(&buf);
        if (iRes < 0)
        {
            return MRTE_ERROR_OSA_FAILURE;
        }

        size_t len = strlen(buf.nodename);
        if (puiRequiredBufferSize)
        {
            *puiRequiredBufferSize = len + 1;
        }
        
        if (uiBufferSize < len + 1)
        {
            return MRTE_ERROR_BUFFER_TOO_SHORT;
        }

        strcpy(szBuffer, buf.nodename);
        
        return MRTE_RESULT_OK;
    }
    
    #if defined(MVS) || defined(__OS400__) || defined(_HPUX)
/*
 * Returns the number of multi-byte char in the converted buffer
 * Arg 0: Pointer to the destination storage of the multi-byte buffer
 * Arg 1: Pointer to the source storage of the multi-byte buffer
 * Arg 2: Number of multi-byte in the buffer
 */
size_t unicode2native(char** out, char* in, size_t size_in) {
	int count;
#if defined(__OS400__)
	/* UTF-8  is IBMCCSID0120800000000000000000000 */
	/* Native is IBMCCSID0000000000000000000000000 */
	return convert(out, in, size_in, "IBMCCSID0000000000000000000000000", "IBMCCSID0120800000000000000000000");
#elif defined(_HPUX)
	return convert(out, in, size_in, nl_langinfo(CODESET), "utf8"); /* HP-UX iconv_open() cannot take "UTF-8" */
#else
	return convert(out, in, size_in, "IBM-1047", "UTF-8");
#endif

	return count;
}
size_t native2unicode(char** out, char* in, size_t size_in) {
#if defined(__OS400__)
	/* UTF-8  is IBMCCSID0120800000000000000000000 */
	/* Native is IBMCCSID0000000000000000000000000 */
	return convert(out, in, size_in, "IBMCCSID0120800000000000000000000", "IBMCCSID0000000000000000000000000");
#elif defined(_HPUX)
	return convert(out, in, size_in, "utf8", nl_langinfo(CODESET)); /* HP-UX iconv_open() cannot take "UTF-8" */
#else
	return convert(out, in, size_in, "UTF-8", "IBM-1047");
#endif
}

int convert(char** out, char* in, size_t size_in, char* to_cp, char* from_cp) {
	iconv_t cd; /* code page converter */
	char* inBuf_copy = 0; /* a copy of the input buffer, so that the original value is not modified */
	char* inBuf_start = 0; /* a pointer to the beginning of the inBuf_copy buffer used for free() */
	char* tempBuf = 0; /* a temporary buffer used to store the output of the conversion, allocated larger than required */
	char* tempBuf_start = 0; /* a pointer to the beginning of the tempBuf buffer used for free() */
	size_t outLen; /* length of the allocated output buffer */
	size_t convLen; /* return value from iconv() */
	size_t tempLen; /* length of the allocated output buffer, will actually be the numnber of bytes left in the buffer */
	size_t inLen_copy; /* a copy of the length of the input buffer */
#if _DEBUG
	int i;
#endif

	setlocale(LC_CTYPE, "");
	cd = iconv_open(to_cp, from_cp); /* to, from */
#if _DEBUG
#pragma convlit(suspend)
		printf("DEBUG: CP(%s) -> CP(%s)\n", from_cp, to_cp);
		#pragma convlit(resume)
#endif

#if defined(__OS400__)
	if(cd.return_value == -1) { /* iconv_t under OS/400 is a structure */
#else
	if(cd == (iconv_t)-1) {
#endif
#if _DEBUG
#pragma convlit(suspend)
		perror("DEBUG: Error calling iconv_open()\n");
		#pragma convlit(resume)
#endif
		*out = 0; /* Bug 80588 */
		return 0;
	}

	inLen_copy = size_in; /* make a copy of the length */
	inBuf_copy = (char*)malloc(sizeof(char) * inLen_copy); /* make a copy of the input buffer */
	memcpy(inBuf_copy, in, size_in);
	inBuf_start = inBuf_copy; /* mark the start of the input buffer */
#if _DEBUG
#pragma convlit(suspend)
	printf("DEBUG: inLen_copy = %d\n", inLen_copy);
	printf("DEBUG: inBuf_copy = ");
	for(i = 0; i < inLen_copy; i++) {
		printf("%c", inBuf_copy[i]);
	}
	printf("\n");
	#pragma convlit(resume)
#endif

	tempLen = inLen_copy * 2; /* create the output buffer, larger than expected since we don't know the actual size yet */
	outLen = tempLen; /* bytes left and buffer size are the same at this point */
	tempBuf = (char*)malloc(sizeof(char) * tempLen);
	tempBuf_start = tempBuf; /* mark the beginning of the output buffer */
	convLen = iconv(cd, &inBuf_copy, &inLen_copy, &tempBuf, &tempLen); /* perform the conversion */
	if(convLen == -1) {
#if _DEBUG
#pragma convlit(suspend)
		printf("DEBUG: Error running iconv(), errno = %d\n", errno);
		switch(errno) {
			case EILSEQ:
				printf("DEBUG: Invalid byte sequence at byte %d\n", inBuf_copy - inBuf_start);
				break;
			case E2BIG:
				printf("DEBUG: Ran out of space in the output buffer\n");
				break;
			case EINVAL:
				printf("DEBUG: Incomplete byte sequence at the end of the input buffer\n");
				break;
		}
		#pragma convlit(resume)
#endif
		*out = 0; /* Bug 80588 */
		return 0; /* return 0 if it fails */
	}
#if _DEBUG
#pragma convlit(suspend)
	printf("DEBUG: Bytes left(tempLen) = %d\n", tempLen);
	printf("DEBUG: Output buffer(tempBuf_start) = ");
	for(i = 0; i < outLen - tempLen; i++) {
		printf("%c", tempBuf_start[i]);
	}
	printf("\n");
	#pragma convlit(resume)
#endif

	*out = (char*)malloc(sizeof(char) * (outLen - tempLen + 1)); /* create the output buffer with the actual size, adding a null */
	bzero(*out, outLen - tempLen + 1);
	memcpy(*out, tempBuf_start, outLen - tempLen);  /* copy the contents from the temp buffer */
#if _DEBUG
#pragma convlit(suspend)
	printf("DEBUG: Copied buffer(out)\n");
	printf("DEBUG: Output buffer size = %d\n", outLen - tempLen);
	#pragma convlit(resume)
#endif

	iconv_close(cd);

	free(inBuf_start); /* free up working buffers */
	free(tempBuf_start);

	return (outLen - tempLen);
}
#endif

void logMessage(char* message)
{
	char* nativeBuffer = 0;
	// Length of the message + '[ ]\n\0'
	char msgBuffer[strlen(message) + 4];
	sprintf(msgBuffer, "[%s]\n", message);
	unicode2native(&nativeBuffer, msgBuffer, strlen(msgBuffer));
    std::cout << nativeBuffer;
    
    if(nativeBuffer != 0)
    	free(nativeBuffer);
}
    
void logErrorMessage(char* message)
{
	char* nativeBuffer = 0;
	// Length of the message + error + extra space
	char msgBuffer[strlen(message) + 32];
	sprintf(msgBuffer, "[Error: %s]\n", message);
	unicode2native(&nativeBuffer, msgBuffer, strlen(msgBuffer));
    std::cerr << nativeBuffer;

    if(nativeBuffer != 0)
    	free(nativeBuffer);

}


}} // namespace 

