/*****************************************************************************
 * Copyright (c) 1997, 2010 Intel Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *
 * $Id: ThreadAdaptor.cpp,v 1.5 2010/05/11 16:32:24 jwest Exp $ 
 *****************************************************************************/

#include "ThreadAdaptor.h"
#include "WideStringUtils.h"
#include "CGProxy.h"
#include "HeapProxy.h"
#include "ThreadProxy.h"
#include "ValidityChecks.h"

#include <assert.h>
#include <string.h>

#include <list>

#define JIE_FIRST_METHOD_ID 0x10000 // first ids are reserved for threads in the prf resolution

using namespace std;

using namespace Martini;
using namespace ThreadAdaptor;
using namespace JIE;
using namespace MIE;
using namespace Infrastructure;
using namespace OSA;
using namespace RTUtil;

struct TProblematicMethodInfo
{
    const char *szClassName;
    const char *szMethodName;
    TProblematicMethodInfo(const char *i_szClassName, const char *i_szMethodName)
        : szClassName(i_szClassName), szMethodName(i_szMethodName) {}
};

struct TNewObjectInfo
{
    TNewObjectInfo(char *i_szClass, TVariableID i_varId)
        : szClass(i_szClass), varId(i_varId) {}

    char *szClass;
    TVariableID varId;
};

extern "C" THREADADAPTOR_API IInstrumentationAdaptor* GetInstrumentationAdaptor()
{
    static CThreadAdaptor s_ThreadAdaptor;
    return (IInstrumentationAdaptor*)&s_ThreadAdaptor;
}

//////////////////////////////////////////////////////////////////////////
// class CThreadAdaptor implementation

//
// Default constructor
//
CThreadAdaptor::CThreadAdaptor()
    : CInstrumentationAdaptorBase(), m_pFilter(NULL)
{
    CWideStringUtils::InitWideString(m_wzThreadProxyCallbackClassName, THREADADAPTOR_CALLBACK_CLASS_NAME);
    CWideStringUtils::InitWideString(m_wzNotifyCallbackName, THREADADAPTOR_CALLBACK_EARLY_CALL_NOTIFY_NAME);
    CWideStringUtils::InitWideString(m_wzNotifyCallbackSig, THREADADAPTOR_CALLBACK_EARLY_CALL_NOTIFY_SIG);
    CWideStringUtils::InitWideString(m_wzNotifyAllCallbackName, THREADADAPTOR_CALLBACK_EARLY_CALL_NOTIFYALL_NAME);
    CWideStringUtils::InitWideString(m_wzNotifyAllCallbackSig, THREADADAPTOR_CALLBACK_EARLY_CALL_NOTIFYALL_SIG);
    CWideStringUtils::InitWideString(m_wzInterruptCallbackName, THREADADAPTOR_CALLBACK_EARLY_CALL_INTERRUPT_NAME);
    CWideStringUtils::InitWideString(m_wzInterruptCallbackSig, THREADADAPTOR_CALLBACK_EARLY_CALL_INTERRUPT_SIG);
    CWideStringUtils::InitWideString(m_wzStartCallbackName, THREADADAPTOR_CALLBACK_EARLY_CALL_START_NAME);
    CWideStringUtils::InitWideString(m_wzStartCallbackSig, THREADADAPTOR_CALLBACK_EARLY_CALL_START_SIG);

    CWideStringUtils::InitWideString(m_wzSleepStartCallbackName, THREADADAPTOR_CALLBACK_EARLY_SLEEP_START_NAME);
    CWideStringUtils::InitWideString(m_wzSleepStartCallbackSig, THREADADAPTOR_CALLBACK_EARLY_SLEEP_START_SIG);

    CWideStringUtils::InitWideString(m_wzSleepEndCallbackName, THREADADAPTOR_CALLBACK_EARLY_SLEEP_END_NAME);
    CWideStringUtils::InitWideString(m_wzSleepEndCallbackSig, THREADADAPTOR_CALLBACK_EARLY_SLEEP_END_SIG);

}

//
// Destructor
//
CThreadAdaptor::~CThreadAdaptor()
{
}

TResult CThreadAdaptor::Init(ILogAssert *pLogger, 
                             MPI::IEventFilter *pFilter,
                             MPI::BitSet configFlags)
{
    // Call common initializer
    TResult res = CInstrumentationAdaptorBase::Init(pLogger, configFlags);
    if (MRTE_FAILED(res))
    {
        return MRTE_ERROR_FAIL;
    }

    // Do ThreadAdaptor-specific initialization
    m_pFilter = (MPI::IThreadInteractionFilter*)pFilter;

    return MRTE_RESULT_OK;
}

TResult CThreadAdaptor::ModifyClass(MPI::TId classId, 
                                  const SClassFile &classToInstrument,
                                  TMemoryAllocatorFunc funcAllocator,
                                  SClassFile *pInstrumentedClass,
                                  IInstrumentationContext *pInsContext)
{
    // Validate arguments
    if (NULL == m_pJIE)
    {
        return MRTE_ERROR_MODULE_NOT_INITIALIZED;
    }
    if (NULL == pInstrumentedClass || NULL == funcAllocator || NULL == pInsContext)
    {
        return MRTE_ERROR_ILLEGAL_ARGUMENT;
    }
    if (NULL == classToInstrument.pClassFile || 0 == classToInstrument.uiSize)
    {
        return MRTE_ERROR_ILLEGAL_ARGUMENT;
    }

    InvalidateInstrumentedClassBuffer(pInstrumentedClass);

    // Parse the class file
    IJavaClass *pJavaClass = m_pJIE->GetClassInterface(classToInstrument.pClassFile,
        classToInstrument.uiSize);
    if (NULL == pJavaClass)
    {
        // Unable to parse class file
        return MRTE_ERROR_ILL_CLASS_FILE;
    }

    SJavaClassInfo jci;
    TResult res = GetJavaClassInfo(&jci, pJavaClass);
    if (MRTE_RESULT_OK != res)
    {
        return MRTE_ERROR_FAIL;
    }

    SAdaptorClassInfo classInfo;
    SetClassInfo(&classInfo, jci, classId);

    res = CanInstrumentClass(classInfo);
    if (MRTE_RESULT_OK == res)
    {
        // Apply basic instrumentation to the class if required at this stage
        if (MRTE_RESULT_OK == res)
        {
            // Write the instrumented class file to pInstrumentedClass
            SBuffer buffer;
            res = pJavaClass->WriteBack(&buffer, funcAllocator);
            if (MRTE_RESULT_OK == res)
            {
                pInstrumentedClass->pClassFile = buffer.pBuffer;
                pInstrumentedClass->uiSize = buffer.uiActualSize;
            }
        }
    }

    CContext *pContext = (CContext*)pInsContext;
    TResult copyRes = CopyInstrumentedClassInfo(&(pContext->classInfo), classInfo, 
        funcAllocator);
    if (MRTE_FAILED(copyRes))
    {
        res = copyRes;
    }

//     Clean-up
    pJavaClass->Free();
    DeallocateJavaClassInfo(&jci);
    DeallocateAdaptorClassInfo(&classInfo);

    return res; 
}



TResult CThreadAdaptor::ModifyByteCodes(MPI::TId classId, 
                                      const SClassFile &classToInstrument,
                                      TMemoryAllocatorFunc funcAllocator,
                                      SClassFile *pInstrumentedClass,
                                      IInstrumentationContext *pContext)
{
    // Validate arguments
    if (NULL == m_pJIE)
    {
        return MRTE_ERROR_MODULE_NOT_INITIALIZED;
    }
    if (NULL == pInstrumentedClass || NULL == funcAllocator)
    {
        return MRTE_ERROR_ILLEGAL_ARGUMENT;
    }
    if (NULL == classToInstrument.pClassFile || 0 == classToInstrument.uiSize)
    {
        return MRTE_ERROR_ILLEGAL_ARGUMENT;
    }

    InvalidateInstrumentedClassBuffer(pInstrumentedClass);

    // Parse the class file
    IJavaClass *pJavaClass = m_pJIE->GetClassInterface(classToInstrument.pClassFile,
        classToInstrument.uiSize);
    if (NULL == pJavaClass)
    {
        // unable to parse class file
        return MRTE_ERROR_ILL_CLASS_FILE;
    }

    CContext *pThreadContext = (CContext*)pContext;
    TResult res = CanInstrumentClass(pThreadContext->classInfo);
    if (MRTE_RESULT_OK == res)
    {
        // Apply thread instrumentation to the class
        res = DoThreadInstrumentation(pJavaClass, pThreadContext->classInfo, m_bJVMInitDone);
        if (MRTE_RESULT_OK == res)
        {
            // Write the instrumented class file to pInstrumentedClass
            SBuffer buffer;
            res = pJavaClass->WriteBack(&buffer, funcAllocator);
            if (MRTE_RESULT_OK == res)
            {
                pInstrumentedClass->pClassFile = buffer.pBuffer;
                pInstrumentedClass->uiSize = buffer.uiActualSize;
            }
        }
    }

    // Clean-up
    pJavaClass->Free();

    return res; 
}

//
// Copies a class file buffer
//
// Parameters:
//      pDest           [out]   : destination buffer
//      src             [in]    : source buffer
//      funcAllocator   [in]    : a function to use for allocating memory for pDest buffers
//
// Returns
//      MRTE_RESULT_OK              : success
//      MRTE_ERROR_OUT_OF_MEMORY    : not enough memory to complete the operation
//
TResult CThreadAdaptor::CopyClassFile(SClassFile *pDest, 
                                    const SClassFile& src,
                                    TMemoryAllocatorFunc funcAllocator)
{
    pDest->uiSize = src.uiSize;
    pDest->pClassFile = (unsigned char *)funcAllocator(src.uiSize);
    if (NULL == pDest->pClassFile)
    {
        return MRTE_ERROR_OUT_OF_MEMORY;
    }
    memcpy(pDest->pClassFile, src.pClassFile, src.uiSize);
    return MRTE_RESULT_OK;
}



//
// Add Thread instrumentation to the class
//
// Parameters:
//      pJavaClass      [in,out]    : Java class to instrument
//      classInfo       [in]        : Java class information, including an array of methods
//                                    that should be instrumented.
//      bJVMInitDone    [in]        : Whether the JVM has finished initialization or not
//
// Returns:
//      MRTE_RESULT_OK              : success
//      MRTE_ERROR_FAIL             : instrumentation failure
//
TResult CThreadAdaptor::DoThreadInstrumentation(IJavaClass *pJavaClass,
                                            const SAdaptorClassInfo &classInfo,
                                            bool bJVMInitDone)
{
    TResult res = MRTE_RESULT_OK;

    if (pJavaClass == NULL)
    {
        return (MRTE_ERROR_ILLEGAL_ARGUMENT);
    }

    TConstantPoolIndex cpNotifyCallback = pJavaClass->RegisterRecorderCallback(
        m_wzThreadProxyCallbackClassName, m_wzNotifyCallbackName, m_wzNotifyCallbackSig);

    TConstantPoolIndex cpNotifyAllCallback = pJavaClass->RegisterRecorderCallback(
        m_wzThreadProxyCallbackClassName, m_wzNotifyAllCallbackName, m_wzNotifyAllCallbackSig);

    TConstantPoolIndex cpInterruptCallback = pJavaClass->RegisterRecorderCallback(
        m_wzThreadProxyCallbackClassName, m_wzInterruptCallbackName, m_wzInterruptCallbackSig);

    TConstantPoolIndex cpStartCallback = pJavaClass->RegisterRecorderCallback(
        m_wzThreadProxyCallbackClassName, m_wzStartCallbackName, m_wzStartCallbackSig);

    TConstantPoolIndex cpSleepStartCallback = pJavaClass->RegisterRecorderCallback(
        m_wzThreadProxyCallbackClassName, m_wzSleepStartCallbackName, m_wzSleepStartCallbackSig);

    TConstantPoolIndex cpSleepEndCallback = pJavaClass->RegisterRecorderCallback(
        m_wzThreadProxyCallbackClassName, m_wzSleepEndCallbackName, m_wzSleepEndCallbackSig);

    if (cpNotifyCallback    == ILLEGAL_CP_ENTRY ||
        cpNotifyAllCallback == ILLEGAL_CP_ENTRY ||
        cpInterruptCallback == ILLEGAL_CP_ENTRY ||
        cpStartCallback     == ILLEGAL_CP_ENTRY ||
        cpSleepStartCallback == ILLEGAL_CP_ENTRY ||
        cpSleepEndCallback   == ILLEGAL_CP_ENTRY )
    {
        return (MRTE_ERROR_ILLEGAL_ARGUMENT);
    }

    SHookMethodCall MethodCallHooks [] = {
        {"notify", cpNotifyCallback},
        {"notifyAll", cpNotifyAllCallback},
        {"interrupt", cpInterruptCallback},
        {"start", cpStartCallback},
    };

    SHookMethodCall SleepMethodCallHooks [] = {
    	// These "method names" are actually placeholders to allow the hooks to be found in the array
        {"sleepStart", cpSleepStartCallback},
        {"sleepEnd", cpSleepEndCallback},
    };

    res = InstrumentMethodCalls(pJavaClass, classInfo,
    		MethodCallHooks, sizeof(MethodCallHooks)/sizeof(SHookMethodCall),
    		SleepMethodCallHooks, sizeof(SleepMethodCallHooks)/sizeof(SHookMethodCall));

    return res;
}



TResult
CThreadAdaptor::InstrumentMethodCalls(IJavaClass *pClass,
                                      const SAdaptorClassInfo &classInfo,
                                      SHookMethodCall *hooks,
                                      size_t hooksNumber,
                                      SHookMethodCall *sleepHooks,
                                      size_t sleepHooksNumber)
{
    TResult res;
    IJavaMethod *pMethod;
    TJavaMethodIterator *pMethIter = pClass->GetMethodIterator();

    SJavaMethodInfo methodInfo;
    CWideStringUtils::AllocateWideString(methodInfo.className, 1000);
    CWideStringUtils::AllocateWideString(methodInfo.methodName, 1000);
    CWideStringUtils::AllocateWideString(methodInfo.methodSignature, 1000);

    bool isNotInstrumented = true;
    while (pMethIter->HasNext())
    {
        pMethod = pMethIter->GetNext();
        res = pMethod->GetMethodInfo(&methodInfo);

        if (MRTE_ERROR_BUFFER_TOO_SHORT == res)
        {
            // One of the buffers are too short. Fix sizes and try again
            CWideStringUtils::ReallocateWideString(methodInfo.className);
            CWideStringUtils::ReallocateWideString(methodInfo.methodName);
            CWideStringUtils::ReallocateWideString(methodInfo.methodSignature);
            res = pMethod->GetMethodInfo(&methodInfo);
        }

        
        if (CanInstrumentMethod(methodInfo) &&
            CheckFilter(classInfo, methodInfo))
        {
                res = AddInstrumentationToMethod(pClass, pMethod,
                    hooks, hooksNumber, sleepHooks, sleepHooksNumber, classInfo, methodInfo);
                if (MRTE_FAILED(res))
                {
                    CWideStringUtils::FreeWideString(methodInfo.className);
                    CWideStringUtils::FreeWideString(methodInfo.methodName);
                    CWideStringUtils::FreeWideString(methodInfo.methodSignature);
                    return res;
                }
                isNotInstrumented = false;
        }
    }

    CWideStringUtils::FreeWideString(methodInfo.className);
    CWideStringUtils::FreeWideString(methodInfo.methodName);
    CWideStringUtils::FreeWideString(methodInfo.methodSignature);

    if (isNotInstrumented)
    {
        return MRTE_ERROR_INSTRUMENTATION_NOT_NEEDED;
    }
    return MRTE_RESULT_OK;
}





TResult
CThreadAdaptor::AddInstrumentationToMethod(IJavaClass *pClass,
                        IJavaMethod *pMethod,
                        SHookMethodCall *hooks,
                        size_t hooksNumber,
                        SHookMethodCall *sleepHooks,
                        size_t sleepHooksNumber,
                        const SAdaptorClassInfo &classInfo,
                        const SJavaMethodInfo &methodInfo)
{

    if (pMethod == NULL)
    {
        return (MRTE_ERROR_ILLEGAL_ARGUMENT);
    }

    TResult res;

    TInstructionListIterator *pInstIter =
        pMethod->GetInstructionListIterator(IET_ALL_INSTRUCTIONS);

    SConstantPoolEntryValue cpMethodValue;
    cpMethodValue.type = CPT_METHOD;

    CWideStringUtils::AllocateWideString(cpMethodValue.u.methodVal.className, 1000);
    CWideStringUtils::AllocateWideString(cpMethodValue.u.methodVal.name, 1000);
    CWideStringUtils::AllocateWideString(cpMethodValue.u.methodVal.signature, 1000);


    while (pInstIter->HasNext())
    {
        IInstruction *pInst = pInstIter->GetNext();
        IInstruction *pNewInst = NULL;
        SOperand opInvokeOper;

        switch (pInst->GetMnemonic())
        {
        case MNM_INVOKEVIRTUAL: {
            pInst->GetOperand(&opInvokeOper);

            if (opInvokeOper.type != OT_JAVA_CP_INDEX)
            {

                char *szMethodName = CWideStringUtils::WideStringToCString(methodInfo.methodName);

                LOG_INFORMATIVE3("ThreadAdaptor", 0, false, 
                    "Error instrumenting '%s.%s'. Reason: "
                    "invoke operand type != OT_JAVA_CP_INDEX", 
                    classInfo.szClassName, szMethodName, res);

                delete[] szMethodName;
                break;
            }

            res = pClass->GetValueFromCPEntry(opInvokeOper.val.cpIndex, &cpMethodValue);
            if (MRTE_ERROR_BUFFER_TOO_SHORT == res)
            {
                // One of the buffers are too short. Fix sizes and try again
                CWideStringUtils::ReallocateWideString(cpMethodValue.u.methodVal.className);
                CWideStringUtils::ReallocateWideString(cpMethodValue.u.methodVal.name);
                CWideStringUtils::ReallocateWideString(cpMethodValue.u.methodVal.signature);
                res = pClass->GetValueFromCPEntry(opInvokeOper.val.cpIndex, &cpMethodValue);
            }
            if (cpMethodValue.type != CPT_METHOD)
            {
                char *szMethodName = CWideStringUtils::WideStringToCString(methodInfo.methodName);

                LOG_INFORMATIVE3("ThreadAdaptor", 0, false, 
                    "Error instrumenting '%s.%s'. Reason: "
                    "invoked method type != CPT_METHOD", 
                    classInfo.szClassName, szMethodName, res);

                delete[] szMethodName;
                break;
            }
                
            char *szInvokedMethod = CWideStringUtils::WideStringToCString(cpMethodValue.u.methodVal.name);
            char *szInvokedMethodSig = CWideStringUtils::WideStringToCString(cpMethodValue.u.methodVal.signature);

            for (int i = 0; i < hooksNumber; i++)
            {
                // Add signature condition for bug 296619.
                // Signatures for notify, notifyAll, interrupt and start are the same
                if (strcmp(szInvokedMethod, hooks[i].methodName) == 0 && 
                	strcmp(szInvokedMethodSig, "()V") == 0)
                {
                    AddNotifyProbeBeforeInstr(pInst, hooks[i].cpCallback);
                }
            }

            delete [] szInvokedMethod;
            delete [] szInvokedMethodSig;
            
            break; }

		case MNM_INVOKESTATIC: {

            pInst->GetOperand(&opInvokeOper);
            if (opInvokeOper.type != OT_JAVA_CP_INDEX)
            {

                char *szMethodName = CWideStringUtils::WideStringToCString(methodInfo.methodName);

                LOG_INFORMATIVE3("ThreadAdaptor", 0, false,
                    "Error instrumenting '%s.%s'. Reason: "
                    "invoke operand type != OT_JAVA_CP_INDEX",
                    classInfo.szClassName, szMethodName, res);

                delete[] szMethodName;
                break;
            }

            res = pClass->GetValueFromCPEntry(opInvokeOper.val.cpIndex, &cpMethodValue);
            if (MRTE_ERROR_BUFFER_TOO_SHORT == res)
            {
                // One of the buffers are too shcdort. Fix sizes and try again
                CWideStringUtils::ReallocateWideString(cpMethodValue.u.methodVal.className);
                CWideStringUtils::ReallocateWideString(cpMethodValue.u.methodVal.name);
                CWideStringUtils::ReallocateWideString(cpMethodValue.u.methodVal.signature);
                res = pClass->GetValueFromCPEntry(opInvokeOper.val.cpIndex, &cpMethodValue);
            }
            if (cpMethodValue.type != CPT_METHOD)
            {
                char *szMethodName = CWideStringUtils::WideStringToCString(methodInfo.methodName);

                LOG_INFORMATIVE3("ThreadAdaptor", 0, false,
                    "Error instrumenting '%s.%s'. Reason: "
                    "invoked method type != CPT_METHOD",
                    classInfo.szClassName, szMethodName, res);

                delete[] szMethodName;
                break;
            }

            char *szInvokedClassName = CWideStringUtils::WideStringToCString(cpMethodValue.u.methodVal.className);
            char *szInvokedMethod = CWideStringUtils::WideStringToCString(cpMethodValue.u.methodVal.name);

            bool parentClassIsThread = false;

            // If the name of the class being instrumented is equal to the name of the class that
            // we are calling sleep() on.....
            if(szInvokedClassName != NULL && classInfo.szClassName != NULL &&
            		strcmp(szInvokedClassName, classInfo.szClassName) == 0) {

            	// We check to see if the method that contains the invokestatic
            	// is calling a member method of itself, and then we are checking if the
            	// parent of the method's class is java/lang/Thread
            	char *szParentClassName;

                SJavaClassInfo jci;
                TResult res = GetJavaClassInfo(&jci, pClass);
    			szParentClassName = CWideStringUtils::WideStringToCString(jci.superClassName);
                DeallocateJavaClassInfo(&jci);

                if(strcmp(szParentClassName, "java/lang/Thread") == 0) {
                	parentClassIsThread = true;
                }
                delete [] szParentClassName;

            }


            // If the Thread.sleep() is being called, or if sleep() is being called on a class that extends Thread
			if(strcmp(szInvokedClassName, "java/lang/Thread") == 0 || parentClassIsThread) {

				if (strcmp(szInvokedMethod, "sleep") == 0)
				{
					TConstantPoolIndex cpStartCallback = 0;
					TConstantPoolIndex cpEndCallback = 0;

					// Locate the start and end callbacks in the list
					for(int x = 0; x < sleepHooksNumber; x++) {
						SHookMethodCall i = sleepHooks[x];
						if(strcmp(i.methodName,"sleepStart") == 0) {
					        cpStartCallback = i.cpCallback;
						}

						if(strcmp(i.methodName,"sleepEnd") == 0) {
					        cpEndCallback = i.cpCallback;
						}
					}

					if(cpStartCallback != 0 && cpEndCallback != 0) {
						AddNotifyProbesBeforeSleepInstr(pInst, cpStartCallback, cpEndCallback);
					} else {
						LOG_ERROR("CThreadAdaptor", "Unable to resolve sleep callbacks");
					}

				}
            }
			delete [] szInvokedClassName;
            delete [] szInvokedMethod;

            break;
		}
        case MNM_INVOKESPECIAL:
        case MNM_INVOKEINTERFACE: {

            break; }
        }
    }

    CWideStringUtils::FreeWideString(cpMethodValue.u.methodVal.className);
    CWideStringUtils::FreeWideString(cpMethodValue.u.methodVal.name);
    CWideStringUtils::FreeWideString(cpMethodValue.u.methodVal.signature);
    
    
    pInstIter->Free();
    res = pMethod->ApplyInstrumentation();

    return (res);
}


/**
 * AddNotifyProbeBeforeInstr is called on an 'invokevirtual' instruction (pInst).
 * At the top of the stack will be an object (objectref), which the invokevirtual
 * instruction is to be called on. The method being called necssarily has no arguments (we check its signature)
 * and so the object being called will necessarily be at the top of the stack.
 *
 * AddNotifyProbeBeforeInstr dups the object on the top of the stack, and then inserts
 * a new invokestatic call, with the dupped object as its first parameter.
 */
TResult CThreadAdaptor::AddNotifyProbeBeforeInstr(IInstruction *pInst,
                                            TConstantPoolIndex cpCallback)
{
    TResult res;
    IInstruction *pNewInst = pInst;
    SOperand opInvokeOper;

    res = pNewInst->AddBefore(&pNewInst, MNM_DUP);

    if (MRTE_FAILED(res))
    {
        return (res);
    }

    opInvokeOper.type = OT_JAVA_CP_INDEX;
    opInvokeOper.val.cpIndex = cpCallback;
    res = pNewInst->AddAfter(&pNewInst, MNM_INVOKESTATIC, &opInvokeOper);

    return (MRTE_RESULT_OK);
}

/** Add an invocation of our thread callback class before and after the invokestatic on the Thread.sleep() call */
TResult
CThreadAdaptor::AddNotifyProbesBeforeSleepInstr(IInstruction *pInst, TConstantPoolIndex cpStartCallback, TConstantPoolIndex cpEndCallback)
{
    TResult res;
    IInstruction * pNewInst = pInst;
    IInstruction * pCreatedInst;
    SOperand opStartInvokeOper;
    SOperand opEndInvokeOper;

    opStartInvokeOper.type = OT_JAVA_CP_INDEX;
    opStartInvokeOper.val.cpIndex = cpStartCallback;
    res = pNewInst->AddBefore(&pCreatedInst, MNM_INVOKESTATIC, &opStartInvokeOper);

    opEndInvokeOper.type = OT_JAVA_CP_INDEX;
    opEndInvokeOper.val.cpIndex = cpEndCallback;
    res = pNewInst->AddAfter(&pCreatedInst, MNM_INVOKESTATIC, &opEndInvokeOper);

    return (MRTE_RESULT_OK);
}

//
// Checks if the class can be instrumented.
//
// Parameters:
//      info    : class information
//
// Returns:
//      MRTE_RESULT_OK                          : class can be instrumented
//      MRTE_ERROR_INSTRUMENTATION_NOT_NEEDED   : the class will not be instrumented
//                                                at all. This value is returned
//                                                for interface classes.
//      MRTE_ERROR_UNABLE_TO_INSTRUMENT         : the class cannot be instrumented.
//                                                This value is returned for classes
//                                                the Thread Adaptor cannot
//                                                instrument.
//
TResult CThreadAdaptor::CanInstrumentClass(const SAdaptorClassInfo &info)
{
    static char *ExcludedClassPatterns[] = 
    {
        CGADAPTOR_CALLBACK_CLASS_NAME,              // CGAdaptor recorder class
        HEAPADAPTOR_CALLBACK_CLASS_NAME,            // HeapAdaptor recorder class
        THREADADAPTOR_CALLBACK_CLASS_NAME,          // ThreadAdaptor recorder class

        "jrockit/",                                 // Internal JRockit classes. 
                                                    // Can't force-load some of them
                                                    // during attach/detach
        "java/lang/J9VM",                           // IBM Internal classes (J9 VM)
        "java/lang/Object",                         // Do not instrument j.l.Object
        "com/ibm/crypto",                           // Cause verification errors. //TODO: fix
        "sun/reflect/Generated",                    // Classes generated during runtime to 
                                                    // support reflection
        "com/Intel/VTune/VTuneAPI",                 // VTune API Proxy class
        "org/eclipse/hyades/collection/profiler",    // TPTP Profiler API Proxy class
        "java/lang/",                               // Patch to IBM redefinition problem
        "java/security/",
        "java/awt/",
        "javax/swing/"
        "com/ibm/jvm/",
        "com/ibm/misc/",
        "com/ibm/oti/",
        "sun/security/",
        "sun/misc/",
        "sun/awt/",
        "sun/java2d/",
        "org/eclipse/hyades/collection/profiler"    // TPTP Profiler API Proxy class
    };

    // Can't instrument interface classes as they contain no code
    if (CAF_INTERFACE & info.uiAttributeFlags)
    {
        return MRTE_ERROR_INSTRUMENTATION_NOT_NEEDED;
    }

    // Check if the class appear in the excluded class patterns list
    const char *szClassName = info.szClassName;
    unsigned int i;
    for (i = 0; i < SIZE_OF_ARRAY(ExcludedClassPatterns); ++i)
    {
        char *szExcludedPattern = ExcludedClassPatterns[i];
        if (strstr(szClassName, szExcludedPattern) != NULL)
        {
            LOG_INFORMATIVE1("CThreadAdaptor", 5, false, "Class '%s' will not be instrumented",
                szClassName);
            return MRTE_ERROR_UNABLE_TO_INSTRUMENT;
        }
    }

    // The class can be instrumented
    return MRTE_RESULT_OK;
}

//
// Checks if the method can be instrumented
//
// Parameters:
//      methodInfo  : method information
//
// Returns:
//      true    : the method can be instrumented
//      false   : the method cannot be instrumented
//
bool CThreadAdaptor::CanInstrumentMethod(const SJavaMethodInfo &methodInfo)
{
    //TODO: log methods that are not instrumented
    if (methodInfo.uiAttributeFlags & MAF_NATIVE ||
        methodInfo.uiAttributeFlags & MAF_ABSTRACT)
    {
        // can't instrument native and abstract methods as they have no code
        return false;
    }

    char *szClassName = CWideStringUtils::WideStringToCString(methodInfo.className);
    char *szMethodName = CWideStringUtils::WideStringToCString(methodInfo.methodName);
    bool bIsProblematic = IsProblematicMethod(szClassName, szMethodName);
    delete [] szClassName;
    delete [] szMethodName;

    return (!bIsProblematic);
}

bool CThreadAdaptor::IsProblematicMethod(const char *szClassName, 
                                       const char *szMethodName)
{
    if (   strcmp(szClassName, "java/lang/ref/Reference") == 0
        || strcmp(szClassName, "java/lang/ref/Finalizer") == 0
        )
    {
        return true;
    }
    return false;
}


bool CThreadAdaptor::CheckFilter(const SAdaptorClassInfo &classInfo,
                                 const SJavaMethodInfo &methodInfo)
{
    bool notified;
    char *szMethodName;
    if (NULL == m_pFilter)
    {
        // No selectivity callback is defined. All methods should be instrumented
        return true;
    }

    szMethodName = CWideStringUtils::WideStringToCString(methodInfo.methodName);
    // Invoke the selectivity callback
    MPI::SThreadInteractionFilterData threadInteractionFilter;
    threadInteractionFilter.szClassName = classInfo.szClassName;
    threadInteractionFilter.szMethodName = szMethodName;

    notified = m_pFilter->ShouldNotify(threadInteractionFilter);

    delete [] szMethodName;
    return notified;
} 
