/*****************************************************************************
 * Copyright (c) 1997-2007, Intel Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *
 * $Id$ 
 *****************************************************************************/
#include <string.h>
#include <stdlib.h>

#include "BootLoaderOptionParser.h"
#include "MString.h"

using namespace Martini::Infrastructure;
using namespace Martini::RTUtil;

//////////////////////////////////////////////////////////////////////
// CBootLoaderOptionParser implementation

//
// Constructor
//
CBootLoaderOptionParser::CBootLoaderOptionParser()
{
}

//
// Destructor
//
CBootLoaderOptionParser::~CBootLoaderOptionParser()
{
}

/**
 * @brief <b>Tokenize</b> Tokenize a string by the specified delimiters
 *
 * @param pStringList  [out] : a list to be filled with string tokens
 * @param szSource      [in] : the string to tokenize
 * @param szDelims      [in] : a string containing the delimiters by which to tokenize
 */
void CBootLoaderOptionParser::Tokenize(TStringList *pStringList, 
                                       const char *szSource,
                                       const char *szDelims)
{
    char *s = strdup(szSource);
    char *szToken = strtok(s, szDelims);
    while (NULL != szToken)
    {
        TMRTEHandle hLast = pStringList->GetLast();
        pStringList->InsertAfter(hLast, strdup(szToken));
        szToken = strtok(NULL, szDelims);
    }
    free(s);
}

/**
 * @brief <b>FreeStringListItems</b> Free memory allocated for StringList items
 *
 * @param stringList    [in] : the list to free
 */
void CBootLoaderOptionParser::FreeStringListItems(TStringList &stringList)
{

    TMRTEHandle h = stringList.GetFirst();
    while (h != NULL)
    {
        char *sz = stringList.GetData(h);
        free(sz);
        h = stringList.GetNext(h);
    }

}

/**
 * @brief <b>ParseModuleSpec</b> Parse module specifications
 * 
 * @param pSpec    [out] : a placeholder for the module specification
 * @param szSpec    [in] : module specification string to parse
 * 
 * @returns MRTE_RESULT_OK  : success
 * @returns MRTE_ERROR_FAIL : failure
 */
TResult CBootLoaderOptionParser::ParseModuleSpec(SModuleSpec *pSpec, const char *szSpec)
{
    // Split the module name and its options around the first ":" character
    const char *szOpt = strchr(szSpec, ':');
    if (NULL == szOpt)
    {
        pSpec->strOptions.Set("");
    }
    else
    {
        pSpec->strOptions.Set(szOpt + 1); // skip the ":"
    }
    char szName[1024];
    memset(szName, 0, 1024);
    size_t cbSpec = (NULL != szSpec) ? strlen(szSpec) : 0;
    size_t cbOpt = (NULL != szOpt) ? strlen(szOpt) : 0;
    strncpy(szName, szSpec, cbSpec - cbOpt);
    pSpec->strName.Set(szName);
    return MRTE_RESULT_OK;
}

/**
 * @brief <b>Parse</b> Parses boot-loader arguments
 * 
 * @param pECSpec              [out] : placeholder for EC module specifications
 * @param pProfilerSpecList    [out] : placeholder for a list of profiler specifications
 * @param szOptions             [in] : command-line arguments passed to the JPI bootloader 
 *
 * @returns MRTE_RESULT_OK      : success
 * @returns MRTE_ERROR_NULL_PTR : one of the parameters is null
 * @returns MRTE_ERROR_FAIL     : failure
 */
TResult CBootLoaderOptionParser::Parse(SModuleSpec *pECSpec, 
                                       TModuleSpecList *pProfilerSpecList,
                                       const char *szOptions)
{
    if (NULL == pECSpec || NULL == pProfilerSpecList || NULL == szOptions)
    {
        return MRTE_ERROR_NULL_PTR;
    }

    //
    // Command line syntax
    // ec_spec;profiler_spec[;profiler_spec]*
    //
    // ec_spec -> ec_module[:param=value[,param=value]*]
    // profiler_spec -> profiler_name[:param=value[,param=value]*]
    //
    TStringList specList;
    Tokenize(&specList, szOptions, ";");
    TMRTEHandle hSpec = specList.GetFirst();
    if (NULL == hSpec)
    {
        return MRTE_ERROR_FAIL;
    }

    // parse EC module spec
    char *szECSpec = specList.GetData(hSpec);
    TResult iRes = ParseModuleSpec(pECSpec, szECSpec);
    if (MRTE_FAILED(iRes))
    {
        FreeStringListItems(specList);
        return MRTE_ERROR_FAIL;
    }

    // parse profiler spec(s)
    hSpec = specList.GetNext(hSpec);
    if (NULL == hSpec)
    {
        // profiler spec is missing
        FreeStringListItems(specList);
        return MRTE_ERROR_FAIL;
    }
    while (NULL != hSpec)
    {
        char *szProfSpec = specList.GetData(hSpec);
        SModuleSpec profSpec;
        TResult iRes = ParseModuleSpec(&profSpec, szProfSpec);
        if (MRTE_SUCCEEDED(iRes))
        {
            pProfilerSpecList->InsertAfter(pProfilerSpecList->GetLast(), profSpec);
        }
        else
        {
            FreeStringListItems(specList);
            return MRTE_ERROR_FAIL;
        }
        hSpec = specList.GetNext(hSpec);
    }

    // cleanup
    FreeStringListItems(specList);
    return MRTE_RESULT_OK;
}
