/*****************************************************************************
 * Copyright (c) 1997-2007, Intel Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *
 * $Id$ 
 *****************************************************************************/


#include "ECAgent.h"
#include "MRTEInfrastructureDefinitions.h"
#include "OSA.h"
#include "LibraryLoader.h"
#include "LogAssert.h"

#include "string.h"

using namespace Martini::ExternalControl;
using namespace Martini::MPI;
using namespace Martini::OSA;
using namespace Martini::Infrastructure;
using namespace Martini::RTUtil;

CECAgent* CECAgent::GetInstance()
{
    static CECAgent instance;
    return &instance;
}


CECAgent::CECAgent() : m_pEventManager(0), m_pRegisterMartiniEvent(0), m_pMartiniMessage(0),
m_bIsECAvailable(false)
{
    memset(m_vuiRegisteredEvents, 0, sizeof(unsigned int) * (int)CMD_LAST);
}

CECAgent::~CECAgent()
{
}

//
// Initializes the EC Agent module
//
// Parameters:
//      clientId        [in] :   the MPI id assigned to the EC Agent module
//      szECName        [in] :   the name of the EC module to load
//      pEventManager   [in] :   pointer to the Event Manager module
//      szOptions       [in] :   EC module options
//
TResult CECAgent::Initialize(TId clientId, 
                             const char* szECName, 
                             IEventManager *pEventManager,
                             const char *szOptions)
{
    m_pEventManager = pEventManager;
    m_iClient = clientId;
    m_szConfigData = NULL;

    ILibraryLoader *pECLib = NULL;
    pECLib = LoadBistroLibrary(szECName, true);
    if (!pECLib)
    {
        MARTINI_INFORMATIVE2("CECAgent", 0, false, 
            "Failed to load External Control library '%s' (options='%s'). Proceeding without "
            "an External Control", szECName, szOptions);
        return MRTE_RESULT_NO_EXTERNAL_CONTROL;
    }
    
    m_pRegisterMartiniEvent = (TRegisterMartiniCallback)pECLib->GetEntry("RegisterMartiniCallback");
    m_pMartiniMessage = (TMartiniMessage)pECLib->GetEntry("MartiniMessage");
    m_pMartiniProcessShutdown = (TMartiniProcessShutdown)pECLib->GetEntry("MartiniProcessShutdown");

    if (!m_pMartiniMessage || !m_pRegisterMartiniEvent || !m_pMartiniProcessShutdown)
    {
        m_pRegisterMartiniEvent = 0;
        m_pMartiniMessage = 0;
        m_pMartiniProcessShutdown = 0;
        return MRTE_ERROR_LIBRARY_FAILURE;
    }

    // invoke the EC module initialization function
    TECInit pfnECInit = (TECInit)pECLib->GetEntry("EC_Init");
    if (pfnECInit)
    {
        char *szWorkingDir = new char[1000];
        memset(szWorkingDir, 0, 1000);
        TResult iRes = pfnECInit(&szWorkingDir, szOptions);
        if (MRTE_RESULT_OK == iRes)
        {
            if (0 != szWorkingDir[0])
            {
                HandleEvent(EV_EC_SET_OUTPUT_DIRECTORY, szWorkingDir);
                delete [] szWorkingDir;
            }
        }
        else
        {
            // the EC module failed to initialize
            delete [] szWorkingDir;
            pECLib->Destroy();
            return MRTE_ERROR_FAIL;
        }
    }
    pECLib->Destroy();
 
    // mark all external control events as implemented by the EC module. This will cause
    // all registration requests for these events to be forwarded to the EC
    m_pEventManager->SetEventImplementor(EV_EC_SET_OUTPUT_DIRECTORY, MPI_EXTERNAL_CONTROL);
    m_pEventManager->SetEventImplementor(EV_EC_START, MPI_EXTERNAL_CONTROL);
    m_pEventManager->SetEventImplementor(EV_EC_STOP, MPI_EXTERNAL_CONTROL);
    m_pEventManager->SetEventImplementor(EV_EC_ATTACH, MPI_EXTERNAL_CONTROL);
    m_pEventManager->SetEventImplementor(EV_EC_DETACH, MPI_EXTERNAL_CONTROL);
    m_pEventManager->SetEventImplementor(EV_EC_CUSTOM_COMMAND, MPI_EXTERNAL_CONTROL);

    // register for the VM_SHUTDOWN event
    m_pEventManager->RegisterEvent(m_iClient, *this);

    m_bIsECAvailable = true;
    MARTINI_INFORMATIVE1("ECAgent", 3, false, "External Control library '%s' initialized", 
        szECName);
    return MRTE_RESULT_OK;
}


void ECConfigureCallback(char* szPath)
{
    CECAgent::GetInstance()->HandleEvent(EV_EC_SET_OUTPUT_DIRECTORY, szPath);
}

void ECStartCallback(void)
{
    CECAgent::GetInstance()->HandleEvent(EV_EC_START, NULL);
}

void ECStopCallback(void)
{
    CECAgent::GetInstance()->HandleEvent(EV_EC_STOP, NULL);
}

void ECAttachCallback(void)
{
    CECAgent::GetInstance()->HandleEvent(EV_EC_ATTACH, NULL);
}

void ECDetachCallback(void)
{
    CECAgent::GetInstance()->HandleEvent(EV_EC_DETACH, NULL);
}

void ECCustomCommandCallback(unsigned int commandId, void *pData)
{
    CECAgent::GetInstance()->NotifyCustomCommandEvent(commandId, pData);
}

TResult CECAgent::RegisterEvent(TEventType event)
{
    if (m_bIsECAvailable == false)
    {
        return MRTE_ERROR_FAIL;
    }

    UCallback callback;
    ECommand command;
    switch(event) 
    {
    case EV_EC_SET_OUTPUT_DIRECTORY:
        callback.pfnConfigure = ECConfigureCallback;
        command = CMD_CONFIGURE;
        break;
    case EV_EC_START:
        callback.pfnStart = ECStartCallback;
        command = CMD_START;
        break;
    case EV_EC_STOP:
        callback.pfnStop = ECStopCallback;
        command = CMD_STOP;
        break;
    case EV_EC_ATTACH:
        callback.pfnAttach = ECAttachCallback;
        command = CMD_ATTACH;
        break;
    case EV_EC_DETACH:
        callback.pfnDetach = ECDetachCallback;
        command = CMD_DETACH;
        break;
    case EV_EC_CUSTOM_COMMAND:
        callback.pfnCustomCommand = ECCustomCommandCallback;
        command = CMD_CUSTOM;
        break;
    default:
        return MRTE_ERROR_ILLEGAL_ARGUMENT;
    }

    if (m_vuiRegisteredEvents[command])
    {
        if (event == EV_EC_SET_OUTPUT_DIRECTORY && m_szConfigData != NULL)
        {
            HandleEvent(EV_EC_SET_OUTPUT_DIRECTORY, m_szConfigData);
        }
        return MRTE_RESULT_PRV_INSTANTIATED;
    }
    m_vuiRegisteredEvents[command] = m_pRegisterMartiniEvent(command, callback);
    if (!m_vuiRegisteredEvents[command])
    {
        return MRTE_ERROR_FAIL;
    }

    return MRTE_RESULT_OK;
}

void CECAgent::HandleEvent(TEventType event, char* szData)
{
    if (event == EV_EC_SET_OUTPUT_DIRECTORY)
    {
        if (NULL != szData)
        {
            if (!m_szConfigData || strcmp(m_szConfigData, szData) != 0)
            {
                if (m_szConfigData)
                {
                    delete [] m_szConfigData;
                }
                m_szConfigData = new char[strlen(szData) + 1];
                strcpy(m_szConfigData, szData);
            }
        }
        
        m_pEventManager->NotifyECEvent(event, m_szConfigData, 0, NULL);
    }
    else
    {
        m_pEventManager->NotifyECEvent(event, NULL, 0, NULL);
    }
}

void CECAgent::NotifyCustomCommandEvent(unsigned int commandId, void *pData)
{
    m_pEventManager->NotifyECEvent(EV_EC_CUSTOM_COMMAND, NULL, commandId, pData);
}

void CECAgent::MartiniMessage(char* szMsg, ESeverityLevel level)
{
    if (m_bIsECAvailable == false)
    {
        return;
    }

    m_pMartiniMessage(szMsg, level);
}

void CECAgent::MartiniProcessShutdown()
{
    if (m_bIsECAvailable == false)
    {
        return;
    }

    m_pMartiniProcessShutdown();
}

//
// VM Shutdown event handler
void CECAgent::HandleEvent()
{
    MartiniProcessShutdown();
}

