/*****************************************************************************
 * Copyright (c) 1997, 2008 Intel Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *
 * $Id$ 
 *****************************************************************************/

#include "InstrumentationAdaptorBase.h"
#include "LibraryLoader.h"
#include "WideStringUtils.h"
#include "MRTEInfrastructureDefinitions.h"
#include "ValidityChecks.h"

#include <assert.h>
#include <string.h>

#pragma warning (disable : 4244)
U32 SAFE_U64_TO_U32(U64 u64Val)
{
    U32 u32Val = (0x00000000FFFFFFFF & u64Val);
    return u32Val;
}
#pragma warning (default : 4244)

using namespace Martini;
using namespace JIE;
using namespace MIE;
using namespace Infrastructure;
using namespace OSA;
using namespace RTUtil;
                           
typedef IJIE* (*TGetJIEInstance)(void);

const unsigned int STR_SIZE = 1024;

//////////////////////////////////////////////////////////////////////////
// class CInstrumentationAdaptorBase implementation

//
// Default constructor
//
CInstrumentationAdaptorBase::CInstrumentationAdaptorBase()
: m_bJVMInitDone(false), m_pLogger(NULL), m_configFlags(MPI::CF_NONE)
{
}

//
// Destructor
//
CInstrumentationAdaptorBase::~CInstrumentationAdaptorBase()
{
}

TResult CInstrumentationAdaptorBase::Init(ILogAssert *pLogger /* = NULL */, 
                                          MPI::BitSet configFlags /* = MPI::CF_NONE */)
{
    m_pLogger = pLogger;
    m_configFlags = configFlags;
    m_pJIE = InitJIE();
    if (NULL == m_pJIE)
    {
        return MRTE_ERROR_FAIL;
    }

    return MRTE_RESULT_OK;
}

void CInstrumentationAdaptorBase::JvmInitDone()
{
    if (IsStackMapCalcEnabled())
    {
        m_pJIE->EnableStackMapCalculation();
    }
    m_bJVMInitDone = true;
}

bool CInstrumentationAdaptorBase::IsStackMapCalcEnabled()
{
    return (m_configFlags & MPI::CF_JAVA_ENABLE_STACKMAP_CALC);
}

IJIE* CInstrumentationAdaptorBase::InitJIE()
{
    const char *JIE_DLL = "JIE";
    IJIE *pJIE = NULL;
   

    ILibraryLoader *pLibrary = LoadBistroLibrary(JIE_DLL);
    if (NULL == pLibrary)
    {
        return NULL;
    }
    TGetJIEInstance pfnGetJIEInstance = (TGetJIEInstance)pLibrary->GetEntry("GetJIEInstance");

    if (pfnGetJIEInstance)
    {
        pJIE = pfnGetJIEInstance();
    }

    pLibrary->Destroy();
    return pJIE;
}

TResult CInstrumentationAdaptorBase::GetJavaClassInfo(SJavaClassInfo *pInfo, 
                                                      IJavaClass *pClass)
{
    CWideStringUtils::AllocateWideString(pInfo->className, STR_SIZE);
    CWideStringUtils::AllocateWideString(pInfo->superClassName, STR_SIZE);
    CWideStringUtils::AllocateWideString(pInfo->sourceFileName, STR_SIZE);

    TResult res;
    res = pClass->GetInfo(pInfo);
    if (MRTE_ERROR_BUFFER_TOO_SHORT == res)
    {
        // One of the buffers are too short. Fix sizes and try again
        CWideStringUtils::ReallocateWideString(pInfo->className);
        CWideStringUtils::ReallocateWideString(pInfo->superClassName);
        CWideStringUtils::ReallocateWideString(pInfo->sourceFileName);
        res = pClass->GetInfo(pInfo);
    }
    return res;
}

TResult CInstrumentationAdaptorBase::GetJavaMethodInfo(SJavaMethodInfo *pInfo, 
                                                       IJavaMethod *pMethod)
{
    CWideStringUtils::AllocateWideString(pInfo->className, STR_SIZE);
    CWideStringUtils::AllocateWideString(pInfo->methodName, STR_SIZE);
    CWideStringUtils::AllocateWideString(pInfo->methodSignature, STR_SIZE);

    TResult res;
    res = pMethod->GetMethodInfo(pInfo);
    if (MRTE_ERROR_BUFFER_TOO_SHORT == res)
    {
        // One of the buffers are too short. Fix sizes and try again
        CWideStringUtils::ReallocateWideString(pInfo->className);
        CWideStringUtils::ReallocateWideString(pInfo->methodName);
        CWideStringUtils::ReallocateWideString(pInfo->methodSignature);
        res = pMethod->GetMethodInfo(pInfo);
    }
    return res;
}

void CInstrumentationAdaptorBase::InvalidateInstrumentedClassBuffer(SClassFile *pClassBuffer)
{
    pClassBuffer->pClassFile = NULL;
    pClassBuffer->uiSize = 0;
}

void CInstrumentationAdaptorBase::DeallocateJavaClassInfo(SJavaClassInfo *pClassInfo)
{
    CWideStringUtils::FreeWideString(pClassInfo->className);
    CWideStringUtils::FreeWideString(pClassInfo->superClassName);
    CWideStringUtils::FreeWideString(pClassInfo->sourceFileName);
}

void CInstrumentationAdaptorBase::DeallocateJavaMethodInfo(SJavaMethodInfo *pMethodInfo)
{
    CWideStringUtils::FreeWideString(pMethodInfo->className);
    CWideStringUtils::FreeWideString(pMethodInfo->methodName);
    CWideStringUtils::FreeWideString(pMethodInfo->methodSignature);
}

//
// Sets pClassInfo members according to the given arguments
//
void CInstrumentationAdaptorBase::SetClassInfo(SAdaptorClassInfo *pClassInfo,
                                               const SJavaClassInfo &jieInfo, 
                                               const MPI_NS::TId uiClassMPIId)
{
    pClassInfo->szClassName = CWideStringUtils::WideStringToCString(jieInfo.className);
    pClassInfo->szSourceFileName = CWideStringUtils::WideStringToCString(
        jieInfo.sourceFileName);
    pClassInfo->classId = uiClassMPIId;
    pClassInfo->uiAttributeFlags = jieInfo.uiAttributeFlags;
    pClassInfo->pMethods = NULL;
    pClassInfo->uiNumMethods = 0;
}

//
// Copies instrumented class information from src to pDest.
//
// NOTE: the function assumes all arguments are valid and does not perform any validity
// checks.
//
// Parameters:
//      pDest           [out]   : unallocated SClassInfo* pointer to which to copy information.
//                                Memory for this parameter is allocated by using funcAllocator
//      src             [in]    : information to copy
//      funcAllocator   [in]    : function to use for memory allocation
//
// Returns:
//      MRTE_RESULT_OK              : success
//      MRTE_ERROR_OUT_OF_MEMORY    : not enough memory
//
TResult CInstrumentationAdaptorBase::CopyInstrumentedClassInfo(SAdaptorClassInfo *pDest, 
                                                               const SAdaptorClassInfo &src,
                                                               TMemoryAllocatorFunc funcAllocator)
{
    memset(pDest, 0, sizeof(SAdaptorClassInfo));
    pDest->classId = src.classId;
    pDest->uiAttributeFlags = src.uiAttributeFlags;
    pDest->uiNumMethods = src.uiNumMethods;
    
    // copy class name
    size_t cbClassName = strlen(src.szClassName);
    char *szDestClassName = (char*)funcAllocator(cbClassName + 1);
    CHECK_ALLOC(szDestClassName)
    strcpy(szDestClassName, src.szClassName);
    pDest->szClassName = szDestClassName;
    
    // copy class source file name
    size_t cbSourceFileName = strlen(src.szSourceFileName);
    char *szDestSourceFileName = (char*)funcAllocator(cbSourceFileName + 1);
    CHECK_ALLOC(szDestSourceFileName)
    strcpy(szDestSourceFileName, src.szSourceFileName);
    pDest->szSourceFileName = szDestSourceFileName;
    
    // copy methods information
    if (pDest->uiNumMethods > 0)
    {
        size_t cbMethodName;
        size_t cbMethodSig;
        char *szDestMethodName;
        char *szDestMethodSig;

        pDest->pMethods = (SAdaptorMethodInfo*)funcAllocator(
            sizeof(SAdaptorMethodInfo) * pDest->uiNumMethods);
        CHECK_ALLOC(pDest->pMethods)
        unsigned int i;
        for (i = 0; i < pDest->uiNumMethods; ++i)
        {
            SAdaptorMethodInfo &srcMethodInfo = src.pMethods[i];
            SAdaptorMethodInfo &destMethodInfo = pDest->pMethods[i];
            destMethodInfo.id = srcMethodInfo.id;
            destMethodInfo.cpAlreadyInvokedFlag = srcMethodInfo.cpAlreadyInvokedFlag;

            // copy method name
            cbMethodName = strlen(srcMethodInfo.szName);
            szDestMethodName = (char*)funcAllocator(cbMethodName + 1);
            CHECK_ALLOC(szDestMethodName)
            strcpy(szDestMethodName, srcMethodInfo.szName);
            destMethodInfo.szName = szDestMethodName;

            // copy method signature
            cbMethodSig = strlen(srcMethodInfo.szSignature);
            szDestMethodSig = (char*)funcAllocator(cbMethodSig + 1);
            CHECK_ALLOC(szDestMethodSig)
            strcpy(szDestMethodSig, srcMethodInfo.szSignature);
            destMethodInfo.szSignature = szDestMethodSig;

            // copy method attributes
            destMethodInfo.uiAttributeFlags = srcMethodInfo.uiAttributeFlags;
        }
    }

    return MRTE_RESULT_OK;
}

void CInstrumentationAdaptorBase::DeallocateAdaptorClassInfo(SAdaptorClassInfo *pInstrClassInfo)
{
    delete [] const_cast<char *>(pInstrClassInfo->szClassName);
    delete [] const_cast<char *>(pInstrClassInfo->szSourceFileName);
    unsigned int i;
    for (i = 0; i < pInstrClassInfo->uiNumMethods; ++i)
    {
        DeallocateAdaptorMethodInfo(&(pInstrClassInfo->pMethods[i]));
    }
    delete [] pInstrClassInfo->pMethods;
}

void CInstrumentationAdaptorBase::DeallocateAdaptorMethodInfo(SAdaptorMethodInfo *pInstrMethodInfo)
{
    delete [] const_cast<char *>(pInstrMethodInfo->szName);
    delete [] const_cast<char *>(pInstrMethodInfo->szSignature);
}

//
// Copies all methods from vecMethods to pClassInstrInfo
//
void CInstrumentationAdaptorBase::SetMethodInfo(SAdaptorClassInfo *pClassInstrInfo, 
                                                TMethodInfoVector &vecMethods)
{
    if (vecMethods.Size() > 0)
    {
        pClassInstrInfo->uiNumMethods = vecMethods.Size();
        pClassInstrInfo->pMethods = new SAdaptorMethodInfo[pClassInstrInfo->uiNumMethods];
        unsigned int i;
        for (i = 0; i < pClassInstrInfo->uiNumMethods; ++i)
        {
            SAdaptorMethodInfo *pmi = vecMethods.GetAt(i);
            pClassInstrInfo->pMethods[i] = *pmi;
        }
    }

}

//
// Deallocates methods vector items
//
void CInstrumentationAdaptorBase::FreeMethodsVectorItems(TMethodInfoVector &vecMethods)
{
    while (vecMethods.Size() > 0)
    {
        SAdaptorMethodInfo *pmi = vecMethods.Pop();
        delete pmi;
    }

}

//
// Initializes a map of method names to their information.
// The key is the method name and signature
//
void CInstrumentationAdaptorBase::InitMethodMap(TMethodNameToInfoMap *pMap, 
                                                const SAdaptorClassInfo &classInfo)
{
    if (classInfo.uiNumMethods == 0)
    {
        return;
    }
    unsigned int i;
    for (i = 0; i < classInfo.uiNumMethods; ++i)
    {
        SAdaptorMethodInfo *pMethodInfo = &classInfo.pMethods[i];
        MCString strKey;
        strKey.Set(pMethodInfo->szName);
        strKey.Append(pMethodInfo->szSignature);
        const char *szKey = strKey.Get();
        pMap->Set(szKey, pMethodInfo);
    }
}

//
// Retrieves method information from the map
//
// Parmameters:
//      theMap      [in]    : the map from which to retrieve information
//      javaMethodInfo  [in]    : the method to search in the map
//
// Returns
//      SMethodInfo*    : if the information was found in the map
//      NULL            : if information for the specified method was not found
//
SAdaptorMethodInfo* 
CInstrumentationAdaptorBase::GetInfoFromMethodMap(TMethodNameToInfoMap &theMap, 
                                                  const SJavaMethodInfo &javaMethodInfo)
{
    MCString strKey;
    char *szMethodName = CWideStringUtils::WideStringToCString(javaMethodInfo.methodName);
    char *szMethodSig = CWideStringUtils::WideStringToCString(javaMethodInfo.methodSignature);
    strKey.Set(szMethodName);
    strKey.Append(szMethodSig);
    const char *szKey = strKey.Get();
    SAdaptorMethodInfo *pResult = theMap.Get(szKey);

    // cleanup
    delete [] szMethodName;
    delete [] szMethodSig;

    return pResult;
}

