/*****************************************************************************
 * Copyright (c) 1997, 2009, Intel Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *
 * $Id$ 
 *****************************************************************************/

#include "LogAssert.h"
#include <string.h>
#include <stdlib.h>
#include "MpiAPI.h"

#include <stdio.h>
#include <stdarg.h>

using namespace Martini::MPI;
using namespace Martini::OSA;
using namespace Martini::Infrastructure;
using namespace Martini::ExternalControl;

#define ENV_VAR_MARTINI_LOGGER_LOG_LEVEL        "MARTINI_LOGGER_LOG_LEVEL" 
#define ENV_VAR_MARTINI_LOGGER_CONSOLE_OUTPUT   "MARTINI_LOGGER_CONSOLE_OUTPUT"
#define ENV_VAR_MARTINI_LOGGER_DIRECTORY        "MARTINI_LOGGER_DIRECTORY"
#define ENV_VAR_MARTINI_LOGGER_REPLACE_IF_EXIST "MARTINI_LOGGER_REPLACE_IF_EXIST"

#define DEFAULT_LOGGER_LOG_LEVEL                3
#define DEFAULT_LOGGER_CONSOLE_OUTPUT           0
#define DEFAULT_LOGGER_REPLACE_IF_EXIST         1

#define MARTINI_LOG_FILENAME                    "MLog" //log file name is: MLog00000.log (the number is PID)
#define MARTINI_LOG_EXTENTION                   ".log"

#define EXTERNAL_CONTROL_FATAL_MESSAGE          "Internal error (see log file)."

#ifdef WIN32
#define vsnprintf _vsnprintf
#endif //WIN32

CLogAssert *CLogAssert::GetInstance()
{
    static CLogAssert instance;
    return &instance;
}

CLogAssert::CLogAssert()
{
    m_pLogger = NULL;
    m_pEC = NULL;
    m_szDirectory = NULL;
    m_csDirectoryChangeSync = NULL;
    m_csLogAssertCall = NULL;
    m_uiConsoleOutput = DEFAULT_LOGGER_CONSOLE_OUTPUT;
    m_iLogLevel = DEFAULT_LOGGER_LOG_LEVEL;
    m_uiReplaceIfExist = DEFAULT_LOGGER_REPLACE_IF_EXIST;
    m_bInitDone = false;
}

CLogAssert::~CLogAssert()
{
    if (m_szDirectory)
    {
        delete [] m_szDirectory;
    }
    if (m_pLogger)
    {
        delete m_pLogger;
    }
    if (m_csDirectoryChangeSync)
    {
        m_csDirectoryChangeSync->Destroy();
    }
    if (m_csLogAssertCall)
    {
        m_csLogAssertCall->Destroy();
    }
}

TResult CLogAssert::Initialize(CECAgent* pEC, bool bInitDone)
{
    m_bInitDone = bInitDone;
    m_pEC = pEC;
    if (NULL == m_pLogger)
    {
        m_pLogger = new CLogger();
        if (NULL == m_pLogger)
        {
            return MRTE_ERROR_OUT_OF_MEMORY;
        }
    }

    if (NULL == m_csDirectoryChangeSync)
    {
        m_csDirectoryChangeSync = CreateThreadSync();
        if (NULL == m_csDirectoryChangeSync)
        {
            return MRTE_ERROR_OSA_FAILURE;
        }
    }

    if (NULL == m_csLogAssertCall)
    {
        m_csLogAssertCall = CreateThreadSync();
        if (NULL == m_csLogAssertCall)
        {
            return MRTE_ERROR_OSA_FAILURE;
        }
    }

    // Query the process environment for the initial logger settings
    unsigned int uiEnvBufferSize = MAX_PATH;
    unsigned int uiEnvActualSize = 0;
    char *szEnvironmentVal = new char[uiEnvBufferSize];
    if (szEnvironmentVal == NULL)
    {
        return MRTE_ERROR_OUT_OF_MEMORY;
    }

    TResult res;

    m_csDirectoryChangeSync->Enter();
    if (m_szDirectory == NULL)
    {
        res = GetEnvironmentVar(ENV_VAR_MARTINI_LOGGER_DIRECTORY, szEnvironmentVal, 
            uiEnvBufferSize, &uiEnvActualSize);
        if (res == MRTE_ERROR_BUFFER_TOO_SHORT)
        {
            delete [] szEnvironmentVal;
            szEnvironmentVal = new char[uiEnvActualSize];
            if (szEnvironmentVal == NULL)
            {
                m_csDirectoryChangeSync->Leave();
                return MRTE_ERROR_OUT_OF_MEMORY;
            }
            uiEnvBufferSize = uiEnvActualSize;
            res = GetEnvironmentVar(ENV_VAR_MARTINI_LOGGER_DIRECTORY, szEnvironmentVal, 
                uiEnvBufferSize, &uiEnvActualSize);
        }  
        if (res == MRTE_RESULT_OK)
        {
            m_szDirectory = new char[uiEnvActualSize + 2];
            if (m_szDirectory == NULL)
            {
                m_csDirectoryChangeSync->Leave();
                return MRTE_ERROR_OUT_OF_MEMORY;
            }
            strcpy(m_szDirectory, szEnvironmentVal);
            if (m_szDirectory[strlen(m_szDirectory) - 1] != DIRECTORY_SEPARATOR)
            {
                strcat(m_szDirectory, DIRECTORY_SEPARATOR_STR);
            }
        }
    }
    m_csDirectoryChangeSync->Leave();

    char *pEndPtr = NULL;
    res = GetEnvironmentVar(ENV_VAR_MARTINI_LOGGER_LOG_LEVEL, szEnvironmentVal, 
        uiEnvBufferSize, &uiEnvActualSize);
    if (res == MRTE_RESULT_OK)
    {
        m_iLogLevel = (int)strtol(szEnvironmentVal, &pEndPtr, 10);
    }

    res = GetEnvironmentVar(ENV_VAR_MARTINI_LOGGER_CONSOLE_OUTPUT, szEnvironmentVal, 
        uiEnvBufferSize, &uiEnvActualSize);
    if (res == MRTE_RESULT_OK)
    {
        m_uiConsoleOutput = (int)strtol(szEnvironmentVal, &pEndPtr, 10);
    }

    res = GetEnvironmentVar(ENV_VAR_MARTINI_LOGGER_REPLACE_IF_EXIST, szEnvironmentVal, 
        uiEnvBufferSize, &uiEnvActualSize);
    if (res == MRTE_RESULT_OK)
    {
        m_uiReplaceIfExist = (int)strtol(szEnvironmentVal, &pEndPtr, 10);
    }

    delete [] szEnvironmentVal;

    unsigned int uiConfig = LCF_HEADER_ALL | LCF_LINE_ALL;
    if (m_uiConsoleOutput)
    {
        uiConfig |= LCF_OUTPUT_TO_STDERR;
    }
    if (m_uiReplaceIfExist)
    {
        uiConfig |= LCF_REPLACE_FILE_IF_EXIST;
    }

    char *szFileName = NULL;
    if (m_szDirectory)
    {
        int iPID = GetCurrProcessID();
        // Use not less than 5 digits of the process id
        char szPID[32];
        size_t iPIDLen = sprintf(szPID, "%05d", iPID);

        size_t fileNameLen = strlen(m_szDirectory) + iPIDLen + strlen(MARTINI_LOG_FILENAME) +
            strlen(MARTINI_LOG_EXTENTION) + /*NULL*/ 1;
        szFileName = new char[fileNameLen];
        if (szFileName == NULL)
        {
            return MRTE_ERROR_OUT_OF_MEMORY;
        }

        sprintf(szFileName, "%s%s%s%s", m_szDirectory, MARTINI_LOG_FILENAME, szPID, MARTINI_LOG_EXTENTION);
    }

    res = m_pLogger->Configure(uiConfig, szFileName, m_iLogLevel);

    if (szFileName)
    {
        delete [] szFileName;
    }
    else if (!m_bInitDone)
    {
        // if no path is configured and the initialization phase is not over - need to buffer
        // the log messages.
        m_pLogger->StartBufferMode();
    }

    if (m_bInitDone)
    {
        m_pLogger->EndBufferMode();
    }
    
    return res;
}

void CLogAssert::Error(const char* szClient, const char* szFormatMsg, ...)
{
    m_csLogAssertCall->Enter();

    va_list argptr;
    va_start(argptr, szFormatMsg);
    char szMsg[MAX_STRING];
    vsnprintf(szMsg, MAX_STRING, szFormatMsg, argptr);
    szMsg[MAX_STRING - 1] = 0;
    m_pLogger->Log(szClient, MT_ERROR, szMsg, 0);
    // we don't want to display in the tool a free from text with details
    if (m_pEC)
    {
        m_pEC->MartiniMessage(EXTERNAL_CONTROL_FATAL_MESSAGE, SL_FATAL);
    }
    va_end(argptr);

    m_csLogAssertCall->Leave();
}

void CLogAssert::Informative(const char* szClient, int iLevel, bool bExposeToEC, const char* szFormatMsg, ...)
{
    m_csLogAssertCall->Enter();

    va_list argptr;
    va_start(argptr, szFormatMsg);
    char szMsg[MAX_STRING];
    vsnprintf(szMsg, MAX_STRING, szFormatMsg, argptr);
    szMsg[MAX_STRING - 1] = 0;
    m_pLogger->Log(szClient, MT_INFORMATIVE, szMsg, iLevel);
    if (bExposeToEC && m_pEC)
    {
        m_pEC->MartiniMessage(szMsg, SL_NOT_ERROR);
    }
    va_end(argptr);

    m_csLogAssertCall->Leave();
}

void CLogAssert::Debug(const char* szClient, int iLevel, const char* szFormatMsg, ...)
{
    m_csLogAssertCall->Enter();

    va_list argptr;
    va_start(argptr, szFormatMsg);
    char szMsg[MAX_STRING];
    vsnprintf(szMsg, MAX_STRING, szFormatMsg, argptr);
    szMsg[MAX_STRING - 1] = 0;
    m_pLogger->Log(szClient, MT_DEBUG, szMsg, iLevel);
    va_end(argptr);

    m_csLogAssertCall->Leave();
}

void CLogAssert::HandleEvent(SSetOutputDirEventData &data)
{
    if (data.szActivityPath == NULL || data.szActivityPath[0] == 0)
    {
        return;
    }
    
    m_csDirectoryChangeSync->Enter();
    if (m_szDirectory)
    {
        delete [] m_szDirectory;
    }
    
    m_szDirectory = new char[strlen(data.szActivityPath) + 2];
    if (m_szDirectory == NULL)
    {
        m_csDirectoryChangeSync->Leave();
        return;
    }
    strcpy(m_szDirectory, data.szActivityPath);
    if (m_szDirectory[strlen(m_szDirectory) - 1] != DIRECTORY_SEPARATOR)
    {
        strcat(m_szDirectory, DIRECTORY_SEPARATOR_STR);
    }
    
    m_csDirectoryChangeSync->Leave();

    // Apply changes by running the initialization process again, but do not change the 
    // initialization status. This status is controlled by the Kernel of the active MPI
    // implementation (CPI, JPI, etc).
    Initialize(m_pEC, m_bInitDone); 
}

unsigned int CLogAssert::EventDataTypes()
{
    return DR_ACTIVITY_PATH;
}


CLogger* CLogAssert::GetLoggerInstance()
{
    return m_pLogger;
}

