/*****************************************************************************
 * Copyright (c) 1997, 2009, Intel Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *
 * $Id$ 
 *****************************************************************************/

#include "Logger.h"
#include <string.h>
#include <time.h>

#include <string.h>
#include <stdlib.h>

using namespace Martini::Infrastructure;
using namespace Martini::MPI;
using namespace Martini::OSA;

#define MAX_LOG_ENTRY_SIZE 1024

CLogger::CLogger()
{
    m_pMessageBuffer = NULL;
    m_csLogLine = NULL;
    m_hFile = NULL;
    m_bForceLogLevel = false;
    m_bOutputToStderr = false;
    m_iLevel = 0;
}

CLogger::~CLogger()
{
    if (m_pMessageBuffer)
    {
        EndBufferMode();
    }
}

TResult CLogger::CreateLogFile(unsigned int configField, const char* szFileName)
{
    if (szFileName == NULL)
    {
        return MRTE_RESULT_OK;
    }
        
    
    if (m_hFile)
    {
        if (strcmp(m_szFileName, szFileName) != 0)
        {
            fclose(m_hFile);
            m_hFile = NULL;
            delete [] m_szFileName;
        }
        else
        {
            return MRTE_RESULT_OK;
        }
    }
    
    if (m_hFile == NULL)
    {
        if (configField & LCF_REPLACE_FILE_IF_EXIST)
        {
            m_hFile = fopen(szFileName, "w");
        }
        else
        {
            m_hFile = fopen(szFileName, "a");
        }
        if (!m_hFile)
        {
            return MRTE_ERROR_FILE_FAILURE;
        }
        m_szFileName = new char[strlen(szFileName) + 1];
        if (!m_szFileName)
        {
            return MRTE_ERROR_OUT_OF_MEMORY;
        }
        strcpy(m_szFileName, szFileName);
    }
    
    m_csLogLine->Enter();

	WriteToFile(m_hFile, "=============================================\n");
    
    if (configField & LCF_HEADER_PID)
    {
        unsigned int uiPID = GetCurrProcessID();
        char uiPIDtext[MAX_LOG_ENTRY_SIZE];
		sprintf(uiPIDtext, "Process ID: %d\n", uiPID);
		WriteToFile(m_hFile, uiPIDtext);
    }
    
    char c;
    unsigned int uiRequiredLen;
    TResult res = MRTE_RESULT_OK;
    
    if (configField & LCF_HEADER_COMMAND_LINE)
    {
        res = GetCurrentProcCommandLine(&c, 0, &uiRequiredLen);
        if (res == MRTE_ERROR_BUFFER_TOO_SHORT)
        {
            unsigned int uiNameLen;
            char* szCommandLine = new char[uiRequiredLen];
            if (!szCommandLine)
            {
                m_csLogLine->Leave();
                return MRTE_ERROR_OUT_OF_MEMORY;
            }
            res = GetCurrentProcCommandLine(szCommandLine, uiRequiredLen, &uiNameLen);
            if (MRTE_SUCCEEDED(res))
            {
            	char * szCommandLineText = (char *)malloc(strlen(szCommandLine) + 64);

            	if (NULL != szCommandLineText)
            	{
				    sprintf(szCommandLineText, "Process command line: %s\n", szCommandLine);
				    WriteToFile(m_hFile, szCommandLineText);
				    free(szCommandLineText);
            	}
            }
            delete [] szCommandLine;
        }
    }
    
    if (configField & LCF_HEADER_MACHINE_NAME)
    {
        res = GetThisMachineName(&c, 0, &uiRequiredLen);
        if (res == MRTE_ERROR_BUFFER_TOO_SHORT)
        {
            unsigned int uiNameLen;
            char* szMachineName = new char[uiRequiredLen];
            if (!szMachineName)
            {
                m_csLogLine->Leave();
                return MRTE_ERROR_OUT_OF_MEMORY;
            }
            res = GetThisMachineName(szMachineName, uiRequiredLen, &uiNameLen);
            if (MRTE_SUCCEEDED(res))
            {
            	char szMachineNameText[MAX_LOG_ENTRY_SIZE];
				sprintf(szMachineNameText, "Machine name: %s\n", szMachineName);
				WriteToFile(m_hFile, szMachineNameText);
            }
            delete [] szMachineName;
        }
    }
    
    WriteToFile(m_hFile, "=============================================\n");


    if (m_uiLineConfig & LCF_LINE_LOG_TYPE)
    {
    	WriteToFile(m_hFile, "Type\t");
    }
    if (m_uiLineConfig & LCF_LINE_DATE)
    {
    	WriteToFile(m_hFile, "Date\t");
    }
    if (m_uiLineConfig & LCF_LINE_TIME)
    {
    	WriteToFile(m_hFile, "Time\t");
    }
    if (m_uiLineConfig & LCF_LINE_TID)
    {
    	WriteToFile(m_hFile, "TID\t");
    }
    if (m_uiLineConfig & LCF_LINE_LOG_LEVEL)
    {
    	WriteToFile(m_hFile, "Level\t");
    }
    if (m_uiLineConfig & LCF_LINE_CLIENT_NAME)
    {
    	WriteToFile(m_hFile, "Client\t");
    }
    WriteToFile(m_hFile, "Message\n");

    fflush(m_hFile);
    
    m_csLogLine->Leave();
    
    return MRTE_RESULT_OK;
}

void CLogger::WriteToFile(FILE * m_hFile, char * text)
{
	#ifdef MVS
		char fileText[strlen(text)+16];
		sprintf(fileText, "%s", text);

		char* nativeBuffer = 0;
		unicode2native(&nativeBuffer, fileText, strlen(fileText));
		fprintf(m_hFile, "%s", nativeBuffer);

		if(nativeBuffer != 0)
			free(nativeBuffer);

	#else
		fprintf(m_hFile, "%s", text);
	#endif
}

TResult CLogger::Configure(unsigned int configField, const char* szFileName, int iLevel)
{
    if (!m_csLogLine)
    {
        m_csLogLine = CreateThreadSync();
        if (!m_csLogLine)
        {
            return MRTE_ERROR_OSA_FAILURE;
        }
    }

    m_iLevel = iLevel;
    m_uiLineConfig = configField & LCF_LINE_ALL;
    TResult res = CreateLogFile(configField, szFileName);
    if (MRTE_FAILED(res))
    {
        return res;
    }
    
    return MRTE_RESULT_OK;
}

void CLogger::Log(const char* szClientName, EMessageType msgType, const char* szMsg, int iLevel)
{
    // a lock free writing is sliced out:
    // keep a counter member in the class that will contain the current size of the file
    // for each log line:
    //      duplicate the file handle
    //      calculate the size of the line
    //      atomically increment the counter (compare exchange)
    //      seek the file with the duplicated handle 
    //      write to the file with the duplicated handle
    //      close the duplicated handle
    
    if (iLevel > m_iLevel || (m_bForceLogLevel && iLevel != m_iLevel))
    {
        return;
    }
    
    m_csLogLine->Enter();
    
    char szLine[MAX_MRTE_STRING_LENGTH] = {0};
    time_t tNow = 0;
    tm* pNow;
    
    if (m_uiLineConfig & LCF_LINE_LOG_TYPE)
    {
        switch(msgType) 
        {
        case MT_ERROR:
            strcat(szLine, "Error\t");
            break;
        case MT_INFORMATIVE:
            strcat(szLine, "Informative\t");
            break;
        case MT_DEBUG:
            strcat(szLine, "Debug\t");
            break;
        default:
            ;
        }
    }

    if (m_uiLineConfig & LCF_LINE_DATE || m_uiLineConfig & LCF_LINE_TIME)
    {
        tNow = time(NULL);
        pNow = gmtime(&tNow);
    }
    if (m_uiLineConfig & LCF_LINE_DATE)
    {
        strftime(&szLine[strlen(szLine)], MAX_MRTE_STRING_LENGTH - strlen(szLine), "%x\t", pNow);
    }
    if (m_uiLineConfig & LCF_LINE_TIME)
    {
        strftime(&szLine[strlen(szLine)], MAX_MRTE_STRING_LENGTH - strlen(szLine), "%X\t", pNow);
    }
    if (m_uiLineConfig & LCF_LINE_TID)
    {
        sprintf(&szLine[strlen(szLine)], "%d\t", GetCurrThreadID());
    }
    if (m_uiLineConfig & LCF_LINE_LOG_LEVEL)
    {
        sprintf(&szLine[strlen(szLine)], "%d\t", iLevel);
    }
    if (m_uiLineConfig & LCF_LINE_CLIENT_NAME)
    {
        sprintf(&szLine[strlen(szLine)], "%s\t", szClientName);
    }

    strcat(szLine, szMsg);
    strcat(szLine, "\n");

    if (m_hFile)
    {
    	WriteToFile(m_hFile, szLine);
        fflush(m_hFile);
    }

    if (m_bOutputToStderr)
    {
    	WriteToFile(stderr, szLine);
    }

    if (m_pMessageBuffer)
    {
        char* pLine = new char[strlen(szLine) + 1];
        strcpy(pLine, szLine);

        m_pMessageBuffer->InsertAfter(m_pMessageBuffer->GetLast(), pLine);
    }
    
    m_csLogLine->Leave();
}

void CLogger::Log(const char* szClientName, EMessageType msgType, const wchar_t* wszMsg, int iLevel)
{
    if (iLevel > m_iLevel || (m_bForceLogLevel && iLevel != m_iLevel))
    {
        return;
    }
    
    // this function replaces each character below 128 with it's corresponding char.
    // other characters are replaced with "/XXXX" where XXXX is the hexadecimal representation
    // of the character
    int iOrigSize = wcslen(wszMsg);
    char* szMsg = new char[iOrigSize * 5 + 1]; // max size is len*5 if all the characters are
    // above 127
    if (szMsg == NULL)
    {
        return;
    }
    
    szMsg[0] = 0;
    for (int i = 0; i < iOrigSize; ++i)
    {
        char szCurChar[6];
        if (wszMsg[i] < 128)
        {
            szCurChar[0] = (char)wszMsg[i];
            szCurChar[1] = 0;
        }
        else
        {
            sprintf(szCurChar, "/%04X", wszMsg[i]);
        }
        strcat(szMsg, szCurChar);
    }
    
    Log(szClientName, msgType, szMsg, iLevel);
    delete [] szMsg;
}


// The buffer is currently used only to buffer messages that arrived before init phase done.
// Since this is the only usage and due to slicing, the buffer will be per process and sync.
TResult CLogger::StartBufferMode()
{
    TResult res = MRTE_RESULT_OK;

    if (m_pMessageBuffer)
    {
        return res;
    }

    m_csLogLine->Enter();
    m_pMessageBuffer = new RTUtil::MList<char*>;
    if (!m_pMessageBuffer)
    {
        res = MRTE_ERROR_OUT_OF_MEMORY;
    }
    m_csLogLine->Leave();

    return res;
}

TResult CLogger::EndBufferMode()
{
    if (NULL == m_pMessageBuffer)
    {
        return MRTE_RESULT_OK;
    }

    m_csLogLine->Enter();

    TMRTEHandle hLine = m_pMessageBuffer->GetFirst();
    while (hLine != NULL)
    {
        char* pLine = m_pMessageBuffer->GetData(hLine);
        if (!pLine)
        {
            continue;
        }

        if (m_hFile)
        {
        	WriteToFile(m_hFile, pLine);
        }

        delete [] pLine;
        hLine = m_pMessageBuffer->GetNext(hLine);
    }

    delete m_pMessageBuffer;
    m_pMessageBuffer = NULL;

    m_csLogLine->Leave();

    return MRTE_RESULT_OK;
}

