/*****************************************************************************
 * Copyright (c) 1997, 2010 Intel Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *
 * $Id$ 
 *****************************************************************************/

#include "MPI.h"
#include "ECAgent.h"
#include "MieAPI.h"

using namespace Martini::MPI;
using namespace Martini::Infrastructure;
using namespace Martini::ExternalControl;
using namespace Martini::MIE;


//
// Constructor
//
CMpiInterface::CMpiInterface(ERuntimeType iRuntimeType, IDataManager *pDataManager, 
                             IEventManager *pEventManager, IKernel *pKernel, 
                             CECAgent* pECAgent, CLogger* pLogger)
{
    m_pDataManager = pDataManager;
    m_pEventManager = pEventManager;
    m_pKernel = pKernel;
    m_iRuntimeType = iRuntimeType;
    m_pECAgent = pECAgent;
    m_pLogger = pLogger;
}

//
// Destructor
//
CMpiInterface::~CMpiInterface()
{
}

TResult CMpiInterface::GetModuleInfo(TId clientId, 
                                     TId moduleId, 
                                     BitSet requestedDataTypes, 
                                     SModuleInfo *pData)
{
    // Verify that the requested data types are supported by this data request
    if ( !((DR_SUPPORTED_MODULE_INFO | requestedDataTypes) == DR_SUPPORTED_MODULE_INFO) )
    {
        return MRTE_ERROR_NOT_SUPPORTED;
    }

    return m_pDataManager->GetModuleInfo(clientId, moduleId, requestedDataTypes, pData);
}

TResult CMpiInterface::GetClassInfo(TId clientId, 
                                    TId classId, 
                                    BitSet requestedDataTypes, 
                                    SClassInfo *pData)
{
    // Verify that the requested data types are supported by this data request
    if ( !((DR_SUPPORTED_CLASS_INFO | requestedDataTypes) == DR_SUPPORTED_CLASS_INFO) )
    {
        return MRTE_ERROR_NOT_SUPPORTED;
    }

    return m_pDataManager->GetClassInfo(clientId, classId, requestedDataTypes, pData);
}

TResult CMpiInterface::GetMethodInfo(TId clientId, 
                                     TId methodId, 
                                     BitSet requestedDataTypes, 
                                     SMethodInfo *pData)
{
    // Verify that the requested data types are supported by this data request
    if ( !((DR_SUPPORTED_METHOD_INFO | requestedDataTypes) == DR_SUPPORTED_METHOD_INFO) )
    {
        return MRTE_ERROR_NOT_SUPPORTED;
    }
    return m_pDataManager->GetMethodInfo(clientId, methodId, requestedDataTypes, pData);
}

TResult CMpiInterface::GetCompiledMethodInfo(TId clientId,
                                              const char* szModuleName,
                                              UIOP uiIpOffset,
                                              BitSet requestedDataTypes,
                                              TId *pMethodId,
                                              SMethodInfo *pData)
{
    return MRTE_ERROR_NOT_SUPPORTED;
}

TResult CMpiInterface::GetThreadInfo(TId clientId, 
                                     TId threadId, 
                                     BitSet requestedDataTypes, 
                                     SThreadInfo *pData,
                                     TDotNetUnmanagedStackWalkCallBack pUnManagedStackWalker)
{
    // Verify that the requested data types are supported by this data request
    if ( !((DR_SUPPORTED_THREAD_INFO | requestedDataTypes) == DR_SUPPORTED_THREAD_INFO) )
    {
        return MRTE_ERROR_NOT_SUPPORTED;
    }

    return m_pDataManager->GetThreadInfo(clientId, threadId, requestedDataTypes, pData, pUnManagedStackWalker);
}

TResult CMpiInterface::GetObjectInfo(TId clientId, 
                                     TId objectId, 
                                     BitSet requestedDataTypes,
                                     SObjectDataRequest *pData)
{
    // Verify that the requested data types are supported by this data request
    if ( !((DR_SUPPORTED_OBJECT_INFO | requestedDataTypes) == DR_SUPPORTED_OBJECT_INFO) )
    {
        return MRTE_ERROR_NOT_SUPPORTED;
    }

    return m_pDataManager->GetObjectInfo(clientId, objectId, requestedDataTypes, pData);
}

TResult CMpiInterface::RegisterEvent(TId clientId, IEventObserver &observer)
{
    return m_pEventManager->RegisterEvent(clientId, observer);
}

TResult CMpiInterface::SetEventGroupFilter(TId clientID, 
                                           EEventGroup group, 
                                           IEventFilter &filter)
{
    return m_pEventManager->SetEventGroupFilter(clientID, group, filter);
}

/** Called by ProfEnv::SetSupportedEG */
TResult CMpiInterface::EnableEventGroup(TId clientId, EEventGroup eventGroup)
{
    return m_pEventManager->EnableEventGroup(clientId, eventGroup);
}

TResult CMpiInterface::DisableEventGroup(TId clientId, EEventGroup eventGroup)
{
    return m_pEventManager->DisableEventGroup(clientId, eventGroup);
}

TResult CMpiInterface::InstantiateClient(TId clientId, 
                                         const char *szRequestedModuleName,
                                         const char *szOptions)
{
    return m_pKernel->InstantiateClient(clientId, szRequestedModuleName, szOptions);
}



TResult CMpiInterface::MessageLog(TId clientId, EMessageType msgType, unsigned int msgDest, 
                                  char* szMessage, unsigned int iLevel)
{
    if (!szMessage || !szMessage[0] || 
        (!(msgDest & MD_EXTERNAL_CONTROL) && !(msgDest & MD_LOG_FILE)))
    {
        return MRTE_ERROR_ILLEGAL_ARGUMENT;
    }
    
    if (msgDest & MD_EXTERNAL_CONTROL)
    {
        ESeverityLevel severity = SL_NOT_ERROR;
        if (msgType == MT_ERROR)
        {
            severity = SL_FATAL;
        }

        m_pECAgent->MartiniMessage(szMessage, severity);
    }

    if (msgDest & MD_LOG_FILE)
    {
        SClientInfo* pClient = m_pKernel->GetClientInfo(clientId);
        m_pLogger->Log(pClient->szClientName.Get(), msgType, szMessage, iLevel);
    }
    
    return MRTE_RESULT_OK;
}

TResult CMpiInterface::GetObjectReferences(TId clientId, SObjectReferenceInfo *pData)
{
    return m_pDataManager->GetObjectReferences(clientId, pData);
}

TResult CMpiInterface::SuspendVM()
{
    return m_pKernel->SuspendVM();
}

TResult CMpiInterface::ResumeVM()
{
    return m_pKernel->ResumeVM();
}

TResult CMpiInterface::RunGC()
{
    return m_pKernel->RunGC();
}

// Accesses the data manager to call heap object data collection on the passed Object.
TResult CMpiInterface::RunHeapObjDataCollection(THeapObjectAnalysis *heapObjectAnalysis) {
	return m_pDataManager->RunHeapObjDataCollection(heapObjectAnalysis);
}

// Accesses the data manager to set the heap instance data collection option.
void CMpiInterface::SetHeapObjDataCollection(bool heapObjDataCollection) {
	m_pDataManager->SetHeapObjDataCollection(heapObjDataCollection);
}

TResult CMpiInterface::GenerateObjectAllocEventForLiveObjects(TId clientId)
{
    return m_pKernel->GenerateObjectAllocEventForLiveObjects(clientId);
}

TResult CMpiInterface::GetAllThreadsInfo(TId clientId, 
                                         U32 maxFrameCount, 
                                         BitSet requestedDataTypes, 
                                         SThreadInfoArray *pData)
{
    // Verify that the requested data types are supported by this data request
    if ( !((DR_SUPPORTED_ALL_THREAD_INFO | requestedDataTypes) == DR_SUPPORTED_ALL_THREAD_INFO) )
    {
        return MRTE_ERROR_NOT_SUPPORTED;
    }

    if (NULL == pData)
    {
        return MRTE_ERROR_NULL_PTR;
    }

    return m_pDataManager->GetAllThreadsInfo(clientId, maxFrameCount, requestedDataTypes, 
        pData);
}
