/*****************************************************************************
 * Copyright (c) 1997, 2008 Intel Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *
 * $Id$ 
 *****************************************************************************/
#ifdef EM64T_ARCH
#pragma runtime_checks( "", off )
#endif

#include <memory.h>
#include <stdlib.h>

#include "ParamChecker.h"

using namespace Martini::Infrastructure;
using namespace Martini::MPI;

//<summary> constructor </summary>
CParamChecker::CParamChecker()
: m_EventDataItems(NULL)
{
    m_MaxClientId = 0;
    m_MinClientId = 0;
    m_MinEvent = (TEventType)0;
    m_MaxEvent = (TEventType)0;
    InitEventDataItems();
}

//<summary> destructor </summary>
CParamChecker::~CParamChecker()
{
    if (m_EventDataItems)
    {
        delete [] m_EventDataItems;
    }
}

TResult CParamChecker::CheckClientAndPtr(TId clientId, const void *ptr)
{
    TResult iRetVal = CheckClient(clientId);
    if (iRetVal != MRTE_RESULT_OK)
    {
        return iRetVal;
    }
    return CheckPtr(ptr);
}

TResult CParamChecker::CheckEventAndPtr(TEventType mpiEvent, const void *ptr)
{
    TResult iRetVal = CheckEvent(mpiEvent);
    if (iRetVal != MRTE_RESULT_OK)
    {
        return iRetVal;
    }
    return CheckPtr(ptr);
}

//<summary> Check parameters </summary>
//<param name='clientId'> client id </param>
//<param name='mpiEvent'> MPI event </param>
//<param name='ptr'> pointer to check </param>
//<returns> Success/Failure indication </returns>
//<remarks>
//<b>Return Values:</b><pre>
//<li>MPI_SUCCESS       valid parameters    </li>
//<li>MPI_ILLEGAL_ID    illegal client id   </li>
//<li>MPI_NULL_PTR      ptr is null         </li>
//<li>MPI_NOT_SUPPORTED event not supported </li></pre>
//</remarks>
TResult CParamChecker::CheckClientAndEventAndPtr(TId clientId, TEventType mpiEvent, 
                                                 const void *ptr)
{
    TResult iRetVal = CheckClientAndEvent(clientId, mpiEvent);
    if (MRTE_FAILED(iRetVal))
    {
        return iRetVal;
    }
    return CheckPtr(ptr);
}

//<summary> Check parameters </summary>
//<param name='clientId'> client id </param>
//<param name='mpiEvent'> MPI event </param>
//<returns> Success/Failure indication </returns>
//<remarks>
//<b>Return Values:</b><pre>
//<li>MPI_SUCCESS       valid parameters    </li>
//<li>MPI_ILLEGAL_ID    illegal client id   </li>
//<li>MPI_NOT_SUPPORTED event not supported </li></pre>
//</remarks>
TResult CParamChecker::CheckClientAndEvent(TId clientId, TEventType mpiEvent)
{
    TResult iRetVal = CheckClient(clientId);
    if (MRTE_FAILED(iRetVal))
    {
        return iRetVal;
    }
    return CheckEvent(mpiEvent);
}


//<summary> Check parameters </summary>
//<param name='clientId'> client id </param>
//<param name='data'> data type </param>
//<returns> Success/Failure indication </returns>
//<remarks>
//<b>Return Values:</b><pre>
//<li>MPI_SUCCESS       valid parameters        </li>
//<li>MPI_ILLEGAL_ID    illegal client id   </li>
//<li>MPI_NOT_SUPPORTED data type not supported </li></pre>
//</remarks>
TResult CParamChecker::CheckClientAndData(TId clientId, unsigned int data)
{
    TResult iRetVal = CheckClient(clientId);
    if (MRTE_FAILED(iRetVal))
    {
        return iRetVal;
    }
    return CheckData(data);
}

//<summary> Check parameters </summary>
//<param name='clientId'> client id </param>
//<param name='mpiEvent'> MPI event </param>
//<param name='data'> data type </param>
//<returns> Success/Failure indication </returns>
//<remarks>
//<b>Return Values:</b><pre>
//<li>MPI_SUCCESS       valid parameters                 </li>
//<li>MPI_ILLEGAL_ID    illegal client id   </li>
//<li>MPI_NOT_SUPPORTED data type or event not supported </li></pre>
//</remarks>
TResult CParamChecker::CheckClientAndEventAndData(TId clientId, TEventType mpiEvent, BitSet data)
{
    TResult iRetVal = CheckClient(clientId);
    if (MRTE_FAILED(iRetVal))
    {
        return iRetVal;
    }
    return CheckEventData(mpiEvent, data);
}

//<summary> Check parameters </summary>
//<param name='clientId'> client id </param>
//<param name='mpiEvent'> MPI event </param>
//<param name='data'> data type </param>
//<param name='ptr'> pointer to check </param>
//<returns> Success/Failure indication </returns>
//<remarks>
//<b>Return Values:</b><pre>
//<li>MPI_SUCCESS       valid parameters                 </li>
//<li>MPI_ILLEGAL_ID    illegal client id                </li>
//<li>MPI_NULL_PTR      ptr is null                      </li>
//<li>MPI_NOT_SUPPORTED data type or event not supported </li></pre>
//</remarks>
TResult CParamChecker::CheckClientAndEventAndDataAndPtr(TId clientId, TEventType mpiEvent, BitSet data,
                                   const void *ptr)
{
    TResult iRetVal = CheckClientAndEventAndData(clientId, mpiEvent, data);
    if (MRTE_FAILED(iRetVal))
    {
        return iRetVal;
    }
    return CheckPtr(ptr);
}

//<summary> Check parameters </summary>
//<param name='clientId'> client id </param>
//<param name='data'> data type </param>
//<param name='ptr'> pointer to check </param>
//<returns> Success/Failure indication </returns>
//<remarks>
//<b>Return Values:</b><pre>
//<li>MPI_SUCCESS       valid parameters                 </li>
//<li>MPI_ILLEGAL_ID    illegal client id                </li>
//<li>MPI_NULL_PTR      ptr is null                      </li>
//<li>MPI_NOT_SUPPORTED data type not supported </li></pre>
//</remarks>
TResult CParamChecker::CheckClientAndDataAndPtr(TId clientId, BitSet data, 
                                                const void *ptr)
{
    TResult iRetVal = CheckClientAndData(clientId, data);
    if (MRTE_FAILED(iRetVal))
    {
        return iRetVal;
    }
    return CheckPtr(ptr);    
}

//<summary> Check parameters </summary>
//<param name='clientId'> client id </param>
//<param name='mpiEvent'> MPI event </param>
//<param name='attr'> selective attribute </param>
//<returns> Success/Failure indication </returns>
//<remarks>
//<b>Return Values:</b><pre>
//<li>MPI_SUCCESS       valid parameters                 </li>
//<li>MPI_ILLEGAL_ID    illegal client id                </li>
//<li>MPI_NOT_SUPPORTED event or attribute not supported </li></pre>
//</remarks>
//TResult CParamChecker::CheckParams(TId clientId, TEventType mpiEvent, TMpiSelectiveAttr attr)
//{   
//    TResult iRetVal = CheckAttr(attr);
//    if (iRetVal != MPI_SUCCESS)
//    {
//        return iRetVal;
//    }
//    return CheckParams(clientId, mpiEvent);
//}

//<summary> set the client id limits: min and max valid values </summary>
//<param name='min'> minimum valid id </param>
//<param name='max'> maximum valid id </param>
//<returns> none </returns>
void CParamChecker::SetClientIdLimits(TId min, TId max)
{
    m_MinClientId = min;
    m_MaxClientId = max;
}

//<summary> set the event limits: min and max valid values </summary>
//<param name='min'> minimum valid event </param>
//<param name='max'> maximum valid event </param>
//<returns> none </returns>
void CParamChecker::SetEventsLimits(TEventType min, TEventType max)
{
    m_MinEvent = min;
    m_MaxEvent = max;
}

//<summary> Check validity of an id </summary>
//<param name='clientId'> client id to check </param>
//<returns> Success/Failure indication </returns>
//<remarks>
//<b>Return Values:</b><pre>
//<li>MPI_SUCCESS       valid parameters    </li>
//<li>MPI_ILLEGAL_ID    illegal client id   </li></pre>
//</remarks>
TResult CParamChecker::CheckClient(TId clientId)
{
    if ((clientId > m_MaxClientId) || (clientId < m_MinClientId))
    {
        return MRTE_ERROR_ILLEGAL_ID;
    }
    return MRTE_RESULT_OK;
}

//<summary> Check validity of an event </summary>
//<param name='event'> event to check </param>
//<returns> Success/Failure indication </returns>
//<remarks>
//<b>Return Values:</b><pre>
//<li>MPI_SUCCESS       valid parameters    </li>
//<li>MPI_NOT_SUPPORTED event not supported </li></pre>
//</remarks>
TResult CParamChecker::CheckEvent(TEventType event)
{
    if ((event < m_MinEvent) || (event >= m_MaxEvent))
    {
        return MRTE_ERROR_NOT_SUPPORTED;
    }
    return MRTE_RESULT_OK;
}

//<summary> Check validity of data type </summary>
//<param name='data'> data type to check </param>
//<returns> Success/Failure indication </returns>
//<remarks>
//<b>Return Values:</b><pre>
//<li>MPI_SUCCESS       valid parameters    </li>
//<li>MPI_NOT_SUPPORTED data type not supported </li></pre>
//</remarks>
TResult CParamChecker::CheckData(unsigned int data)
{
    if ((data & DR_VALID_DATA) == 0)
    {
        return MRTE_ERROR_NOT_SUPPORTED;
    }
    return MRTE_RESULT_OK;
}

//
// Check that the specified data items match the event
//
// Parameters:
//      event   [in]: the event to check
//      data    [in]: a bit-mask of data items
//
// Returns:
//      MRTE_RESULT_OK  : the event supports the data item(s)
//      MRTE_ERROR_FAIL : the event does not support the data item(s)
//
TResult CParamChecker::CheckEventData(TEventType event, unsigned int data)
{
    if (DR_NONE == data)
    {
        return MRTE_RESULT_OK;
    }
    if (event >= EV_LAST)
    {
        // Do not validate data items of internal events
        return MRTE_RESULT_OK;
    }
    if (event <= EV_NONE)
    {
        return MRTE_ERROR_FAIL;
    }
    unsigned int uiSupportedItems = m_EventDataItems[event];
    if ( !((uiSupportedItems | data) == uiSupportedItems) )
    {
        return MRTE_ERROR_FAIL;
    }
    return MRTE_RESULT_OK;
}

//<summary> Check validity of a pointer </summary>
//<param name='ptr'> pointer to check </param>
//<returns> Success/Failure indication </returns>
//<remarks>
//<b>Return Values:</b><pre>
//<li>MPI_SUCCESS       valid parameters    </li>
//<li>MPI_NULL_PTR      pointer is NULL     </li></pre>
//</remarks>
TResult CParamChecker::CheckPtr(const void *ptr)
{
    if (!ptr)
    {
        return MRTE_ERROR_NULL_PTR;
    }
    return MRTE_RESULT_OK;
}

void CParamChecker::InitEventDataItems()
{
    //TODO: store this information in Event Manager's descriptors array
    m_EventDataItems = new unsigned int[EV_LAST];
    memset(m_EventDataItems, 0, sizeof(unsigned int) * EV_LAST);
    m_EventDataItems[EV_METHOD_ENTER] = DR_THREAD_ID | DR_METHOD_ID | DR_THREAD_ELAPSED_TIME | 
        DR_THREAD_CPU_TIME | DR_MODULE_ID;
    m_EventDataItems[EV_METHOD_LEAVE] = DR_THREAD_ID | DR_METHOD_ID | DR_THREAD_ELAPSED_TIME |
        DR_THREAD_CPU_TIME;
    m_EventDataItems[EV_NEW_METHOD] = DR_METHOD_ID;
    m_EventDataItems[EV_JITTED_METHOD_LOADED] = DR_METHOD_ID | DR_NATIVE_TO_MANAGED_LINE_MAP | 
        DR_MANAGED_TO_SRC_LINE_MAP | DR_NATIVE_TO_SRC_LINE_MAP | DR_METHOD_CODE 
        | DR_DOTNET_MODULE_PREJIT;
    m_EventDataItems[EV_JITTED_METHOD_UNLOADED] = DR_METHOD_ID;
	m_EventDataItems[EV_DOTNET_METHOD_INSTRUMENTATION_POINT] = DR_METHOD_ID;
    m_EventDataItems[EV_THREAD_START] = DR_THREAD_ID | DR_OBJECT_ID | DR_CLASS_ID;
    m_EventDataItems[EV_THREAD_END] = DR_THREAD_ID | DR_OBJECT_ID | DR_CLASS_ID;
    m_EventDataItems[EV_VM_SHUTDOWN] = DR_NONE;
    m_EventDataItems[EV_VM_INIT] = DR_THREAD_ID;
    m_EventDataItems[EV_EC_SET_OUTPUT_DIRECTORY] = DR_ACTIVITY_PATH;
	m_EventDataItems[EV_DOTNET_MODULE_LOAD_FINISHED] = DR_MODULE_ID;
	m_EventDataItems[EV_DOTNET_MODULE_UNLOAD_STARTED] = DR_MODULE_ID;
    m_EventDataItems[EV_OBJECT_ALLOC] = DR_THREAD_ID | DR_OBJECT_ID | DR_OBJECT_INFO |
        DR_METHOD_ID | DR_ALLOC_VM_INSTRUCTION_OFFSET | DR_OBJECT_AGE;
    m_EventDataItems[EV_OBJECT_FREE] = DR_THREAD_ID | DR_OBJECT_ID | DR_OBJECT_INFO |
        DR_OBJECT_AGE;
    m_EventDataItems[EV_EC_CUSTOM_COMMAND] = DR_COMMAND_ID | DR_COMMAND_DATA;
    m_EventDataItems[EV_MONITOR_WAIT] = DR_THREAD_ID | DR_OBJECT_ID | DR_MONITOR_TIMEOUT;
    m_EventDataItems[EV_MONITOR_WAITED] = DR_THREAD_ID | DR_OBJECT_ID | DR_MONITOR_TIMED_OUT;
    m_EventDataItems[EV_CONTENDED_MONITOR_ENTER] = DR_THREAD_ID | DR_OBJECT_ID | 
        DR_MONITOR_OWNER_THREAD_ID;
    m_EventDataItems[EV_CONTENDED_MONITOR_ENTERED] = DR_THREAD_ID | DR_OBJECT_ID;
    m_EventDataItems[EV_THREAD_INTERACTION] = DR_THREAD_ID | DR_OBJECT_ID | DR_THREAD_INTERACTION_TYPE;
    m_EventDataItems[EV_JAVA_DYNAMIC_CODE_GENERATED] = DR_JAVA_DYNAMIC_CODE;
	m_EventDataItems[EV_GC_MOVED_REFERENCES] = DR_DOTNET_GC_MOVED_REF_INFO;
	m_EventDataItems[EV_GC_END] = DR_NONE;
	m_EventDataItems[EV_GC_START] = DR_NONE;
}
