/************************************************************************
 * Copyright (c) 2007, 2010 OC Systems Inc. and others
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Vsevolod Sandomirskiy, OC Systems - Probekit support
 *
 * $Id$ 
 ************************************************************************/
 
#include "Instrumenter.h"
#include "Globals.h"
#include "ProbekitAgent.h"
#include "Util.h"

#include <stdio.h>
#include <assert.h>
#include <string>
#include <set>

#ifdef WIN32
#include <Winsock2.h> // for htons
#endif

#ifdef _UNIX_
#include <dlfcn.h> // for dlopen
	#ifdef MVS
		#include <sys/types.h>
	    #include <arpa/inet.h>
	#endif
#include <netinet/in.h> // for htons
#endif 
 
// Martini
#include <OSA.h>
#include "log.h"

using namespace Martini::OSA;
using namespace std;

// Old BCI

#define PROBESCRIPT_FILE_EXTENSION ".probescript"
#define PROBESCRIPT_CLASS_EXTENSION ".class"

#include <BCI/BCIEng/BCIEngInterface.h>

using namespace Martini::ProbekitAgent;

// BCI function types
typedef unsigned (*CreateBCIEngine_t)(pbcieng_t* o_eng);
typedef unsigned (*DestroyBCIEngine_t)(pbcieng_t i_eng);
typedef unsigned (*Initialize_t)(pbcieng_t i_pbcieng, const char* i_pchOptions, size_t i_cbOptions);
typedef unsigned (*SetAllocator_t)(pbcieng_t i_pbcieng, pfnMalloc_t i_pfnMalloc);
typedef unsigned (*SetCallback_t)(pbcieng_t i_pbcieng, pfnCallback_t i_pfnCallback, unsigned i_uFlags);
typedef unsigned (*Instrument_t)(pbcieng_t i_pbcieng, 
                                  const char* i_pInClass, size_t i_cbInClass, 
                                  char** o_ppOutClass, size_t* o_pcbOutClass);
typedef unsigned (*Instrument2_t)(pbcieng_t i_pbcieng, 
                                  const char* i_pInClass, size_t i_cbInClass, 
							      char** o_ppOutClass, size_t* o_pcbOutClass,
                                  void *(*)(void **));
                                       
// BCI functions

static CreateBCIEngine_t    CreateBCIEngine_fn;
static DestroyBCIEngine_t   DestroyBCIEngine_fn;
static Initialize_t         Initialize_fn;
static SetAllocator_t       SetAllocator_fn;
static SetCallback_t        SetCallback_fn;
static Instrument_t         Instrument_fn;
static Instrument2_t        Instrument2_fn;

#define RESOLVE_FUNCTION(F) \
F##_fn = (F##_t)pLibrary->GetEntry(#F); \
if (F##_fn == NULL) { \
    LOG_ERROR("can't resolve " #F); \
    result = MRTE_ERROR_FAIL; \
    goto out; \
}

static pbcieng_t BCIEngine = NULL;

#define DEFAULT_BCI_LIBRARY "BCIEngProbe"

#define CMDLINE_PARAM_BCI "ext-pk-BCILibraryName"
#define CMDLINE_PARAM_PROBESCRIPT "ext-pk-probescript"
#define CMDLINE_PARAM_DUMP "ext-pk-dump"

static std::set<std::string> s_probeClasses;

/////////////////////////////////////////////////////////////////

static Martini::OSA::IThreadSync* lockBCI = Martini::OSA::CreateThreadSync();

static const char* logLocation = 0; // path to dump debug files

/////////////////////////////////////////////////////////////////

//
// Probe classes support
//

struct SDelayedClass {
	SDelayedClass( char* pname, // 0 terminated
				   char* pdata, // binary data
				   int plen )
		: name( strdup(pname) ), data( pdata ), len(plen), next( 0 )	
	{}
	
	~SDelayedClass()
	{
		free( (void*)name );
		free( (void*)data );
	}
	
	char* name;
	char* data;
	int len;
	
	SDelayedClass* next;
	
	static SDelayedClass* head;
	static SDelayedClass* tail;
	
	static void AddClass( SDelayedClass* newClass );
	static void FreeAllClasses();

};

SDelayedClass* SDelayedClass::head = 0;
SDelayedClass* SDelayedClass::tail = SDelayedClass::head;

// add new class in the list
void SDelayedClass::AddClass( SDelayedClass* newClass )
{
	if( SDelayedClass::head == 0 ) {
		SDelayedClass::head = newClass;
	}
	
	if( SDelayedClass::tail == 0 ) {
		SDelayedClass::tail = newClass;
	} else {
		SDelayedClass::tail->next = newClass;
		SDelayedClass::tail = newClass;
	}
}

// clear list
void SDelayedClass::FreeAllClasses() 
{	
	SDelayedClass* clsNext;
	SDelayedClass* cls = SDelayedClass::head;
	
	while( cls ) {
		clsNext = cls->next;
		delete cls;
		cls = clsNext;
	}
		
	SDelayedClass::head = 0;
	SDelayedClass::tail = 0;
}

//
// util
//

// Load shared library that implements BCI
static TResult LoadBCI(const char* options)
{
    TResult result = MRTE_RESULT_OK;
    char* BCIEngineName = Util::GetCmdlineOption( options, CMDLINE_PARAM_BCI );    
    if( BCIEngineName == NULL ) {
    	LOG_MESSAGE( PROFILER_NAME " BCI engine not defined, using default " 
    		DEFAULT_BCI_LIBRARY ); 
        BCIEngineName = strdup( DEFAULT_BCI_LIBRARY );
    }
    LOG_TRACE( PROFILER_NAME " BCIEngine=" << BCIEngineName );
    
    ILibraryLoader *pLibrary = LoadBistroLibrary( BCIEngineName, true );	
	        
    if (pLibrary == NULL) {
#ifdef __unix    	
    	LOG_ERROR( PROFILER_NAME " Instrumenter error: " << dlerror());
#endif    	
		LOG_ERROR( PROFILER_NAME "Could not load " << BCIEngineName ); 
		free( BCIEngineName );
        return MRTE_ERROR_FAIL;
    }    

    RESOLVE_FUNCTION(CreateBCIEngine);
    RESOLVE_FUNCTION(DestroyBCIEngine);
    RESOLVE_FUNCTION(Initialize);
    RESOLVE_FUNCTION(SetAllocator);
    RESOLVE_FUNCTION(SetCallback);
    RESOLVE_FUNCTION(Instrument);
	RESOLVE_FUNCTION(Instrument2);

        
    if( CreateBCIEngine_fn((pbcieng_t*)&BCIEngine) != 0 ) {
        LOG_ERROR( PROFILER_NAME " Can't initialize BCI engine implementation");
        result = MRTE_ERROR_FAIL;        
    }
    
out:        
    pLibrary->Destroy();    
    free( BCIEngineName );
    
    return result;
}

// send probescript to BCI
static TResult SendProbescriptToBCI( const char* probescript, int len )
{
	TResult result = MRTE_RESULT_OK;
	
	LOG_TRACE( PROFILER_NAME " content starts with " << (probescript ? probescript : "") );
    if( probescript ) {
    	int errCode = Initialize_fn(BCIEngine, probescript, len);

		if (errCode == 0) {
    		LOG_TRACE( PROFILER_NAME " probe script file processed successfully");
		} else {
        	LOG_ERROR( PROFILER_NAME " error processing probe script file");
            result = MRTE_ERROR_FAIL;
		}        
	}
	return result;
}

static TResult LoadProbeScriptFromOptions(const char* options)
{
	LOG_ASSERT(BCIEngine);
	TResult result = MRTE_RESULT_OK;
	
	char* probeScriptName = Util::GetCmdlineOption( options, CMDLINE_PARAM_PROBESCRIPT );
        
    if( probeScriptName ) {
    	
    	LOG_TRACE( PROFILER_NAME " probeScriptName=" << probeScriptName );
    
    	char* content;
    	int len;
    	Util::ReadFile( probeScriptName, content, len ); 
    	
    	result = SendProbescriptToBCI( content, len );
    	
    	free( content );
    	free( probeScriptName );
    }
    
    return result;
}

static void *
get_vm_pointer(void **pJniEnv) 
{
    bool bAttached = false;
	
	TResult res = CGlobals::Instance()->pfnJPI_AttachCurrentThread(
        (JNIEnv**)pJniEnv, &bAttached);
    if (MRTE_SUCCEEDED(res))
	{        	
		return *pJniEnv;
	}
	else {
		LOG_ERROR( PROFILER_NAME " Error attaching the jvm" );
	}
    
    
}  
// BCI callback
int BCICallback(const char* i_pInfo, size_t i_cbInfo, unsigned i_wFlags)
{
    int result = 1;    
    
    if(!CGlobals::Instance()->m_VMInitSeen)
    {
        result = 0;
    }
    else if(i_wFlags == BCIENGINTERFACE_CALLBACK_MODULE)
    {
    	// let it instrument
    	// NOTE: is this is a probekit class, the condition above
    	// (m_VMInitSeen) disables its instrumentation
    	
		/* Check to see if the current class is the probe class. If it is, we need to veto
		 * it to prevent infinite instrumentation. */
    	char* nameBuf = Util::StrNDup( i_pInfo, (int)i_cbInfo );
    	std::set<std::string>::iterator iter;
    	iter = s_probeClasses.find( nameBuf );

    	if ( iter != s_probeClasses.end() ) 
    		return 0;    	
    }
    else if (
    	i_wFlags == BCIENGINTERFACE_CALLBACK_MODULE_INSTR ||
    	i_wFlags == BCIENGINTERFACE_CALLBACK_METHOD_INSTR ) {
    	
        /*
         * This is a report that this class or method was actually instrumented.
         * */      
#if defined(DEBUG) || defined(_DEBUG)         
		char* nameBuf = Util::StrNDup( i_pInfo, (int)i_cbInfo );  		
		if( nameBuf ) {
			LOG_TRACE( PROFILER_NAME " instrumented " << nameBuf );
			free( nameBuf );
		} else {
			LOG_ERROR( PROFILER_NAME " out of memory" );
		}
#endif        
    }
    
    return result;   
}

// process data sent by client - it's either probescript or .class
static 
TResult ProcessClientFile( const char* name, const char* data, int len )
{
	TResult retVal = MRTE_RESULT_OK;
		
	// name is either a class name or something.probescript
	if( strstr( name, PROBESCRIPT_FILE_EXTENSION ) ) {
		// probescript
		retVal = SendProbescriptToBCI( data, len );			
	} else if( const char* ext = strstr( name, PROBESCRIPT_CLASS_EXTENSION ) ) {
		// assume class
		char* nameBuf = Util::StrNDup( name, ext - name ); // strip extension		
		if( nameBuf == 0 ) {
			return MRTE_ERROR_FAIL;
		}
		char* dataBuf = Util::StrNDup( data, len );
		if( dataBuf == 0 ) {
			return MRTE_ERROR_FAIL;
		}
		SDelayedClass* delayedClass = new SDelayedClass( nameBuf, dataBuf, len );		
		SDelayedClass::AddClass( delayedClass );

		/* We're being passed the probe class. Add it to the probeClasses list so
		 * that we can veto instrumentation of this class */
		s_probeClasses.insert( nameBuf );
		free( nameBuf );
	} else {
		LOG_ERROR( PROFILER_NAME " unsupported file " << name );
		retVal = MRTE_ERROR_FAIL;
	}	
	return retVal;
}

// read probekit settings from EC; return MRTE_RESULT_TRUE if such 
// settings found, MRTE_RESULT_FALSE if not found, an error code
// otherwise.	
static TResult ReadClientProbekitData()
{
	TResult retVal = MRTE_RESULT_FALSE;
	
   	// read probekit options
	CProbekitAgent* agent = CProbekitAgent::getInstance();
	
	// iterate and find all probekit options	
	for( int count = 0;; count++ ) {
		char optName[64];
		sprintf( optName, "PROBEKIT_DATA_%d", count );
		
    	const char* data = agent->getEC()->getUnknownOptionByName( optName );       
    
    	if( data == 0 ) {
    		break;
    	}    	    
    	
    	LOG_TRACE( PROFILER_NAME " " << optName << "=" << data	);
    	
    	int len = (int)strlen( data );
    	LOG_TRACE( PROFILER_NAME " total len = " << len );
    	Util::DumpBuffer( logLocation, "base64", data, len );
    	
		int decoded = Util::DecodeBufferLength( data, len );
		if( decoded < 1 ) {
			LOG_TRACE( PROFILER_NAME " base64 decoder returned invalid length " <<
				decoded );
			break;
		}
		char* out = (char*)malloc( decoded + 1 ); 
    	int decodedResult = Util::DecodeBuffer( data, len, out );     	
		if( decodedResult != 0 ) {
			LOG_TRACE( PROFILER_NAME " base64 decoder failed with code " <<
				decodedResult );
			free( out );
			break;
		}
    	
    	// java is big-endian (i.e. network byte order)
    	int magic = ntohl(*(int*)out);
    	
    	char* ptr = out + sizeof(magic);
    	
    	LOG_TRACE( PROFILER_NAME " decoded " << decoded << 
    		" bytes, magic=" << magic);
    	    	
    	// ptr alignment may be inefficient
    	
    	while( ptr < out + decoded ) {

    		int tmpAligned;

    		// SPARC doesn't allow unaligned dereferences (i.e. *(int*)ptr)
    		// so we must use memcpy
    		memcpy( &tmpAligned, ptr, sizeof(tmpAligned) );
    	
    		int nameLen = ntohl( tmpAligned );
    		ptr += sizeof( nameLen );
    		
    		char* name = Util::StrNDup( ptr, nameLen ); 
			if( name == 0 ) {
				return MRTE_ERROR_FAIL;
			}
    		ptr += nameLen;
    		
    		memcpy( &tmpAligned, ptr, sizeof(tmpAligned) );
    		int dataLen = ntohl( tmpAligned );
    		ptr += sizeof(dataLen);
    		
    		// read data
    		LOG_TRACE( PROFILER_NAME " nameLen=" << nameLen
    			<< " name = " << name
    			<< " dataLen = " << dataLen );
    			
    		retVal = ProcessClientFile( name, ptr, dataLen );
    		free( name );    		
    		if( MRTE_FAILED(retVal) ) {
				return retVal;    			
    		}
    		
    		ptr += dataLen;
    	}
    	 
    	free(out); 
    	
    	// read at least one option block
    	retVal = MRTE_RESULT_TRUE;    	
	} // for
    
    return retVal;
}
                                
//
// CBCIInstrumenter
//


CBCIInstrumenter::CBCIInstrumenter()
    : m_initialized(false), m_stackMapCalcEnabled(false),m_JVMInitDone(false)
{}

CBCIInstrumenter::~CBCIInstrumenter() 
{
    if( m_initialized && BCIEngine ) {
        DestroyBCIEngine_fn( BCIEngine );           
    }
}

TResult CBCIInstrumenter::Initialize(const char* options,
                                     bool enableStackMapCalc)
{
    m_stackMapCalcEnabled = enableStackMapCalc;
	TResult retVal = MRTE_RESULT_OK;
	
    LOG_TRACE( PROFILER_NAME " Initialize BCI Instrumenter, options=" << options);
            
    if( m_initialized ) {
    	LOG_MESSAGE( PROFILER_NAME " Instrumenter already initialized");
        return retVal;
    }        
                 
	// load BCI library                 
	retVal = LoadBCI(options);                 
    if( MRTE_FAILED( retVal ) ) {
        LOG_ERROR( PROFILER_NAME " Can't load BCI engine, error=" << retVal ); 
        return retVal;       
    } 
    
    SetCallback_fn(BCIEngine, BCICallback, 0xFFFF);

	logLocation = Util::GetCmdlineOption( options, CMDLINE_PARAM_DUMP );
    
    // read and process probekit settings from EC first
    retVal = ReadClientProbekitData();
    if( MRTE_FAILED(retVal) ) {
    	LOG_ERROR( PROFILER_NAME " Can't load probescript data from EC" ); 
        return retVal;    	
    }
    if( retVal == MRTE_RESULT_FALSE ) {
    	// didn't find probescript data in EC - read options     
    	retVal = LoadProbeScriptFromOptions(options);
    }
    if( MRTE_FAILED( retVal ) ) {
        LOG_ERROR( PROFILER_NAME " Can't load probescript from options" ); 
        return retVal;       
    }
    
    m_initialized = true;
    return retVal;    
}

TResult CBCIInstrumenter::Instrument(
    TMemoryAllocatorFunc memAllocFunc,
	const char* className,
    const char* inClass, S32 inClassLength, 
    char** outClass, S32* outClassLength)
{
	TResult result = MRTE_RESULT_FALSE;
    
    if( !m_initialized ) {
        LOG_ERROR( PROFILER_NAME " BCI engine is not initialized" );
        return ( result = MRTE_ERROR_FAIL );
    }
    
    char* outClassTmp = 0;
    size_t outClassLengthTmp = 0;
    
    lockBCI->Enter(); // lock
    
    SetAllocator_fn( BCIEngine, (pfnMalloc_t)memAllocFunc );

	unsigned retVal = 0;
	
    if (m_stackMapCalcEnabled && m_JVMInitDone) {
		retVal= Instrument2_fn( BCIEngine, inClass, inClassLength, 
    	    &outClassTmp, &outClassLengthTmp, get_vm_pointer);
    } else {
        retVal = Instrument_fn( BCIEngine, inClass, inClassLength, 
    	    &outClassTmp, &outClassLengthTmp);
    }
	
		SetAllocator_fn( BCIEngine, 0 );
	    
		lockBCI->Leave(); // unlock
	    
		if(retVal > 0) {
			result = MRTE_RESULT_FALSE;
		}
		else {
			if( outClassLengthTmp == inClassLength ) {
			; // did nothing
			} else {   
				// instrumented        

				Util::DumpBuffer( logLocation, className,
					(const char*)outClassTmp, (int)outClassLengthTmp );        

				*outClass = outClassTmp;
				*outClassLength = outClassLengthTmp; 
		        
				result = MRTE_RESULT_TRUE;         
			}
		}
	return result;
}                            

void CBCIInstrumenter::OnVMInit()
{
	m_JVMInitDone = true;
	   
	LOG_TRACE( PROFILER_NAME " CBCIInstrumenter::OnVMInit" );
	loadDelayedClass();
	
}

void CBCIInstrumenter::OnDataCollectionEnabled() {
	lockBCI->Enter();
	ReadClientProbekitData();
	loadDelayedClass();
	lockBCI->Leave();
}

void CBCIInstrumenter::loadDelayedClass() {
	bool bAttached = false;
	JNIEnv *pEnv = 0;
	TResult res = CGlobals::Instance()->pfnJPI_AttachCurrentThread(
        (JNIEnv**)&pEnv, &bAttached);
    if (MRTE_SUCCEEDED(res))
    {        	
		SDelayedClass* cls = SDelayedClass::head;
		while( cls ) {
	
			LOG_TRACE( PROFILER_NAME " process delayed class " << cls->name );
			
        	Util::DumpBuffer( logLocation, cls->name, cls->data, cls->len );
        	 
        	jclass c = pEnv->DefineClass(
        		cls->name, (jobject)NULL, (const jbyte*)cls->data, cls->len);
			LOG_TRACE( PROFILER_NAME " DefineClass returned " << (void*)c );        		
			if(c != 0 ) {
				pEnv->NewGlobalRef(c);
			}
			
			cls = cls->next;
			
		} // while
		
		SDelayedClass::FreeAllClasses();
				
		if (bAttached)
        {
            // Need to detach the thread from the JVM for proper cleanup
            CGlobals::Instance()->pfnJPI_DetachCurrentThread();
        }
	}

}

