/************************************************************************
 * Copyright (c) 2007, 2010 OC Systems Inc. and others
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Vsevolod Sandomirskiy, OC Systems - Probekit support
 *
 * $Id$ 
 ************************************************************************/
 
#include "Util.h"
#include "ProbekitAgent.h"

#include <log.h>

#ifdef _UNIX_
#define FILE_SEPARATOR '/'
#else
#define FILE_SEPARATOR '\\'
#endif

namespace Martini { namespace ProbekitAgent { namespace Util {

void DumpBuffer( const char* dir, const char* fileName, 
				 const char* buffer, int len )
{
	if( dir ) {
		char* fileNameClean = strdup( fileName );
		
		// replace all slashes
		char* p = fileNameClean;
		while( *p ) {
			if( *p == FILE_SEPARATOR ) *p = '_';
			p++;
		}
		
		char* path = (char*)malloc( strlen(dir) + strlen(fileName) + 16 );
		sprintf( path, "%s%c%s.dump", dir, FILE_SEPARATOR, fileNameClean );
    
    	FILE* handle = fopen( path, "wb+" );    	
    	if( handle == 0 ) {
     		LOG_ERROR( PROFILER_NAME " can't create dump file " << path );
     		free( path );
     		return;   
		}
    	fwrite( buffer, len, 1, handle );
	    fclose( handle );
	    
	    free( fileNameClean );
	    free( path );	    
	}
}

// caller must free result
void ReadFile( const char* fname, char*& result, int& resultLen )
{
	resultLen = 0;
	
	FILE* handle = fopen( fname, "r" );
	if( handle == 0 ) {
		LOG_ERROR( PROFILER_NAME " can't open " << fname );
		return;
	}
	
	fseek(handle, 0L, SEEK_END);
	long len = ftell( handle );
	LOG_TRACE( PROFILER_NAME " length of probescript file is " << len ); 
    if( len < 1 ) {
    	fclose( handle );
		return;
    }	
    fseek(handle, 0L, SEEK_SET);
    
    result = (char*)malloc( len + 1 );
    resultLen = (int)fread( result, 1, len, handle );
    LOG_TRACE( PROFILER_NAME " read " << resultLen );
    result[ resultLen ] = 0;

	// remove     
    char* ptr = result;
    while( *ptr ) {
		if( *ptr == '\n') {
			*ptr = 0;
		}
		ptr++;
	}
    
    fclose( handle );
    return;
}


#define CMDLINE_PARAM_NAME_SEPARATOR '='
#define CMDLINE_PARAM_SEPARATOR ','

// caller must free result
char* GetCmdlineOption( const char* options, const char* name )
{
	char* result = NULL;
	const char* p0 = strstr( options, name );
	if(	p0 ) {
		p0 = strchr( p0, CMDLINE_PARAM_NAME_SEPARATOR );
		if( p0 && *(++p0) ) {
			const char* p1 = strchr( p0, CMDLINE_PARAM_SEPARATOR );
			int len = (int)(p1 ? (p1 - p0) : strlen(p0));
			result = (char*)malloc( len + 1 );
			strncpy( result, p0, len );
			result[ len ] = 0;
		}
	}
	
	return result;
}

int lookup(char c) {
	switch(c) {
		/* A-Z */
		case 'A': return 0;
		case 'B': return 1;
		case 'C': return 2;
		case 'D': return 3;
		case 'E': return 4;
		case 'F': return 5;
		case 'G': return 6;
		case 'H': return 7;
		case 'I': return 8;
		case 'J': return 9;
		case 'K': return 10;
		case 'L': return 11;
		case 'M': return 12;
		case 'N': return 13;
		case 'O': return 14;
		case 'P': return 15;
		case 'Q': return 16;
		case 'R': return 17;
		case 'S': return 18;
		case 'T': return 19;
		case 'U': return 20;
		case 'V': return 21;
		case 'W': return 22;
		case 'X': return 23;
		case 'Y': return 24;
		case 'Z': return 25;

		/* a-z */
		case 'a': return 26;
		case 'b': return 27;
		case 'c': return 28;
		case 'd': return 29;
		case 'e': return 30;
		case 'f': return 31;
		case 'g': return 32;
		case 'h': return 33;
		case 'i': return 34;
		case 'j': return 35;
		case 'k': return 36;
		case 'l': return 37;
		case 'm': return 38;
		case 'n': return 39;
		case 'o': return 40;
		case 'p': return 41;
		case 'q': return 42;
		case 'r': return 43;
		case 's': return 44;
		case 't': return 45;
		case 'u': return 46;
		case 'v': return 47;
		case 'w': return 48;
		case 'x': return 49;
		case 'y': return 50;
		case 'z': return 51;

		/* 0-9, +, / */
		case '0': return 52;
		case '1': return 53;
		case '2': return 54;
		case '3': return 55;
		case '4': return 56;
		case '5': return 57;
		case '6': return 58;
		case '7': return 59;
		case '8': return 60;
		case '9': return 61;
		case '+': return 62;
		case '/': return 63;

		/* = */
		case '=': return -1;
	}

	return 0;
}

int tptp_decodeBase64(const char* base64Data, int encodedLen, unsigned char* decodedData, int bufferLen) {
	int i, j;
	int out_count = 0;

	char *in_buffer; /* A modified buffer with all newlines removed */
	int in_len; /* A modified buffer size without all the newlines */

	int required;

	/* Need to handle the newline in MIME format */
	in_len = encodedLen;
	in_buffer = (char*)malloc(sizeof(char) * encodedLen); /* Upperbound max size equals to the original */
	memset(in_buffer, 0, encodedLen);
	for(i = 0, j = 0; i < encodedLen; i++) {
		if((base64Data[i] == '\n') || (base64Data[i] == '\r')) {
			in_len--; /* Decrement the length by one whenever a newline is encountered */
		}
		// Bug 324334 -The zOS XML toolkit tends to translate the \n\r sequence into the
		// following meaningless code points. So, if we see them in MVS-land, we skip 
		// them and pretend they are whitespace like \n\r.
#ifdef MVS
		else if( 0xc2 == base64Data[i] || 0x85 == base64Data[i] ) {
			LOG_TRACE( PROFILER_NAME " decodeBase64: skipping busted char '" << base64Data[i] << "' at index=" << i );
			in_len--;
		}
#endif
		else {
			in_buffer[j] = base64Data[i]; /* Otherwise copy the byte to the modified buffer */
			j++;
		}
	}

	/* Check if the input buffer contains a valid Base64 representation*/
	if(in_len % 4 != 0) {
		free(in_buffer);
		return -1;
	}

	/* Check the output buffer if it can contain the result */
	required = in_len / 4 * 3; /* The upperbound */
	if(in_buffer[in_len - 1] == '=') {
		required--; /* One '=' means one zero padded byte */
		if(in_buffer[in_len - 2] == '=') {
			required--; /* Two '=' means two zero padded bytes */
		}
	}

	if(bufferLen < required || decodedData == NULL) {
		free(in_buffer);
		return required; /* Add a null at the end */
	}


	for(i = 0; i < (in_len / 4); i++) {
		int b64[4];

		b64[0] = lookup(in_buffer[i * 4]);
		b64[1] = lookup(in_buffer[i * 4 + 1]);
		b64[2] = lookup(in_buffer[i * 4 + 2]);
		b64[3] = lookup(in_buffer[i * 4 + 3]);

		/* Process the 1st byte */
		decodedData[out_count] = (b64[0] << 2) + (b64[1] >> 4);
		out_count++;

		/* No more byte if the 3rd and 4th Base64 equal to the '=' character */
		if((b64[2] == -1) && (b64[3] == -1)){
			/* done */
		}
		else {
			decodedData[out_count] = (b64[1] << 4) + (b64[2] >> 2);
			out_count++;

			/* No more byte if the 4th Base64 equals to the '=' character */
			if(b64[3] == -1) {
				/* done */
			}
			else {
				decodedData[out_count] = (b64[2] << 6) + b64[3];
				out_count++;
			}
		}

	}

	/* decodedData[out_count] = 0; */ /* Decoding does not need to terminate with a null */
	free(in_buffer);

	return 0;
}

int DecodeBuffer( const char* inBuffer, int inLen, char* outBuffer)
{
	return tptp_decodeBase64( inBuffer, inLen,
			(unsigned char*)outBuffer, inLen );
}

int DecodeBufferLength( const char* inBuffer, int inLen )
{
	return tptp_decodeBase64( inBuffer, inLen, 0, 0 );
}


char* StrNDup( const char* src, int len )
{
	char* dst = (char*)malloc( len + 1 );
	if( dst == 0 ) {
		return 0;
	}
	memcpy( dst, src, len );
    dst[ len ] = 0;
	return dst;
}	

} /*namespace Martini*/  } /*namespace ProbekitAgent*/ } /*namespace Util*/
