/************************************************************************
 * Copyright (c) 2006, 2010 Intel Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *    Viacheslav Rybalov, Intel - Initial API and implementation
 *
 * $Id: ThreadProfiler.cpp,v 1.16 2010/08/17 16:30:00 jwest Exp $ 
 ************************************************************************/

#include "ThreadProfiler.h"

#ifdef MVS
	#include <strings.h>
#endif

#include <string.h>

using namespace Martini::BaseProf;
using namespace Martini::ThreadProf;
using namespace Martini::MPI;

/*
 *    MPICL_Instantiate - The profiler initialization method. 
 *    The profiler initializes itself and registers for MPI events 
 */
extern "C" API_EXPORT TResult 
MPICL_Instantiate(IMpi *pMpiApi, TId clientId, const char *szOptions)
{
    LOG_ASSERT(pMpiApi != 0);
    LOG_ASSERT(clientId != 0);
    static Martini::ThreadProf::CThreadProfiler s_ThreadProfiler;
    return s_ThreadProfiler.Init(pMpiApi, clientId, szOptions);
}

//=======================================================================================

CThreadProfiler::CThreadProfiler()
{
    m_profilerName = "Thread profiler";
    m_pProfEnv.profName = "org.eclipse.tptp.analysisType.jvmti.thread";  // fixed for 190684
    m_pProfEnv.AddSupportedEG(EG_MONITOR);
}

CThreadProfiler::~CThreadProfiler()
{
}

/*
 * InitEvents - initialize object event - register events
 */
TResult 
CThreadProfiler::InitProfilerSpecificEvents()
{
    TResult retVal;
    retVal = m_customCommandHandler.Init(&m_pProfEnv);
    if (MRTE_FAILED(retVal)) {
        LOG_ERROR("Initialization of CustomCommand event failed: " << retVal);
        return retVal;
    }
    retVal = m_threadStartHandler.Init(&m_pProfEnv);
    if (MRTE_FAILED(retVal)) {
        LOG_ERROR("Initialization of ThreadStart event failed: " << retVal);
        return retVal;
    }
    retVal = m_threadEndHandler.Init(&m_pProfEnv);
    if (MRTE_FAILED(retVal)) {
        LOG_ERROR("Initialization of ThreadEnd event failed: " << retVal);
        return retVal;
    }
    retVal = m_monitorWaitHandler.Init(&m_pProfEnv);
    if (MRTE_FAILED(retVal)) {
        LOG_ERROR("Initialization of MonitorWait event failed: " << retVal);
        return retVal;
    }
    retVal = m_monitorWaitedHandler.Init(&m_pProfEnv);
    if (MRTE_FAILED(retVal)) {
        LOG_ERROR("Initialization of MonitorWaited event failed: " << retVal);
        return retVal;
    }
    retVal = m_contendedMonitorEnterHandler.Init(&m_pProfEnv);
    if (MRTE_FAILED(retVal)) {
        LOG_ERROR("Initialization of ContendedMonitorEnter event failed: " << retVal);
        return retVal;
    }
    retVal = m_contendedMonitorEnteredHandler.Init(&m_pProfEnv);
    if (MRTE_FAILED(retVal)) {
        LOG_ERROR("Initialization of ContendedMonitorEntered event failed: " << retVal);
        return retVal;
    }
    retVal = m_newMethodHandler.Init(&m_pProfEnv);
    if (MRTE_FAILED(retVal)) {
        LOG_ERROR("Initialization of NewMethod event failed: " << retVal);
        return retVal;
    }

    if (m_pProfEnv.ec_env->isContentionAnalysisSupported()) {
        m_pProfEnv.AddSupportedEG(EG_THREAD_INTERACTION);

        retVal = m_threadInteractionHandler.Init(&m_pProfEnv);
        if (MRTE_FAILED(retVal)) {
            LOG_ERROR("Initialization of callNotify event failed: " << retVal);
            return retVal;
        }
    }

    return MRTE_RESULT_OK;
}

TResult 
CThreadProfiler::InitFilter()
{
    // register to filter
    static CThreadInteractionFilter filter(&m_pProfEnv);

    TResult retVal = m_pProfEnv.m_pMpiApi->SetEventGroupFilter(
        m_pProfEnv.m_clientId,
        Martini::MPI::EG_THREAD_INTERACTION,
        filter);

    if (MRTE_SUCCEEDED(retVal)) {
        LOG_TRACE(m_profilerName << " filter: init success");
    } else {
        LOG_ERROR(m_profilerName << " filter: init failed");
    }
    return retVal;
}

TResult 
CThreadProfiler::ParceOptions(const char *szOptions)
{
#ifndef _WIN32
#define stricmp strcasecmp 
#endif
	TResult result = MRTE_RESULT_OK;
	int size = strlen(szOptions);
	if(size != 0){
		size += 1; //include \0
		char *buffCopy, *pnext;
		buffCopy = pnext = new char[size];
		if(pnext == 0){
			return MRTE_ERROR_OUT_OF_MEMORY;
		}
		
		memcpy(pnext, szOptions, size);
		while (pnext) {
    		char* pname = pnext;	// current tag
    		
    		pnext = strchr(pnext, ',');
    		if (pnext != 0) {
    			*pnext++ = '\0';	// next tag
    		}
    		
    		char* pvalue = strchr(pname, '=');
    		if (pvalue != 0) {
    			*pvalue++ = '\0';
    		}

			if (pvalue == 0){
				result = MRTE_ERROR_ILLEGAL_ARGUMENT;
				break;
			}

			/* Specied contanalysis */
			if (stricmp(pname, "contanalysis") == 0) {
				if(stricmp(pvalue, "true") == 0){
					m_pProfEnv.ec_env->SetProfileOption("CONTANALYSIS", "true");
				} else if (stricmp(pvalue, "false") == 0){
					m_pProfEnv.ec_env->SetProfileOption("CONTANALYSIS", "false");
				} else {
					result = MRTE_ERROR_ILLEGAL_ARGUMENT;
					break;
				}
			} /* Stack depth */
			else if (stricmp(pname, "maxstackdepth") == 0) {
				m_pProfEnv.ec_env->SetProfileOption("MAXSTACKDEPTH", pvalue);
			}
			else {
				fprintf(stderr, "Unknown profiler option %s=%s. Use defaults.\n", pname, pvalue);
			} 
		}
		
		delete buffCopy;
	}
	
	return result;
}


//=======================================================================================

CThreadInteractionFilter::CThreadInteractionFilter(CProfEnv* profEnv)
{
    m_pProfEnv = profEnv;
}

bool 
CThreadInteractionFilter::ShouldNotify(SThreadInteractionFilterData &data)
{
    if (m_pProfEnv->ec_env->IsExcluded(data.szClassName, data.szMethodName)) {
        return false;
    }
    return true;
}

