/*******************************************************************************
 * Copyright (c) 2006, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: BIRTReportGenerator.java,v 1.17 2010/11/25 08:18:19 bjerome Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.tptp.platform.report.birt.core.provisional;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PrintStream;
import java.util.HashMap;
import java.util.logging.Level;

import org.eclipse.birt.core.exception.BirtException;
import org.eclipse.birt.core.framework.Platform;
import org.eclipse.birt.report.data.oda.xml.Constants;
import org.eclipse.birt.report.engine.api.EngineConfig;
import org.eclipse.birt.report.engine.api.EngineConstants;
import org.eclipse.birt.report.engine.api.EngineException;
import org.eclipse.birt.report.engine.api.HTMLActionHandler;
import org.eclipse.birt.report.engine.api.HTMLEmitterConfig;
import org.eclipse.birt.report.engine.api.HTMLRenderOption;
import org.eclipse.birt.report.engine.api.IReportEngine;
import org.eclipse.birt.report.engine.api.IReportEngineFactory;
import org.eclipse.birt.report.engine.api.IReportRunnable;
import org.eclipse.birt.report.engine.api.IRunAndRenderTask;
import org.eclipse.core.runtime.FileLocator;
import org.eclipse.core.runtime.Path;
import org.eclipse.tptp.platform.report.birt.core.BIRTPlugin;
import org.eclipse.tptp.platform.report.birt.core.internal.impl.BIRTResourceLocator;
import org.eclipse.tptp.platform.report.birt.core.internal.impl.HtmlSortWrapper;
import org.eclipse.tptp.platform.report.birt.core.internal.impl.TPTPHTMLRenderContext;
import org.eclipse.tptp.platform.report.birt.core.internal.impl.TPTPImageHandler;

/**
 * BIRTReportGenerator provides an entry point to generate BIRT reports based on BIRT report template design files.
 * This class provides method that takes in a IReportElement object that contains the template file, datasource and output
 * stream to generate the HTML report.
 * <p/>
 * 
 * 
 * @author  Sheldon Lee-Loy
 * @author  Paul E. Slauenwhite
 * @author  Jerome Bozier
 * @version November 23, 2010
 * @since   March 24, 2006
 * @see     org.eclipse.tptp.platform.report.core.provisional.IReportElement.
 */
public class BIRTReportGenerator {

	private static BIRTReportGenerator INSTANCE;
	protected String birtHome;
	
	/**
	 * engine used to execute the tasks.
	 */
	private IReportEngine engine;
	
	/**
	 * Singleton that returns the BIRTReportGenerator instance.
	 */
	public static BIRTReportGenerator getInstance(){
		if (INSTANCE == null)
			INSTANCE = new BIRTReportGenerator();
		return INSTANCE;
		
	}

	/**
	 * Generator api that takes in an output stream that will generate will write to, an input stream that holds
	 * the datasource needed by the template file, an input stream that contains the BIRT report template file and 
	 * the location to write any images for the report.
	 * 
	 * @param os the output stream that the generate will write the html content to.
	 * @param imageDir the image directory that will hold any images produced by the generator.
	 * @param datasource the datasource that contains the raw data needed by the report template file.
	 * @param templateFile the report template file that contains the cosmetic and layout information for the report.
	 * @throws Exception when an error occurs report generation.
	 */
	public void generateHTMLReport(OutputStream os, String imageDir, InputStream datasource, InputStream templateFile) throws Exception{
		generateHTMLReportFragment(os, imageDir, null, datasource, templateFile, false, null);		
	}
	
	public void generateHTMLReport(OutputStream os, String imageDir, String baseImageURL, InputStream datasource, InputStream templateFile, BIRTResourceLocator resourceLocator) throws Exception{
		generateHTMLReportFragment(os, imageDir, baseImageURL, datasource, templateFile, false, resourceLocator);		
	}
	
	/**
	 * Generator api that takes in a IReportElement object that contains the report information needed 
	 * to generate a reoprt and the location to write any images for the report.
	 * 
	 * @param reportElement IReportElement that contians report data.
	 * @param imageDir the image directory that will hold any images produced by the generator.
	 * @param baseImageURL the image directory that will hold any images produced by the generator.
	 * @param resourcePath the path that contains the resource properties file.
	 * @throws Exception when an error occurs report generation.
	 * @see org.eclipse.tptp.platform.report.core.provisional.IReportElement.
	 */
	public void generateHTMLReport(IReportElement reportElement, String imageDir, String baseImageURL, BIRTResourceLocator resourceLocator) throws Exception{
		OutputStream reportStream = reportElement.getOutputStream();
		PrintStream ps = null;
		ps = new PrintStream(reportStream);
		if (reportElement.isSortable()){
			copySortIcons(reportElement.isSortable(), baseImageURL+File.separator+imageDir);
			ps.print(HtmlSortWrapper.getHeader(imageDir));
			reportElement.setEmbedded(true);
			
		}
		processReportElement(ps, reportElement, imageDir, baseImageURL, resourceLocator);
		if (reportElement.isSortable()){
			reportElement.setEmbedded(false);
			if (ps != null){
				ps.print(HtmlSortWrapper.getFooter());
			}
			
		}
		if (ps != null){
			ps.close();
		}
	}
	
	private void copySortIcons(boolean sortable, String outputFolder) throws IOException{
		
		if (sortable){
		
			File file = new File(outputFolder);
			
			if(!file.exists()){
				file.mkdir();
			}
			
			if (file.isDirectory()){
				copyImageFiles(outputFolder, "arrow.gif");
				copyImageFiles(outputFolder, "darrow.gif");
				copyImageFiles(outputFolder, "uarrow.gif");
			}
		}
	}
	
	private void copyImageFiles(String outputFolder, String filename) throws IOException{
		int len;
		InputStream in = FileLocator.find(BIRTPlugin.getDefault().getBundle(),new Path("images"+File.separator+filename), null).openStream();
		byte[] buffer = new byte[2*1024];
		OutputStream out = new FileOutputStream(outputFolder+File.separator + filename);
		while ((len = in.read(buffer)) >= 0)
				out.write(buffer, 0, len);
		out.close();
		
	}
	
	
	private  void processReportElement(OutputStream parentStream, IReportElement reportElement, String imageDir, String baseImageURL,  BIRTResourceLocator resourceLocator) throws Exception{
		OutputStream reportStream = null;
		PrintStream ps = null;
		boolean embedded = reportElement.isEmbedded();
		if ((parentStream != null) && (embedded)){
			reportStream = parentStream;
		}
		else if (reportElement.isSortable()){
			ps = new PrintStream(reportElement.getOutputStream());
			ps.print(HtmlSortWrapper.getHeader(imageDir));
			reportStream = ps;
			embedded = true;
		}
		else{
			reportStream = reportElement.getOutputStream();			
		}
		this.generateHTMLReportFragment(reportStream, imageDir, baseImageURL, reportElement.getDataSource(), reportElement.getTemplateFile(),embedded, resourceLocator);
		if (reportElement.isSortable()){
			if (ps != null){
				ps.print(HtmlSortWrapper.getFooter());
				ps.close();
			}			
		}
	
		if ((reportElement.getChildElements() != null) && (reportElement.getChildElements().size() > 0)){
			//recursively generate report
			for (int x = 0; x < reportElement.getChildElements().size(); x++){
				IReportElement childReportElement = (IReportElement)reportElement.getChildElements().get(x);
				processReportElement(reportStream, childReportElement, imageDir, baseImageURL, resourceLocator);
			}
		}
	}
	
	public  void generateHTMLReport(OutputStream out, InputStream templateFile, InputStream dataSource, boolean sortable, String imageDir, String baseImageURL,  BIRTResourceLocator resourceLocator) throws Exception{
		OutputStream reportStream = null;
		PrintStream ps = null;
		ps = new PrintStream(out);
		reportStream = ps;
		if (sortable){
			copySortIcons(sortable, baseImageURL+File.separator+imageDir);		
			ps.print(HtmlSortWrapper.getHeader(imageDir));
		}
		this.generateHTMLReportFragment(reportStream, imageDir, baseImageURL, dataSource, templateFile ,false, resourceLocator);
		if (sortable){
			if (ps != null){
				ps.print(HtmlSortWrapper.getFooter());
				ps.close();
			}			
		}
		if (ps != null){
			ps.close();
		}			
	
	}
	
	private IReportEngine getEngine() throws BirtException{
		if (engine == null){
			EngineConfig config = new EngineConfig( );
			if (birtHome != null)
				config.setProperty("BIRT_HOME", birtHome); //$NON-NLS-1$
			Platform.startup( config );
			IReportEngineFactory factory = (IReportEngineFactory) Platform
					.createFactoryObject( IReportEngineFactory.EXTENSION_REPORT_ENGINE_FACTORY );
			engine = factory.createReportEngine( config );

		    HTMLEmitterConfig emitterConfig = new HTMLEmitterConfig( );
            emitterConfig.setImageHandler( new TPTPImageHandler( ) );
            emitterConfig.setActionHandler(new HTMLActionHandler());
            config.getEmitterConfigs( ).put( "html", emitterConfig ); 
            
			// JRE default level is INFO, which may reveal too much internal
			// logging
			// information.
			engine.changeLogLevel( Level.SEVERE );	
		}
		
		return engine;
		
		
	}
	
	public boolean initializationOK(){
		try {
			Class.forName("org.eclipse.birt.report.engine.api.IReportEngine");
		} catch (ClassNotFoundException e) {
			return false;
		}
		return true;
	}
	private  void generateHTMLReportFragment(OutputStream os, String imageDir, String baseImageURL, InputStream xmlInput, InputStream templateFile, boolean embedded,  BIRTResourceLocator resourceLocator) throws Exception{
		
		engine = getEngine();
		
		TPTPHTMLRenderContext renderContext = new TPTPHTMLRenderContext();

		if (imageDir != null){
			renderContext.setImageDirectory(imageDir);
			renderContext.setImageDirectory(baseImageURL+File.separator+imageDir);
			renderContext.setTptpImageDir(imageDir);
		}

		//Set rendering options - such as file or stream output, 
		//output format, whether it is embeddable, etc
		HTMLRenderOption options = new HTMLRenderOption();

		//Set output format
		options.setOutputFormat("html");//$NON-NLS-1$

		renderContext.setSupportedImageFormats("SVG;PNG;GIF;JPG;BMP"); //$NON-NLS-1$ 
		renderContext.setRenderOption(options);

		HashMap contextMap = new HashMap();
		contextMap.put(EngineConstants.APPCONTEXT_HTML_RENDER_CONTEXT,
				renderContext);
		contextMap.put(Constants.APPCONTEXT_INPUTSTREAM , xmlInput);
		contextMap.put(Constants.APPCONTEXT_CLOSEINPUTSTREAM , "true");

		IReportRunnable report = null;
		try {
			
			report = engine.openReportDesign(templateFile);
			report.getDesignHandle().getModule().getSession().setResourceLocator(resourceLocator);
		} catch (EngineException e) {
			engine.destroy();
			throw e;
		}

		IRunAndRenderTask task = engine.createRunAndRenderTask(report);
		task.setAppContext(contextMap);
		//HTMLRenderOption options = new HTMLRenderOption();
		options.setEmbeddable(embedded);
		OutputStream out = os;
		if (embedded){
			out = new ByteArrayOutputStream();			
		}

		options.setOutputFormat("html");//$NON-NLS-1$
		options.setOutputStream(out);
		task.setRenderOption(options);

		task.setAppContext(contextMap);

		task.run();
		
		if (embedded){
			os.write(((ByteArrayOutputStream)out).toString("UTF-8").getBytes());
		}

	}
	

	/**
	 * If in standalone mode this method returns the location of the BIRT report engine
	 * @return
	 */
	public String getBirtHome() {
		return birtHome;
	}

	/**
	 * In stand alone mode this method should be called to set the location of the BIRT report engine.
	 * @param birtHome the location of the BIRT report engine
	 */
	public void setBirtHome(String birtHome) {
		this.birtHome = birtHome;
	}
	
}
