/**********************************************************************
 * Copyright (c) 2006,2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.tptp.platform.jvmti.client.internal;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.emf.common.util.EList;
import org.eclipse.hyades.loaders.util.LoadersUtils;
import org.eclipse.hyades.models.hierarchy.TRCAgentProxy;
import org.eclipse.hyades.models.hierarchy.TRCConfiguration;
import org.eclipse.hyades.models.hierarchy.TRCOption;
import org.eclipse.hyades.models.hierarchy.TRCProcessProxy;
import org.eclipse.hyades.trace.ui.ProfileEvent;
import org.eclipse.tptp.platform.jvmti.client.internal.launcher.TIDelegateHelper.ACDataProcessorAdapter;
import org.eclipse.tptp.trace.ui.internal.launcher.core.LauncherUtility;

/**
 * A utility class that contains common static methods used by multiple
 * classes.
 * 
 * @author Ali Mehregani
 */
public class TIUtility
{
	/** Common constants used to construct XML fragments */
	public static final String OPEN_ANGLE_BRACKET = "<";
	public static final String FORWARD_SLASH = "/";
	public static final String EMPTY_STRING = "";
	public static final String CLOSE_ANGLE_BRACKET = ">";
	public static final String SINGLE_SPACE = " ";
	public static final String EQUAL_SIGN = "=";
	public static final String DOUBLE_QUOTE = "\"";


	/**
	 * Terminates the agent proxy passed in.  If terminatedProcessProxy happens to be true,
	 * then the associated process proxy agentProxy is also terminated if doesn't contain
	 * any active agents.
	 *  
	 * @param agentProxy The agent proxy to be terminated
	 * @param terminateProcessProxy Indicates whether the associated process proxy should
	 * also be terminated 
	 */
	public static void terminateAgentProxy(TRCAgentProxy agentProxy, boolean terminateProcessProxy)
	{
		agentProxy.setActive(false);
		agentProxy.setAttached(false);
		agentProxy.setMonitored(false);
		LauncherUtility.sendProfileEvent(ProfileEvent.TERMINATE, agentProxy);
		
		
		closeFileStream(agentProxy);
		LoadersUtils.deregisterAgentInstance(agentProxy);
		if(!terminateProcessProxy)
			return;
		
		TRCProcessProxy processProxy = agentProxy.getProcessProxy();
		if (processProxy == null)
			return;
		
		EList agentProxies = processProxy.getAgentProxies();
		boolean agentsAreInactive = true;
		for (int i = 0, agentCount = agentProxies.size(); i < agentCount; i++)
		{
			if (((TRCAgentProxy)agentProxies.get(i)).isActive())
			{
				agentsAreInactive = false;
				break;
			}
		}
		
		if (agentsAreInactive)
			terminateProcessProxy(processProxy);
	}


	public static void closeFileStream(TRCAgentProxy trcAgentProxy)
	{
		if (trcAgentProxy == null || !trcAgentProxy.isToProfileFile())
			return;
		
		final ACDataProcessorAdapter dataProcessor = (ACDataProcessorAdapter) LoadersUtils.locateDataProcessor(trcAgentProxy);
		if (dataProcessor != null && dataProcessor.getProfileFileName() == null)
		{
			dataProcessor.cleanUp();
		}
			
		if (dataProcessor != null && dataProcessor.getProfileFileName() != null)
		{
			/* Start a thread that will periodically check the activity of the processor.  If it does not detect any activity
			 * after PERIODIC_CHECK milliseconds OR number of waits reaches MAXIMUM_NUM_OF_WAIT, then it will close the trace */
			dataProcessor.setProcessAlive(false);
			new Thread (new Runnable()
			{
				public void run() 
				{
					int currentActivityLevel = dataProcessor.getActivity();			
					final int MAXIMUM_NUM_OF_WAIT = 5;
					final int INCREMENTAL_WAIT = 2000;
					int numOfWaits = 0;
					do
					{
						numOfWaits++;
						try 
						{
							Thread.sleep(INCREMENTAL_WAIT);
						} catch (InterruptedException e) 
						{
							/* Doesn't need to be handled */
						}
					}while (numOfWaits < MAXIMUM_NUM_OF_WAIT && currentActivityLevel != dataProcessor.getActivity());

					dataProcessor.setEnabled(false);
					dataProcessor.endTrace(dataProcessor.getWriter());
				}}).start();
			
		}
	}
	
	/**
	 * Terminates the process proxy passed in
	 * 
	 * @param processProxy The process proxy to terminate
	 */
	public static void terminateProcessProxy(TRCProcessProxy processProxy)
	{
		processProxy.setActive(false);
		LauncherUtility.sendProfileEvent(ProfileEvent.TERMINATE, processProxy);
	}
	
	
	/**
	 * Return the set of options that contain the option name passed in
	 * 
	 * @param agentProxy The agent proxy
	 * @param optionName The option name
	 * 
	 * @return The set of options that have the same name as optionName
	 */
	public static TRCOption[] getAgentOptions (TRCAgentProxy agentProxy, String optionName)
	{
		EList agentProxyConf = agentProxy.getConfigurations();
		ArrayList optionList = new ArrayList();
		for (int i = 0, confCount = agentProxyConf.size(); i < confCount; i++)
		{
			TRCConfiguration config = (TRCConfiguration)agentProxyConf.get(i);
			List options = config.getOptions();
			for (int opt = 0; opt < options.size(); opt++)
			{		
				TRCOption option = (TRCOption)options.get(opt);
				if (option.getKey().indexOf(optionName) != -1)
					optionList.add(option);				
			}
		}
		
		TRCOption[] optionsFound = new TRCOption[optionList.size()];
		optionList.toArray(optionsFound);
		return optionsFound;
	}
	
	
	/**
	 * Adds an XML element to the string buffer passed in.
	 * 
	 * @param stringBuffer The string buffer that the element will be written to
	 * @param elementName The element name
	 * @param close Indicates whether this element should be closed or not.  If set, 
	 * a forward slash is added before theelement is ended.  
	 * @param end Indicates if the element should be ended with a closed angle bracket.  
	 * For example elements requring attributes are not ended.
	 */
	public static void addElement (StringBuffer stringBuffer, String elementName, boolean close, boolean end)
	{
		stringBuffer.append(OPEN_ANGLE_BRACKET);
		stringBuffer.append(close ? FORWARD_SLASH : EMPTY_STRING);		
		stringBuffer.append(elementName);
		stringBuffer.append(end ? CLOSE_ANGLE_BRACKET: EMPTY_STRING);		
	}
	
	
	/**
	 * Adds XML attributes to the string buffer passed in.  Attributes with a null value will
	 * not be written.
	 * 
	 * Pre-condition:
	 * <ul>
	 * 	<li> attributes and attributeValues must be valid (i.e. non-null) </li>
	 * 	<li> attributes and attributeValues length must be the same </li>
	 * </ul>
	 *  
	 * @param stringBuffer The string buffer that the attributes will be written to
	 * @param attributeNames The attribute names
	 * @param attributeValues The attribute values
	 * @param close If set, a forward slash is added after the attributes.  
	 * @param end If set, a closed angle bracket is added after the attributes
	 */
	public static void addAttribute(StringBuffer stringBuffer, String[] attributeNames, String[] attributeValues, boolean close, boolean end)
	{
		for (int i = 0; i < attributeValues.length; i++)
		{
			if (attributeValues[i] == null)
			{
				continue;
			}
			stringBuffer.append(SINGLE_SPACE);
			stringBuffer.append(attributeNames[i]);
			stringBuffer.append(EQUAL_SIGN);
			stringBuffer.append(DOUBLE_QUOTE);
			stringBuffer.append(attributeValues[i]);
			stringBuffer.append(DOUBLE_QUOTE);
		}
		
		stringBuffer.append(close ? FORWARD_SLASH : EMPTY_STRING);
		stringBuffer.append(end ? CLOSE_ANGLE_BRACKET : EMPTY_STRING);
	}
}
