/**********************************************************************
 * Copyright (c) 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 * @author ldancy
 **********************************************************************/
package org.eclipse.tptp.platform.jvmti.client.internal.analysistypes;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Vector;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.hyades.internal.execution.local.control.AgentConfigurationEntry;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.internal.launcher.IProfileLaunchConfigurationConstants;
import org.eclipse.hyades.trace.ui.internal.launcher.ProfilingSetsManager;
import org.eclipse.hyades.trace.ui.launcher.IProfilingSet;
import org.eclipse.hyades.trace.ui.launcher.ProfilingAttribute;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.widgets.Display;
import org.eclipse.tptp.platform.probekit.launch.internal.ProbekitLaunchPlugin;
import org.eclipse.tptp.platform.probekit.launch.internal.config.ProbekitLaunchMessages;
import org.eclipse.tptp.platform.probekit.registry.ProbeRegistry;
import org.eclipse.tptp.platform.probekit.registry.ProbeRegistryEntry;
import org.eclipse.tptp.platform.probekit.util.InvalidProbeBundleException;
import org.eclipse.tptp.platform.probekit.util.ProbeLaunchConfigString;
import org.eclipse.tptp.trace.ui.internal.launcher.core.LauncherMessages;
import org.eclipse.tptp.trace.ui.provisional.launcher.ILaunchValidator;
/**
 * A Validator for the Probe Launch Configuration in JVMTI
 * This was necessary due to the deprecated ProfilingType 
 * (JVMTI uses Analysis Type)
 * @author ldancy
 *
 */
public class TIProbekitLaunchValidator implements ILaunchValidator 
{
		/**
		 * Called before we launch the process for this launch configuration
		 * so this is the final check of whether or not we have any probes selected.
		 * Must return an error message if there are no probes selected.
		 * @param configuration The LaunchConfiguration
		 */
		public IStatus validateConfiguration(ILaunchConfiguration configuration)
		{
			Status noProbesSelected = null;
			try{
				boolean probesSelected = configuration.getAttribute("anyProbesSelected", false);
				if (!probesSelected)
				{
					// then we want to display an error message indicating that no probes have been selected for insertion 
					// and we want the Profile button to be disabled
					 noProbesSelected = new Status(Status.ERROR, UIPlugin.getPluginId(),LauncherMessages.ERROR_NO_PROBES_SELECTED);
				}
			}
			catch(Exception e){
				
			}
			
			return noProbesSelected;
		}
		
		/**
		 * This method is called right before we launch so it is a good time
		 * to check the validity of the probe selections rather than the
		 * validity of the launch itself which will have already passed to get to this 
		 * point.
		 * @see org.eclipse.tptp.platform.probekit.launch.internal.config.ProbekitProfilingType#launch(ILaunchConfiguration)
		 */
		public IStatus launchNotification(ILaunchConfiguration configuration)
		{
			final ProfilingSetsManager psm = ProfilingSetsManager.instance();
			final ArrayList invalidProbes = new ArrayList();
			String strMissingOrInvalidProbes = ""; //$NON-NLS-1$
			String strLineSeparator = System.getProperty("line.separator");
			final ILaunchConfiguration config = configuration;
			// we need to step through all of the probes selected for deployment and 
			// validate that all of the files are there for each one
			Map mapProbeSpecs= getSelectedProbes(configuration, ProbeLaunchConfigString.AGENT_CONFIG_NAME_PREFIX
					+ "PROBEUI");
			
			Iterator iterKeys = mapProbeSpecs.keySet().iterator();
			while (iterKeys.hasNext()) {
				String key = iterKeys.next().toString();
				String strProbeSpec = (String) mapProbeSpecs.get(key);
				ProbeLaunchConfigString probeConfigDeploy = ProbeLaunchConfigString
						.fromString(strProbeSpec);
				String strProbeID = probeConfigDeploy.getRegistryId();
				ProbeRegistryEntry entry = (ProbeRegistryEntry) ProbeRegistry.getRegistry()
						.lookupById(strProbeID);
				if (entry == null) {
					strMissingOrInvalidProbes = strMissingOrInvalidProbes
							+ strProbeID + strLineSeparator;
					invalidProbes.add(strProbeID);
				} else {
					try {
						entry.fullValidate();
					} catch (InvalidProbeBundleException e) {
						strMissingOrInvalidProbes = strMissingOrInvalidProbes
								+ strProbeID + strLineSeparator;
						invalidProbes.add(strProbeID);

						// Silently remove from registry so it will not show up
						// in subsequent lists, etc.
						ProbeRegistry.getRegistry().remove(entry);
					}
				}
			}

		if (strMissingOrInvalidProbes.length() > 0) {
			strMissingOrInvalidProbes = strLineSeparator
					+ strMissingOrInvalidProbes;
			final String strWarningQuery = NLS.bind(ProbekitLaunchMessages._26,
					strMissingOrInvalidProbes); //$NON-NLS-1$

			Display.getDefault().asyncExec(new Runnable() {
				public void run() {
					boolean bOK = MessageDialog.openQuestion(Display
							.getDefault().getActiveShell(),
							ProbekitLaunchMessages._27, strWarningQuery);
					if (bOK) {
						try {
							String setID = config
									.getAttribute(
											IProfileLaunchConfigurationConstants.ATTR_PROFILING_SET,
											(String) null);
							IProfilingSet set = (IProfilingSet) psm
									.getProfilingSets().get(setID);
							if (set != null) {
								String strProbeOptionPrefix = ProbeLaunchConfigString.LAUNCH_CONFIG_NAME_PREFIX
										+ "PROBEUI";
								boolean bRemovedProbes = false;
								Map attributes = set.getAttributes();
								Iterator iterKeys = attributes.keySet()
										.iterator();

								while (iterKeys.hasNext()) {
									String key = iterKeys.next().toString();
									if (key.startsWith(strProbeOptionPrefix)) {
										ProfilingAttribute profAttribute = (ProfilingAttribute) attributes
												.get(key);
										String strValue = profAttribute
												.getValue();
										ProbeLaunchConfigString probeConfigDeploy = ProbeLaunchConfigString
												.fromString(strValue);
										String strID = probeConfigDeploy
												.getRegistryId();
										// Check if invalid or missing probe
										if (invalidProbes.contains(strID)) {
											// Remove from attributes
											iterKeys.remove();
											bRemovedProbes = true;
										}
									}
								}
								// Update preferences to save profiling set
								// changes
								if (bRemovedProbes) {
									set.setAttributes(attributes);
									psm.writeSetsToPreferences();
								}
							}
						} catch (Exception e) {
							// TODO: Report more specific message indicating
							// removing probe from profiling set was not
							// successful
							e.printStackTrace();
							ProbekitLaunchPlugin.reportException(e);
						}
					}
				}
			});
		}

		return null;
	}// end method

			
	
		/**
		 * A method to return the selected Probes 
		 * @param launchConfig The LaunchConfiguration
		 * @param strPrefix The special prefix to help us find probes we are interested in
		 * @return Map representing the selected Probes
		 * @see org.eclipse.tptp.platform.probekit.launch.internal.config.ProbekitProfilingType#getSelectedProbes(ILaunchConfiguration, String)
		 */
		protected Map getSelectedProbes(ILaunchConfiguration launchConfig, String strPrefix) 
		{
			Map map = new HashMap();
			try 
			{
				ProfilingSetsManager psm = ProfilingSetsManager.instance();
				Vector configOptions = psm.getOptions(launchConfig);
				Iterator iterOptions = configOptions.iterator();
	
				while (iterOptions.hasNext()) 
				{
					AgentConfigurationEntry entry = (AgentConfigurationEntry) iterOptions
							.next();
					if (entry != null) 
					{
						String strName = entry.getName();
						if ((strName != null) && strName.startsWith(strPrefix)) 
						{
							String strValue = entry.getValue();
							map.put(strName, strValue);
						}
					}
				}
			} 
			catch (CoreException e) 
			{
			e.printStackTrace();
			}

		return map;
	}// end method
}
