/**********************************************************************
 * Copyright (c) 2006, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.tptp.platform.jvmti.client.internal.analysistypes;

import java.util.Vector;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;
import org.eclipse.hyades.internal.execution.local.common.Options;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.tptp.platform.jvmti.client.internal.ITILaunchConfigurationConstants;
import org.eclipse.tptp.platform.jvmti.client.internal.TIMessages;
import org.eclipse.tptp.trace.ui.internal.launcher.application.AbstractConfiguration;
import org.eclipse.tptp.trace.ui.internal.launcher.application.PITimeAnalysisConf.ExecutionTimeConfigurationPage;
import org.eclipse.tptp.trace.ui.internal.launcher.core.LauncherMessages;
import org.eclipse.tptp.trace.ui.internal.launcher.core.LauncherUtility;
import org.eclipse.tptp.trace.ui.provisional.launcher.DataCollectionEngineAttribute;
import org.eclipse.tptp.trace.ui.provisional.launcher.IStatusListener;


/**
 * The configuration wizard for the execution time analysis belonging to the
 * TI data collector.
 * 
 * @author Ali Mehregani
 */
public class TITimeAnalysisConfig extends AbstractConfiguration
{

	
	public TITimeAnalysisConfig()
	{
		super(new Class[]{TIExecutionTimeAnalysisOpt.class});
	}
	
	
	/**
	 * The execution time analysis options are displayed in this
	 * configuration page.
	 */
	public static class TIExecutionTimeAnalysisOpt extends ExecutionTimeConfigurationPage
	{
		/** The execution time analysis UI */
		private TITimeAnalysisUI execTimeAnalysisUI;
		
		/** The launch configuration */
		private ILaunchConfiguration launchConfiguration;
		
		/** Indicates whether CPU time should be collected */
		private boolean collectCPUTime;
		
		/** Indicates whether execution flow/statistics should be collected */
		private boolean showExecutionFlow;
		
		/** Indicates auto polling is used */
		private boolean autoPolling;
		
		/** Indicates manual polling period is used */
		private boolean manualPolling;
		
		/** Indicates frequency at manual polling*/
		private int pollingFrequency;
		
		
		public TIExecutionTimeAnalysisOpt()
		{
			execTimeAnalysisUI = new TITimeAnalysisUI();
		}

		
		public void reset(ILaunchConfiguration launchConfiguration)
		{
			this.launchConfiguration = launchConfiguration;
			try
			{
				collectCPUTime = launchConfiguration.getAttribute(ITILaunchConfigurationConstants.ATTR_EXEC_CPU_TIME, false);
				showExecutionFlow = launchConfiguration.getAttribute(ITILaunchConfigurationConstants.ATTR_EXEC_FLOW, true);
				autoPolling = launchConfiguration.getAttribute(ITILaunchConfigurationConstants.ATTR_EXEC_AUTO_POLLING, false);
				manualPolling = launchConfiguration.getAttribute(ITILaunchConfigurationConstants.ATTR_EXEC_MANUAL_POLLING, false);
				pollingFrequency =  launchConfiguration.getAttribute(ITILaunchConfigurationConstants.ATTR_EXEC_POLLING_FREQ, ExecutionTimeConfigurationPage.DEFAULT_POLLING_PERIOD);
			}
			catch (CoreException e)
			{
				LauncherUtility.openErrorWithDetail(
						LauncherMessages.LAUNCHER_COMMON_ERROR_TITLE, 
						TIMessages.ERROR_TI_CONFIGURATION_RESET, 
						e);
			}
		}
		
		public Vector getAttributes()
		{
			Vector options = null;
			try
			{
				options = new Vector();				
				options.add(new DataCollectionEngineAttribute(Options.OPTION_CPU_TIME, String.valueOf(launchConfiguration.getAttribute(ITILaunchConfigurationConstants.ATTR_EXEC_CPU_TIME, false))));
				options.add(new DataCollectionEngineAttribute(Options.OPTION_STACK_INFORMATION, Options.OPTION_VALUE_NORMAL));
				options.add(new DataCollectionEngineAttribute(ITILaunchConfigurationConstants.ATTR_EXEC_DATA, Options.OPTION_VALUE_TRUE));
				options.add(new DataCollectionEngineAttribute(ITILaunchConfigurationConstants.ATTR_EXEC_FLOW, String.valueOf(launchConfiguration.getAttribute(ITILaunchConfigurationConstants.ATTR_EXEC_FLOW, true))));
				options.add(new DataCollectionEngineAttribute(ITILaunchConfigurationConstants.ATTR_EXEC_AUTO_POLLING, String.valueOf(launchConfiguration.getAttribute(ITILaunchConfigurationConstants.ATTR_EXEC_AUTO_POLLING, false))));
				options.add(new DataCollectionEngineAttribute(ITILaunchConfigurationConstants.ATTR_EXEC_MANUAL_POLLING, String.valueOf(launchConfiguration.getAttribute(ITILaunchConfigurationConstants.ATTR_EXEC_MANUAL_POLLING, false))));
				options.add(new DataCollectionEngineAttribute(ITILaunchConfigurationConstants.ATTR_EXEC_POLLING_FREQ, String.valueOf(launchConfiguration.getAttribute(ITILaunchConfigurationConstants.ATTR_EXEC_POLLING_FREQ, ExecutionTimeConfigurationPage.DEFAULT_POLLING_PERIOD))));
			}
			catch (CoreException e)
			{
				LauncherUtility.openErrorWithDetail(
						LauncherMessages.LAUNCHER_COMMON_ERROR_TITLE,
						LauncherMessages.ERROR_LAUNCH_ATTRIBUTES, e);
			}
			
			return options;
		}
		
		
		public boolean storeAttributes(ILaunchConfigurationWorkingCopy configurationWC)
		{
			configurationWC.setAttribute(ITILaunchConfigurationConstants.ATTR_EXEC_CPU_TIME, execTimeAnalysisUI.isCollectCPUTimeSelected());
			configurationWC.setAttribute(ITILaunchConfigurationConstants.ATTR_EXEC_FLOW, execTimeAnalysisUI.isShowExecutionFlowSelected());
			configurationWC.setAttribute(ITILaunchConfigurationConstants.ATTR_EXEC_AUTO_POLLING, execTimeAnalysisUI.isAutoPollingSelected());
			configurationWC.setAttribute(ITILaunchConfigurationConstants.ATTR_EXEC_MANUAL_POLLING, execTimeAnalysisUI.isManualPollingSelected());
			configurationWC.setAttribute(ITILaunchConfigurationConstants.ATTR_EXEC_POLLING_FREQ, execTimeAnalysisUI.getPollingFrequency());
			return true;
			
		}

		public void addErrorListener(IStatusListener statusListener)
		{
			/* Doesn't need to be implemented */
		}

		public void createControl(Composite parent)
		{
			execTimeAnalysisUI.createControl(parent);
			execTimeAnalysisUI.setCollectCPUTime(collectCPUTime);
			execTimeAnalysisUI.setShowExecutionFlow(showExecutionFlow);
			execTimeAnalysisUI.setAutoPolling(autoPolling);
			execTimeAnalysisUI.setManualPolling(manualPolling);
			execTimeAnalysisUI.setPollingFrequency(pollingFrequency);
		}
	}

}
