/**********************************************************************
 * Copyright (c) 2006, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 * 
 * $Id: TITimeAnalysisUI.java,v 1.11 2010/05/07 19:11:35 jwest Exp $
 **********************************************************************/
package org.eclipse.tptp.platform.jvmti.client.internal.analysistypes;

import org.eclipse.hyades.trace.ui.internal.util.TraceMessages;
import org.eclipse.hyades.ui.util.GridUtil;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Spinner;
import org.eclipse.tptp.platform.jvmti.client.internal.TIMessages;
import org.eclipse.tptp.trace.ui.internal.launcher.application.PITimeAnalysisConf.ExecutionTimeConfigurationPage;


/**
 * The UI of the execution time analysis configuration page
 * 
 * @author Ali Mehregani
 */
public class TITimeAnalysisUI
{
	/** The cpu time option */
	private Button collectCPUTime; 
	
	/** Show execution statistics option */
	private Button showExecutionStat;
	
	/** Show execution flow option */
	private Button showExecutionFlow;
	
	/* The automatically calculate polling frequency option */
	private Button _autoFrequency;
	
	/* The manually set polling frequency option */
	private Button _manualFrequency;
	
	/* The value of the polling frequency */
	private Spinner _pollingFreq;
	
	public void createControl(Composite parent)
	{
		/* Create the level of detail group */
		createLevelOfDetailGroup(parent);
		
		/* Create the type of data group */
		createTypeOfDataGroup(parent);
	}

	private void createLevelOfDetailGroup(Composite parent)
	{
		Group grp = createGroup (parent, TIMessages.EXEC_LEVEL_OF_DETAIL);

		GridLayout gridLayout = new GridLayout();
		gridLayout.marginBottom = 10; 
    	grp.setLayout(gridLayout);
		
		collectCPUTime = createButton (grp, TIMessages.EXEC_COLLECT_CPU_TIME, SWT.CHECK);
	}


	private void createTypeOfDataGroup(Composite parent)
	{
		Group grp = createGroup (parent, TIMessages.EXEC_TYPE_OF_DATA);
		((GridData)grp.getLayoutData()).verticalAlignment = SWT.END;	
		showExecutionStat = createButton (grp, TIMessages.EXEC_COLLECT_STATISTICS, SWT.RADIO);
		addExecutionOption(grp);
		showExecutionFlow = createButton (grp, TIMessages.EXEC_COLLECT_FLOW, SWT.RADIO);
		showExecutionStat.setSelection(false);
		showExecutionFlow.setSelection(true);
	}
	
	private void addExecutionOption(Group ctx) {
		
		GridLayout layout = new GridLayout(2, false);
		GridData data;
		Composite _detailsExecGroup;
		_detailsExecGroup = new Composite(ctx, SWT.NULL);
		data = GridUtil.createHorizontalFill();
		data.horizontalIndent = 20;
		_detailsExecGroup.setLayoutData(data);
		_detailsExecGroup.setLayout(layout);
		
		/* Create the controls for allowing users to set a polling frequency */
		Composite pollingFreqComp = new Composite (_detailsExecGroup, SWT.NONE);
		GridLayout pollingFreqCompGL =  new GridLayout(2, false);
		pollingFreqCompGL.marginLeft = 20;
		pollingFreqComp.setLayout(pollingFreqCompGL);
		pollingFreqComp.setLayoutData(new GridData(SWT.FILL, SWT.DEFAULT, true, false));
		
		/* The auto calculate button */
		_autoFrequency = new Button (pollingFreqComp, SWT.RADIO);
		GridData autoFrequencyGD = new GridData(SWT.FILL, SWT.DEFAULT, true, false);
		autoFrequencyGD.horizontalSpan = 2;		
		_autoFrequency.setLayoutData(autoFrequencyGD);
		_autoFrequency.setText(TraceMessages.EXECUTION_ANALYSIS_AUTO_FREQ);
		//_autoFrequency.addListener(SWT.Selection, modificationNotifier);
		_autoFrequency.setSelection(true);
		
		/* The manual set button */
		_manualFrequency = new Button (pollingFreqComp, SWT.RADIO);		
		_manualFrequency.setLayoutData(new GridData(SWT.FILL, SWT.DEFAULT, true, false));
		_manualFrequency.setText(TraceMessages.EXECUTION_ANALYSIS_MANUAL_FREQ);
		_manualFrequency.setSelection(false);
		
		/* The spinner for the manual set button */
		_pollingFreq = new Spinner (pollingFreqComp, SWT.BORDER);
		_pollingFreq.setLayout(new GridLayout());
		_pollingFreq.setLayoutData(new GridData(SWT.FILL, SWT.DEFAULT, true, false));
		_pollingFreq.setValues(ExecutionTimeConfigurationPage.DEFAULT_POLLING_PERIOD, 1, 10000, 0, 10, 100);
		_pollingFreq.setEnabled(true);
		//_pollingFreq.addListener(SWT.Selection, modificationNotifier);		
		/* Enable the polling frequency spinner when the manual set option is selected */
		_manualFrequency.addSelectionListener(new SelectionAdapter()
		{
			public void widgetSelected(SelectionEvent e) 
			{
				_pollingFreq.setEnabled(true);
				// Bug 273634
				_pollingFreq.setFocus();
			}
		});
		_autoFrequency.addSelectionListener(new SelectionAdapter()
		{
			public void widgetSelected(SelectionEvent e) 
			{
				_pollingFreq.setEnabled(false);
			}
		});
		showExecutionStat.addSelectionListener(new SelectionAdapter(){
			public void widgetSelected(SelectionEvent e) 
			{				
				executionStateChange(((Button)e.widget).getSelection());
			}
		});
	}
	private Group createGroup(Composite parent, String name)
	{
		Group grp = new Group(parent, SWT.NONE);
		grp.setLayout(new GridLayout());
		grp.setLayoutData(new GridData(SWT.FILL, SWT.DEFAULT, true, true));
		grp.setText(name);
		
		return grp;
	}
	
	
	private Button createButton(Composite parent, String name, int style)
	{
		Button button = new Button(parent, style);
		button.setLayoutData(new GridData(SWT.FILL, SWT.DEFAULT, true, false));
		button.setText(name);
		
		return button;
	}
	
	/**
	 * @return the collectCPUTime
	 */
	public boolean isCollectCPUTimeSelected()
	{
		return collectCPUTime == null || collectCPUTime.isDisposed() ? false : collectCPUTime.getSelection();
	}
	
	/**
	 * @return the AutoPolling option
	 */
	public boolean isAutoPollingSelected()
	{
		return _autoFrequency == null || _autoFrequency.isDisposed() ? false : _autoFrequency.getSelection();
	}

	/**
	 * @return the ManualPolling option
	 */
	public boolean isManualPollingSelected()
	{
		return _manualFrequency == null || _manualFrequency.isDisposed() ? false : _manualFrequency.getSelection();
	}

	/**
	 * @return the polling frequency option
	 */
	public int getPollingFrequency() {
		return _pollingFreq.getSelection();
	}
	
	/**
	 * @param collectCPUTime the collectCPUTime to set
	 */
	public void setCollectCPUTime(boolean collectCPUTime)
	{
		this.collectCPUTime.setSelection(collectCPUTime);
	}

	/**
	 * @return the showExecutionFlow
	 */
	public boolean isShowExecutionFlowSelected()
	{
		return showExecutionFlow == null || showExecutionFlow.isDisposed() ? false : showExecutionFlow.getSelection();
	}

	/**
	 * @param showExecutionFlow the showExecutionFlow to set
	 */
	public void setShowExecutionFlow(boolean showExecutionFlow)
	{
		this.showExecutionFlow.setSelection(showExecutionFlow);
		this.showExecutionStat.setSelection(!showExecutionFlow);
		executionStateChange(!showExecutionFlow);		
	}

	/**
	 * @param  frequency set the polling frequency 
	 */
	public void setPollingFrequency(int frequency) {
		_pollingFreq.setSelection(frequency);
	}

	/**
	 * @param autoPolling the autoPolling to set
	 */
	public void setAutoPolling(boolean autoPolling) {
		_autoFrequency.setSelection(autoPolling);
	}

	/**
	 * @param manualPolling the manualPolling to set
	 */
	public void setManualPolling(boolean manualPolling) {
		_manualFrequency.setSelection(manualPolling);
		if (!showExecutionFlow.getSelection()) {
			_pollingFreq.setEnabled(manualPolling);
		}
		else {
			_pollingFreq.setEnabled(false);
		}
		
	}
	
	
	
	private void executionStateChange(boolean selected)
	{
		if (selected && !_autoFrequency.getSelection() && !_manualFrequency.getSelection()) {
			_autoFrequency.setSelection(selected);
		}
		_autoFrequency.setEnabled(selected);
		_manualFrequency.setEnabled(selected);
		_pollingFreq.setEnabled(selected && _manualFrequency.getSelection());
	}

	
}
