/**********************************************************************
 * Copyright (c) 2007, 2010 IBM Corporation, Intel Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.tptp.platform.jvmti.client.internal.controlproviders;

import java.util.Iterator;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.hyades.models.hierarchy.TRCAgentProxy;
import org.eclipse.hyades.models.hierarchy.TRCNode;
import org.eclipse.hyades.trace.ui.ProfileEvent;
import org.eclipse.hyades.trace.ui.internal.core.TraceUIImages;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.osgi.util.NLS;
import org.eclipse.tptp.platform.common.ui.trace.internal.CommonUITraceConstants;
import org.eclipse.tptp.platform.execution.client.agent.ICollector;
import org.eclipse.tptp.platform.execution.client.core.IAgentController;
import org.eclipse.tptp.platform.execution.exceptions.InactiveAgentException;
import org.eclipse.tptp.platform.jvmti.client.internal.TIMessages;
import org.eclipse.tptp.platform.jvmti.client.internal.TIPlugin;
import org.eclipse.tptp.platform.jvmti.client.internal.TIUtility;
import org.eclipse.tptp.platform.jvmti.client.internal.actions.RunGCActionDelegate;
import org.eclipse.tptp.platform.jvmti.client.internal.controlproviders.TIProcessControlProvider.TIProcessStateModifier;
import org.eclipse.tptp.platform.jvmti.client.internal.launcher.TIDelegateHelper;
import org.eclipse.tptp.platform.jvmti.client.internal.launcher.util.AgentControllerDelegate;
import org.eclipse.tptp.trace.ui.internal.control.provider.application.ControlMessages;
import org.eclipse.tptp.trace.ui.internal.launcher.core.LauncherMessages;
import org.eclipse.tptp.trace.ui.internal.launcher.core.LauncherUtility;
import org.eclipse.tptp.trace.ui.provisional.control.provider.AbstractAgentControlProvider;
import org.eclipse.tptp.trace.ui.provisional.control.provider.ControlItem;
import org.eclipse.tptp.trace.ui.provisional.control.provider.IAgentStateModifier;
import org.eclipse.tptp.trace.ui.provisional.control.provider.IControlItem;
import org.eclipse.tptp.trace.ui.provisional.control.provider.IProcessStateModifier;


/**
 * This is the TI agent control provider.  Its purpose is to modify the states that the TI agent
 * can be in (e.g. Pause/Resume/Detach/Attach).
 * 
 * @author Ali Mehregani
 */
public class TIAgentControlProvider extends AbstractAgentControlProvider
{
    
    /** The id of the contributed items */
    private static final String RUNGC_ITEM = "org.eclipse.tptp.trace.jvmti.client.internal.popupMenu.RunGCAction";
	
    /** The runGC item */
    private IControlItem runGCControlItem;

    
    protected void initializeControlItems() {
	super.initializeControlItems();
	    	
	/* Add the GC run option */
	addControlItem (createRunGCControlItem());
    }

    /**
     * Overwrite this method to modify the terminate control
     * item.
     * 
     * @return Ther terminate control item
     */
    protected IControlItem createRunGCControlItem() {
	if (runGCControlItem != null)
	    return runGCControlItem;

	runGCControlItem = new RunGCControlItem();
	return runGCControlItem;
    }

    static class RunGCControlItem extends ControlItem {
	
	private RunGCActionDelegate gcAction;
	
	public RunGCControlItem() {
	    super(RUNGC_ITEM, CommonUITraceConstants.PROFILE_GC_GROUP,
		    ControlMessages.CONTROL_ITEM_GC_ACTION,
		    TraceUIImages.INSTANCE.getImageDescriptor("c",
			    TraceUIImages.IMG_GC));
	    setDisabledImageDescriptor(TraceUIImages.INSTANCE
		    .getImageDescriptor("d", TraceUIImages.IMG_GC));
	    gcAction = new RunGCActionDelegate();
	}

	public void run() {
	    try {
		for (Iterator agents = input.iterator(); agents.hasNext();)
		    gcAction.doAction(agents.next());

	    } catch (Exception e) {
		LauncherUtility.openMessageWithDetail(IStatus.ERROR, "", e
			.getMessage(), e);
	    }
	}

	public boolean isEnabled() {
	    if (input == null || input.size() == 0)
		return false;

	    boolean isEnable = true;
	    for (Iterator agents = input.iterator(); isEnable
		    && agents.hasNext();)
		isEnable = isEnable && gcAction.isEnabledFor(agents.next());
	    return isEnable;
	}
    }
	
	public IAgentStateModifier getAgentStateModifier()
	{
		return TIAgentStateModifier.getInstance();
	}
	
	
	public IProcessStateModifier getProcessStateModifier()
	{		
		return TIProcessStateModifier.getInstance();
	}
	
	
	/**
	 * The actual state modifier for the TI agent.  This is a singleton class
	 * that should be accessed via the <code>getInstance</code> method.  All public
	 * methods of this class are synchornized.
	 * 
	 * @author Ali Mehregani
	 */
	public static class TIAgentStateModifier implements IAgentStateModifier
	{
		/** Used for the enable action operations */
		private static final byte CAN_ATTACH 	= 0x01;
		private static final byte CAN_DETACH 	= 0x02;
		private static final byte CAN_RESUME 	= 0x03;
		private static final byte CAN_PAUSE 	= 0x04;
		
		/** Used for each action */
		private static final byte ATTACH 		= 0x05;
		private static final byte DETACH 		= 0x06;
		private static final byte RESUME 		= 0x07;
		private static final byte PAUSE 		= 0x08;

		/** The instance of this singleton class */
		private static TIAgentStateModifier agentStateModifier = new TIAgentStateModifier();
		
		/** The input of this state modifier */
		private StructuredSelection input;
		
		/** The iterator representing the input */
		private Iterator inputIterator;
		
		/**
		 * Limit the visibility of the constructor
		 */
		private TIAgentStateModifier()
		{			
		}
		
		
		/**
		 * Return the only instance of this class
		 * 
		 * @return The instance of this singleton class
		 */
		public static TIAgentStateModifier getInstance()
		{
			return agentStateModifier;
		}



		private TRCAgentProxy getNextAgent()
		{
			if (inputIterator == null)
				return null;
			
			Object object;
			while (inputIterator.hasNext())
			{
				object = inputIterator.next();
				if (object instanceof TRCAgentProxy)
					return (TRCAgentProxy)object;				
			}
			
			return null;
		}


		public synchronized boolean canAttach()
		{
			return performCollectiveAction(CAN_ATTACH);
		}

		public synchronized boolean canDetach()
		{
			return performCollectiveAction(CAN_DETACH);
		}

		public synchronized boolean canPause()
		{
			return performCollectiveAction(CAN_PAUSE);
		}

		public synchronized boolean canResume()
		{
			return performCollectiveAction(CAN_RESUME);
		}

		public synchronized void attach() throws CoreException
		{
			performCollectiveAction(ATTACH);
		}
		
		public synchronized void detach() throws CoreException
		{
			performCollectiveAction(DETACH);
		}

		public synchronized void pauseMonitoring() throws CoreException
		{
			performCollectiveAction(PAUSE);
		}

		public synchronized void startMonitoring() throws CoreException
		{
			performCollectiveAction(RESUME);
		}
	
		public synchronized void setInput(StructuredSelection input)
		{
			this.input = input;			
		}
		
		private boolean performCollectiveAction(byte operation)
		{
			inputIterator = (input == null ? null : input.iterator());
			boolean status = true;
			TRCAgentProxy currentAgentProxy;
			boolean agentSelected = false;
			
			while ((currentAgentProxy = getNextAgent()) != null)
			{
				agentSelected = true;
				status = status && currentAgentProxy.isActive();
				switch(operation)
				{
					case CAN_ATTACH:
						status = status && !currentAgentProxy.isAttached() && !currentAgentProxy.isMonitored();
						break;
					case CAN_DETACH:
						status = status && currentAgentProxy.isAttached();
						break;
					case CAN_RESUME:
						status = status && !currentAgentProxy.isMonitored();
						break;
					case CAN_PAUSE:
						status = status && currentAgentProxy.isAttached() && currentAgentProxy.isMonitored();
						break;
						
					default:
						doAction(currentAgentProxy, operation);
						break;
				}
			}
			
			return status && agentSelected;
		}


		private void doAction(TRCAgentProxy agentProxy, byte operation)
		{
			agentProxy.setMonitored(operation == RESUME);
			agentProxy.setAttached(!(operation == DETACH));
			agentProxy.setActive(true);
							
			/* Get the TI execution agent that corresponds to the agent proxy */
			TRCNode trcNode = agentProxy.getProcessProxy().getNode();
			IAgentController agentController = null;
			try
			{
				agentController = AgentControllerDelegate.getInstance().getConnection(trcNode.getName(), trcNode.getPort());
			} 
			catch (Exception e)
			{			
				LauncherUtility.openErrorWithDetail(LauncherMessages.LAUNCHER_COMMON_ERROR_TITLE, NLS.bind(TIMessages.ERROR_TI_AC_UNAVAILABLE, String.valueOf(trcNode.getPort())), e);
			}
			ICollector tiAgent = null;
			if (agentController != null)
				tiAgent = TIDelegateHelper.locateTIAgent(agentController, agentProxy.getProcessProxy().getPid()); 
			
			/* If in case the agent was not found, then terminate it and show a message */
			if (tiAgent == null)
			{
				TIUtility.terminateAgentProxy(agentProxy, true);
				
				MessageDialog.openWarning(
						TIPlugin.getDefault().getWorkbench().getActiveWorkbenchWindow().getShell(), 
						LauncherMessages.LAUNCHER_COMMON_WARNING_TITLE, 
						TIMessages.ERROR_TI_AGENT_INACTIVE);
			}
			/* Otherwise perform the requested operation */
			else
			{
				switch(operation)
				{
					case ATTACH:
						if (tiAgent != null)
						{
							final ICollector ctiAgent = tiAgent;
							final TRCAgentProxy cagentProxy = agentProxy;
							final IAgentController cagentController = agentController;
							/* We need to terminate the current agent proxy and create a new one for the attach */
							TIUtility.terminateAgentProxy(agentProxy, false);
							new Job(TIMessages.TASK_CONNECTING_TO_AGENT) {
							protected IStatus run(IProgressMonitor monitor) {
								TIDelegateHelper.getInstance().attachToAgent(
										cagentController, cagentProxy,
										ctiAgent.getProcess(),
										new NullProgressMonitor());
								TIDelegateHelper.getInstance().startPolling(
										ctiAgent);
								return Status.OK_STATUS;
							}
						}.schedule();
						}
					
						break;
					case DETACH:
						TIUtility.closeFileStream(agentProxy);
						LauncherUtility.sendProfileEvent(ProfileEvent.DETACH, agentProxy);
						if (tiAgent != null)
						{
							tiAgent.stop();
							tiAgent.cancel();
							try
							{
								//Wait for a little time before stop monitoring. Some data may still be flushed
								Thread.sleep(500);
								tiAgent.stopMonitoring();
							} 
							catch (InactiveAgentException e)
							{
							}
							catch(Exception e)
							{
							}
						}
						
						break;
					case RESUME:
						LauncherUtility.sendProfileEvent(ProfileEvent.START_MONITOR, agentProxy);
						if (tiAgent != null)
							tiAgent.resume();
						break;
					case PAUSE:
						LauncherUtility.sendProfileEvent(ProfileEvent.STOP_MONITOR, agentProxy);
						if (tiAgent != null)
							tiAgent.pause();
						break;
					
					default:
						break;
				}
				
			}
			
		}		
	}
}
