/**********************************************************************
 * Copyright (c) 2006 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.tptp.platform.jvmti.client.internal.controlproviders;

import java.util.Hashtable;
import java.util.Iterator;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.emf.common.util.EList;
import org.eclipse.hyades.models.hierarchy.TRCAgentProxy;
import org.eclipse.hyades.models.hierarchy.TRCNode;
import org.eclipse.hyades.models.hierarchy.TRCProcessProxy;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.internal.util.TraceMessages;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.osgi.util.NLS;
import org.eclipse.tptp.platform.execution.client.core.IAgentController;
import org.eclipse.tptp.platform.execution.client.core.IProcess;
import org.eclipse.tptp.platform.execution.exceptions.InactiveProcessException;
import org.eclipse.tptp.platform.jvmti.client.internal.TIConstants;
import org.eclipse.tptp.platform.jvmti.client.internal.TIUtility;
import org.eclipse.tptp.platform.jvmti.client.internal.launcher.util.AgentControllerDelegate;
import org.eclipse.tptp.trace.ui.internal.control.provider.application.ControlMessages;
import org.eclipse.tptp.trace.ui.provisional.control.provider.AbstractProcessControlProvider;
import org.eclipse.tptp.trace.ui.provisional.control.provider.IProcessStateModifier;

/**
 * This is the process control provider for the JVMTI agent.
 * 
 * @author Ali Mehregani
 */
public class TIProcessControlProvider extends AbstractProcessControlProvider 
{

	public IProcessStateModifier getProcessStateModifier()
	{
		return TIProcessStateModifier.getInstance();
	}
	
	
	/**
	 * The process state modifier for the TI agent's process.  This is a singleton
	 * class that should be accessed via <code>getInstance</code>.  All
	 * public methods of this class (except for getInstance) are synchronized.
	 *  
	 * @author Ali Mehregani
	 */
	public static class TIProcessStateModifier implements IProcessStateModifier
	{
		/** The instance of this singleton class */
		private static TIProcessStateModifier instance = new TIProcessStateModifier();;
		
		/** The input for the state modifier */
		private StructuredSelection input;
		
		/** The input iterator */
		private Iterator inputIterator;
		
		/** Stores the inputs that have already been processed */
		private Hashtable processedInputs;
		
		
		/**
		 * Limit the visibility of the constructor 
		 */
		private TIProcessStateModifier()
		{			
			processedInputs = new Hashtable();
		}
		
		
		public static TIProcessStateModifier getInstance()
		{
			return instance;
		}
		
		public synchronized void setInput(StructuredSelection input)
		{
			this.input = input;
		}
		
		
		public synchronized boolean canTerminate()
		{
			inputIterator = (input == null ? null : input.iterator());
			processedInputs.clear();
			TRCProcessProxy processProxy;
			boolean canTerminate = true;
			boolean activeProcessOrAgentSelected = false;
			while ((processProxy = getNextProcessProxy()) != null)
			{
				activeProcessOrAgentSelected = true;
				canTerminate = canTerminate && processProxy.isActive();
			}
				
			
			return canTerminate && activeProcessOrAgentSelected;
		}



		public synchronized void terminate() throws CoreException
		{
			inputIterator = (input == null ? null : input.iterator());
			processedInputs.clear();
			TRCProcessProxy currentProcessProxy;
			CoreException error = null;
			long processId = 0;
			while ((currentProcessProxy = getNextProcessProxy()) != null)
			{				
				try
				{
					String title = TraceMessages.TRC_MSGT;	
					processId = currentProcessProxy.getPid();
					String msg = NLS.bind(TraceMessages.TERMINATE_Q, new Object[] {currentProcessProxy.getName(), String.valueOf(processId)});
					
					if(MessageDialog.openQuestion(UIPlugin.getActiveWorkbenchShell(), title, msg))
					{
						TRCNode trcNode = currentProcessProxy.getNode();
						IAgentController ac = AgentControllerDelegate.getInstance().getConnection(trcNode.getName(), trcNode.getPort());
						if(ac != null)
						{
							IProcess process = ac.getProcess(processId);
							
							if(process != null && process.isActive())
								process.kill();		
						}						
					}	
					/* User clicked on "NO" */
					else
					{
						return;
					}
				}
				catch (InactiveProcessException e)
				{
					/* If the process happens to already be inactive, we don't need to do anything */
				}
				catch (Exception e)
				{
					/* Wrap the exception and return */
					error = new CoreException (
							new Status(IStatus.ERROR, 
								UIPlugin.getPluginId(), 
								IStatus.ERROR, 
								NLS.bind(ControlMessages.ERROR_PROCESS_TERMINATE, 
										String.valueOf(processId)),
								e));
				}
				
				terminateProcessProxy(currentProcessProxy);
				
				if (error != null)
				{
					throw error;
				}
			}			
		}
		
		public void terminateProcessProxy (TRCProcessProxy processProxy)
		{
			TIUtility.terminateProcessProxy(processProxy);
			
			/* Walk through the bounded TI agents and terminate them */
			EList agentProxies = processProxy.getAgentProxies();
			for (int i = 0, agentCount = agentProxies.size(); i < agentCount; i++)
			{
				TRCAgentProxy agentProxy = (TRCAgentProxy)agentProxies.get(i);

				if (TIConstants.TI_AGENT_NAME.equals(agentProxy.getName()) && agentProxy.isActive())
				{
					TIUtility.terminateAgentProxy(agentProxy, false);
				}
			}
		}
		
		private TRCProcessProxy getNextProcessProxy()
		{
			if (inputIterator == null)
			{
				return null;
			}
			
			Object currentInput = null;
			while (inputIterator.hasNext())
			{
				Object input = inputIterator.next();
				TRCAgentProxy agentProxy = null;
				if (input instanceof TRCAgentProxy && (agentProxy = ((TRCAgentProxy)input)).isActive())
				{
					currentInput = agentProxy.getProcessProxy();
					
					/* Return the process proxy if it has not yet been processed */
					if (processedInputs.get(currentInput) == null)
					{
						processedInputs.put(currentInput, Boolean.TRUE);
						return (TRCProcessProxy)currentInput;
					}
				}
				else if (input instanceof TRCProcessProxy)
				{
					if (processedInputs.get(input) == null)
					{
						processedInputs.put(input, Boolean.TRUE);
						return (TRCProcessProxy)input;
					}
				}
			}
			
			return null;
		}
		
	}

}
