/**********************************************************************
 * Copyright (c) 2006, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.tptp.platform.jvmti.client.internal.launcher;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.InetAddress;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.debug.core.DebugException;
import org.eclipse.debug.core.DebugPlugin;
import org.eclipse.debug.core.ILaunch;
import org.eclipse.debug.core.IStreamListener;
import org.eclipse.debug.core.model.IStreamMonitor;
import org.eclipse.debug.core.model.IStreamsProxy;
import org.eclipse.debug.core.model.ITerminate;
import org.eclipse.hyades.execution.core.IDataProcessor;
import org.eclipse.hyades.trace.ui.internal.util.ProcessAdapter;
import org.eclipse.tptp.platform.execution.client.core.IConsole;
import org.eclipse.tptp.platform.execution.client.core.IProcess;
import org.eclipse.tptp.platform.execution.client.core.internal.IProcessListener;
import org.eclipse.tptp.platform.execution.exceptions.InactiveAgentException;
import org.eclipse.tptp.platform.execution.exceptions.InactiveProcessException;
import org.eclipse.tptp.platform.execution.exceptions.NotConnectedException;

import com.ibm.icu.text.DateFormat;
import com.ibm.icu.util.ULocale;

/**
 * This class provides an adapter from AC IProcess (TPTP) to the debug IProcess (Debug). 
 * It's primary purpose is to capture the process' output stream and display it in the console 
 * view. This is an almost identical clone of org.eclipse.hyades.trace.ui.internal.util.ProcessAdapter. 
 * These two classes need to eventually be merged to provide one adapter for the Process 
 * (old execution framework) and the IProcess (new execution framework) objects.
 * 
 * @author Ali Mehregani
 */
public class ACProcessToDebugProcessAdapter extends ProcessAdapter
{
	/** The attributes of this debug process */
	private Map attributes = new HashMap();

	/** Indicates whether the process is alive */
	private boolean isAlive = false;

	/** The human readable label of the process */
	private String label;

	/** The AC process instance */
	private IProcess process;

	/** The input/output proxy */
	private IOProxy proxy;

	/** The launch */
	private ILaunch launch;

	/** The launch mode */
	protected int launchMode = 1;


	class ProcessListener implements IProcessListener {
		public void processLaunched(IProcess p) {
			isAlive = true;
		}
		public void processExited(IProcess p) {
			isAlive = false;
		}
	}

	/**
	 * Constructor
	 * 
	 * @param launch The launch
	 * @param process The AC process
	 * @param launchMode The launch mode.
	 * 
	 */
	public ACProcessToDebugProcessAdapter(ILaunch launch, IProcess process, int launchMode, boolean isAlive)
	{
		super(launch, process);
		
		process.addProcessListener(new ProcessListener());

		DateFormat format = DateFormat.getDateTimeInstance(DateFormat.LONG, DateFormat.MEDIUM, ULocale.getDefault());
		final String SPACE = " ";
		final String OPEN_BRACKET = "(";
		final String CLOSE_BRACKET = ")";
		
		this.launchMode = launchMode;
		this.label = process.getExecutable() + SPACE + OPEN_BRACKET + format.format(new Date(System.currentTimeMillis())) + CLOSE_BRACKET; //$NON-NLS-1$
		this.launch = launch;
		this.process = process;
		this.isAlive = isAlive;
	}


	/**
	 * @see IAdaptable#getAdapter()
	 */
	public Object getAdapter(Class adapter)
	{
		if (IDataProcessor.class.equals(adapter))
		{
			return proxy;
		}
		return null;
	}


	/**
	 * @see IProcess#getAttribute()
	 */
	public String getAttribute(String key)
	{
		return (String) attributes.get(key);
	}


	/**
	 * @see IProcess#setAttribute()
	 */
	public void setAttribute(String key, String value)
	{
		attributes.put(key, value);
	}


	/**
	 * @see IProcess#getExitValue()
	 */
	public int getExitValue() throws DebugException
	{
		return 0;
	}


	/**
	 * @see IProcess#getLabel()
	 */
	public String getLabel()
	{
		return label;
	}


	/**
	 * @see IProcess#getLaunch()
	 */
	public ILaunch getLaunch()
	{
		return launch;
	}


	/**
	 * @see IProcess#getStreamsProxy()
	 */
	public IStreamsProxy getStreamsProxy()
	{
		if (proxy == null)
		{
			// BUG 312125: Pass the console encoding on to the IO proxy
			//  so we interpret console output properly
			proxy = new IOProxy(process, launch.getAttribute(DebugPlugin.ATTR_CONSOLE_ENCODING));
		}

		return proxy;
	}


	/**
	 * @see ITerminate#canTerminate()
	 */
	public boolean canTerminate()
	{
		return isAlive && getLaunchMode() != 0;
	}


	/**
	 * @see ITerminate#isTerminated()
	 */
	public boolean isTerminated()
	{
		return !isAlive;
	}


	/**
	 * @see ITerminate#terminate()
	 */
	public void terminate() throws DebugException
	{

		if (canTerminate())
		{
			try
			{
				process.kill();
			} 
			catch (InactiveProcessException e)
			{
				e.printStackTrace();
			}
			catch (NotConnectedException e)
			{
				e.printStackTrace();
			}
			catch (InactiveAgentException e)
			{
				e.printStackTrace();
			}
			isAlive = false;
		}
	}


	/**
	 * Return the launch mode of the launch
	 * 
	 * @return The launch mode
	 */
	public int getLaunchMode()
	{
		return launchMode;
	}


	/**
	 * Set the launch mode of the launch
	 * 
	 * @param launchMode The launch mode
	 */
	public void setLaunchMode(int launchMode)
	{
		this.launchMode = launchMode;
	}
}

/**
 * A proxy to the streams provided by the <code>Process</code>. Needed by the console view.
 */
class IOProxy implements IStreamsProxy, IStreamMonitor, org.eclipse.tptp.platform.execution.client.core.IDataProcessor
{
	/** The listeners of this IO proxy */
	private List listeners;

	/** The buffer of this IO Proxy */
	private StringBuffer buffer;

	/** The AC console of this proxy */
	private IConsole console;

	/** BUG 312125 - Store the console encoding */
	private String encoding;
	
	/**
	 * Creates a new proxy.
	 */
	public IOProxy(IProcess process, String consoleEncoding)
	{
		console = process.getConsole();
		console.setDataProcessor(this);
		encoding = consoleEncoding;
		buffer = new StringBuffer();
		listeners = new ArrayList();
	}


	/**
	 * @see IStreamsProxy#getErrorStreamMonitor()
	 */
	public IStreamMonitor getErrorStreamMonitor()
	{
		return null;
	}


	/**
	 * @see IStreamsProxy#getOutputStreamMonitor()
	 */
	public IStreamMonitor getOutputStreamMonitor()
	{
		return this;
	}


	/**
	 * @see IStreamsProxy#write()
	 */
	public void write(String input) throws IOException
	{
		console.write(input);
	}


	/**
	 * @see IStreamMonitor#addListener()
	 */
	public void addListener(IStreamListener listener)
	{
		synchronized (listeners)
		{
			listeners.add(listener);
		}
	}


	/**
	 * @see IStreamMonitor#removeListener()
	 */
	public void removeListener(IStreamListener listener)
	{
		synchronized (listeners)
		{
			listeners.remove(listener);
		}
	}


	/**
	 * @see IStreamMonitor#getContents()
	 */
	public String getContents()
	{
		return buffer.toString();
	}


	/**
	 * @see IDataProcessor#incommingData()
	 */
	public void incomingData(byte[] buffer, int length, InetAddress peer)
	{
		// BUG 312125: General solution to bug 228936
		//  Convert the buffer into a string using the encoding specified in the
		//  ILaunch. This should be the "right thing" in 99% of the cases and if
		//  it isn't, the user can explicitly select the character encoding in the
		//  launch config (on the "Common" tab)
		String str = null;
		try {
			// BUG 312125: Create the string using the console encoding specified in the ILaunch
			str = new String(buffer, 0, length, this.encoding);
		} catch (UnsupportedEncodingException e) {
			// If that fails, just use the JVM default encoding. 
			str = new String(buffer, 0, length);
		}
		this.buffer.append(str);
		synchronized (listeners)
		{
			Iterator i = listeners.iterator();
			while (i.hasNext())
			{
				((IStreamListener) i.next()).streamAppended(str, this);
			}
		}
	}


	/**
	 * @see IDataProcessor#incommingData()
	 */
	public void incomingData(char[] buffer, int length, InetAddress peer)
	{
		byte[] buf = new String(buffer).getBytes();
		incomingData(buf, buf.length, peer);
	}


	/**
	 * @see IDataProcessor#invalidDataType()
	 */
	public void invalidDataType(byte[] arg0, int arg1, InetAddress arg2)
	{
	}


	/**
	 * @see IDataProcessor#waitingForData()
	 */
	public void waitingForData()
	{
	}
}