 /**********************************************************************
 * Copyright (c) 2006, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.tptp.platform.jvmti.client.internal.launcher;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.hyades.trace.ui.internal.launcher.IProfileLaunchConfigurationConstants;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.osgi.util.NLS;
import org.eclipse.tptp.platform.common.ui.internal.CommonUIConstants;
import org.eclipse.tptp.platform.common.ui.internal.CommonUIPlugin;
import org.eclipse.tptp.platform.execution.client.agent.ICollector;
import org.eclipse.tptp.platform.execution.client.core.IAgentController;
import org.eclipse.tptp.platform.execution.client.core.IProcess;
import org.eclipse.tptp.platform.execution.client.core.internal.IProcessListener;
import org.eclipse.tptp.platform.execution.exceptions.NotConnectedException;
import org.eclipse.tptp.platform.jvmti.client.internal.TIConstants;
import org.eclipse.tptp.platform.jvmti.client.internal.TIMessages;
import org.eclipse.tptp.platform.jvmti.client.internal.TIPlugin;
import org.eclipse.tptp.platform.jvmti.client.internal.launcher.util.AgentControllerDelegate;
import org.eclipse.tptp.trace.ui.provisional.launcher.IAvailabilityTester;


/**
 * Tests the availability of Agent Controller and the JVMTI agent
 * 
 * @author Ali Mehregani
 */
public class AgentAvailibilityTester implements IAvailabilityTester
{
	/* The host name that the agent should be present on */
	private String hostName;
	
	/* The port number that the AC is running on */
	private int port;
	
	/* The previous launch configuration that this tester was reseted with */
	private ILaunchConfiguration configuration;
	
	/* Indicates if the TI agent is available */
	private boolean isAgentAvailable;
	
	/**
	 * Reset this tester using the passed configuration.
	 * 
	 * @param configuration The current configuration that should
	 * be used as the context.
	 */
	public void reset (ILaunchConfiguration configuration)
	{
		if (configuration == null || configuration.equals(this.configuration))
		{
			return;
		}
		
		try
		{
			this.configuration = configuration;
			hostName = configuration.getAttribute(IProfileLaunchConfigurationConstants.ATTR_HOSTNAME, (String)null);
			port = configuration.getAttribute(IProfileLaunchConfigurationConstants.ATTR_PORT, -1);
			
			/* Assume localhost if the hostname is not saved as part of the configuration */
			if (hostName == null)
			{
				hostName = TIConstants.LOCAL_HOST;
			}
			
			/* Determine the port number from the preference page if it's not stored as part of the configuration */
			if (port == -1)
			{
				IPreferenceStore store = CommonUIPlugin.getDefault().getPreferenceStore();					
				port = store.getInt(CommonUIConstants.LOCALHOST_PORT);
			}
		} catch (CoreException e)
		{
			hostName = null;
		}
	}
	
	
	/**
	 * Return a status indicating the availability of the JVMTI
	 * agent.
	 * 
	 * @return Availability status of JVMTI agent
	 */
	public IStatus isServiceAvailable()
	{
		IStatus status;
		isAgentAvailable = false;
		
		/* Display an error if we don't know the host name that we should connect to */
		if (hostName == null)
		{
			return new Status (IStatus.ERROR, TIPlugin.getId(), IStatus.ERROR, TIMessages.ERROR_TI_AC_UNKNOWN_HOST, null);
		}
 		
		AgentControllerDelegate acDelegate = AgentControllerDelegate.getInstance();
		try
		{
			final IAgentController ac = acDelegate.getConnection(hostName, port);		
			final String SPACE = " ";
			final String NON_EXISTING_CLASS = "org.eclipse.tptp.martini.HeapProxy";
			final String CURRENT_DIRECTORY = ".";
			final IProcess process = ac.createProcess();
			
			process.setExecutable(TIDelegateHelper.EXEC_JAVA);
			process.setParameter(SPACE + TIDelegateHelper.TI_JVM_FLAG + SPACE + NON_EXISTING_CLASS);
			process.setLocation(CURRENT_DIRECTORY);
	
			process.addProcessListener(new IProcessListener()
			{
				public void processLaunched(IProcess process) 
				{
					
					try 
					{
						ICollector tiAgent = (ICollector)ac.getAgent(TIConstants.TI_AGENT_NAME, ICollector.class.getName(), process.getProcessId());
						//There may be a delay in the agent registration process.
						//Account for the same - Guru
						int retrycount = 0;
						while((tiAgent == null) && (retrycount <= TIConstants.TOTAL_WAIT))
						{
							tiAgent = (ICollector)ac.getAgent(TIConstants.TI_AGENT_NAME, ICollector.class.getName(), process.getProcessId());
							Thread.sleep(TIConstants.INCREMENTAL_WAIT);
							retrycount++;
						}
						if (tiAgent != null)
							isAgentAvailable = true;
						process.kill();
					} 
					catch (Exception e) 
					{
						/* Doesn't need to be handled */
					} 				
				}

				public void processExited(IProcess process) 
				{
					/* Doesn't need to be implemented */
				}
			});
			
			/* We run it as a thread just in case the call blocks */
			new Thread (new Runnable()
			{
				public void run() 
				{
					try 
					{
						process.launch();
					} 
					catch (Exception e) 
					{
						/* Doesn't need to be handled */
					} 					
				}
			}).start();
			

			int totalWaits = 0;
			while (!isAgentAvailable && totalWaits < TIConstants.TOTAL_WAIT)
			{
				totalWaits++;
				Thread.sleep(TIConstants.INCREMENTAL_WAIT);
			}
			
			if (!isAgentAvailable)
			{
				return new Status (IStatus.ERROR, TIPlugin.getId(), IStatus.ERROR, 
						NLS.bind(TIMessages.ERROR_TI_AGENT_UNAVAILABLE, TIConstants.TI_AGENT_NAME), 
						null);
			}
			
			status = new Status (IStatus.OK, TIPlugin.getId(), IStatus.OK, TIMessages.TI_AGNET_CONNECTION_SUCCESS, null);
				
		} 
		catch (NotConnectedException e)
		{
			/* Indicate that the agent controller is not available */
			status = new Status (IStatus.ERROR, TIPlugin.getId(), IStatus.ERROR, TIMessages.ERROR_TI_AC_NOT_CONNECTED, e);			
		}
		catch (Exception e)
		{
			/* Indicate an unexpected error */
			status = new Status (IStatus.ERROR, TIPlugin.getId(), IStatus.ERROR, TIMessages.ERROR_TI_AC_UNKNOWN, e);
		} 
		
		return status;
	}

}
