/**********************************************************************
 * Copyright (c) 2006, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.tptp.platform.jvmti.client.internal.launcher;

import java.util.Date;
import java.util.Iterator;

import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.hyades.models.hierarchy.HierarchyFactory;
import org.eclipse.hyades.models.hierarchy.TRCAgent;
import org.eclipse.hyades.models.hierarchy.TRCAgentProxy;
import org.eclipse.hyades.models.hierarchy.TRCConfiguration;
import org.eclipse.hyades.models.hierarchy.TRCMonitor;
import org.eclipse.hyades.models.hierarchy.TRCNode;
import org.eclipse.hyades.models.hierarchy.TRCOption;
import org.eclipse.hyades.models.hierarchy.TRCProcessProxy;
import org.eclipse.hyades.models.hierarchy.util.SaveUtil;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.internal.util.PDCoreUtil;
import org.eclipse.hyades.trace.ui.internal.util.TString;
import org.eclipse.tptp.platform.common.ui.internal.CommonUIConstants;
import org.eclipse.tptp.platform.execution.client.agent.IAgent;
import org.eclipse.tptp.platform.execution.client.core.IProcess;
import org.eclipse.tptp.platform.execution.client.core.internal.IProcessListener;
import org.eclipse.tptp.platform.execution.exceptions.InactiveProcessException;

import com.ibm.icu.text.DateFormat;


/**
 * A factory for the hierarchical model entities that is used to display the monitor,
 * node, process, and agent relationship in the profiling monitor view.  This class should
 * eventually be moved to a more generalized plug-in (e.g. <code>org.eclipse.hyades.trace.ui</code>) 
 * and replace the create methods available under <code>PDCoreUtil</code>
 * 
 * @author Ali Mehregani
 */
public class HierarchicalModelFactory 
{
	
	/**
	 * Create and return the process proxy represeting the execution process passed in.
	 * If a corresponding process proxy already exists, then it is returned without creating a new one.
	 *  
	 * @param trcNode The trace node
	 * @param process The execution process
	 * @return The process proxy representing the execution process
	 */
	public static TRCProcessProxy createTRCProcessProxy (TRCNode trcNode, IProcess process)
	{
        try 
        {            
            long processId = process.getProcessId();
            String processUUID = process.getUUID();
            boolean isProcessLaunched = processId > 0 && process.isActive();
            	
            /* If the execution process happens to be valid (i.e. already launched), then walk 
             * through the node to see if the process has already been created. */
            if (isProcessLaunched)
            {
            	TRCProcessProxy currentTRCProcessProxy;
            	/* For every process proxy registered under the node */
                for (Iterator processes = trcNode.getProcessProxies().iterator(); processes.hasNext();)
                {
                	currentTRCProcessProxy = (TRCProcessProxy) processes.next();
                                        
                    if (!currentTRCProcessProxy.eIsProxy() && currentTRCProcessProxy.getPid() == processId)
                    {
                      	/* Return current process proxy if it meets the required criteria */
                   		return currentTRCProcessProxy;
                    }                                                                                                         
                }
            }

            
            /* The process pro doesn't exist.  We need to create it */
            TRCMonitor monitor = trcNode.getMonitor();

            final String UNDER_SCORE = "_";
            StringBuffer processName = new StringBuffer(monitor.getName()).append(UNDER_SCORE).append(trcNode.getName()).append(UNDER_SCORE).append(processId);
            EList pExt = createResource(trcNode, processName, CommonUIConstants.PROCESS_EXT);

            HierarchyFactory factory = UIPlugin.getDefault().getPerftraceFactory();
            
            /* Create the process proxy */
            final TRCProcessProxy trcProcessProxy = factory.createTRCProcessProxy();
           
            /* Ensure the PID and runtime UUID is set on the process proxy if they are valid. */
            if (isProcessLaunched)
            {
            	trcProcessProxy.setPid((int)processId);
            	trcProcessProxy.setRuntimeId(processUUID);
            }
            /* Otherwise we'll need to set them when the process gets launched */
            else
            {
            	process.addProcessListener(new IProcessListener()
            	{
            		private boolean enabled = true;            	
					public void processLaunched(IProcess process)
					{
						if (enabled)
						{
							trcProcessProxy.setRuntimeId(process.getUUID());
			            	try 
			            	{
								trcProcessProxy.setPid((int)process.getProcessId());
							} 
			            	catch (InactiveProcessException e) 
			            	{
			            		/* Shouldn't happen */
								e.printStackTrace();
							}			            	
						}
					}
					public void processExited(IProcess process)
					{
						enabled = false;						
					}
            	});
            }
            	
            /* Set the process proxy's attributes based on the execution process */
            String executionProcessName = process.getName();
            if (executionProcessName != null) 
            {
                trcProcessProxy.setName(executionProcessName);
            }
            else
            {
            	trcProcessProxy.setName(process.getExecutable());
            }
            
                        

            trcProcessProxy.setLocation(process.getLocation());
            trcProcessProxy.setActive(true);
            trcProcessProxy.setLaunchMode(0); /* attach */                
            trcProcessProxy.setNode(trcNode);
            pExt.add(trcProcessProxy);

            return trcProcessProxy;
        } 
        catch (InactiveProcessException e)
		{
        	/* Should never happen */
			e.printStackTrace();
		}

        return null;
	}
	
	
	public static TRCAgentProxy createTRCAgentProxy (TRCProcessProxy processProxy, IAgent agent, String agentType)
	{
		/* Iterate through the process proxy's agents and determine if the agent is already there */
		for (Iterator agentProxies = processProxy.getAgentProxies().iterator(); agentProxies.hasNext();)
		{
			TRCAgentProxy currentAgentProxy = (TRCAgentProxy) agentProxies.next();
			if (!currentAgentProxy.eIsProxy() && currentAgentProxy.getName().equals(agent.getName()) && agentType.equals(currentAgentProxy.getType()) && currentAgentProxy.isActive() && currentAgentProxy.isMonitored())
			{
				return currentAgentProxy;
			}
				
		}

		/* The agent proxy needs to be created */
		final String UNDER_SCORE = "_";
        TRCNode trcNode = processProxy.getNode();
        StringBuffer resourceName = new StringBuffer(trcNode.getMonitor().getName()).append(UNDER_SCORE).append(trcNode.getName()).append(UNDER_SCORE).append(processProxy.getPid()).append(UNDER_SCORE).append(agent.getName());
        EList agExt = createResource (processProxy, resourceName, CommonUIConstants.AGENT_EXT);

        HierarchyFactory factory = UIPlugin.getDefault().getPerftraceFactory();
        TRCAgentProxy trcAgentProxy;
        trcAgentProxy = factory.createTRCAgentProxy();
        
        trcAgentProxy.setName(agent.getName());
        trcAgentProxy.setType(agentType);
        trcAgentProxy.setProcessProxy(processProxy);

        TRCAgent iAgent = factory.createTRCAgent();
        iAgent.setAgentProxy(trcAgentProxy);
        agExt.add(iAgent);
		PDCoreUtil.markModifiedResources(iAgent,true);

        iAgent.setName(agent.getName());
        iAgent.setType(agentType);
        
        return trcAgentProxy;
	}

	
	private static EList createResource(EObject eobject, StringBuffer resourceName, String resourceExtension)
	{
        String timestamp = DateFormat.getDateTimeInstance(DateFormat.FULL, DateFormat.FULL).format(new Date());
        String rID = timestamp;
        
        final String EMPTY_STRING = "";
        final String SPACE = " ";
        final String COLON = ":";
        final String PERIOD = ".";
        final String UNDER_SCORE = "_";
        final String POUND = "#";
                
        rID = TString.change(rID, SPACE, EMPTY_STRING);
        rID = TString.change(rID, COLON, EMPTY_STRING);

        String pPath = eobject.eResource().getURI().toString();
        IPath path = new Path(pPath);
        if (path.segmentCount() > 1) 
        {
            pPath = path.removeLastSegments(1).toString();
        }
        

        String fileName = resourceName.append(UNDER_SCORE).append(rID).toString() + PERIOD + resourceExtension;
        IPath filePath = new Path(pPath).append(fileName);

        URI uri = SaveUtil.createURI(filePath.toString()+POUND).trimFragment();

        Resource agDoc = Resource.Factory.Registry.INSTANCE.getFactory(uri).createResource(uri);
        EList agExt = agDoc.getContents();
        
        UIPlugin.getDefault().getResourceSet().getResources().add(agDoc); // prevents reloading later
        return agExt;
	}
	
	
	
	/**
	 * Creates a configuration with the name passed in and associates it with the trcAgentProxy.
	 * If a configuration with the name configurationName is already associated with trcAgentProxy,
	 * then the same configuration is returned.
	 *  
	 * @param trcAgentProxy The agent proxy
	 * @param configuraitonName The configuration name
	 * @return A configuration associated with the agent proxy
	 */
	public static TRCConfiguration createTRCConfiguration (TRCAgentProxy trcAgentProxy, String configuraitonName)
	{
		/* Check to see if the configuration already exists */
		EList configurations = trcAgentProxy.getConfigurations();
		TRCConfiguration currentConfiguration;
		for (int i = 0, configurationCount = configurations.size(); i < configurationCount; i++)
		{
			currentConfiguration = (TRCConfiguration)configurations.get(i);
			if (configuraitonName.equals(currentConfiguration.getName()))
			{
				return currentConfiguration;
			}
		}
		
		/* We need to create the configuration */
		TRCConfiguration agentProxyConfiguration = UIPlugin.getDefault().getPerftraceFactory().createTRCConfiguration();

		agentProxyConfiguration.setName(configuraitonName);   
		agentProxyConfiguration.setActive(true);
		agentProxyConfiguration.setAgentProxy(trcAgentProxy);
		trcAgentProxy.getConfigurations().add(agentProxyConfiguration);
		
		return agentProxyConfiguration;
	}
	
	
	
	/**
	 * Creates an option and associates it with the configuration that is passed in.
	 *  
	 * @param configuration The configuration
	 * @param optionName The name of the option
	 * @param optionValue The value of the option
	 * 
	 * @return The option created
	 */
	public static TRCOption createOption (TRCConfiguration configuration, String optionName, String optionValue)
	{
        TRCOption trcAgentProxyOption = UIPlugin.getDefault().getPerftraceFactory().createTRCOption();
        trcAgentProxyOption.setKey(optionName);
        trcAgentProxyOption.setValue(optionValue);
        trcAgentProxyOption.setConfiguration(configuration);
        
        return trcAgentProxyOption;
	}
}
