/*******************************************************************************
 * Copyright (c) 2006, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.tptp.platform.jvmti.client.internal.launcher;

import java.util.List;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.debug.core.ILaunch;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.hyades.internal.execution.local.control.Node;
import org.eclipse.hyades.models.hierarchy.TRCAgentProxy;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.internal.launcher.IProfileLaunchConfigurationConstants;
import org.eclipse.hyades.trace.ui.internal.launcher.SelectedAgent;
import org.eclipse.tptp.platform.common.ui.internal.CommonUIConstants;
import org.eclipse.tptp.platform.common.ui.internal.CommonUIPlugin;
import org.eclipse.tptp.platform.common.ui.trace.internal.CommonUITraceConstants;
import org.eclipse.tptp.platform.execution.client.agent.IAgent;
import org.eclipse.tptp.platform.execution.client.agent.ICollector;
import org.eclipse.tptp.platform.execution.client.core.IAgentController;
import org.eclipse.tptp.platform.execution.client.core.IProcess;
import org.eclipse.tptp.platform.execution.client.core.internal.AgentController;
import org.eclipse.tptp.platform.execution.client.core.internal.ConsoleImpl;
import org.eclipse.tptp.platform.execution.client.core.internal.IProcessListener;
import org.eclipse.tptp.platform.execution.exceptions.InactiveAgentException;
import org.eclipse.tptp.platform.execution.exceptions.InactiveProcessException;
import org.eclipse.tptp.platform.execution.util.TPTPDataPath;
import org.eclipse.tptp.platform.jvmti.client.internal.TIConstants;
import org.eclipse.tptp.platform.jvmti.client.internal.TIMessages;
import org.eclipse.tptp.platform.jvmti.client.internal.launcher.util.AgentControllerDelegate;
import org.eclipse.tptp.platform.probekit.launch.launchpad.LaunchPadException;
import org.eclipse.tptp.platform.probekit.launch.launchpad.LaunchPadExtensionSlot;
import org.eclipse.tptp.trace.ui.internal.launcher.core.LauncherUtility;
import org.eclipse.tptp.trace.ui.provisional.launcher.AbstractDCMutualLauncher;
import org.eclipse.tptp.trace.ui.provisional.launcher.AgentSelection;
import org.eclipse.tptp.trace.ui.provisional.launcher.IDataCollectorAgentAttach;
import org.eclipse.tptp.trace.ui.provisional.launcher.IParameterExtractor;
import org.eclipse.tptp.trace.ui.provisional.launcher.ProcessParameters;

/**
 * The PI attach launcher delegate.
 * 
 * @author Ali Mehregani
 */
public class TIAttachLauncherDelegate extends AbstractDCMutualLauncher implements IDataCollectorAgentAttach, IParameterExtractor
{
	/**
	 * The connection to the agent controller
	 */
	private IAgentController agentController;
	
	/**
	 * The launch configuration
	 */
	private ILaunchConfiguration configuration;
	private ILaunch launch;

	final private int ATTACH_POLL_TIME = 10;
	
	public Node createNode()
	{
		/* Doesn't need to be implemented */
		return null;
	}

	public boolean isProcessLauncher()
	{		
		return false;
	}

	/**
	 * @see org.eclipse.tptp.trace.ui.provisional.launcher.IDataCollectorMutualLauncher#preLaunch(org.eclipse.debug.core.ILaunchConfiguration, java.lang.String, org.eclipse.debug.core.ILaunch, org.eclipse.core.runtime.IProgressMonitor)
	 */
	public void preLaunch(ILaunchConfiguration configuration, String mode, ILaunch launch, IProgressMonitor monitor) throws CoreException
	{		
		String hostName = configuration.getAttribute(IProfileLaunchConfigurationConstants.ATTR_HOSTNAME, CommonUITraceConstants.LOCAL_HOST);
		int port = configuration.getAttribute(IProfileLaunchConfigurationConstants.ATTR_PORT, CommonUIPlugin.getDefault().getPreferenceStore().getDefaultInt(CommonUIConstants.LOCALHOST_PORT));
		
		try
		{
			agentController = AgentControllerDelegate.getInstance().getConnection(hostName, port);
			this.configuration = configuration;
			this.launch = launch;
		} 
		catch (Exception e)
		{
			throw LauncherUtility.createCoreException(IStatus.ERROR, e.getLocalizedMessage(), e);
		}
	}

	/**
	 * @see org.eclipse.tptp.trace.ui.provisional.launcher.IDataCollectorAgentAttach#launch(java.util.ArrayList)
	 */
	public void launch(AgentSelection[] selection) throws CoreException
	{		
		for (int i = 0; i < selection.length; i++) {
			Object agentObject = selection[i].getAgent();
			final TRCAgentProxy agentProxy = selection[i].getAgentProxy();
			agentProxy.setType(CommonUIConstants.PROFILE_AGENT_TYPE);
			if (!(agentObject instanceof IAgent) || agentProxy == null)
				continue;
			final IAgent agent = (IAgent) agentObject;
			new Job(TIMessages.TASK_CONNECTING_TO_AGENT) {
				protected IStatus run(IProgressMonitor monitor) {
					try {
						launchAgent(agent, agentProxy);
					} catch (Exception e) {
						UIPlugin.getDefault().log(e);
						return Status.CANCEL_STATUS;
					}
					return Status.OK_STATUS;
				}
			}.schedule();
		}
	}
	
	private void launchAgent(IAgent agent, final TRCAgentProxy agentProxy) throws CoreException, InactiveAgentException, InactiveProcessException
	{
		final ICollector tiAgent = TIDelegateHelper.locateTIAgent(agentController, agent.getProcess().getProcessId());
		if (tiAgent == null)
			return;
		SelectedAgent[] selectedAgents = LauncherUtility.retrieveSelectedAgents(configuration);
		for (int j = 0; j < selectedAgents.length; j++) {
			if (!selectedAgents[j].getPid().equals(String.valueOf(tiAgent.getProcess().getProcessId())))
				return;
			String[] selectedAnalysisTypes = selectedAgents[j].getAnalysisTypes();
			if (selectedAnalysisTypes!=null) {
				for (int k = 0; k < selectedAnalysisTypes.length; k++) {
					String analysisType = selectedAnalysisTypes[k].toString();
					if (analysisType != null && analysisType.equals(TIConstants.PROBE_INSERTION_ID)) {
						LaunchPadExtensionSlot extension = new LaunchPadExtensionSlot();
						try {
							extension.handlePreAgentActiveJVMTI(tiAgent, configuration);
						} catch (LaunchPadException e) {
							// ignore
						}
					}
				}
			}
		}
		tiAgent.startMonitoring(TPTPDataPath.DATA_PATH_TWO_WAY);
		TIDelegateHelper.getInstance().processAttached(tiAgent, launch);
		IProcessListener processListener = new IProcessListener() {
			public void processLaunched(IProcess process) {
				// System.out.println("Attach process launched.");
			}
			public void processExited(IProcess process) {
				// System.out.println("Attach process exited.");
				TIDelegateHelper.getInstance().processDetached(agentProxy, tiAgent);
			}
		};
		agent.getProcess().addProcessListener(processListener);
		TIDelegateHelper.getInstance().sendFilters(configuration, tiAgent);
		TIDelegateHelper.getInstance().sendOptions(configuration, agentProxy, tiAgent);

		ConsoleImpl console = (ConsoleImpl) agent.getProcess().getConsole();
		((AgentController)agentController).addDataListener(console.getDataConnectionID(), console.getDataProcessor());
		
		boolean isEnableDataCollection = configuration == null ? true : configuration.getAttribute(IProfileLaunchConfigurationConstants.ATTR_AUTO_MONITORING, true);
		if (isEnableDataCollection) {
			TIDelegateHelper.getInstance().sendEnableDataCollection(configuration, tiAgent);
		}		
				
		TIDelegateHelper.getInstance().startPolling(tiAgent, ATTACH_POLL_TIME);
		
		TIDelegateHelper.configureDataProcessor(configuration, agentProxy, tiAgent);
		if (!tiAgent.isAgentRunning())
			tiAgent.run();
		
		if (!isEnableDataCollection) {
			TIDelegateHelper.getInstance().setPausedState(agentProxy);
		}		
	}
	
	/**
	 * @see org.eclipse.tptp.trace.ui.provisional.launcher.IDataCollectorMutualLauncher#postLaunch(org.eclipse.core.runtime.IStatus)
	 */
	public void postLaunch(IStatus launchStatus) throws CoreException
	{
		TIAnalysisTypeFiltration.clearCache();
	}

	public ProcessParameters getProcessParameters(ILaunchConfiguration launchConfiguration) {
		ProcessParameters params = new ProcessParameters();
		List<String> args = TIDelegateHelper.getInstance().getVMArgList(launchConfiguration);
		params.addExecutableParameter(args);
		return params;
	}

}
