/**********************************************************************
 * Copyright (c) 2006, 2010 IBM Corporation, Intel Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.tptp.platform.jvmti.client.internal.launcher;

import java.io.IOException;
import java.io.OutputStream;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.variables.VariablesPlugin;
import org.eclipse.debug.core.DebugEvent;
import org.eclipse.debug.core.DebugPlugin;
import org.eclipse.debug.core.ILaunch;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchManager;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.hyades.loaders.trace.TraceUtils;
import org.eclipse.hyades.loaders.util.BinaryLoader;
import org.eclipse.hyades.loaders.util.LoadersUtils;
import org.eclipse.hyades.models.hierarchy.TRCAgentProxy;
import org.eclipse.hyades.models.hierarchy.TRCCollectionMode;
import org.eclipse.hyades.models.hierarchy.TRCConfiguration;
import org.eclipse.hyades.models.hierarchy.TRCMonitor;
import org.eclipse.hyades.models.hierarchy.TRCNode;
import org.eclipse.hyades.models.hierarchy.TRCOption;
import org.eclipse.hyades.models.hierarchy.TRCProcessProxy;
import org.eclipse.hyades.trace.internal.ui.PDPerspective;
import org.eclipse.hyades.trace.internal.ui.PDProjectExplorer;
import org.eclipse.hyades.trace.internal.ui.TraceConstants;
import org.eclipse.hyades.trace.ui.ProfileEvent;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.internal.core.TraceFilterManager;
import org.eclipse.hyades.trace.ui.internal.launcher.IProfileLaunchConfigurationConstants;
import org.eclipse.hyades.trace.ui.internal.launcher.ProfilingSetsManager;
import org.eclipse.hyades.trace.ui.internal.launcher.SelectedAgent;
import org.eclipse.hyades.trace.ui.internal.util.FilterSetElement;
import org.eclipse.hyades.trace.ui.internal.util.FilterTableElement;
import org.eclipse.hyades.trace.ui.internal.util.PDCoreUtil;
import org.eclipse.hyades.trace.ui.internal.util.ProcessMap;
import org.eclipse.hyades.trace.ui.internal.util.TraceMessages;
import org.eclipse.hyades.trace.ui.internal.wizard.TraceWizardMessages;
import org.eclipse.hyades.trace.ui.launcher.IProfilingSetType;
import org.eclipse.hyades.ui.HyadesUI;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.widgets.Display;
import org.eclipse.tptp.platform.common.internal.CommonPlugin;
import org.eclipse.tptp.platform.common.ui.internal.CommonUIConstants;
import org.eclipse.tptp.platform.common.ui.internal.CommonUIPlugin;
import org.eclipse.tptp.platform.common.ui.trace.internal.CommonUITraceConstants;
import org.eclipse.tptp.platform.execution.client.agent.ICollector;
import org.eclipse.tptp.platform.execution.client.core.ConnectionInfo;
import org.eclipse.tptp.platform.execution.client.core.IAgentController;
import org.eclipse.tptp.platform.execution.client.core.IDataProcessor;
import org.eclipse.tptp.platform.execution.client.core.INode;
import org.eclipse.tptp.platform.execution.client.core.IProcess;
import org.eclipse.tptp.platform.execution.client.core.internal.IProcessListener;
import org.eclipse.tptp.platform.execution.exceptions.InactiveAgentException;
import org.eclipse.tptp.platform.execution.exceptions.InactiveProcessException;
import org.eclipse.tptp.platform.execution.exceptions.NoSuchApplicationException;
import org.eclipse.tptp.platform.execution.exceptions.NotConnectedException;
import org.eclipse.tptp.platform.execution.exceptions.ProcessActiveException;
import org.eclipse.tptp.platform.execution.exceptions.TimeoutException;
import org.eclipse.tptp.platform.execution.util.ICommandElement;
import org.eclipse.tptp.platform.execution.util.ICommandHandler;
import org.eclipse.tptp.platform.execution.util.TPTPDataPath;
import org.eclipse.tptp.platform.execution.util.internal.CommandFragment;
import org.eclipse.tptp.platform.execution.util.internal.Constants;
import org.eclipse.tptp.platform.execution.util.internal.TPTPXMLParse;
import org.eclipse.tptp.platform.jvmti.client.internal.ITILaunchConfigurationConstants;
import org.eclipse.tptp.platform.jvmti.client.internal.TIConstants;
import org.eclipse.tptp.platform.jvmti.client.internal.TIMessages;
import org.eclipse.tptp.platform.jvmti.client.internal.TIPlugin;
import org.eclipse.tptp.platform.jvmti.client.internal.TIUtility;
import org.eclipse.tptp.platform.jvmti.client.internal.controlproviders.TIAgentControlProvider.TIAgentStateModifier;
import org.eclipse.tptp.platform.jvmti.client.internal.controlproviders.TIProcessControlProvider.TIProcessStateModifier;
import org.eclipse.tptp.platform.jvmti.client.internal.launcher.util.AgentControllerDelegate;
import org.eclipse.tptp.platform.jvmti.client.internal.launcher.util.RunCommand;
import org.eclipse.tptp.platform.jvmti.client.internal.launcher.util.SetOutputFormatCommand;
import org.eclipse.tptp.platform.jvmti.client.provisional.extension.ICustomAnalysisProvider;
import org.eclipse.tptp.trace.ui.internal.launcher.core.AnalysisType;
import org.eclipse.tptp.trace.ui.internal.launcher.core.AnalysisTypeManager;
import org.eclipse.tptp.trace.ui.internal.launcher.core.DataCollectorManager;
import org.eclipse.tptp.trace.ui.internal.launcher.core.LauncherConstants;
import org.eclipse.tptp.trace.ui.internal.launcher.core.LauncherMessages;
import org.eclipse.tptp.trace.ui.internal.launcher.core.LauncherUtility;
import org.eclipse.tptp.trace.ui.internal.launcher.core.LightConfigurationLoader;
import org.eclipse.tptp.trace.ui.internal.launcher.deleg.application.StatGeneratorDataProcessor;
import org.eclipse.tptp.trace.ui.provisional.control.provider.IAgentStateModifier;
import org.eclipse.tptp.trace.ui.provisional.launcher.DataCollectionEngineAttribute;
import org.eclipse.tptp.trace.ui.provisional.launcher.IConfiguration;
import org.eclipse.tptp.trace.ui.provisional.launcher.IConfigurationPage;
import org.eclipse.tptp.trace.ui.provisional.launcher.ILightConfiguration;
import org.eclipse.tptp.trace.ui.provisional.launcher.ProcessParameters;
import org.eclipse.ui.IViewPart;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.console.ConsolePlugin;
import org.eclipse.ui.console.IConsole;
import org.eclipse.ui.console.IConsoleConstants;
import org.eclipse.ui.console.IConsoleManager;
import org.eclipse.ui.console.IConsoleView;
/**
 * This delegate helper is shared between all launch configuration types (e.g. local Java application, external Java application, Eclipse application, and etc...). The logic behind the launch is the same. The only difference is the process parameters. The following sequence of tasks starts the profiling session:
 * <ul>
 * <li> Connect to the agent controller </li>
 * <li> Create the process to be launched </li>
 * <li> Register the necessary process listeners </li>
 * <li> Launch the process </li>
 * <li> Get the associated agent that results from the launch </li>
 * <li> Get the data channel of the agent </li>
 * <li> Feed the data to the loader </li>
 * </ul>
 * 
 * This is a singleton class whose instance is locked when the <code>launch</code> method is invoked.
 * 
 * @author Ali Mehregani
 */
public class TIDelegateHelper
{
	/** Agent commands */
	private static final String CMD_APPLY_FILTERS = "applyFilters";
	private static final String CMD_APPLY_OPTIONS = "applyOptions";
	private static final String CMD_COLLECT_DATA = "CollectData";
	private static final String CMD_SET_OUTPUT_FORMAT = "setOutputFormat";
	
	/** Elements required to construct agent commands */
	private static final String ELEMENT_FILTER = "Filter";
	private static final String ELEMENT_OPTION = "option";
	private static final String ELEMENT_NAME = "name";
	private static final String ELEMENT_VALUE = "value";
	
	/** Attributes required to construct agent commands */
	private static final String ATTRIBUTE_IID = "iid";

	
	/** The executable that's run */
	protected static final String EXEC_JAVA = "java.exe";
	
	/** The TI profiler variables*/
	private static final String TI_VARIABLE_OUTPUT_FORMAT="TPTP_OUTPUT_FORMAT"; // Can be XML or BINARY
	
	/** The AC/JVM variables */
	private static final String AC_VARIALBE_CLASSPATH = "CLASSPATH";
	
	/** The JVM argument used to launch a Java application with the JVMTI Call Graph Profiler flag */
	protected static final String TI_CGPROF_JVM_FLAG = "-agentlib:JPIBootLoader=JPIAgent:server=controlled;CGProf:execdetails=false "; //bug 299591
	
	/** The JVM argument used to launch a Java application with the JVMTI Call Graph Profiler flag */
	protected static final String TI_EXEC_CGPROF_JVM_FLAG = "-agentlib:JPIBootLoader=JPIAgent:server=controlled;CGProf:execdetails=true ";
	
	/** The JVM argument used to launch a Java application with the JVMTI HEAP flag */
	protected static final String TI_HEAP_JVM_FLAG = "-agentlib:JPIBootLoader=JPIAgent:server=controlled;HeapProf";
	
	/** The JVM argument used to launch a Java application with the JVMTI HEAP flag */
	protected static final String TI_HEAP_ALLOC_JVM_FLAG = "-agentlib:JPIBootLoader=JPIAgent:server=controlled;HeapProf:allocsites=true";
	
	/** The JVM argument used to launch a Java application with the JVMTI THREAD flag */
	protected static final String TI_THRD_JVM_FLAG = "-agentlib:JPIBootLoader=JPIAgent:server=controlled;ThreadProf";
	
	/** The JVM argument used to launch a Java application with the JVMTI THREAD flag */
	protected static final String TI_EXEC_THRD_JVM_FLAG = "-agentlib:JPIBootLoader=JPIAgent:server=controlled;ThreadProf:contanalysis=true";
	
	/** The JVM argument used to launch a Java application with the JVMTI PROBE flag */
	protected static final String TI_PROBE_JVM_FLAG = "-agentlib:JPIBootLoader=JPIAgent:server=controlled;ProbekitAgent:ext-pk-BCILibraryName=BCIEngProbe";
	
	/** The JVM argument used to launch a Java application with the JVMTI flag */
	protected static String TI_JVM_FLAG = TI_CGPROF_JVM_FLAG;
	
	/** A no-op command handler */
	private static final NOOPCommandHandler NOOP_COMMAND_HANDLER = new NOOPCommandHandler();
	
	/** The instance of this singleton class */
	private static TIDelegateHelper instance = new TIDelegateHelper();

	private static int POLL_TIME = 10;
	private boolean isCGProf = true;
	private boolean isAggProf = false;
	private boolean isProcessActive = false;
	private static final int RETRY_THRESHOLD = 60;
	private static final int WAITING_TIME = 100;
	private int errCount = 0;
	private Object lock = new Object();
	private long errorCode = 0;
	private String errorMsg;
	
	private org.eclipse.tptp.platform.execution.client.agent.internal.AgentState _agentState = null;
	
	/**
	 * Provider for the customAnalysisProvider extension point.
	 */
	private ICustomAnalysisProvider analysisProvider = null;
	
	/**
	 * Restrict the visibility of the constructor
	 */
	private TIDelegateHelper()
	{
	}


	/**
	 * Returns the instance of this singleton class
	 */
	public static TIDelegateHelper getInstance()
	{
		return instance;
	}

	/**
	 * This method resets configuration pages.
	 * 
	 * @param launchConfiguration
	 */
	private void resetConfigurationPages(ILaunchConfiguration launchConfiguration) {
		List analysisTypes = getSelectedTIAnalysisTypes(launchConfiguration);
		int analysisTypesCount = analysisTypes == null ? 0 : analysisTypes.size();
		for (int i = 0; i < analysisTypesCount; i++) {
			Object currentAnalysisTypeObject = analysisTypes.get(i);
			LightConfigurationLoader confLoader = null;
			if (currentAnalysisTypeObject instanceof AnalysisType
					&& (confLoader = ((AnalysisType) currentAnalysisTypeObject).getConfigurationLoader()) != null) {
				ILightConfiguration analysisTypeConfiguraiton = confLoader.getConfigurationClass();
				if (confLoader.isEditable()) {
					IConfigurationPage[] configurationPages = ((IConfiguration) analysisTypeConfiguraiton).getConfigurationPages();
					for (int j = 0; j < configurationPages.length; j++)
						configurationPages[j].reset(launchConfiguration);
				}
			}
		}
	}

	/**
	 * Launches a profiling session
	 * 
	 * @param launcher The launcher
	 * @param launchConfiguration The launch configuration
	 * @param mode The mode of the launch
	 * @param launch The launch
	 * @param monitor The progress monitor
	 * 
	 * @throws CoreException In case of an error
	 */
	public synchronized void launch(ILauncherParameterProvider launcher, ILaunchConfiguration launchConfiguration, String mode, ILaunch launch, IProgressMonitor monitor) throws CoreException
	{
		resetConfigurationPages(launchConfiguration);
		java.util.Map mp = launchConfiguration.getAttributes();
		int mapsize = mp.size();
		java.util.Iterator keyValuePairs = mp.entrySet().iterator();
		isCGProf = true;
		for (int i = 0; i < mapsize; i++) {
		  java.util.Map.Entry entry = (java.util.Map.Entry) keyValuePairs.next();
		  Object key = entry.getKey();
		  Object value = entry.getValue();
		  addOption(new StringBuffer(), key.toString(), value.toString());
		  
		  //defaults to call graph profiler 
		  TI_JVM_FLAG = TI_CGPROF_JVM_FLAG;
		  
		  //You have concurrent profilers - so break out once you hit this attribute.
		  if(("org.eclipse.hyades.trace.ui.ATTR_DATA_COLLECTOR").equals(key.toString()))
		  {
			// Handling a heap analysis request
			if(value.toString().indexOf("heap") != -1)
			{
				// Create a flag to determine if an attribute has been added for data collection.
				boolean hasAttributes = false;
				
				// Check if the Agent should run with Object Allocation
				if(checkAllocOptions(launchConfiguration, ITILaunchConfigurationConstants.ATTR_EXEC_OBJ_ALLOC_SITE))
				{
					// Run with Object Allocation enabled
					TI_JVM_FLAG = TI_HEAP_ALLOC_JVM_FLAG;
					hasAttributes = true;
				}
				else
				{
					// Run without Object Allocation
					TI_JVM_FLAG = TI_HEAP_JVM_FLAG;
				}
				// Sets the JVM arguments, depending on if Object Allocation (allocSites) is enabled, a different String is used
				ICustomAnalysisProvider[] providers = TIPlugin.getProviders();
				StringBuilder jvmArgs = new StringBuilder();
				StringBuilder tiJVMFlags = new StringBuilder(TI_JVM_FLAG);
				
				if(providers != null) {
					for(int provider = 0; provider < providers.length; provider++) {
						analysisProvider = providers[provider];
						// Checks if the custom option should be appended to the JVM flag
						if(checkAllocOptions(launchConfiguration, analysisProvider.getAllocOptionToCheck())){
							tiJVMFlags.append(analysisProvider.updateJVMArg(hasAttributes));
							// Update so that attributes are appended
							if(!hasAttributes)
								hasAttributes = true;
							
							// Store any additional JVM arguments to add to the command.
							List<String> extraJVMArgs = analysisProvider.addExtraJVMArgs();
							if(extraJVMArgs == null || extraJVMArgs.size() == 0) {
								continue;
							}
							for(String extraJVMArg : extraJVMArgs) {
								extraJVMArg = extraJVMArg.trim();
								if(extraJVMArg != null && extraJVMArg.length() != 0) {
									jvmArgs.append(extraJVMArg);
									// Add a trailing space between the arguments or command.
									jvmArgs.append(" ");
								}
							}
						}	
					}
				}
				
				// Reset analysisProvider to ensure old information is not used
				analysisProvider = null;
				// Restores the trailing space when an attribute is set
				if(hasAttributes)
					tiJVMFlags.append(" ");
				
				// Update the JVMTI options and append any additional JVM arguments to the command.
				TI_JVM_FLAG = tiJVMFlags.toString() + jvmArgs.toString();
				isCGProf = false;
				break;
			}
			if(value.toString().indexOf("thread") != -1) {
				TI_JVM_FLAG = launchConfiguration.getAttribute(ITILaunchConfigurationConstants.ATTR_EXEC_CONTENTION_ANALYSIS, false) ?
						TI_EXEC_THRD_JVM_FLAG : TI_THRD_JVM_FLAG;

				isCGProf = false;
				break;
			}
			// added for enhancement 141540
			// the attribute "executableArgs" is retrieved from the
			// attributes stored when the probes to launch are selected
			// Liz Dancy
			if(value.toString().indexOf("probe") != -1)
			{
				TI_JVM_FLAG = TI_PROBE_JVM_FLAG;
				String VMArgs = launchConfiguration.getAttribute("executab1leArgs", "");
				//255813 need to deal with the Remote Java Application use case
				//where the specified classpath needs to be added to the bootclasspath
				String classpath = launchConfiguration.getAttribute(IProfileLaunchConfigurationConstants.ATTR_CLASSPATH, (String)null);
				if(classpath != null) {
					StringBuffer bootClassPath = new StringBuffer();
					bootClassPath.append("-Xbootclasspath/p:"); //$NON-NLS-1$
					/* If this is a Windows operating system, add quotes to the beginning and end of
					 * the value passed to bootclasspath.  This is done in case there are spaces in the
					 * path(s) specified. */
					boolean isWindows = false;
					if (System.getProperty("os.name").toLowerCase().indexOf("windows") != -1) //$NON-NLS-1$ //$NON-NLS-2$
					{
						bootClassPath.append("\""); //$NON-NLS-1$
						isWindows = true;
					}
					bootClassPath.append(classpath);
					if (isWindows)
					{
						bootClassPath.append("\""); //$NON-NLS-1$
					}
					VMArgs += " " + bootClassPath.toString(); //$NON-NLS-1$
				}
				TI_JVM_FLAG = VMArgs + " " + TI_JVM_FLAG ; //$NON-NLS-1$
				isCGProf = false;
				break;
			}
		  }
		}
		//If it is not Heap or thread check for the CGProf Alloc execdetails true
		if(isCGProf)
		{
			if(checkAllocOptions(launchConfiguration, ITILaunchConfigurationConstants.ATTR_EXEC_FLOW))
			{
				TI_JVM_FLAG = TI_EXEC_CGPROF_JVM_FLAG;
			}
		}
		
		launcher.preLaunch(launchConfiguration, mode, launch, monitor);
		
		monitor.setTaskName(TIMessages.TASK_CONNECTING_TO_AC);
		IAgentController agentController = connectToAgentController(launcher);
		monitor.worked(2);
		monitor.setTaskName(TIMessages.TASK_LAUNCHING_PROCESS);
		launchProcess(agentController, launcher, launchConfiguration, launch, 1, monitor);		
	}

	private static void switchPerspective() {
        Display.getDefault().syncExec(new Runnable() {
        	public void run() {
        		try{
    		 		IWorkbenchWindow window = PlatformUI.getWorkbench().getActiveWorkbenchWindow();
    		 		boolean shouldSwitch = CommonUIPlugin.getDefault().openPerspectiveSwitchDialog(
    						window.getShell(), PDPerspective.ID_TRACE_PERSPECTIVE,
    						CommonUIPlugin.getDefault().getPreferenceStore(), 
    						HyadesUI.SWITCH_TO_PERSPECTIVE_KEY, 
    						TraceWizardMessages.CONFIRM_PSWITCH);
    			  	if(shouldSwitch) {
    			  		UIPlugin.getActivePage().showView(PDPerspective.ID_PD_NAVIGATOR_VIEW);
    			  	}
    		    	if (PDPerspective.ID_TRACE_PERSPECTIVE.equals(window.getActivePage().getPerspective().getId())) {
   						PDProjectExplorer nav = (PDProjectExplorer)UIPlugin.getActivePage().showView(PDPerspective.ID_PD_NAVIGATOR_VIEW);
   						nav.getViewer().refresh();
    		        }
        		} catch (Exception exc) {
        			CommonPlugin.logError(exc);
        		}
        	}
        });

	}
	
	/**
	 * @see org.eclipse.tptp.platform.jvmti.client.internal.launcher.TIDelegateHelper.launch(ILauncherParameterProvider, ILaunchConfiguration, String, ILaunch, IProgressMonitor)
	 **/
	public String getVMArg(ILaunchConfiguration launchConfiguration) {
		List<String> args = this.getVMArgList(launchConfiguration);
		if(args != null) {
			StringBuilder result = new StringBuilder();
			for(String arg : args) {
				if(arg == null || arg.trim().length() == 0)
					continue;
				result.append(arg.trim() + " ");
			}
			return result.toString();
		}
		return null;
	}
	
	/**
	 * @see org.eclipse.tptp.platform.jvmti.client.internal.launcher.TIDelegateHelper.launch(ILauncherParameterProvider, ILaunchConfiguration, String, ILaunch, IProgressMonitor)
	 **/
	public List<String> getVMArgList(ILaunchConfiguration launchConfiguration) {
		try {
			String collectors = launchConfiguration.getAttribute(IProfileLaunchConfigurationConstants.ATTR_PROFILING_COLLECTOR_AND_ANALYSIS,"");
			ArrayList<String> args = new ArrayList<String>();
			//defaults to call graph profiler 
			 String JVM_FLAG = TI_CGPROF_JVM_FLAG;
			  
			if(collectors.indexOf("heap") != -1)
			{
				// Create a flag to determine if an attribute has been added for data collection.
				boolean hasAttributes = false;
				// Check if the Agent should run with Object Allocation
				if(checkAllocOptions(launchConfiguration, ITILaunchConfigurationConstants.ATTR_EXEC_OBJ_ALLOC_SITE))
				{
					// Run with Object Allocation enabled
					JVM_FLAG = TI_HEAP_ALLOC_JVM_FLAG;
					hasAttributes = true;
				}
				else
				{
					// Run without Object Allocation
					JVM_FLAG = TI_HEAP_JVM_FLAG;
				}

				// Sets the JVM arguments, depending on if Object Allocation (allocSites) is enabled, a different String is used
				ICustomAnalysisProvider[] providers = TIPlugin.getProviders();
				StringBuilder tiJVMFlags = new StringBuilder(JVM_FLAG);
				
				if(providers != null) {
					for(int provider = 0; provider < providers.length; provider++) {
						analysisProvider = providers[provider];
						// Checks if the custom option should be appended to the JVM flag
						if(checkAllocOptions(launchConfiguration, analysisProvider.getAllocOptionToCheck())){
							tiJVMFlags.append(analysisProvider.updateJVMArg(hasAttributes));
							// Update so that attributes are appended
							if(!hasAttributes)
								hasAttributes = true;
							
							// Store any additional JVM arguments to add to the command.
							List<String> extraJVMArgs = analysisProvider.addExtraJVMArgs();
							if(extraJVMArgs == null || extraJVMArgs.size() == 0)
								continue;
							for(String extraJVMArg : extraJVMArgs) {
								extraJVMArg = extraJVMArg.trim();
								if(extraJVMArg != null && extraJVMArg.length() != 0) {
									args.add(extraJVMArg);
								}
							}
						}	
					}
				}
				
				// Reset analysisProvider to ensure old information is not used
				analysisProvider = null;
				// Restore the trailing space when an attribute is set
				if(hasAttributes)
					tiJVMFlags.append(" ");

				// Update the JVMTI options.
				JVM_FLAG = tiJVMFlags.toString();
				isCGProf = false;
			}
			if(collectors.indexOf("thread") != -1) {
				JVM_FLAG = launchConfiguration.getAttribute(ITILaunchConfigurationConstants.ATTR_EXEC_CONTENTION_ANALYSIS, false) ?
						TI_EXEC_THRD_JVM_FLAG : TI_THRD_JVM_FLAG;
				
				isCGProf = false;
			}
			if(collectors.indexOf("probe") != -1)
			{
				JVM_FLAG = TI_PROBE_JVM_FLAG;
				isCGProf = false;
			}

			//If it is not Heap or thread check for the CGProf Alloc execdetails true
			if(isCGProf)
			{
				if(checkAllocOptions(launchConfiguration, ITILaunchConfigurationConstants.ATTR_EXEC_FLOW))
				{
					JVM_FLAG = TI_EXEC_CGPROF_JVM_FLAG;
				}
			}

			// Insert the JVM argument at the beginning.
			args.add(0, JVM_FLAG);
			return args;
		}
		catch (CoreException e) {
			e.printStackTrace();
		}
		return null;
	}
	private IAgentController connectToAgentController(ILauncherParameterProvider launcher) throws CoreException
	{
		try
		{
			String hostname = launcher.getHostName();
			int port = launcher.getPortNumber();			
			return AgentControllerDelegate.getInstance().getConnection(hostname, port);
		} catch (Exception e)
		{
			if (e instanceof CoreException)
				throw (CoreException)e;
			
			/* Wrap the exception in a core exception */
			throw LauncherUtility.createCoreException(IStatus.ERROR, NLS.bind(TIMessages.ERROR_TI_AC_UNAVAILABLE, String.valueOf(launcher.getPortNumber())), e);
		}
		
		
	}

	private static boolean isLocal(String hostName)
	{
		hostName = AgentControllerDelegate.getUnifiedHostName(hostName);
		String localHostName = "";
		try {
			localHostName = InetAddress.getLocalHost().getHostName();
		} catch (UnknownHostException e) {
			e.printStackTrace();
		}
		return hostName.equalsIgnoreCase(localHostName);
	}
	
	private String parseArguments(String args) {
		if (args == null)
			return null;

		int ch= -1;
		int index= 0;
		int argLength = args.length();
		String argsRet = "";
		
		while (argLength > 0) {
			ch = args.charAt(index++);
			if (Character.isWhitespace((char)ch)) {
				argsRet += TIUtility.SINGLE_SPACE;
			} else {
				argsRet += String.valueOf((char)ch); 
			}
			argLength --;
		}

		return argsRet;		
	}	
	
	private void launchProcess(final IAgentController agentController, ILauncherParameterProvider launcher, final ILaunchConfiguration launchConfiguration, ILaunch launch, int launchMode, final IProgressMonitor monitor) throws CoreException
	{
		try
		{
			IProcess process = agentController.createProcess();
			ProcessParameters processParameters = launcher.getProcessParameters(launchConfiguration);
	
			/* Configure the process to be launched */			
			String programParameter = processParameters.getApplicationParametersStr();
			String processParameter = processParameters.getExecutableParametersStr() + TIUtility.SINGLE_SPACE + processParameters.getApplicationName() + (programParameter.length() > 0 ? TIUtility.SINGLE_SPACE + programParameter.trim() : TIUtility.EMPTY_STRING);			
			processParameter = parseArguments(processParameter).trim();	
			
			process.setExecutable(EXEC_JAVA);			
			process.setParameter(processParameter);

			setEnvironmentVariables(process, launcher, launchConfiguration);
			
			String hostName = launcher.getHostName();
			if (isLocal(hostName))
				process.setLocation(launcher.getLocation());			
			else
				process.setLocation(".");
			process.setName(processParameters.getApplicationName());

	        final org.eclipse.debug.core.model.IProcess adapter = new ACProcessToDebugProcessAdapter (launch, process, launchMode, true);	       
	        UIPlugin.getDefault().registerLaunchProcess(adapter);
	        IProcessListener processListener = new IProcessListener() 
	        {
	        	private TRCAgentProxy tiTRCAgentProxy;
				private ICollector agent = null;

				public synchronized void processLaunched(IProcess process)
				{					
					final IProcess cproc = process;	
					/* Once the agent is launched, then attach to the agent */
					monitor.setTaskName(TIMessages.TASK_CONNECTING_TO_AGENT);
					isProcessActive = true;
					tiTRCAgentProxy = attachToAgent(agentController, process, launchConfiguration, monitor);
					try {
					    agent = locateTIAgent(agentController, process.getProcessId());
					} catch (InactiveProcessException e) {
					    // just ignore
					}
					if (agent == null) {
						//Need to throw an exception here
						return;
					}
					LoadersUtils.registerAgentInstance(tiTRCAgentProxy, agent);
					Display.getDefault().asyncExec(new Runnable() {
						public void run() {
							IConsole console = getConsole(cproc);
							showConsole(console);
						}
					});
					/* We need to add the mapping between the TRCProcessProxy (TPTP) and the IProcess (Eclipse) */
					if (tiTRCAgentProxy != null) {
						ProcessMap.put(tiTRCAgentProxy.getProcessProxy(), adapter);
					}
					monitor.worked(1);
				}

				public synchronized void processExited(IProcess process)
				{
					
					UIPlugin.getDefault().deregisterLaunchProcess(adapter);
					
					// Bug 331801 - Give the process a few moments to complete before we inform Eclipse
					// to close the console window streams
					new Thread() {
						public void run() {
							try { sleep(5000); } catch (InterruptedException e) { }
							DebugPlugin.getDefault().fireDebugEventSet(
									new DebugEvent[] { new DebugEvent(adapter, DebugEvent.TERMINATE)	});
							
						}
					}.start();
					
		        	// We need to terminate the agent proxy
					cleanupAgent(tiTRCAgentProxy, agent);
				}
	        };
	        process.addProcessListener(processListener);
			process.launch();
			monitor.worked(1);			
		}
		catch (ProcessActiveException e) 
		{
			handleError(monitor, TIMessages.ERROR_TI_PROCESS_LAUNCH_ACTIVE, e);
		} 
		catch (NotConnectedException e) 
		{
			handleError(monitor, NLS.bind(TIMessages.ERROR_TI_AC_UNAVAILABLE, String.valueOf(agentController.getConnectionInfo().getPort())), e);
		} 
		catch (NoSuchApplicationException e) 
		{
			handleError(monitor, TIMessages.ERROR_TI_PROCESS_LAUNCH_NO_APP, e);
		} 
		catch (InactiveAgentException e) 
		{
			handleError(monitor, TIMessages.ERROR_TI_PROCESS_LAUNCH_INACTIVE_AG, e);
		} 
		catch (TimeoutException e) 
		{
			handleError(monitor, TIMessages.ERROR_TI_PROCESS_LAUNCH_TIMEOUT, e);
		}
	}

	private void cleanupAgent(TRCAgentProxy tiTRCAgentProxy, ICollector agent) {
		// We need to terminate the agent proxy
		if (tiTRCAgentProxy == null) {
			return;
		}
		
		TraceUtils.removeSnapshot(tiTRCAgentProxy);
		
		LoadersUtils.deregisterAgentInstance(tiTRCAgentProxy);
		TIProcessStateModifier processStateModifier = TIProcessStateModifier.getInstance();
		processStateModifier.terminateProcessProxy(tiTRCAgentProxy.getProcessProxy());
		isProcessActive = false;
		// wait some time to detect that data is collected
		for (int i = 1; i <= 100; i++) {
			if (tiTRCAgentProxy.getStopTime() >
				tiTRCAgentProxy.getStartTime()) {
				break;
			}
			//System.out.println("@@@ waiting for agent to stop..." + i);
			try {
				Thread.sleep(100);
			} catch (InterruptedException e) {
				break;
			}
		}
		// cleanup internal resources
		if (agent != null) {
			// wait some time for collecting remaining data
			try {
				//System.out.println("@@@ agent.stopMonitoring");
				// stopMonitoring: cleanup threads								
				ACDataProcessorAdapter dataProcessor = (ACDataProcessorAdapter) LoadersUtils.locateDataProcessor(tiTRCAgentProxy);
				if (dataProcessor != null)
					dataProcessor.cleanUp();
				agent.stopMonitoring();								
			} catch (InactiveAgentException e) {
				// ignore
			}
		}
	}
	
	public synchronized void processAttached(ICollector agent, ILaunch launch) 
	{
		isAggProf = true;
		isProcessActive = true;
	}

	public synchronized void processDetached(TRCAgentProxy tiTRCAgentProxy, ICollector agent) 
	{
		isProcessActive = false;
		cleanupAgent(tiTRCAgentProxy, agent);
	}

	private void setEnvironmentVariables(IProcess process,
			ILauncherParameterProvider launcher,
			ILaunchConfiguration launchConfiguration) throws CoreException, ProcessActiveException{
		Map configEnv = launchConfiguration.getAttribute(ILaunchManager.ATTR_ENVIRONMENT_VARIABLES, (Map) null);
		if (configEnv != null) {
			Iterator iter = configEnv.entrySet().iterator();
			Map.Entry entry = null;
			String key = null;
			String value = null;
			while (iter.hasNext()) {
				entry = (Map.Entry) iter.next();
				key = (String) entry.getKey();
				value = (String) entry.getValue();
				if (value != null) {
					value = VariablesPlugin.getDefault().getStringVariableManager().performStringSubstitution(value);
				}
				process.prependEnvironmentVariable(key, value);
			}
		}
		process.prependEnvironmentVariable(AC_VARIALBE_CLASSPATH, launcher.getClassPath());
	}

	private void handleError(IProgressMonitor monitor, String message, Exception e) throws CoreException
	{
		monitor.setCanceled(true);
		
		/* Wrap the error in a core exception and throw it to the caller */
		throw LauncherUtility.createCoreException(IStatus.ERROR, message);
	}


	/**
	 * Attach to the agent and create the necessary model entities to represent the agent.  The following
	 * tasks need to be accomplished in this order:
	 * 
	 * <ul>
	 * 	<li> Get the agent </li>
	 * 	<li> Start monitoring it </li>
	 *  <li> Create the model entities </li>
	 *  <li> Send the filters and options </li>
	 *  <li> Start the data collection process </li>
	 * </ul>
	 *  	
	 * @param agentController The agent controller
	 * @param process The process launched
	 */
	private TRCAgentProxy attachToAgent(final IAgentController agentController, IProcess process, ILaunchConfiguration launchConfiguration, final IProgressMonitor monitor)
	{		
		return attachToAgent(agentController, null, launchConfiguration, process, monitor);
	}

	
	/**
	 * Attaches to the JVMTI agent based on a previous attach.  This method is used to reattach to the 
	 * detached agent "agentProxy".  The configuration of "agentProxy" is used to re-attach to the JVMTI 
	 * agent.
	 * 
	 * @param agentController The agent controller connection
	 * @param agentProxy The agent proxy that the user is re-attaching to
	 * @param process The execution process 
	 * @param monitor The monitor.
	 * 
	 * @return The agent proxy that is created as a result of a re-attach.  This value can be null if 
	 * an unexpected error occurrs.
	 */
	public TRCAgentProxy attachToAgent (IAgentController agentController, TRCAgentProxy agentProxy, IProcess process, final IProgressMonitor monitor)
	{
		return attachToAgent(agentController, agentProxy, null, process, monitor);
	}
	

	private TRCAgentProxy attachToAgent(IAgentController agentController, TRCAgentProxy agentProxy, ILaunchConfiguration launchConfiguration, IProcess process, IProgressMonitor monitor)
	{
		try
		{	
			errCount = 0;
 			ICollector tiAgent = null;

			tiAgent = locateTIAgent(agentController, process.getProcessId());
			/* The agent wasn't found - show an error and return */
			while (tiAgent == null && errCount < RETRY_THRESHOLD)
 			{
				Thread.sleep(WAITING_TIME);
 				tiAgent = locateTIAgent(agentController, process.getProcessId());
				errCount++;
			}
			
			if (tiAgent == null)
			{
				handleErrorCondition(LauncherMessages.LAUNCHER_COMMON_ERROR_TITLE, NLS.bind(TIMessages.ERROR_TI_AGENT_UNAVAILABLE, TIConstants.TI_AGENT_NAME), null, process, monitor);
				return null;
			}
			
			if (launchConfiguration != null)
			{
				boolean addProbeListener = probesSelected(launchConfiguration);						
				if (addProbeListener)
				{
					ProbeControlListener probeController = new ProbeControlListener(launchConfiguration);
					tiAgent.addAgentListener(probeController);
				}
			}			
						
			/* Start Monitoring */
			tiAgent.startMonitoring(TPTPDataPath.DATA_PATH_TWO_WAY);
			
			
			/* Create the model entities */
			TRCAgentProxy tiAgentProxy = agentProxy == null ? createModelEntities(launchConfiguration, agentController, process, tiAgent) : createModelEntities(agentProxy, agentController, process, tiAgent);
					
			
			final TRCAgentProxy trcAgentProxy = tiAgentProxy; 			
			if (launchConfiguration != null)
			{
				LauncherUtility.addLaunchConfigId(trcAgentProxy.getProcessProxy(), launchConfiguration.getType().getIdentifier());
			}
			if(tiAgent != null)
			{
				_agentState = org.eclipse.tptp.platform.execution.client.agent.internal.AgentStatePool.getAgentInfo(agentController.getConnectionInfo(), process.getProcessId());
			}
			/* Send filters and options */
			if (launchConfiguration != null)
			{
				sendFilters(launchConfiguration, tiAgent);
				sendOptions(launchConfiguration, trcAgentProxy, tiAgent);				
			}	
			else
			{
				sendFilters(tiAgentProxy, tiAgent);
				sendOptions(agentProxy != null ? agentProxy : tiAgentProxy, trcAgentProxy, tiAgent);
			}
		                  	
			switchPerspective();
          	configureDataProcessor(launchConfiguration, tiAgentProxy, tiAgent);
          	
    		boolean isEnableDataCollection = launchConfiguration == null ? true : launchConfiguration.getAttribute(IProfileLaunchConfigurationConstants.ATTR_AUTO_MONITORING, true);
    		if (isEnableDataCollection) {
    			TIDelegateHelper.getInstance().sendEnableDataCollection(launchConfiguration, tiAgent);
    		}		
    				
    		if (!tiAgent.isAgentRunning())
    			tiAgent.run();
    		
    		if (!isEnableDataCollection) {
    			setPausedState(tiAgentProxy);
    		}		
          	          	
			return trcAgentProxy;
		} 
		catch (InactiveAgentException e)
		{
			handleErrorCondition (LauncherMessages.LAUNCHER_COMMON_ERROR_TITLE, TIMessages.ERROR_TI_AGENT_INACTIVE, e, process, monitor);
		}
		catch (Exception e)
		{
			handleErrorCondition (LauncherMessages.LAUNCHER_COMMON_ERROR_TITLE, TIMessages.ERROR_TI_AGENT_UNKNOWN, e, process, monitor);
		}
		
		return null;
	}

	static class SetOutputFormatHandler implements ICommandHandler {
		
		private final Object lock = new Object();
		
		private String format = SetOutputFormatCommand.XML_FORMAT;
		
		public String getFormat() {
			return format;
		}
		
		public Object getLock() {
			return lock;
		}

		public void incomingCommand(INode node, ICommandElement command) {
			String commandStr = new String(((CommandFragment) command)
					.getCommandData());
			if (Constants.TPTP_DEBUG)
				System.out.println("The Agent response - " + commandStr);
			TPTPXMLParse ParseObj = new TPTPXMLParse();
			ParseObj.setParser(commandStr);
			Hashtable CommandHash = ParseObj.getHashTable();
			if (CommandHash.containsKey(SetOutputFormatCommand.FORMAT_ELEMENT)) {
				format = (String) CommandHash.get(SetOutputFormatCommand.FORMAT_ELEMENT);
			} else {
				format = SetOutputFormatCommand.XML_FORMAT; 
			}
			
			synchronized(lock) {
				lock.notifyAll();
			}
		}
	}

	public static void configureDataProcessor(ILaunchConfiguration launchConfiguration, TRCAgentProxy agentProxy, ICollector tiAgent) throws CoreException
	{
		/* Start the data collection process */
		
		ACDataProcessorAdapter dataProcessor;
		SetOutputFormatHandler handler = new SetOutputFormatHandler(); 
		// Possible enhancement: let user choose the format from UI 
		String outputFormat = System.getenv(TI_VARIABLE_OUTPUT_FORMAT);
		if (outputFormat != null ) {
			if (!(SetOutputFormatCommand.BINARY_FORMAT.equals(outputFormat) || SetOutputFormatCommand.XML_FORMAT.equals(outputFormat))) {
				throw new IllegalArgumentException("Environment variable " + TI_VARIABLE_OUTPUT_FORMAT + " should be 'xml' or 'binary'");
			}
		} else {
			outputFormat = SetOutputFormatCommand.BINARY_FORMAT;
		}
		SetOutputFormatCommand command = new SetOutputFormatCommand(outputFormat);
		
		synchronized(handler.getLock()) {
			try {
				tiAgent.sendCommand(command.buildCommand(), handler);
				handler.getLock().wait();
			} catch (InactiveAgentException e) {
				e.printStackTrace();
			} catch (NotConnectedException e) {
				e.printStackTrace();
			} catch (InterruptedException e) {
				e.printStackTrace();
			}
		}
		
		if (SetOutputFormatCommand.XML_FORMAT.equals(handler.getFormat())) {
			dataProcessor = (ACDataProcessorAdapter) LoadersUtils.locateDataProcessor(agentProxy);
	      	if (dataProcessor == null) 
	        {
	      		dataProcessor = new ACDataProcessorAdapter(agentProxy);
	        }
		} else { // Binary format
			dataProcessor = new BFDataProcessorAdapter(agentProxy);
		}
        LoadersUtils.registerDataProcessor(agentProxy, dataProcessor);                        
      	
      	/* Does the data need to be forwarded to a file? */
      	boolean isProfileToLaunch = launchConfiguration == null ? agentProxy.isToProfileFile() : launchConfiguration.getAttribute(IProfileLaunchConfigurationConstants.ATTR_PROFILE_TO_FILE, false);
		if (isProfileToLaunch)
		{
			String filePath = launchConfiguration == null ? agentProxy.getProfileFile() : launchConfiguration.getAttribute(IProfileLaunchConfigurationConstants.ATTR_DESTINATION_FILE, (String)null);
			if (filePath != null)
			{
				agentProxy.setProfileFile(filePath);
				dataProcessor.setProfileFileName(filePath);
				dataProcessor.createWriter();

                if (dataProcessor.isNewProfileFile()) {
                	dataProcessor.writeXMLVersion(dataProcessor.getWriter());
                	dataProcessor.startTrace(dataProcessor.getWriter());
                }
			}
		}
		
		tiAgent.addDataListener(dataProcessor);

		/* Start monitoring the agent */
	    IAgentStateModifier agentStateModifier = TIAgentStateModifier.getInstance();
	    agentStateModifier.setInput(new StructuredSelection(agentProxy));
//        try
//		{
			agentStateModifier.startMonitoring();
//		} catch (CoreException e)
//		{
//			LauncherUtility.openErrorWithDetail(
//					LauncherMessages.LAUNCHER_COMMON_ERROR_TITLE, 
//					TIMessages.ERROR_TI_AGENT_MONITOR, e);
//		}	
	}
	
	public static ICollector locateTIAgent(IAgentController agentController, long processId)
	{
		/* Find the TI agent that corresponds to the process launched. */
		try
		{
			return (ICollector)agentController.getAgent(TIConstants.TI_AGENT_NAME, ICollector.class.getName(), processId);			
		} 
		catch (NotConnectedException e)
		{
			ConnectionInfo connectionInfo = agentController.getConnectionInfo();
			String port = LauncherMessages.LAUNCHER_COMMON_UNKNOWN;
			if (connectionInfo != null)
			{
				port = String.valueOf(connectionInfo.getPort());
			}

			LauncherUtility.openErrorWithDetail(LauncherMessages.LAUNCHER_COMMON_ERROR_TITLE, NLS.bind(TIMessages.ERROR_TI_AC_UNAVAILABLE, port), e);
		} 
		return null;
	}

	private boolean checkAllocOptions(ILaunchConfiguration launchConfiguration,String optionCheck)
	{
		try
		{
			List analysisTypes = getSelectedTIAnalysisTypes(launchConfiguration);
			int analysisTypesCount = analysisTypes == null ? 0 : analysisTypes.size();
			boolean retValue = false;
			for (int i = 0; i < analysisTypesCount; i++)
			{
				/* Find out the attributes */
				Object currentAnalysisTypeObject = analysisTypes.get(i);
				LightConfigurationLoader confLoader = null;
				if (currentAnalysisTypeObject instanceof AnalysisType && (confLoader = ((AnalysisType)currentAnalysisTypeObject).getConfigurationLoader()) != null)
				{			
					ILightConfiguration analysisTypeConfiguraiton = confLoader.getConfigurationClass();
					
					DataCollectionEngineAttribute[] attributes = analysisTypeConfiguraiton.getAttributes();
					int attributeCount = attributes == null ? 0 : attributes.length;
					
					/* For analysis type attribute */
					for (int j = 0; j < attributeCount; j++)
					{
						if(optionCheck.equals(ITILaunchConfigurationConstants.ATTR_EXEC_OBJ_ALLOC_SITE))
						{
							if (ITILaunchConfigurationConstants.ATTR_EXEC_OBJ_ALLOC_SITE.equals(attributes[j].getName()))
							{
								retValue = launchConfiguration.getAttribute(ITILaunchConfigurationConstants.ATTR_EXEC_OBJ_ALLOC_SITE, false);
								if(retValue)
								{
									break;
								}
							}
						}
						
						if(analysisProvider != null) {
							retValue = analysisProvider.checkAllocOptions(launchConfiguration, optionCheck, attributes[j].getName());
							if(retValue)
							{
								break;
							}
						}
						
						if(optionCheck.equals(ITILaunchConfigurationConstants.ATTR_EXEC_FLOW))
						{
							if (ITILaunchConfigurationConstants.ATTR_EXEC_FLOW.equals(attributes[j].getName()))
							{
								retValue = launchConfiguration.getAttribute(ITILaunchConfigurationConstants.ATTR_EXEC_FLOW, true);
								if(retValue)
								{
									break;
								}
							}
						}
						if(optionCheck.equals(ITILaunchConfigurationConstants.ATTR_EXEC_AUTO_POLLING))
						{
							if (ITILaunchConfigurationConstants.ATTR_EXEC_AUTO_POLLING.equals(attributes[j].getName()))
							{
								retValue = launchConfiguration.getAttribute(ITILaunchConfigurationConstants.ATTR_EXEC_AUTO_POLLING, false);
								if(retValue)
								{
									break;
								}
							}
						}
					}
				}
			}
		
		return retValue;
	}
	catch (Exception e)
	{
		return false;			
	}	
  }

	/**
	 * Read in the option set from the launch configuration and sends it to the tiAgent
	 * 
	 * @param launchConfiguration The launch configuration
	 * @param agentProxy The agent proxy.
	 * @param tiAgent The TI agent
	 */
	public void sendOptions(ILaunchConfiguration launchConfiguration, TRCAgentProxy agentProxy, ICollector tiAgent) 
	{
		Exception error;
		try
		{				
			/* The option command.  The initial value should be <applyOptions iid="org.eclipse.tptp.jvmti"> */
			StringBuffer optionCommand = new StringBuffer();
			TIUtility.addElement(optionCommand, CMD_APPLY_OPTIONS, false, false);
			TIUtility.addAttribute(optionCommand, new String[]{ATTRIBUTE_IID}, new String[]{TIConstants.TI_AGENT_NAME}, false, true);
			
			/* Get the selected analysis types */
			List analysisTypes = getSelectedTIAnalysisTypes(launchConfiguration, tiAgent);
			int analysisTypesCount = analysisTypes == null ? 0 : analysisTypes.size();
			
			
			/* For every selected TI analysis type */
			boolean collectExecData = false;
			boolean collectHeapData = false;
			boolean isAutoPolling = false;
			boolean isManualPolling = false;
						
			for (int i = 0; i < analysisTypesCount; i++)
			{
				/* Find out the attributes */
				Object currentAnalysisTypeObject = analysisTypes.get(i);
				LightConfigurationLoader confLoader = null;
				if (currentAnalysisTypeObject instanceof AnalysisType && (confLoader = ((AnalysisType)currentAnalysisTypeObject).getConfigurationLoader()) != null)
				{			
					ILightConfiguration analysisTypeConfiguraiton = confLoader.getConfigurationClass();
					if (confLoader.isEditable())
					{
						IConfigurationPage[] configurationPages = ((IConfiguration)analysisTypeConfiguraiton).getConfigurationPages();
						for (int j = 0; j < configurationPages.length; configurationPages[j++].reset(launchConfiguration));		
					}
					
					DataCollectionEngineAttribute[] attributes = analysisTypeConfiguraiton.getAttributes();
					int attributeCount = attributes == null ? 0 : attributes.length;
					
	
					/* For analysis type attribute */
					String value;
					for (int j = 0; j < attributeCount; j++)
					{
						/* Append to the option command */
						value = attributes[j].getValue();						
						if (ITILaunchConfigurationConstants.ATTR_EXEC_DATA.equals(attributes[j].getName()))
						{
							collectExecData = true;
						}
						else if (ITILaunchConfigurationConstants.ATTR_HEAP_DATA.equals(attributes[j].getName()))
						{
							collectHeapData = true;
						}
						else if(ITILaunchConfigurationConstants.ATTR_EXEC_MANUAL_POLLING.equals(attributes[j].getName()))
						{
							try
							{
								isManualPolling = launchConfiguration.getAttribute(ITILaunchConfigurationConstants.ATTR_EXEC_MANUAL_POLLING, false);
							}
							catch(Exception e)
							{
								isManualPolling = false;
							}
							value = String.valueOf(isManualPolling);
						}
						else if(ITILaunchConfigurationConstants.ATTR_EXEC_AUTO_POLLING.equals(attributes[j].getName()))
						{
							try
							{
								isAutoPolling = launchConfiguration.getAttribute(ITILaunchConfigurationConstants.ATTR_EXEC_AUTO_POLLING, false);
							}
							catch(Exception e)
							{
								isAutoPolling = false;
							}
							value = String.valueOf(isAutoPolling);
						}
						else if (ITILaunchConfigurationConstants.ATTR_EXEC_POLLING_FREQ.equals(attributes[j].getName()))
						{
							POLL_TIME = Integer.valueOf(value).intValue();

							// this attribute should not added into options passed to backend
							continue;
						}
						addOption (optionCommand, attributes[j].getName(), value);							
					}
				}
			}
			
			TIUtility.addElement(optionCommand, CMD_APPLY_OPTIONS, true, true);
			tiAgent.sendCommand(optionCommand.toString(), NOOP_COMMAND_HANDLER);
			
			agentProxy.setCollectionMode(computeCollectionMode(launchConfiguration, collectExecData, collectHeapData));
			if(tiAgent != null && _agentState != null)
			{
					//After the first invocation of the Execution Stat set the token.
					//See a few lines backward and there is where the token is set
					if(_agentState.token != null)
					{
						if(_agentState.token.equals("CGProf"))
						{
							isAggProf = true;
						}
					}
			}
			if(isAggProf)
			{
				PeriodicPoll pollthread = new PeriodicPoll(tiAgent);
				pollthread.start();
			}
			
			if(tiAgent != null && _agentState != null)
			{
					//After the first invocation of the Execution Stat set the token.
					//See a few lines backward and there is where the token is set
					if(((isCGProf)|| (isAggProf)) && _agentState.token == null)
					{
						_agentState.token = "CGProf";
					}
			}
			return;
		}
		catch (InactiveAgentException e)
		{
			error = e;			
		}	
		catch (NotConnectedException e)
		{
			error = e;
		}	
		LauncherUtility.openErrorWithDetail(LauncherMessages.LAUNCHER_COMMON_ERROR_TITLE, TIMessages.ERROR_TI_AGENT_OPTIONS, error);
	}

	
	private void addOption(StringBuffer optionCommand, String name, String value)
	{
		/* <option><name><NAME-OF-OPTION></name><value><OPTION-VALUE></value></option> */		
		TIUtility.addElement(optionCommand, ELEMENT_OPTION, false, true);
		TIUtility.addElement(optionCommand, ELEMENT_NAME, false, true);
		optionCommand.append(name);
		TIUtility.addElement(optionCommand, ELEMENT_NAME, true, true);
		
		TIUtility.addElement(optionCommand, ELEMENT_VALUE, false, true);
		optionCommand.append(value);
		TIUtility.addElement(optionCommand, ELEMENT_VALUE, true, true);
		TIUtility.addElement(optionCommand, ELEMENT_OPTION, true, true);
	}


	/**
	 * Read in the option set from the launch configuration and sends it to the tiAgent
	 * 
	 * @param oldAgentProxy The detached agent proxy
	 * @param agentProxy The agent proxy.
	 * @param tiAgent The TI agent
	 */
	public void sendOptions(TRCAgentProxy oldAgentProxy, TRCAgentProxy agentProxy, ICollector tiAgent) 
	{
		Exception error;
		try
		{					
			/* The option command.  The initial value should be <applyOptions iid="org.eclipse.tptp.jvmti"> */
			StringBuffer optionCommand = new StringBuffer();
			TIUtility.addElement(optionCommand, CMD_APPLY_OPTIONS, false, false);
			TIUtility.addAttribute(optionCommand, new String[]{ATTRIBUTE_IID}, new String[]{TIConstants.TI_AGENT_NAME}, false, true);		
			
			EList agentProxyConf = oldAgentProxy.getConfigurations();
			for (int i = 0, confCount = agentProxyConf.size(); i < confCount; i++)
			{
				TRCConfiguration config = (TRCConfiguration)agentProxyConf.get(i);
				List options = config.getOptions();
				for (int opt = 0; opt < options.size(); opt++)
				{		
					TRCOption option = (TRCOption)options.get(opt);
					
					/* Append to the option command */
					addOption(optionCommand, option.getKey(), option.getValue());
				}
			}
												
			TIUtility.addElement(optionCommand, CMD_APPLY_OPTIONS, true, true);
			tiAgent.sendCommand(optionCommand.toString(), NOOP_COMMAND_HANDLER);			
			agentProxy.setCollectionMode(oldAgentProxy.getCollectionMode());
			return;
		}		
		catch (InactiveAgentException e)
		{
			error = e;			
		}	
		catch (NotConnectedException e)
		{
			error = e;
		}	
		LauncherUtility.openErrorWithDetail(LauncherMessages.LAUNCHER_COMMON_ERROR_TITLE, TIMessages.ERROR_TI_AGENT_OPTIONS, error);
				
	}
	
	private TRCCollectionMode computeCollectionMode(ILaunchConfiguration launchConfiguration, boolean collectExecData, boolean collectHeapData)
	{
		boolean collectExecutionFlow;
		boolean collectExecInstanceLevelInfo;
		
		boolean collectHeapInstanceLevelInfo;
		try
		{
			collectExecutionFlow = launchConfiguration.getAttribute(ITILaunchConfigurationConstants.ATTR_EXEC_FLOW, true);
			isAggProf = !collectExecutionFlow;
			collectExecInstanceLevelInfo = launchConfiguration.getAttribute(ITILaunchConfigurationConstants.ATTR_EXEC_COLLECT_INSTANCE, false);		
			collectHeapInstanceLevelInfo = launchConfiguration.getAttribute(ITILaunchConfigurationConstants.ATTR_HEAP_COLLECT_INSTANCE, false);
		}
		catch (CoreException e)
		{
			/* Return the literal execution collection mode */
			
			return TRCCollectionMode.EXECUTION_STATISTICS_ONLY_LITERAL;
		}			
        if (collectExecData) 
        {
            if (collectHeapData) /* heap + execution */
            {
            	if (collectHeapInstanceLevelInfo) /* collect heap instance */
            	{
            		return collectExecutionFlow ? 
            			TRCCollectionMode.HEAP_AND_EXECUTION_FULL_LITERAL :					/* exec full, no need to worry about instance level up to this level */ 
            			TRCCollectionMode.HEAP_FULL_AND_EXECUTION_STATISTICS_ONLY_LITERAL;  /* exec compressed, no instance collection available */
            	}
            	else /* no heap instance */
            	{
            		return collectExecutionFlow ? /* exectution full */ 
            				(collectExecInstanceLevelInfo ? 
            						TRCCollectionMode.HEAP_AND_EXECUTION_FULL_LITERAL:	/* execution full and execution instance */
            						TRCCollectionMode.HEAP_STATISTICS_AND_EXECUTION_NO_INSTANCES_LITERAL): /* execution full but no execution instance */            							
            				TRCCollectionMode.HEAP_AND_EXECUTION_STATISTICS_ONLY_LITERAL; /* execution compress, no instance collection available */ 
            	}
            } 
            else /* execution only */
            {
            	return collectExecutionFlow ? /* execution full */
            			(collectExecInstanceLevelInfo ? TRCCollectionMode.EXECUTION_FULL_LITERAL: /* execution full and execution instance */
						TRCCollectionMode.EXECUTION_NO_INSTANCES_LITERAL): /* execution full but no execution instance */
						TRCCollectionMode.EXECUTION_STATISTICS_ONLY_LITERAL;
            }
        }
        else /* heap only */
        {
        
        	// Bug 219379 - If the user has selected heap profiling + obj alloc site option
        	// then use HEAP_FULL_LITERAL, otherwise, use HEAP_STATISTICS_ONLY
	        boolean heapObjAllocSites = true;
			
			try {
				// The default value is false if the edit options dialog has not yet been opened. Otherwise, obtain the launch attribute.
				heapObjAllocSites = launchConfiguration.getAttribute(ITILaunchConfigurationConstants.ATTR_EXEC_OBJ_ALLOC_SITE, false);
				
				/*
				 * Bug 323868
				 * If heapObjAllocSites is false, then the Object data will not be created. Otherwise, a provider can 
				 * store the Object data in the Class without collecting the allocation site information using HEAP_FULL.
				 */
				if(!heapObjAllocSites) {
					ICustomAnalysisProvider[] providers = TIPlugin.getProviders();
					
					for(ICustomAnalysisProvider provider : providers) {
						if(provider.getCollectAnalysisOption()) {
							heapObjAllocSites = true;
							break;
						}
					}
				}
			} catch (CoreException e1) { /** Leave heapObjAllocSites at default */}

			return TRCCollectionMode.HEAP_AND_EXECUTION_FULL_LITERAL;

			// This block of code is from 219379, commented out for 330942
//			if(heapObjAllocSites) {
//				// Store the Object data. If the allocSites profiling option is not set, this will only 
//				// create the TRCHeapObject Objects in the TRCClass Object array.
//				return TRCCollectionMode.HEAP_FULL_LITERAL;
//			} else {
//				// Stores only the statistic information about the Heap with no TRCObjects in the TRCClass.
//				return TRCCollectionMode.HEAP_STATISTICS_ONLY_LITERAL;
//			}

			
			// return TRCCollectionMode.HEAP_AND_EXECUTION_FULL_LITERAL;
        	//return collectHeapInstanceLevelInfo ?
            //    	TRCCollectionMode.HEAP_AND_EXECUTION_FULL_LITERAL:
            //    	TRCCollectionMode.HEAP_STATISTICS_ONLY_LITERAL;
        	//return collectHeapInstanceLevelInfo ?
        	//		TRCCollectionMode.HEAP_FULL_LITERAL: /* collect heap instance */
        	//			TRCCollectionMode.HEAP_STATISTICS_ONLY_LITERAL;

			
        }
	}
	
	
	/**
	 * Read in the filters set from the launch configuration and sends it to the tiAgent
	 * 
	 * @param launchConfiguration The launch configuration
	 * @param tiAgent The TI agent
	 */
	public void sendFilters(String filterSetId, ICollector tiAgent) 
	{
		/* Read in filter set and construct the filter command XML */
		Exception error;
		try 
		{			
			FilterSetElement filterSet = (FilterSetElement)TraceFilterManager.getInstance().getFilterSet().get(filterSetId);
			if (filterSet == null && (filterSet = ProfilingSetsManager.instance().findNextAvailableFilterSet()) == null)
				return;

			ArrayList filterElements = filterSet.getChildren();

			/* The command.  Initial value is: "<applyFilters iid="org.eclipse.tptp.jvmti">" */
			StringBuffer filterCommand = new StringBuffer();
			TIUtility.addElement(filterCommand, CMD_APPLY_FILTERS, false, false);			
			TIUtility.addAttribute(filterCommand, new String[]{ATTRIBUTE_IID}, new String[]{TIConstants.TI_AGENT_NAME}, false, true);
			if (filterElements != null)
			{
				FilterTableElement currentElement;
				for (int i = 0, filterElementCount = filterElements.size(); i < filterElementCount; i++)
				{
					currentElement = (FilterTableElement)filterElements.get(i);
					
					/* Add the filter to the command.  The added filter fragment should be: 
					 * <Filter>CLASS_NAME METHOD_NAME INCLUDE|EXCLUDE</Filter> */ 
					TIUtility.addElement(filterCommand, ELEMENT_FILTER, false, true);
					filterCommand.append(currentElement.getText()).append(TIUtility.SINGLE_SPACE).append(currentElement.getMethod()).append(TIUtility.SINGLE_SPACE).append(currentElement.getVisibility().equals(TraceMessages.INCLUDE)?"INCLUDE":"EXCLUDE");
					TIUtility.addElement(filterCommand, ELEMENT_FILTER, true, true);
				}
			}
			
			TIUtility.addElement(filterCommand, CMD_APPLY_FILTERS, true, true);
			tiAgent.sendCommand(filterCommand.toString(), NOOP_COMMAND_HANDLER);
			
			return;
		} 
		catch (NotConnectedException e)
		{
			error = e;			
		}
		catch (InactiveAgentException e) 
		{
			error = e;			
		}
		
		LauncherUtility.openErrorWithDetail(LauncherMessages.LAUNCHER_COMMON_ERROR_TITLE, TIMessages.ERROR_TI_AGENT_FILTERS, error);
	}
	
	
	public void sendFilters(ILaunchConfiguration launchConfiguration, ICollector tiAgent) 
	{
		try
		{
			String filterSetId = launchConfiguration.getAttribute(ITILaunchConfigurationConstants.ATTR_FILTERSET_ID, getDefaultFilterSetName(launchConfiguration));
			sendFilters (filterSetId, tiAgent);
			if (LauncherUtility.isAutoFilteringCriteria(launchConfiguration)) {
				LauncherUtility.removeAutogeneratedFilterSet();
			}
		} 
		catch (CoreException e)
		{
			LauncherUtility.openErrorWithDetail(LauncherMessages.LAUNCHER_COMMON_ERROR_TITLE, TIMessages.ERROR_TI_AGENT_FILTERS, e);
		}		
	}
	

	private String getDefaultFilterSetName(final ILaunchConfiguration config)
			throws CoreException {
		return !LauncherUtility.isAutoFilteringCriteria(config) ? LauncherConstants.DEFAULT_FILTER_ID : TraceConstants.AUTO_GENERATED_FILTER_SET;
	}
	
	public void sendFilters(TRCAgentProxy agentProxy, ICollector tiAgent) 
	{
		TRCOption[] options = TIUtility.getAgentOptions(agentProxy, ITILaunchConfigurationConstants.ATTR_FILTERSET_ID);
		if (options.length >= 1)
		{
			sendFilters(options[0].getValue(), tiAgent);
		}
	}
	

	public void sendEnableDataCollection(ILaunchConfiguration configuration, 
			ICollector tiAgent) {
		Exception error = null;
		String cmd = new String("<EnableDataCollection iid=\"org.eclipse.tptp.jvmti\"></EnableDataCollection>");
		boolean isEnableDataCollection = true;
		
		if (configuration != null) {
			try {
				isEnableDataCollection = configuration == null ? true : configuration.getAttribute(IProfileLaunchConfigurationConstants.ATTR_AUTO_MONITORING, true);
			} catch (CoreException e) {
				LauncherUtility.openErrorWithDetail(LauncherMessages.LAUNCHER_COMMON_ERROR_TITLE, TIMessages.ERROR_TI_AGENT_FILTERS, e);
				return;
			}
		}
		
		if (isEnableDataCollection) {
			try {
				tiAgent.sendCommand(cmd, NOOP_COMMAND_HANDLER);
			} catch (InactiveAgentException e) {
				error = e;
			} catch (NotConnectedException e) {
				error = e;
			}
			if (error != null) {
				LauncherUtility.openErrorWithDetail(LauncherMessages.LAUNCHER_COMMON_ERROR_TITLE, TIMessages.ERROR_TI_AGENT_FILTERS, error);
				return;
			}
			return;
		}
		
	}
	
	
	private TRCAgentProxy createModelEntities(ILaunchConfiguration launchConfiguration, IAgentController agentController, IProcess process, ICollector tiAgent) throws CoreException
	{		
		/* Get the required attributes for the model entities */
		IPreferenceStore store = UIPlugin.getDefault().getPreferenceStore();
		String projectName = launchConfiguration.getAttribute(IProfileLaunchConfigurationConstants.ATTR_DESTINATION_PROJECT, store.getString(CommonUITraceConstants.TRACE_PROJECT_NAME));
		String monitorName = launchConfiguration.getAttribute(IProfileLaunchConfigurationConstants.ATTR_DESTINATION_MONITOR, store.getString(CommonUITraceConstants.TRACE_MONITOR_NAME));			
		
		/* Get the analysis types */		
		List analysisTypes = getSelectedTIAnalysisTypes(launchConfiguration);
		return createModelEntities (projectName, monitorName, analysisTypes, agentController, process, tiAgent); 
	}
	
	
	private TRCAgentProxy createModelEntities(TRCAgentProxy agentProxy, IAgentController agentController, IProcess process, ICollector tiAgent) throws CoreException
	{
		/* Get the required attributes for the model entities */
		TRCMonitor monitor = agentProxy.getProcessProxy().getNode().getMonitor();
		URI uri = monitor.eResource().getURI();
		if (uri.segmentCount() - 2 < 0)
		{
			/* Show an error about not being able to resolve the project name */
			throw LauncherUtility.createCoreException(IStatus.ERROR, TIMessages.ERROR_TI_AGENT_PROJECT_NAME);
		}
		
		String projectName = monitor.eResource().getURI().segment(uri.segmentCount() - 2);
		String monitorName = monitor.getName();			
		
		/* Get the analysis types */
		List analysisTypes = new ArrayList();
		TRCOption[] analysisTypeOptions = TIUtility.getAgentOptions(agentProxy, LauncherConstants.ANALYSIS_TYPE_ASSOCIATION);
		for (int i = 0; i < analysisTypeOptions.length; i++)
		{
			AnalysisType analysisType = AnalysisTypeManager.getInstance().getAnalysisType(analysisTypeOptions[i].getValue());
			if (analysisType != null && !analysisTypes.contains(analysisType))
			{
				analysisTypes.add (analysisType);
			}
		}
					
		TRCAgentProxy newAgentProxy = createModelEntities (projectName, monitorName, analysisTypes, agentController, process, tiAgent);
		
		// copy destination file name for trace
		newAgentProxy.setProfileFile(agentProxy.getProfileFile());
		return newAgentProxy; 
	
	}
	
	
	
	private TRCAgentProxy createModelEntities(String projectName, String monitorName, List analysisTypes, IAgentController agentController, IProcess process, ICollector tiAgent) throws CoreException
	{			
		ConnectionInfo connectionInfo = agentController.getConnectionInfo();		
					
		/* Create the heirarchical model entities */
        IContainer container = PDCoreUtil.createContainer(new Path(projectName));
        TRCMonitor monitor = PDCoreUtil.createMonitor(container, monitorName);
        TRCNode trcNode = PDCoreUtil.createNode(monitor, connectionInfo.getHostName(), String.valueOf(connectionInfo.getPort()));	        
    	TRCProcessProxy trcProcessProxy = HierarchicalModelFactory.createTRCProcessProxy(trcNode, process);
        TRCAgentProxy trcAgentProxy = HierarchicalModelFactory.createTRCAgentProxy(trcProcessProxy, tiAgent, CommonUIConstants.PROFILE_AGENT_TYPE);
                
        
		/* Associate the analysis type to the agent proxy */		
		int analysisTypeCount = analysisTypes == null ? 0 : analysisTypes.size();
		TRCConfiguration agentProxyConfiguration = null;
		final String OPEN_BRACKET = "(";
		final String CLOSE_BRACKET = ")";
		for (int i = 0; i < analysisTypeCount; i++)
		{
			/* Create the configuration */
			if (i == 0)
			{
				agentProxyConfiguration = HierarchicalModelFactory.createTRCConfiguration(trcAgentProxy, LauncherConstants.SET_OPTION_PREFIX);
			}
								
			Object currentAnalysisType = analysisTypes.get(i);
			String id = null;
			if (currentAnalysisType instanceof AnalysisType)
			{
				id = ((AnalysisType)currentAnalysisType).getId();
			}
			else if (currentAnalysisType instanceof IProfilingSetType)
			{
				id = ((IProfilingSetType)currentAnalysisType).getId();
			}
			
			if (id != null)
			{
				HierarchicalModelFactory.createOption(agentProxyConfiguration, LauncherConstants.ANALYSIS_TYPE_ASSOCIATION + OPEN_BRACKET + id + CLOSE_BRACKET, id);
			}
		}
	            
        return trcAgentProxy;
	}

	private List getSelectedTIAnalysisTypes(ILaunchConfiguration launchConfiguration)
	{
		return getSelectedTIAnalysisTypes(launchConfiguration, null);
	}
	
	private List getSelectedTIAnalysisTypes(ILaunchConfiguration launchConfiguration, ICollector tiAgent)
	{
		List analysisTypes = new ArrayList();
		try 
		{
			/* Is this the attach launch configuration? */
			if (TIConstants.CONFIGURATION_ATTACH_ID.equals(launchConfiguration.getType().getIdentifier()))
			{
				SelectedAgent[] selectedAgents = LauncherUtility.retrieveSelectedAgents(launchConfiguration);
				for (int i = 0; i < selectedAgents.length; i++)
				{
					if (!selectedAgents[i].getPid().equals(String.valueOf(tiAgent.getProcess().getProcessId())))
						continue;
					
					String[] selectedAnalysisTypes = selectedAgents[i].getAnalysisTypes();
					if (selectedAnalysisTypes!=null) {
						for (int j = 0; j < selectedAnalysisTypes.length; j++)
						{
							AnalysisType analysisType = AnalysisTypeManager.getInstance().getAnalysisType(selectedAnalysisTypes[j]);
							if (analysisType != null)
								analysisTypes.add(analysisType);
						}
					}
				}
			}
			else
			{
				String launchItemSelections = launchConfiguration.getAttribute(IProfileLaunchConfigurationConstants.ATTR_PROFILING_COLLECTOR_AND_ANALYSIS, (String)null);
				Hashtable dataCollectionSelections;
				if (launchItemSelections != null && (dataCollectionSelections = LauncherUtility.unserializeSelection(launchItemSelections)) != null)
				{
					return (List)dataCollectionSelections.get(DataCollectorManager.getInstance().getDataCollector(TIConstants.TI_DATA_COLLECTOR_ID));
				}
			}
		} 
		catch (CoreException e) 
		{			
			UIPlugin.getDefault().log(e);
		} 
		catch (InactiveProcessException e)
		{
			UIPlugin.getDefault().log(e);
		}


		return analysisTypes;
	}
	
	
	private boolean handleErrorCondition(String title, String message, Exception e, IProcess process, IProgressMonitor monitor)
	{
		monitor.setCanceled(true);
		LauncherUtility.openErrorWithDetail(title, message, e);
		try
		{
			if (process != null)
			{
				process.kill();
			}
		}
		catch (InactiveProcessException e1)
		{
			/* Doesn't need to be handled */
		}
		catch (NotConnectedException e1)
		{
			/* Doesn't need to be handled */
		}
		catch (InactiveAgentException e1)
		{
			/* Doesn't need to be handled */
			
		}
		
		/* Always returns false to indicate an error condition */
		return false;
	}

	public static ProcessParameters getProcessParameter(ProcessParameters processParameters)
	{
		processParameters.addExecutableParameter(TI_JVM_FLAG);
		return processParameters;
	}
	
	private  boolean probesSelected(ILaunchConfiguration launchConfiguration)
	{
		boolean addProbeListener = false;
		List selectedTypes = getSelectedTIAnalysisTypes(launchConfiguration);
		 for (int i=0; i<selectedTypes.size(); i++){
			 AnalysisType analysisType = (AnalysisType)selectedTypes.get(i);
			 if (analysisType.getId().equals(TIConstants.PROBE_INSERTION_ID))
			 {
				 addProbeListener = true;
			 }
		 }
		 return addProbeListener;
	}

	public void startPolling(ICollector tiAgent) {
		PeriodicPoll pollthread = new PeriodicPoll(tiAgent);
		pollthread.start();
	}

	
	public void startPolling(ICollector tiAgent, int POLL_TIME) {
		this.POLL_TIME = POLL_TIME;
		PeriodicPoll pollthread = new PeriodicPoll(tiAgent);
		pollthread.start();
	}	
	
	/**
	 * A data process adapter that turns the current data processor that we use for our
	 * trace model to a data processor that is acceptable by the new Agent Controller.
	 * 
	 * @author Ali Mehregani
	 */
	public static class ACDataProcessorAdapter extends StatGeneratorDataProcessor implements IDataProcessor
	{
		/** Used to indicate if the process associted with this processor has exited.  Used only when
		 * a the user is profiling to a file */
		private boolean processAlive;

		/** Used to monitor the activity of the processor when processAlive is set to false.  This is
		 * used to determine when the output streams associated with processor can be closed.  We can't
		 * close a processor right after we get a process exit event because the agent sends some events. */
		private int activity;
		
		/** Indicates if this processor is enabled/disabled */
		private boolean enabled;
		
		public ACDataProcessorAdapter(TRCAgentProxy agentProxy)
		{
			super(agentProxy);		
			processAlive = true;
			enabled = true;
			BinaryLoader bfLoader = this;
			bfLoader.setLoadXml(true);
		}

		public void incomingData(byte[] buffer, int length, InetAddress peer)
		{
			if (!enabled)
				return;
			preIncomingData();
			super.incommingData(buffer, length, peer);
		}

		public void incomingData(char[] buffer, int length, InetAddress peer)
		{		
			if (!enabled)
				return;
			preIncomingData();
			super.incommingData(buffer, length, peer);
		}
		
		protected void preIncomingData() 
		{
			if (!processAlive)
				activity++;
		}
		
		public void setProcessAlive (boolean processAlive)
		{
			this.processAlive = processAlive;
		}
		
		public boolean isProcessAlive()
		{
			return processAlive;
		}
		
		public int getActivity()
		{
			return activity;
		}
		
		public void setEnabled(boolean enabled)
		{
			this.enabled = enabled;
		}
		
		public boolean isEnabled()
		{
			return enabled;
		}
	}
	
	public static class BFDataProcessorAdapter extends ACDataProcessorAdapter {

		public BFDataProcessorAdapter(TRCAgentProxy agentProxy) {
			super(agentProxy);
			BinaryLoader bfLoader = this;
			bfLoader.setLoadXml(false);
		}

		protected void loadEventHook(byte[] buffer, int offset, int length) {
			loadEvent(buffer, offset, length);
		}

		protected boolean isValidTag(byte[] buf, int offset, int length) {
			// Do not check binary stream for XML tags
			return true;
		}

		protected String getDumpFileName(){
			return "TraceContents-" + getTimeStamp() + ".bin";
		}
		
		protected void writeByte(byte[] data, int offset, int length) {
			try {
				if (fileWriter != null) {
					fileWriter.write(data, offset, length);
					fileWriter.flush();
				}
			} catch (IOException e) {
				e.printStackTrace();
			}
		}
		public void endTrace(OutputStream randomFile) {
			try {
				if (randomFile != null) {
					randomFile.flush();
					randomFile.close();
				}
			} catch (IOException e) {
			}
		}

		public void writeXMLVersion(OutputStream randomFile) {
			// Do not write XML version into binary stream
		}

		public void startTrace(OutputStream randomFile) {
			// Do not write XML tags into binary stream
		}
	
	}
	
	
	/**
	 * A dummy command handler that doesn't do anything in response to a command
	 * that is sent
	 */
	public static final class NOOPCommandHandler implements ICommandHandler
	{

		public void incomingCommand(INode node, ICommandElement command)
		{
			/* Don't do anything */
		}
		
	}
	private static IConsole getConsole(IProcess process) {
		IConsoleManager manager = ConsolePlugin.getDefault()
				.getConsoleManager();
		IConsole[] consoles = manager.getConsoles();
		for (int i = 0; i < consoles.length; i++) {
			IConsole console = consoles[i];
			if (console instanceof org.eclipse.debug.ui.console.IConsole) {
				if (((org.eclipse.debug.ui.console.IConsole) console)
						.getProcess().equals(process)) {
					return console;
				}
			}
		}
		return null;
	}
	private static void showConsole(final IConsole console) {

		if (console == null)
			return;

		IWorkbenchWindow window = UIPlugin.getActiveWorkbenchWindow();
		if (window != null) {
			IWorkbenchPage page = window.getActivePage();
			if (page != null) {
				IViewPart consoleView = page
						.findView(IConsoleConstants.ID_CONSOLE_VIEW);
				if (consoleView != null) {
					// display (ie. update the content) the console, if it
					// exists
					// but do not bring it to the front
					if (consoleView instanceof IConsoleView) {
						((IConsoleView) consoleView).display(console);
					}
				}
			}
		}
	}
	
	private class PeriodicPoll extends Thread {
		boolean isMonitored = true;
		ICollector tiAgent;

		public PeriodicPoll(ICollector agent) {
			tiAgent = agent;
		}

		public void run() {
			while (isMonitored && isProcessActive) {
				isMonitored = tiAgent.isMonitored();
				StringBuffer optionCommand = new StringBuffer();

				TIUtility.addElement(optionCommand, CMD_COLLECT_DATA, false,
						false);
				TIUtility
						.addAttribute(optionCommand,
								new String[] { ATTRIBUTE_IID },
								new String[] { TIConstants.TI_AGENT_NAME },
								false, true);
				TIUtility.addElement(optionCommand, CMD_COLLECT_DATA, true,
						true);

				try {
					tiAgent.sendCommand(optionCommand.toString(),
							NOOP_COMMAND_HANDLER);
					Thread.sleep(POLL_TIME * 1000);
				} catch (Exception e) {
					e.printStackTrace(System.out);
				}
			}
			System.gc();
		}
	}
	class RunCommandHandler implements ICommandHandler {
		
		public void incomingCommand(INode node, ICommandElement command) {
			String commandStr = new String(((CommandFragment) command)
					.getCommandData());
			if (Constants.TPTP_DEBUG)
				System.out.println("The Agent response - " + commandStr);
			TPTPXMLParse ParseObj = new TPTPXMLParse();
			ParseObj.setParser(commandStr);
			Hashtable CommandHash = ParseObj.getHashTable();
			if (CommandHash.containsKey("ErrCode")) {
				try {
					if(commandStr.indexOf("Wrong Phase") ==-1)
					{
						errorCode = Long.parseLong((String) CommandHash
								.get("ErrCode"));
					}
				} catch (NumberFormatException e) {
					// just ignore
				}
				if(commandStr.indexOf("Wrong Phase") ==-1)
				{
					errorMsg = (String) CommandHash.get("ErrInfo");	
				}
			}
			synchronized (lock) {
				lock.notifyAll();
			}
		}
	}

	public void runAgent(IAgentController agentController, ICollector tiAgent,
			IProgressMonitor monitor) {

		errCount = 0;
		
		RunCommand runCommand = new RunCommand();

		synchronized (lock) {
			while (errCount < RETRY_THRESHOLD) {
				errorCode = 0;
				errorMsg = "";
				if (Constants.TPTP_DEBUG)
					System.out.println("Sending run() command");

				try {
					tiAgent.sendCommand(runCommand.buildCommand(), new RunCommandHandler());
					tiAgent.resume();
					lock.wait(WAITING_TIME);
				} catch (Exception e) {
					// just ignore
				}
				if (errorCode > 0) {
					errCount++;
				} else {
					break;
				}
			}
		}
		if (errorCode > 0) {
			LauncherUtility.openErrorWithDetail(
					LauncherMessages.LAUNCHER_COMMON_ERROR_TITLE, 
					errorMsg, null);
		}
	}


	public void setPausedState(TRCAgentProxy agentProxy) {
		if (agentProxy != null) {
			LauncherUtility.sendProfileEvent(ProfileEvent.STOP_MONITOR, agentProxy);
			agentProxy.setMonitored(false);
			agentProxy.setAttached(true);
			agentProxy.setActive(true);
		}
	}

}
