/**********************************************************************
 * Copyright (c) 2007 IBM Corporation, Intel Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.tptp.platform.jvmti.client.internal.launcher.util;

import org.eclipse.tptp.platform.execution.client.agent.internal.AgentStatePool;
import org.eclipse.tptp.platform.execution.client.core.IAgentController;
import org.eclipse.tptp.platform.execution.client.core.IConsole;
import org.eclipse.tptp.platform.execution.client.core.IDataProcessor;
import org.eclipse.tptp.platform.execution.client.core.IProcess;
import org.eclipse.tptp.platform.execution.client.core.internal.IProcessListener;
import org.eclipse.tptp.platform.execution.util.internal.AgentControllerPool;
import org.eclipse.tptp.platform.jvmti.client.internal.launcher.ILauncherParameterProvider;

/**
 * This delegate is used to keep track of connections with Agent Controller.  Only one
 * connection should be established with any one host.  Establishing connections is
 * expensive and should only be done once.  This is a singleton class.
 *  
 * @author Ali Mehregani
 */
public class AgentControllerDelegate
{
	/* The instance of this class */
	private static AgentControllerDelegate instance = new AgentControllerDelegate();

	/**
	 * Hide the constructor
	 */
	private AgentControllerDelegate()
	{		
	}

	public static AgentControllerDelegate getInstance() {
		return instance;
	}

	public IAgentController getConnection(String hostName, int portNumber,
			boolean reuseExistingConnection) throws Exception {
		return AgentControllerPool.getInstance().getConnection(hostName, portNumber);
	}

	public IAgentController getConnection(String hostName, int portNumber)
			throws Exception {
		return getConnection(hostName, portNumber, true);
	}

	static public String getUnifiedHostName(String hostName) {
		return AgentStatePool.getUnifiedHostName(hostName);
	}

	static class DataProcessor implements IDataProcessor {

	private final static String JAVA_VERSION = "java version";

	public void incomingData(byte[] buffer, int length,
		java.net.InetAddress peer) {
	    String line = new String(buffer, 0, length);
	    int index = line.indexOf(JAVA_VERSION);
	    if (index >= 0) {
		index = line.indexOf("\"");
		if (index >= 0 && line.length() >= index + 4) {
		    int version = Integer.parseInt(line.substring(index + 3,
			    index + 4));
		    isTigerAndHigher = (version >= 5);
		}
	    }
	}

	public void incomingData(char[] buffer, int length,
		java.net.InetAddress peer) {
	}

	public void invalidDataType(byte[] data, int length,
		java.net.InetAddress peer) {
	}

	public void waitingForData() {
	}
    }

    private static boolean isTigerAndHigher = false;
	
    /**
     * The method checks a version of the target VM and returns
     * <code>true</code> if VM version is 1.5
     * 
     * @param hostName
     *                The hostname that Agent Controller is running on
     * @param portNumber
     *                The port number that Agent Controller is running on
     * @param location
     *                The home directory of the process.
     * @return <code>true</code> if VM version is 1.5 or higher otherwise -
     *         <code>false</code>
     */
    public static boolean checkVMVersion(String hostName, int portNumber,
	    String location) {

	try {
	    IAgentController ac = AgentControllerPool.getInstance().getConnection(hostName, portNumber);

	    IProcess process = ac.createProcess();
	    process.setExecutable("java.exe");
	    process.setParameter("-version");
	    process.setLocation(location);

	    IConsole console = process.getConsole();
	    DataProcessor dp = new DataProcessor();
	    console.setDataProcessor(dp, IConsole.STDOUT);

	    process.addProcessListener(new IProcessListener() {

		public void processExited(IProcess process) {
		    synchronized (process) {
			process.notify();
		    }
		}

		public void processLaunched(IProcess process) {
		    // do nothing
		}
	    });
	    
	    process.launch();
	    synchronized (process) {
		process.wait();
	    }

	    return isTigerAndHigher;
	} catch (Exception exp) {
	    exp.printStackTrace();
	    System.out.println("Error occurred while connecting to " + hostName
		    + ":" + exp);
	}
	return false;
    }

    /**
     * The method checks a version of the target VM and returns
     * <code>true</code> if VM version is 1.5. This method is equivalent to
     * <code>checkVMversion(launcher.getHostName(), launcher.getPortNumber(), launcher.getLocation())</code> -
     * see the {@link #checkVMVersion(String, int, String) checkVMVersion}
     * method for more information.
     * 
     * @param launcher
     *                provides the expected getter methods whose values are
     *                needed to perform a profile launch.
     * @return <code>true</code> if VM version is 1.5 or higher otherwise -
     *         <code>false</code>
     */
    public static boolean checkVMVersion(ILauncherParameterProvider launcher) {
		String hostName = launcher.getHostName();
		int portNumber = launcher.getPortNumber();
		String location = launcher.getLocation();
		return checkVMVersion(hostName, portNumber, location);
	}
}
