/********************************************************************** 
 * Copyright (c) 2006, 2008 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: AutomationClientAdapter.java,v 1.9 2008/04/28 15:45:40 paules Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/

package org.eclipse.hyades.automation.client.adapters.shell;

import java.io.File;
import java.util.Properties;

import org.eclipse.hyades.automation.client.internal.resources.AutomationClientResourceBundle;

/**
 * Shell automation client adapter -- provides a command line interface to the
 * TPTP features that are exposed as automatable services. The test execution
 * service is not only functional but an exemplar automatable service.
 * 
 * This client automation adapter adapts a shell command line interface to the
 * standard Java automation client interface that is used to communicate with
 * the automation server. There are currently other adapters as well such as an
 * ant task adapter and a Java adapter. The pool of adapter types is not fixed
 * and additional developers can offer new ones with then immediate access to
 * all TPTP published services in a particular Eclipse instance.
 * 
 * @author Duwayne J. Morris
 * @author Scott E. Schneider
 */
public class AutomationClientAdapter {

	/**
	 * Arbitrary service that is either an improved test execution service
	 * implementation or a different service
	 */
	private static String arbitraryService;

	/**
	 * Eclipse home used for this operation, multiple eclipse homes might be
	 * available on a particular machine and can be selected via this specifier,
	 * without this the operation would not know which Eclipse instance to
	 * automate and command
	 */
	private static String eclipseHome;

	/**
	 * Indicates whether a fatal error has occurred during this operation
	 */
	private static boolean fatalError = false;

	/**
	 * The file separator for this particular platform is stored
	 */
	private static String fileSeparator;

	/**
	 * Indicates if an arbitrary service is desired, if this is true, then the
	 * expectation is that the arbitrary service instance variable would also be
	 * set to a valid value
	 */
	private static boolean isArbitraryService = false;

	/**
	 * The line separator for this particular platform is stored
	 */
	private static String lineSeparator;

	/**
	 * The message built up to display to the user, typically to list errors
	 * that might have occurred to the user
	 */
	private static StringBuffer message;

	/**
	 * The operating system type name in use for this operation
	 */
	private static String osName;

	/**
	 * The properties state for this operation instance, contains the parsed
	 * parameters and values input from the command line interface (the main
	 * input arguments in this class)
	 */
	private static Properties properties;

	/**
	 * The workspace to be used by the specified automatable service or the test
	 * execution service by default
	 */
	private static String workspace;

	/**
	 * Check required parameters, tries to calculate eclipse homes based on some
	 * simple heuristics. Eclipse home and workspace must be specified.
	 */
	private static void checkRequiredParameters() {

		eclipseHome = properties
				.getProperty(ICommandLineParameters.CMD_ECLIPSEHOME);
		workspace = properties
				.getProperty(ICommandLineParameters.CMD_WORKSPACE);
		arbitraryService = properties
				.getProperty(ICommandLineParameters.CMD_SERVICENAME);

		if (arbitraryService != null) {
			isArbitraryService = true;
		}

		if (eclipseHome == null) {
			int iStart;
			boolean bError = false;
			String strWorking = System.getProperty("user.dir");//$NON-NLS-1$
			iStart = strWorking.lastIndexOf("eclipse");//$NON-NLS-1$
			if (iStart > 0) {
				String strEclipseExe;
				strWorking = strWorking.substring(0, iStart);
				if (osName.startsWith("Windows") == true) {//$NON-NLS-1$
					strEclipseExe = "eclipse.exe";//$NON-NLS-1$
				} else {
					strEclipseExe = "eclipse";//$NON-NLS-1$
				}
				String strEclipse = strWorking + fileSeparator + fileSeparator
						+ strEclipseExe;//$NON-NLS-1$
				File test = new File(strEclipse);

				if (!test.exists())
					bError = true;
			} else {
				bError = true;
			}
			if (bError == true)
				message.append(AutomationClientResourceBundle.getString("AutomationClientAdapter.5") + //$NON-NLS-1$
						lineSeparator);

		} else {
			File file = new File(eclipseHome);
			if (!file.isDirectory()) {
				Object[] obj = new Object[1];
				obj[0] = eclipseHome;
				message.append(AutomationClientResourceBundle.getString("AutomationClientAdapter.6", obj));//$NON-NLS-1$
			}
		}

		if (arbitraryService != null)
			return;

		if (workspace != null) {
			File file = new File(workspace);
			if (!file.isDirectory()) {
				Object[] obj = new Object[1];
				obj[0] = workspace;
				message
						.append(AutomationClientResourceBundle.getString("AutomationClientAdapter.9", obj) + lineSeparator);//$NON-NLS-1$
			}
		}

	}

	/**
	 * The main entry point for command line execution, the command line
	 * interface arguments are parsed coming in as program arguments.
	 * 
	 * @param args
	 *            the program arguments being adapted to standard automation
	 *            client interface
	 */
	public static void main(String[] args) {

		if (args.length == 0
				|| args[0].compareTo(ICommandLineParameters.CMD_HELP1) == 0
				|| args[0].compareTo(ICommandLineParameters.CMD_HELP2) == 0
				|| args[0].compareTo(ICommandLineParameters.CMD_HELP3) == 0
				|| args[0].compareTo(ICommandLineParameters.CMD_HELP4) == 0) {
			printHelp();
			return;
		}

		fileSeparator = System.getProperty("file.separator");//$NON-NLS-1$
		lineSeparator = System.getProperty("line.separator");//$NON-NLS-1$
		osName = System.getProperty("os.name");//$NON-NLS-1$

		if (args[0].compareTo(ICommandLineParameters.COMMENT) == 0) {
			Object[] obj = new Object[1];
			obj[0] = args[1];
			System.out
					.println(AutomationClientResourceBundle.getString("AutomationClientAdapter.1", obj) + lineSeparator); //$NON-NLS-1$
		}

		properties = new Properties();
		CommandLineArgumentParser parser = new CommandLineArgumentParser();
		parser.parse(properties, args);
		parser.parseConfigFile(properties);
		message = parser.getLastError();
		if (message == null) {
			message = new StringBuffer();
		}

		checkRequiredParameters();

		if (message.length() > 0) {
			fatalError = true;
			message.insert(0, lineSeparator);
			message.append(lineSeparator
					+ AutomationClientResourceBundle.getString("AutomationClientAdapter.0") + //$NON-NLS-1$
					lineSeparator);
			System.out.println(message);
		}

		if (!fatalError) {
			writeMessagesToUser();
			Object[] obj = new Object[1];
			try {
				org.eclipse.hyades.automation.client.adapters.java.AutomationClientAdapter adapter = new org.eclipse.hyades.automation.client.adapters.java.AutomationClientAdapter(
						eclipseHome);
				obj[0] = eclipseHome;
				if (isArbitraryService == true)
					adapter.execute(arbitraryService, properties);
				else
					adapter.execute(ICommandLineParameters.SERVICE_NAME,
							properties);

			} catch (Exception e) {
				System.out.println(e);
			}
		}

	}

	/**
	 * Print help to the standard output, invoked based on --help, -help, ?, -?
	 * or simply by not specifying anything after automation-client-adapter on
	 * the command line.
	 */
	private static void printHelp() {
		StringBuffer bufHelp = new StringBuffer();
		bufHelp.append(AutomationClientResourceBundle.getString("AutomationClientAdapter.21")); //$NON-NLS-1$
		bufHelp.append(AutomationClientResourceBundle.getString("AutomationClientAdapter.22")); //$NON-NLS-1$
		bufHelp.append(AutomationClientResourceBundle.getString("AutomationClientAdapter.221")); //$NON-NLS-1$
		bufHelp.append(AutomationClientResourceBundle.getString("AutomationClientAdapter.222")); //$NON-NLS-1$
		bufHelp.append(AutomationClientResourceBundle.getString("AutomationClientAdapter.25")); //$NON-NLS-1$
		bufHelp.append(AutomationClientResourceBundle.getString("AutomationClientAdapter.23")); //$NON-NLS-1$
		bufHelp.append(AutomationClientResourceBundle.getString("AutomationClientAdapter.26")); //$NON-NLS-1$
		bufHelp.append(AutomationClientResourceBundle.getString("AutomationClientAdapter.27")); //$NON-NLS-1$
		bufHelp.append(AutomationClientResourceBundle.getString("AutomationClientAdapter.28")); //$NON-NLS-1$
		bufHelp.append(AutomationClientResourceBundle.getString("AutomationClientAdapter.29")); //$NON-NLS-1$
		bufHelp.append(AutomationClientResourceBundle.getString("AutomationClientAdapter.30")); //$NON-NLS-1$
		bufHelp.append(AutomationClientResourceBundle.getString("AutomationClientAdapter.31")); //$NON-NLS-1$
		bufHelp.append(AutomationClientResourceBundle.getString("AutomationClientAdapter.59")); //$NON-NLS-1$
		bufHelp.append(AutomationClientResourceBundle.getString("AutomationClientAdapter.60")); //$NON-NLS-1$
		bufHelp.append(AutomationClientResourceBundle.getString("AutomationClientAdapter.61")); //$NON-NLS-1$
		bufHelp.append(AutomationClientResourceBundle.getString("AutomationClientAdapter.53")); //$NON-NLS-1$
		bufHelp.append(AutomationClientResourceBundle.getString("AutomationClientAdapter.54")); //$NON-NLS-1$
		bufHelp.append(AutomationClientResourceBundle.getString("AutomationClientAdapter.55")); //$NON-NLS-1$
		bufHelp.append(AutomationClientResourceBundle.getString("AutomationClientAdapter.32")); //$NON-NLS-1$
		bufHelp.append(AutomationClientResourceBundle.getString("AutomationClientAdapter.33")); //$NON-NLS-1$
		bufHelp.append(AutomationClientResourceBundle.getString("AutomationClientAdapter.34")); //$NON-NLS-1$
		bufHelp.append(AutomationClientResourceBundle.getString("AutomationClientAdapter.35")); //$NON-NLS-1$
		bufHelp.append(AutomationClientResourceBundle.getString("AutomationClientAdapter.36")); //$NON-NLS-1$
		bufHelp.append(AutomationClientResourceBundle.getString("AutomationClientAdapter.37")); //$NON-NLS-1$
		bufHelp.append(AutomationClientResourceBundle.getString("AutomationClientAdapter.41")); //$NON-NLS-1$
		bufHelp.append(AutomationClientResourceBundle.getString("AutomationClientAdapter.42")); //$NON-NLS-1$
		bufHelp.append(AutomationClientResourceBundle.getString("AutomationClientAdapter.43")); //$NON-NLS-1$
		bufHelp.append(AutomationClientResourceBundle.getString("AutomationClientAdapter.50")); //$NON-NLS-1$
		bufHelp.append(AutomationClientResourceBundle.getString("AutomationClientAdapter.51")); //$NON-NLS-1$
		bufHelp.append(AutomationClientResourceBundle.getString("AutomationClientAdapter.52")); //$NON-NLS-1$
		bufHelp.append(AutomationClientResourceBundle.getString("AutomationClientAdapter.44")); //$NON-NLS-1$
		bufHelp.append(AutomationClientResourceBundle.getString("AutomationClientAdapter.45")); //$NON-NLS-1$
		bufHelp.append(AutomationClientResourceBundle.getString("AutomationClientAdapter.46")); //$NON-NLS-1$
		bufHelp.append(AutomationClientResourceBundle.getString("AutomationClientAdapter.47")); //$NON-NLS-1$
		bufHelp.append(AutomationClientResourceBundle.getString("AutomationClientAdapter.48")); //$NON-NLS-1$
		bufHelp.append(AutomationClientResourceBundle.getString("AutomationClientAdapter.49")); //$NON-NLS-1$
		bufHelp.append(AutomationClientResourceBundle.getString("AutomationClientAdapter.88")); //$NON-NLS-1$
		bufHelp.append(AutomationClientResourceBundle.getString("AutomationClientAdapter.89")); //$NON-NLS-1$
		bufHelp.append(AutomationClientResourceBundle.getString("AutomationClientAdapter.90")); //$NON-NLS-1$
		bufHelp.append(AutomationClientResourceBundle.getString("AutomationClientAdapter.91")); //$NON-NLS-1$
		bufHelp.append(AutomationClientResourceBundle.getString("AutomationClientAdapter.92")); //$NON-NLS-1$
		bufHelp.append(AutomationClientResourceBundle.getString("AutomationClientAdapter.93")); //$NON-NLS-1$
		bufHelp.append(AutomationClientResourceBundle.getString("AutomationClientAdapter.94")); //$NON-NLS-1$
		bufHelp.append(AutomationClientResourceBundle.getString("AutomationClientAdapter.85")); //$NON-NLS-1$
		bufHelp.append(AutomationClientResourceBundle.getString("AutomationClientAdapter.86")); //$NON-NLS-1$
		bufHelp.append(AutomationClientResourceBundle.getString("AutomationClientAdapter.87")); //$NON-NLS-1$
		bufHelp.append(AutomationClientResourceBundle.getString("AutomationClientAdapter.62")); //$NON-NLS-1$
		bufHelp.append(AutomationClientResourceBundle.getString("AutomationClientAdapter.63")); //$NON-NLS-1$
		bufHelp.append(AutomationClientResourceBundle.getString("AutomationClientAdapter.66")); //$NON-NLS-1$
		bufHelp.append(AutomationClientResourceBundle.getString("AutomationClientAdapter.67")); //$NON-NLS-1$
		bufHelp.append(AutomationClientResourceBundle.getString("AutomationClientAdapter.68")); //$NON-NLS-1$
		bufHelp.append(AutomationClientResourceBundle.getString("AutomationClientAdapter.69")); //$NON-NLS-1$
		bufHelp.append(AutomationClientResourceBundle.getString("AutomationClientAdapter.70")); //$NON-NLS-1$
		bufHelp.append(AutomationClientResourceBundle.getString("AutomationClientAdapter.71")); //$NON-NLS-1$
		bufHelp.append(AutomationClientResourceBundle.getString("AutomationClientAdapter.72")); //$NON-NLS-1$
		bufHelp.append(AutomationClientResourceBundle.getString("AutomationClientAdapter.73")); //$NON-NLS-1$
		bufHelp.append(AutomationClientResourceBundle.getString("AutomationClientAdapter.74")); //$NON-NLS-1$
		bufHelp.append(AutomationClientResourceBundle.getString("AutomationClientAdapter.79")); //$NON-NLS-1$
		bufHelp.append(AutomationClientResourceBundle.getString("AutomationClientAdapter.81")); //$NON-NLS-1$
		bufHelp.append(AutomationClientResourceBundle.getString("AutomationClientAdapter.84")); //$NON-NLS-1$
		System.out.println(bufHelp);
	}

	/**
	 * Write message to user based on the accumulated contents of the message
	 * instance variable.
	 */
	private static void writeMessagesToUser() {
		if (message.length() > 0) {
			message.append(AutomationClientResourceBundle.getString("AutomationClientAdapter.18") + //$NON-NLS-1$
					lineSeparator);
			System.out.println(message);
		}

	}

}
