/********************************************************************** 
 * Copyright (c) 2005, 2009 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: OutOfProcessStrategy.java,v 1.31 2009/03/02 13:45:41 paules Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/

package org.eclipse.hyades.automation.client.strategies;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.rmi.Naming;
import java.rmi.RemoteException;
import java.util.Properties;

import org.eclipse.hyades.automation.client.adapters.java.AutomationClientAdapter;
import org.eclipse.hyades.automation.client.internal.resources.AutomationClientResourceBundle;
import org.eclipse.hyades.automation.core.AbstractService;
import org.eclipse.hyades.automation.core.IAutomationServer;
import org.eclipse.hyades.automation.core.Service;
import org.eclipse.hyades.automation.core.utils.ProgressiveTask;
import org.eclipse.hyades.automation.core.utils.ProgressiveTask.Synchronicity;

/**
 * <p>The out-of-process strategy is used as the default and launches the
 * automation server and executes the automation command via the Eclipse
 * command-line in a separate Eclipse process.</p>
 * 
 * 
 * @author  Scott E. Schneider
 * @author  Paul Slauenwhite
 * @version February 26, 2009
 * @since   March 30, 2005
 */
class OutOfProcessStrategy extends AbstractExecutionStrategy {

	/**
	 * The out-of-process keep-alive specialization strategy, this specializes
	 * the non keep-alive strategy by only using the out-of-process strategy to
	 * start up the keep-alive server, then using the remote invocation methods
	 * to talk across process boundary
	 * 
	 * @author Scott E. Schneider
	 */
	public static class KeepAlive extends OutOfProcessStrategy {

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.hyades.automation.client.strategies.OutOfProcessStrategy
		 *      #execute(org.eclipse.hyades.automation.core.Service,
		 *      org.eclipse.hyades.automation.core.utils.ProgressiveTask.Synchronicity)
		 */
		public Object execute(Service service, Synchronicity synchronicity) {

			// Port to use for the automation server
			final int AUTOMATION_SERVER_PORT = 2099;

			// Retrieve automation server if running
			IAutomationServer server = this
					.connectToServer(AUTOMATION_SERVER_PORT);

			// If server not available then start it
			if (server == null) {
				this.startServer(service, AUTOMATION_SERVER_PORT);
			}

			// Keep trying until server is ready
			try {
				while (server == null) {
					Thread.sleep(3000);
					server = this.connectToServer(AUTOMATION_SERVER_PORT);
				}
			} catch (InterruptedException e) {
				e.printStackTrace();
			}

			// Execute service using non-null server
			try {

				// Execute service
				this.execute(service, synchronicity, server);

			} catch (RemoteException e) {
				e.printStackTrace();
			}

			// In exceptional cases
			return null;

		}

		/**
		 * Start RMI automation server on the specified port
		 * 
		 * @param the
		 *            service to use and base a new service call on
		 * @param port
		 *            the port to start the server on
		 */
		private void startServer(Service service, int port) {

			// Instantiate aonther instance of adapter
			AutomationClientAdapter adapter = new AutomationClientAdapter(
					service.getRoot());

			// Configure service properties for this service invocation
			Properties properties = new Properties();
			properties.setProperty("port", String.valueOf(port));//$NON-NLS-1$

			// Execute test results interrogation service
			adapter.execute("org.eclipse.hyades.execution.keepAlive",//$NON-NLS-1$
					properties, Synchronicity.ASYNCHRONOUS);

		}

		/**
		 * Execute service using the specified synchronicity setting and server
		 * instance (this is a direct connection use case where the
		 * out-of-process Eclipse instance is already running as an RMI server
		 * 
		 * @param service
		 *            the service to execute
		 * @param synchronicity
		 *            the synchronicity preference
		 * @param server
		 *            the non-null server instance
		 */
		private void execute(Service service, Synchronicity synchronicity,
				IAutomationServer server) throws RemoteException {
			if (server == null) {
				throw new IllegalArgumentException(
						AutomationClientResourceBundle.getString("OutOfProcessStrategy_AUTOMATION_SERVER_VALUE_")); //$NON-NLS-1$
			}
			Object object = server.run(service.createMemento());
			service.configure((Service.Memento) object);
			service.execute(synchronicity);
		}

		/**
		 * Attempt to connect directly to the automation server using remote
		 * method invocation
		 * 
		 * @param port
		 *            to attempt connection on
		 * @return the server to use, null if a server had timed out or had not
		 *         yet been running
		 */
		private IAutomationServer connectToServer(int port) {

			// Establish direct connection to automation server
			try {
				String uri = "//localhost:" + String.valueOf(port);//$NON-NLS-1$
				IAutomationServer server = (IAutomationServer) Naming
						.lookup(uri + "/server");//$NON-NLS-1$
				return server;
			} catch (Throwable t) {
				return null;
			}

		}

	}

	/**
	 * This debug flag is used by developers to turn on helpful debug support,
	 * in this case it sets up the eclipse process to be remotely debugged via
	 * Eclipse
	 */
	private static final boolean DEBUG = false;

	/**
	 * The service memento binary filename to use when passing service state
	 * from client to server, all non-transient state is sent
	 */
	private static final String SERVICE_MEMENTO_PREFIX = "memento";//$NON-NLS-1$

	/**
	 * The file name suffix to use when creating the temporary state memento.
	 */
	private static final String SERVICE_MEMENTO_SUFFIX = ".dat";//$NON-NLS-1$

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.automation.core.Service.Executable#execute(org.eclipse.hyades.automation.core.Service,
	 *      org.eclipse.hyades.automation.core.utils.ProgressiveTask.Synchronicity)
	 */
	public Object execute(Service service,
			ProgressiveTask.Synchronicity synchronicity) {

		try {

			// Create and write the memento object
			Service.Memento memento = service.createMemento();

			AbstractService sp = (AbstractService)service;
			Properties p = sp.getPublicProperties();
			
			
			//AbstractService abs = new AbstractService(memento);
			
			File file = File.createTempFile(
					OutOfProcessStrategy.SERVICE_MEMENTO_PREFIX,
					OutOfProcessStrategy.SERVICE_MEMENTO_SUFFIX);
			file.deleteOnExit();
			FileOutputStream fileOutput = new FileOutputStream(file);
			ObjectOutputStream objectOutput = new ObjectOutputStream(fileOutput);
			objectOutput.writeObject(memento);
			objectOutput.close();
			fileOutput.close();
			
			// Windows requires quotes surrounding path
			boolean useQuotes = false;
			if (System.getProperty("os.name").toLowerCase().indexOf("windows") != -1) {//$NON-NLS-1$ //$NON-NLS-2$
				useQuotes = true;
			}

			// Build the command-line string to launch Eclipse
			StringBuffer command = new StringBuffer(service.getRoot());
			command.append(File.separator);
			command.append("eclipse -nosplash ");//$NON-NLS-1$
			command.append("-application ");//$NON-NLS-1$
			command
					.append(AbstractExecutionStrategy.AUTOMATION_SERVER_IDENTIFIER
							+ " "); //$NON-NLS-1$
			String strWorkspace = p.getProperty("workspace");//$NON-NLS-1$
			if (strWorkspace == null)
				strWorkspace = p.getProperty("tptp.test.workspace");//$NON-NLS-1$
			if (strWorkspace != null){
				if ((useQuotes == true)  && (strWorkspace.startsWith("\"") == false)){ //$NON-NLS-1$
					strWorkspace = "\"" + strWorkspace + "\""; //$NON-NLS-1$ //$NON-NLS-2$
				}
				command.append("-data " + strWorkspace + " ");//$NON-NLS-1$ //$NON-NLS-2$
			}
			command.append("-vmargs ");//$NON-NLS-1$
			
		
			// Optional debugging of automation server
			if (OutOfProcessStrategy.DEBUG) {
				command.append("-Xdebug -Xnoagent -Djava.compiler=NONE ");//$NON-NLS-1$
				command
						.append("-Xrunjdwp:transport=dt_socket,server=y,suspend=y,address=8000 ");//$NON-NLS-1$
			}

			String strVMargs = p.getProperty("vmargs");//$NON-NLS-1$
			if (strVMargs == null)
				strVMargs = p.getProperty("tptp.automation.vmargs");//$NON-NLS-1$
			if (strVMargs != null)
				command.append(strVMargs + " ");			 //$NON-NLS-1$

			// Other -D options to pass in
			command.append("-D" //$NON-NLS-1$
					+ AbstractExecutionStrategy.AUTOMATION_COMMAND_IDENTIFIER);
			command.append("=execute ");//$NON-NLS-1$
			command
					.append("-D" //$NON-NLS-1$
							+ AbstractExecutionStrategy.AUTOMATION_SYNCHRONICITY_DIRECTIVE);
			command.append("=" + synchronicity.toString()); //$NON-NLS-1$
			command.append(" -D" //$NON-NLS-1$
					+ AbstractExecutionStrategy.AUTOMATION_DATA_IDENTIFIER);

			command.append("=" + (useQuotes ? "\"" : "") //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
					+ file.getCanonicalPath() + (useQuotes ? "\"" : "")); //$NON-NLS-1$ //$NON-NLS-2$
			


			// Render the string and execute the automation server
			String commandString = command.toString();

			// Debug output of command string used to launch Eclipse
			if (OutOfProcessStrategy.DEBUG) {
				System.out.println(commandString);
			}

			// Launch and wait for process to exit
			try {

				final Process process = Runtime.getRuntime()
						.exec(commandString);

				// Redirect process out to local system out
				AbstractExecutionStrategy.redirectConsole(process, process
						.getInputStream(), System.out);

				// Redirect process err to local system err
				AbstractExecutionStrategy.redirectConsole(process, process
						.getErrorStream(), System.err);

				// Wait for process to exit if synchronicity set to synchronous
				if (synchronicity == Synchronicity.SYNCHRONOUS) {

					// Launch and wait for process (0 is success)
					int returnCode = process.waitFor();

					// Restore memento and retrieve service affected properties
					service.configure(this.restoreMemento(null, file));

					// Return using integer wrapped return code
					return new Integer(returnCode);

				} else {
					new Thread() {
						public void run() {
							try {
								process.waitFor();
							} catch (Throwable t) {
								t.printStackTrace();
							}
						}
					}.start();
				}

			} catch (Throwable e) {
				e.printStackTrace();
			}
		} catch (IOException e) {
			e.printStackTrace();
		}

		// The automation execution failed, return null
		return null;

	}

	/**
	 * Restores the service memento, either returns the input candidate if it
	 * meets the requirements or restores one from the disk, this is the
	 * captured state of the service from the client (such as service specific
	 * properties)
	 * 
	 * @param candidate
	 *            a potential memento that is acquired by other means via
	 *            strategy
	 * @param dataFile
	 *            dataFile where if candidate is not null, will hold the file
	 *            that contains the persisted memento to restore
	 * 
	 * @return the opaque memento
	 */
	private Service.Memento restoreMemento(Object candidate, File dataFile) {

		// If the candidate is not null and is a memento then use it
		if (candidate != null && candidate instanceof Service.Memento) {
			return (Service.Memento) candidate;
		}

		// If not, load the memento from the disk given the filename
		try {

			// Construct return file name
			String returnFileName = dataFile.getCanonicalPath();
			int length = returnFileName.length();
			returnFileName = returnFileName.substring(0, length - 4).concat(
					"-out").concat(returnFileName.substring(length - 4)); //$NON-NLS-1$
 
			File returnFile = new File(returnFileName);

			//Request for the file to be deleted when the JVM exits:
			returnFile.deleteOnExit();
			
			// Read file contents (persisted objects)
			FileInputStream fileInput = new FileInputStream(returnFile);
			ObjectInputStream objectInput = new ObjectInputStream(fileInput);
			Service.Memento memento = (Service.Memento) objectInput
					.readObject();
			objectInput.close();
			fileInput.close();

			// Return reconstituted memento instance
			return memento;

		} catch (FileNotFoundException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		} catch (ClassNotFoundException e) {
			e.printStackTrace();
		}

		// In exceptional cases
		return null;

	}

}