/********************************************************************** 
 * Copyright (c) 2006 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: Service.java,v 1.7 2006/10/30 18:25:58 jptoomey Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/

package org.eclipse.hyades.automation.core;

import java.io.Serializable;
import java.util.List;
import java.util.Properties;

import org.eclipse.hyades.automation.core.utils.ProgressiveTask;

/**
 * The service interface is a key abstraction in the automation framework,
 * provides the central abstraction for constructing, defining and executing
 * services.
 * 
 * Service support in this release is provisional and may deviate from the
 * current API and design in subsequent releases
 * 
 * @author Scott E. Schneider
 */
public interface Service extends Serializable {

	/**
	 * The nested interface used to discover if this service exists and is
	 * runnable or not
	 * 
	 * @author Scott E. Schneider
	 */
	public static interface Discoverable {

		/**
		 * Discover published services available for command, other peer methods
		 * will be added to constrain the list membership returned.
		 * 
		 * @return the list of services available for command, the list contains
		 *         the service identifiers available and not the services
		 *         themselves -- a service identifier can then be requested and
		 *         executed once an identifier is known
		 */
		public List discover();

		/**
		 * Discover the given service on the server (within Eclipse)
		 * 
		 * @param identifier
		 *            the service identifier
		 * @return true if the service is available
		 */
		public boolean discover(String identifier);

	}

	/**
	 * The executable interface, used to execute services on both client and
	 * server
	 * 
	 * @author Scott E. Schneider
	 */
	public static interface Executable {

		/**
		 * The most used method within the services framework, a key method that
		 * executes a remote server (Eclipse instance)
		 * 
		 * @param service
		 *            the service to execute
		 * @return the service to execute
		 */
		public Object execute(Service service);

		/**
		 * The most used method within the services framework, a key method that
		 * executes a remote server (Eclipse instance)
		 * 
		 * @param service
		 *            the service to execute
		 * @param the
		 *            synchronicity setting to use
		 * @return the service to execute
		 */
		public Object execute(Service service,
				ProgressiveTask.Synchronicity synchronicity);

	}

	/**
	 * A marker interface to tag a memento and remark it as serializable
	 * 
	 * @author Scott E. Schneider
	 */
	public interface Memento extends Serializable {
	}

	/**
	 * The nested requestable interface, to request a service to command
	 * 
	 * @author Scott E. Schneider
	 */
	public static interface Requestable {

		/**
		 * Requests a service to command
		 * 
		 * @param identifier
		 *            the identifier to identify the service uniquely
		 * @return returns the service interface
		 */
		public Service request(String identifier);

	}

	/**
	 * Configure the properties of this service
	 * 
	 * @param properties
	 *            the service properties to attach to this service
	 */
	public void configure(Properties properties);

	/**
	 * Configure the properties of this service, using a memento that holds
	 * these properties -- populate the existing state with state from this
	 * memento instead of the equivalent of the set memento call which replaces
	 * the state with that memento. This method does a replace of the properties
	 * only.
	 * 
	 * @param memento
	 *            the memento to use as the source
	 */
	public void configure(Service.Memento memento);

	/**
	 * Creates a memento for storing and communicating the state between service
	 * instances (from outside Eclipse to inside Eclipse)
	 * 
	 * @return the opaque memento used to transfer state
	 */
	public Service.Memento createMemento();

	/**
	 * Execute the service
	 * 
	 * @return return the return value, the return value currently only contains
	 *         the java process execute command, unidirectional communication is
	 *         only possible
	 */
	public Object execute();

	/**
	 * Execute the service
	 * 
	 * @param synchronicity
	 *            specifies if this calls blocks or not
	 * @return return the return value, the return value currently only contains
	 *         the java process execute command, unidirectional communication is
	 *         only possible
	 */
	public Object execute(ProgressiveTask.Synchronicity synchronicity);

	/**
	 * Returns the identifier for this service
	 * 
	 * @return the service identifier
	 */
	public String getIdentifier();

	/**
	 * Returns the Eclipse root or home directory
	 * 
	 * @return returns the root directory
	 */
	public String getRoot();

	/**
	 * Sets the state of this service, given the opaque memento
	 * 
	 * @param state
	 *            the opaque memento, not opaque to the abstract service code
	 *            itself though
	 */
	public void setMemento(Service.Memento state);

}