/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: MutableObject.java,v 1.1 2005/10/07 14:49:18 sschneid Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package org.eclipse.hyades.automation.core.utils;

/**
 * A mutable object is mainly used for the accumulation of state from local or
 * anonymously defined inner classes into the surrounding context (to be
 * accessed variables of this type must be defined final at point-of-use). A
 * java.lang.Object is immutable once declared final and therefore cannot be
 * used for this purpose.
 * 
 * This is almost equivalent to a pointer to an object rather than just the
 * object itself.
 * 
 * @author Scott E. Schneider
 * @since 3.0.1
 */
public class MutableObject {
	/**
	 * The internally stored state of the mutable Object
	 */
	private Object value;

	/**
	 * Constructs a mutable Object
	 */
	public MutableObject() {
	}

	/**
	 * Constructs a mutable Object with the given value.
	 * 
	 * @param value
	 *            the value the internally stored state of the mutable Object
	 *            will be initialized to
	 */
	public MutableObject(Object value) {
		this.value = value;
	}

	/**
	 * It is a good practice to call the clear method after the mutable object
	 * is done being used, this will reset the internal value to null and allow
	 * the object to be garbage collected.
	 */
	public void clear() {
		this.value = null;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	public boolean equals(Object object) {
		if (object instanceof MutableObject) {
			if (object == this) {
				return true;
			}
			return (((MutableObject) object).value == this.value);
		}
		return false;
	}

	/**
	 * Returns the current value of the internally stored mutable Object state.
	 * 
	 * @return the current value of the mutable Object
	 */
	public Object get() {
		return this.value;
	}

	/**
	 * Combines the operations of get and then clear, to avoid memory leaks of
	 * underlying held on to object
	 * 
	 * @return same return as the get method would return
	 * 
	 * @see #get()
	 */
	public Object getAndClear() {
		Object object = this.get();
		this.clear();
		return object;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.lang.Object#hashCode()
	 */
	public int hashCode() {
		return this.value.hashCode();
	}

	/**
	 * Indicates if the value is set to null
	 * 
	 * @return null if the mutable object's contained value is set to null
	 */
	public boolean isNull() {
		return this.value == null;
	}

	/**
	 * Sets the value of this mutable Object.
	 * 
	 * @param value
	 *            the value to set the mutable Object to
	 */
	public void set(Object value) {
		this.value = value;
	}
}