/********************************************************************** 
 * Copyright (c) 2005, 2006 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: IFileManagerExtended.java,v 1.20 2006/07/29 01:57:51 sschneid Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/

package org.eclipse.hyades.execution.core.file;

import java.io.File;
import java.io.IOException;
import java.util.Arrays;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.hyades.internal.execution.core.file.ServerNotAvailableException;

/**
 * New and improved file manager interface that supports easier feature
 * extension in the future. The get, put and delete operations have been
 * improved with plurality support, progress monitor support and increased
 * type-safety for method signatures. File manager options are now available to
 * further fine-tune the semantics of a given method such as get or put. Options
 * are described in more detail under the option type-safe enumeration in this
 * interface.
 * 
 * @author Scott E. Schneider
 */
public interface IFileManagerExtended extends IFileManager {

	/**
	 * Cookie class used for the sake of uniqueness when identifing a set of
	 * state on the file server. Might need to add algorithm in the future to
	 * create a unique identifier. The to string method is called before this is
	 * sent over the wire, using that as the unique identifier. Override the to
	 * string method in the future for greater uniqueness.
	 * 
	 * @author Scott E. Schneider
	 */
	public static class Cookie {

		/**
		 * Shared cookie instance representing the void or null cookie, none
		 */
		public static final Cookie NONE = new Cookie();

		/**
		 * Creates a new cookie, used for operations that take an optional
		 * cookie, basically the cookie is just an identity object used to
		 * differentiate and group the effects of operations on the server
		 * (state that is stored on the server using the cookie as a key that
		 * can later be retrieved by the server using the same key)
		 * 
		 * @return the cookie instance created
		 */
		public static Cookie create() {
			return new Cookie();
		}

		/**
		 * Limit cookie instantiation to the factory method
		 * 
		 * @see #create()
		 */
		private Cookie() {
		}
	}

	/**
	 * A lightweight list class to use when passing one or many file identifier
	 * names into the file manager extended interface. An already existing
	 * string array can be used to construct an instance of a name list -- for
	 * those without a pre- existing array, list operations to add to the list
	 * are available.
	 * 
	 * @author Scott E. Schneider
	 */
	public static class FileIdentifierList {
		/**
		 * Filter interface, indicates if an element should be filtered out from
		 * the list when creating a new one
		 * 
		 * @author Scott E. Schneider
		 */
		public static interface Filter {
			/**
			 * If identifier pair A and B should be filtered when creating a new
			 * identifier list (condition can use just identifierA or both
			 * incoming argument values to determine if the values should be
			 * filtered and not be put into the new lists)
			 * 
			 * @param identifierA
			 *            an identifier next on list A
			 * @param identifierB
			 *            an identifier next on list B (matched in position with
			 *            A)
			 * @return true if this pair of identifiers should be ignored when
			 *         creating the new lists
			 */
			public boolean filter(String identifierA, String identifierB);
		}

		/**
		 * Stream-Unique IDentifier (SUID) of this class
		 */
		private static final long serialVersionUID = 3977859592212198201L;

		/**
		 * Create an empty file identifier list
		 * 
		 * @return an empty file identifier list ready for additions
		 */
		public static FileIdentifierList create() {
			return new FileIdentifierList();
		}

		/**
		 * Create a new file identifier list with one entry of the given name
		 * 
		 * @param name
		 *            the name of the element to add to the list initially
		 * @return the file identifier list
		 */
		public static FileIdentifierList create(String name) {
			return new FileIdentifierList(new String[] { name });
		}

		/**
		 * Create a new file identifier list with the entries specified
		 * 
		 * @param names
		 *            the elements to intially add to the list
		 * @return the new file identifier list
		 */
		public static FileIdentifierList create(String[] names) {
			return new FileIdentifierList(names);
		}

		/**
		 * Filter two identifiers lists into two resultant lists, the lists are
		 * assumed to be linked.
		 * 
		 * @param originalListA
		 *            the first original list
		 * @param resultantListA
		 *            the first resultant list
		 * @param originalListB
		 *            the second original list
		 * @param resultantListB
		 *            the second resultant list
		 * @param filter
		 *            the filter functor to use, indicates which elements are to
		 *            be filtered by invoking it once for each pair
		 */
		public static void filter(FileIdentifierList originalListA,
				FileIdentifierList resultantListA,
				FileIdentifierList originalListB,
				FileIdentifierList resultantListB, Filter filter) {

			// Iterate through the identifiers in identifier list A
			for (Iterator iteratorA = originalListA.iterator(), iteratorB = originalListB
					.iterator(); iteratorA.hasNext() && iteratorB.hasNext();) {

				// Pull identifiers from the lists
				String identifierA = (String) iteratorA.next();
				String identifierB = (String) iteratorB.next();

				// If identifier (from list A and list B) should not be filtered
				if (!filter.filter(identifierA, identifierB)) {
					resultantListA.add(identifierA);
					resultantListB.add(identifierB);
				}

			}

		}

		/**
		 * Filter two identifiers list leaving the original lists filtered down
		 * appropriately
		 * 
		 * @param listA
		 *            the first list
		 * @param listB
		 *            the matched list in the pair
		 * @param filter
		 *            the filter functor to use to determine if an identifier
		 *            should be filtered or not
		 */
		public static void filter(FileIdentifierList listA,
				FileIdentifierList listB, Filter filter) {

			// Enumerate list backwards to avoid problems with removes
			for (int i = listA.size() - 1; i >= 0; i--) {
				if (filter.filter((String) listA.get(i), (String) listB.get(i))) {
					listA.remove(i);
					listB.remove(i);
				}
			}

		}

		/**
		 * Internal list to store file identifiers
		 */
		private List list;

		/**
		 * Constructs a file identifier list using a linked list as the
		 * underlying data structure
		 */
		private FileIdentifierList() {
			this.list = new LinkedList();
		}

		/**
		 * Constructs a file identifier list using the given array of strings as
		 * names
		 * 
		 * @param names
		 *            the names that will initially form the list
		 */
		private FileIdentifierList(String[] names) {
			this.list = new LinkedList(Arrays.asList(names));
		}

		/**
		 * Add file to the file identifier list
		 * 
		 * @param file
		 *            the file identifying the file identifier to add to the
		 *            list
		 */
		public void add(File file) {
			if (file == null) {
				file = new File("");
			}
			this.list.add(file.getAbsolutePath());
		}

		/**
		 * Add file identifier to the list inserted at the given index
		 * 
		 * @param index
		 *            the index to insert the new element at
		 * @param identifier
		 *            the identifier to add
		 */
		public void add(int index, String identifier) {
			this.list.add(index, identifier);
		}

		/**
		 * Add an identifier to the file identifier list
		 * 
		 * @param identifier
		 *            the identifier to add to the list
		 */
		public void add(String identifier) {
			this.list.add(identifier);
		}

		/**
		 * Get the file identifier off the list
		 * 
		 * @param index
		 *            the index to get the identifier from
		 * @return the file identifier retrieved from the file identifier list
		 */
		public String get(int index) {
			return (String) this.list.get(index);
		}

		/**
		 * Returns a new file identifier list using this list as the basis for
		 * deriving the parent identifiers to return
		 */
		public FileIdentifierList getParentIdentifiers() {
			FileIdentifierList parentIdentifiers = FileIdentifierList.create();
			for (Iterator identifiers = list.iterator(); identifiers.hasNext();) {
				String identifier = (String) identifiers.next();
				File parentFile = new File(identifier).getParentFile();
				parentIdentifiers.add(parentFile);
			}
			return parentIdentifiers;
		}

		/**
		 * Get the array of strings representing the file identifiers on the
		 * list
		 * 
		 * @return the file identifier list as an array
		 */
		public String[] getArray() {
			String[] identifiers = new String[this.size()];
			return (String[]) this.list.toArray(identifiers);
		}

		/**
		 * Retrieve an iterator to iterate the file identifier list with
		 * 
		 * @return an iterator to iterate file identifiers
		 */
		public Iterator iterator() {
			return this.list.iterator();
		}

		/**
		 * Remove the file identifier at the given index
		 * 
		 * @param index
		 *            the index specifying the file identifier to remove
		 * @return the identifier removed from the list
		 */
		public String remove(int index) {
			return (String) this.list.remove(index);
		}

		/**
		 * Remove the file identifier specified by the given identifier argument
		 * 
		 * @param identifier
		 *            the identifier to remove from the file identifier list
		 * @return true if the item was removed
		 */
		public boolean remove(String identifier) {
			return this.list.remove(identifier);
		}

		/**
		 * The number of file identifiers comprising the list
		 * 
		 * @return an integer of the number of elements on the list
		 */
		public int size() {
			return this.list.size();
		}

	}

	/**
	 * Type-safe enumeration class used for file manager operations, not final
	 * since enumeration values can be added in the future by classes in the
	 * local package-only. This allows options to evolve in the future without
	 * the main file manager interface breaking compatability. Another class or
	 * interface in this package would be created that can extend this Option
	 * class in order to add more enumeration values and hence options.
	 * 
	 * @author Scott E. Schneider
	 */
	public static final class Option {
		/**
		 * Enable verification of operations performed, results in some type of
		 * enhanced validation handshaking that decreases performance but
		 * increase recoverability and reliability
		 */
		public static final Option ENABLE_VERIFICATION = new Option();

		/**
		 * A type safe constant representing no options are specified or
		 * required
		 */
		public static final Option[] NONE = new Option[0];

		/**
		 * Operate with atomicity in operations performed, all or nothing, if
		 * one of out one hundred files are not successful in a put file command
		 * operation then all changes are rolled back (which means n of one
		 * hundred already written files in the put file command operation are
		 * removed)
		 */
		public static final Option OPERATE_WITH_ATOMICITY = new Option();

		/**
		 * Overwrites existing files in a put file command operation, without
		 * this specified files will not be overwritten
		 */
		public static final Option OVERWRITE_ALL_EXISTING = new Option();

		/**
		 * If only a portion of the file operands are existing then overwrite
		 * all the files, if all the file operands already exist do not
		 * overwrite, this ensures all files are existing and matching
		 * (basically if any of the file operands already exist on the server
		 * they must all exist or not be considered valid)
		 */
		public static final Option OVERWRITE_PARTIAL_EXISTING = new Option();

		/**
		 * Use transfer packaging on operations, this enables commands to
		 * process their operands before and after the operation, for example,
		 * an compression packaging could be employed and therefore the operand
		 * content is compressed, sent over the wire and then uncompressed on
		 * the other side. Due to the already optimized speed of file transfer
		 * this might not be beneficial in most cases.
		 */
		public static final Option USE_TRANSFER_PACKAGING = new Option();

		/**
		 * Limit instantiation of option, used for type safe enumeration
		 * purposes only
		 */
		Option() {
		} // can be instantiated/subclassed from this package
	}

	/**
	 * Delete directories identified by the remote names specified. The
	 * directories are deleted recursively if they are not empty.
	 * 
	 * @param remoteIdentifiers
	 *            the remote directories to delete
	 * @throws IOException,
	 *             ServerNotAvailableException
	 */
	public void deleteDirectory(FileIdentifierList remoteIdentifiers)
			throws IOException, ServerNotAvailableException;

	/**
	 * Delete directories identified by the remote names specified using the
	 * given progress monitor for progres tracking and cancelability. The
	 * directories are deleted recursively if they are not empty.
	 * 
	 * @param remoteIdentifiers
	 *            the names identifying the remote directories to be deleted
	 * @param monitor
	 *            the progress monitor to use for progress query and canceling
	 * @throws IOException,
	 *             ServerNotAvailableException
	 */
	public void deleteDirectory(FileIdentifierList remoteIdentifiers,
			IProgressMonitor monitor) throws IOException,
			ServerNotAvailableException;

	/**
	 * Delete files identified by the state on the remote file server specified
	 * by the given state identifier (cookie)
	 * 
	 * @param cookie
	 *            identifies the state on the server for this command, for
	 *            example, when a put file command is executed, optionally a
	 *            state identifier (cookie) may be given, when this is the case,
	 *            the file server stores state concerning the particular put
	 *            file command -- it will know for example which files have been
	 *            created and then it will know how to use that state to delete
	 *            them again
	 * @param monitor
	 *            the progress monitor to use for progress query and canceling
	 * @throws IOException,
	 *             ServerNotAvailableException
	 */
	public void deleteFile(Cookie cookie, IProgressMonitor monitor)
			throws IOException, ServerNotAvailableException;

	/**
	 * Delete files identified by the remote names specified.
	 * 
	 * @param remoteIdentifiers
	 *            the remote files to delete
	 * @throws IOException,
	 *             ServerNotAvailableException
	 */
	public void deleteFile(FileIdentifierList remoteIdentifiers)
			throws IOException, ServerNotAvailableException;

	/**
	 * Delete files identified by the remote names specified using the given
	 * progress monitor for progres tracking and cancelability.
	 * 
	 * @param remoteIdentifiers
	 *            the names identifying the remote files to be deleted
	 * @param monitor
	 *            the progress monitor to use for progress query and canceling
	 * @throws IOException,
	 *             ServerNotAvailableException
	 */
	public void deleteFile(FileIdentifierList remoteIdentifiers,
			IProgressMonitor monitor) throws IOException,
			ServerNotAvailableException;

	/**
	 * Delete files identified by the state on the remote file server specified
	 * by the given state identifier (cookie)
	 * 
	 * @param cookie
	 *            identifies the state on the server for this command, for
	 *            example, when a put file command is executed, optionally a
	 *            state identifier (cookie) may be given, when this is the case,
	 *            the file server stores state concerning the particular put
	 *            file command -- it will know for example which files have been
	 *            created and then it will know how to use that state to delete
	 *            them again
	 * @throws IOException,
	 *             ServerNotAvailableException
	 */
	public void deleteFile(IFileManagerExtended.Cookie cookie)
			throws IOException, ServerNotAvailableException;

	/**
	 * Determine the server's reach, the server's reach is its ability to
	 * connect to other hosts (if a firewall blocks the server attempting
	 * connection then the reach cannot be extended for this particular host and
	 * port number) -- the primary use for this function is to determine if the
	 * agent controller machine can connect successfully to the workbench
	 * machine on a given host and port combination
	 * 
	 * @param host
	 *            the host to attempt to connect back on (from the server to
	 *            this client initiating the command)
	 * @param port
	 *            the port the server will attempt to connect back on
	 * @return true if the server can reach given the combination host and port
	 *         combination
	 */
	public boolean determineServerReach(String host, int port);

	/**
	 * Transfer files identified by remote names into files identified by local
	 * names, basically download files from remote to local.
	 * 
	 * @param localIdentifiers
	 *            the names identifying the local files to be copied to
	 * @param remoteNames
	 *            the names identifying the remote files to be copied from
	 * @throws IOException,
	 *             ServerNotAvailableException
	 */
	public void getFile(FileIdentifierList localIdentifiers,
			FileIdentifierList remoteIdentifiers) throws IOException,
			ServerNotAvailableException;

	/**
	 * Transfer files identified by remote names into files identified by local
	 * names, basically download files from remote to local. Use the given
	 * options to determine the exact semantics of the operation.
	 * 
	 * @param localIdentifiers
	 *            the names identifying the local files to be copied to
	 * @param remoteIdentifiers
	 *            the names identifying the remote files to be copied from
	 * @param options
	 *            the options to use for the file manager operation
	 * @throws IOException,
	 *             ServerNotAvailableException
	 */
	public void getFile(FileIdentifierList localIdentifiers,
			FileIdentifierList remoteIdentifiers,
			IFileManagerExtended.Option[] options) throws IOException,
			ServerNotAvailableException;

	/**
	 * Transfer files identified by remote names into files identified by local
	 * names, basically download files from remote to local. Use the given
	 * options to determine the exact semantics of the operation. Use the
	 * progress monitor to query status and support cancelability.
	 * 
	 * @param localIdentifiers
	 *            the names identifying the local files to be copied to
	 * @param remoteIdentifiers
	 *            the names identifying the remote files to be copied from
	 * @param options
	 *            the options to use for the file manager operation
	 * @param monitor
	 *            the progress monitor to use for progress and execution control
	 * @throws IOException,
	 *             ServerNotAvailableException
	 */
	public void getFile(FileIdentifierList localIdentifiers,
			FileIdentifierList remoteIdentifiers,
			IFileManagerExtended.Option[] options, IProgressMonitor monitor)
			throws IOException, ServerNotAvailableException;

	/**
	 * Transfer files identified by remote names into files identified by local
	 * names, basically download files from remote to local. Use the progress
	 * monitor to query status and support cancelability.
	 * 
	 * @param localIdentifiers
	 *            the names identifying the local files to be copied to
	 * @param remoteIdentifiers
	 *            the names identifying the remote files to be copied from
	 * @param monitor
	 *            the progress monitor to use for progress and execution control
	 * @throws IOException,
	 *             ServerNotAvailableException
	 */
	public void getFile(FileIdentifierList localIdentifiers,
			FileIdentifierList remoteIdentifiers, IProgressMonitor monitor)
			throws IOException, ServerNotAvailableException;

	/**
	 * Determines if the directory path specified resolves to an existing
	 * directory, used to validate if the given path represents a valid existing
	 * directory on the server
	 * 
	 * @param remoteIdentifier
	 *            the directory to validate on the server
	 * @return true if the directory exists on the server
	 * @throws IOException,
	 *             ServerNotAvailableException
	 */
	public boolean[] isDirectoryExistent(FileIdentifierList remoteIdentifier)
			throws IOException, ServerNotAvailableException;

	/**
	 * Determines if the directory path specified resolves to an existing
	 * directory, used to validate if the given path represents a valid existing
	 * directory on the server
	 * 
	 * @param remoteIdentifier
	 *            the directory to validate on the server
	 * @param monitor
	 *            the progres monitor for querying progress and canceling
	 * @return true if the directory exists on the server
	 * @throws IOException,
	 *             ServerNotAvailableException
	 */
	public boolean[] isDirectoryExistent(FileIdentifierList remoteIdentifier,
			IProgressMonitor monitor) throws IOException,
			ServerNotAvailableException;

	/**
	 * Indicates whether the file server is up and running and available for
	 * commands to be sent
	 * 
	 * @return true if the file server is ready for commands to be sent to it,
	 *         it should always be available unless some exception condition
	 *         brought it down, it is initialized and brought online as the
	 *         agent controller starts up
	 * @throws IOException
	 */
	public boolean isServerAvailable() throws IOException;

	/**
	 * List the contents of the file (which is a directory) identified by the
	 * remote names given. The contents of the directories are returned in a
	 * file location list that can be iterated through to retrieve the
	 * individual strings stored in the list. Each string represents the textual
	 * identity of the file or file directory that is contained in the
	 * directories identified by the given remote names.
	 * 
	 * @param remoteIdentifiers
	 *            the names identifying the remote file directories to list
	 *            contents of
	 * @return the contents of the specified file directories are returned to
	 *         the caller
	 * @throws IOException,
	 *             ServerNotAvailableException
	 */
	public FileIdentifierList listContent(FileIdentifierList remoteIdentifiers)
			throws IOException, ServerNotAvailableException;

	/**
	 * List the contents of the file (which is a directory) identified by the
	 * remote names given. The contents of the directories are returned in a
	 * file location list that can be iterated through to retrieve the
	 * individual strings stored in the list. Each string represents the textual
	 * identity of the file or file directory that is contained in the
	 * directories identified by the given remote names.
	 * 
	 * @param remoteIdentifiers
	 *            the names identifying the remote file directories to list
	 *            contents of
	 * @param monitor
	 *            the progres monitor for querying progress and canceling
	 * @return the contents of the specified file directories are returned to
	 *         the caller
	 * @throws IOException,
	 *             ServerNotAvailableException
	 */
	public FileIdentifierList listContent(FileIdentifierList remoteIdentifiers,
			IProgressMonitor monitor) throws IOException,
			ServerNotAvailableException;

	/**
	 * Modify the permissions of a file or file directory given the names that
	 * identify the remote files and/or file directories and a permission
	 * directive in the form of a platform-specific string. The
	 * platform-specific string will be interpreted by the server-side and
	 * examples of usage will be included here when available.
	 * 
	 * @param remoteIdentifiers
	 *            the names identifying the remote files and file directories to
	 *            apply the given permission directive to
	 * @param permissionDirective
	 *            the permission directive to pass, unchanged, to the
	 *            server-side file manager implementation
	 * @throws IOException,
	 *             ServerNotAvailableException
	 */
	public void modifyPermission(FileIdentifierList remoteIdentifiers,
			String permissionDirective) throws IOException,
			ServerNotAvailableException;

	/**
	 * Modify the permissions of a file or file directory given the names that
	 * identify the remote files and/or file directories and a permission
	 * directive in the form of a platform-specific string. The
	 * platform-specific string will be interpreted by the server-side and
	 * examples of usage will be included here when available.
	 * 
	 * @param remoteIdentifiers
	 *            the names identifying the remote files and file directories to
	 *            apply the given permission directive to
	 * @param permissionDirective
	 *            the permission directive to pass, unchanged, to the
	 *            server-side file manager implementation
	 * @param monitor
	 *            the progress monitor for querying progress and canceling
	 * @throws IOException,
	 *             ServerNotAvailableException
	 */
	public void modifyPermission(FileIdentifierList remoteIdentifiers,
			String permissionDirective, IProgressMonitor monitor)
			throws IOException, ServerNotAvailableException;

	/**
	 * Transfer files identified by local names into files identified by remote
	 * names, basically upload files from local to remote, the remote file
	 * identifiers can be relative or absolute paths on the remote file system.
	 * Relative paths will be made absolute to the temporary location deemed
	 * appropriate on the target, remote operating system. Relative paths can be
	 * 
	 * @param cookie
	 *            the state identifier (cookie) to use for this one particular
	 *            command invocation, can be saved and then passed in to other
	 *            methods to access this stored server state (such as the delete
	 *            file method)
	 * @param localIdentifiers
	 *            the names identifying the local files to be copied from
	 * @param remoteIdentifiers
	 *            the names identifying the remote files to be copied to,
	 *            supports relative paths in addition to absolute paths,
	 *            resolving the relative paths into absolute paths as required,
	 *            this argument serves as an in and out argument returning the
	 *            resolved relative file identifiers back (as absolute paths on
	 *            the remote system) -- use forward slashes '/' to separate path
	 *            segments to be as platform-agnostic as possible (this will
	 *            work on the majority of file systems).
	 * @throws IOException,
	 *             ServerNotAvailableException
	 */
	public void putFile(Cookie cookie, FileIdentifierList localIdentifiers,
			FileIdentifierList remoteIdentifiers) throws IOException,
			ServerNotAvailableException;

	/**
	 * Transfer files identified by local names into files identified by remote
	 * names, basically upload files from local to remote. Use the given options
	 * to determine the exact semantics of the operation.
	 * 
	 * @param cookie
	 *            the state identifier (cookie) to use for this one particular
	 *            command invocation, can be saved and then passed in to other
	 *            methods to access this stored server state (such as the delete
	 *            file method)
	 * @param localIdentifiers
	 *            the names identifying the local files to be copied from
	 * @param remoteIdentifiers
	 *            the names identifying the remote files to be copied to,
	 *            supports relative paths in addition to absolute paths,
	 *            resolving the relative paths into absolute paths as required,
	 *            this argument serves as an in and out argument returning the
	 *            resolved relative file identifiers back (as absolute paths on
	 *            the remote system)
	 * @param options
	 *            the options to use for the file manager operation
	 * @throws IOException,
	 *             ServerNotAvailableException
	 */
	public void putFile(Cookie cookie, FileIdentifierList localIdentifiers,
			FileIdentifierList remoteIdentifiers,
			IFileManagerExtended.Option[] options) throws IOException,
			ServerNotAvailableException;

	/**
	 * Transfer files identified by local names into files identified by remote
	 * names, basically upload files from local to remote. Use the given options
	 * to determine the exact semantics of the operation. The progress monitor
	 * is used for progress reporting and cancelability.
	 * 
	 * @param cookie
	 *            the state identifier (cookie) to use for this one particular
	 *            command invocation, can be saved and then passed in to other
	 *            methods to access this stored server state (such as the delete
	 *            file method)
	 * @param localIdentifiers
	 *            the names identifying the local files to be copied from
	 * @param remoteIdentifiers
	 *            the names identifying the remote files to be copied to,
	 *            supports relative paths in addition to absolute paths,
	 *            resolving the relative paths into absolute paths as required,
	 *            this argument serves as an in and out argument returning the
	 *            resolved relative file identifiers back (as absolute paths on
	 *            the remote system)
	 * @param options
	 *            the options to use for the file manager operation
	 * @param monitor
	 *            the progress monitor to use for reporting and canceling
	 * @throws IOException,
	 *             ServerNotAvailableException
	 */
	public void putFile(Cookie cookie, FileIdentifierList localIdentifiers,
			FileIdentifierList remoteIdentifiers,
			IFileManagerExtended.Option[] options, IProgressMonitor monitor)
			throws IOException, ServerNotAvailableException;

	/**
	 * Transfer files identified by local names into files identified by remote
	 * names, basically upload files from local to remote. The progress monitor
	 * is used for progress reporting and cancelability.
	 * 
	 * @param cookie
	 *            the state identifier (cookie) to use for this one particular
	 *            command invocation, can be saved and then passed in to other
	 *            methods to access this stored server state (such as the delete
	 *            file method)
	 * @param localIdentifiers
	 *            the names identifying the local files to be copied from
	 * @param remoteIdentifiers
	 *            the names identifying the remote files to be copied to,
	 *            supports relative paths in addition to absolute paths,
	 *            resolving the relative paths into absolute paths as required,
	 *            this argument serves as an in and out argument returning the
	 *            resolved relative file identifiers back (as absolute paths on
	 *            the remote system)
	 * @param monitor
	 *            the progress monitor to use for reporting and canceling
	 * @throws IOException,
	 *             ServerNotAvailableException
	 */
	public void putFile(Cookie cookie, FileIdentifierList localIdentifiers,
			FileIdentifierList remoteIdentifiers, IProgressMonitor monitor)
			throws IOException, ServerNotAvailableException;

	/**
	 * Transfer files identified by local names into files identified by remote
	 * names, basically upload files from local to remote.
	 * 
	 * @param localIdentifiers
	 *            the names identifying the local files to be copied from
	 * @param remoteIdentifiers
	 *            the names identifying the remote files to be copied to,
	 *            supports relative paths in addition to absolute paths,
	 *            resolving the relative paths into absolute paths as required,
	 *            this argument serves as an in and out argument returning the
	 *            resolved relative file identifiers back (as absolute paths on
	 *            the remote system)
	 * @throws IOException,
	 *             ServerNotAvailableException
	 */
	public void putFile(FileIdentifierList localIdentifiers,
			FileIdentifierList remoteIdentifiers) throws IOException,
			ServerNotAvailableException;

	/**
	 * Transfer files identified by local names into files identified by remote
	 * names, basically upload files from local to remote. Use the given options
	 * to determine the exact semantics of the operation.
	 * 
	 * @param localIdentifiers
	 *            the names identifying the local files to be copied from
	 * @param remoteIdentifiers
	 *            the names identifying the remote files to be copied to,
	 *            supports relative paths in addition to absolute paths,
	 *            resolving the relative paths into absolute paths as required,
	 *            this argument serves as an in and out argument returning the
	 *            resolved relative file identifiers back (as absolute paths on
	 *            the remote system)
	 * @param options
	 *            the options to use for the file manager operation
	 * @throws IOException,
	 *             ServerNotAvailableException
	 */
	public void putFile(FileIdentifierList localIdentifiers,
			FileIdentifierList remoteIdentifiers,
			IFileManagerExtended.Option[] options) throws IOException,
			ServerNotAvailableException;

	/**
	 * Transfer files identified by local names into files identified by remote
	 * names, basically upload files from local to remote. Use the given options
	 * to determine the exact semantics of the operation. The progress monitor
	 * is used for progress reporting and cancelability.
	 * 
	 * @param localIdentifiers
	 *            the names identifying the local files to be copied from
	 * @param remoteIdentifiers
	 *            the names identifying the remote files to be copied to,
	 *            supports relative paths in addition to absolute paths,
	 *            resolving the relative paths into absolute paths as required,
	 *            this argument serves as an in and out argument returning the
	 *            resolved relative file identifiers back (as absolute paths on
	 *            the remote system)
	 * @param options
	 *            the options to use for the file manager operation
	 * @param monitor
	 *            the progress monitor to use for reporting and canceling
	 * @throws IOException,
	 *             ServerNotAvailableException
	 */
	public void putFile(FileIdentifierList localIdentifiers,
			FileIdentifierList remoteIdentifiers,
			IFileManagerExtended.Option[] options, IProgressMonitor monitor)
			throws IOException, ServerNotAvailableException;

	/**
	 * Transfer files identified by local names into files identified by remote
	 * names, basically upload files from local to remote. The progress monitor
	 * is used for progress reporting and cancelability.
	 * 
	 * @param localIdentifiers
	 *            the names identifying the local files to be copied from
	 * @param remoteIdentifiers
	 *            the names identifying the remote files to be copied to,
	 *            supports relative paths in addition to absolute paths,
	 *            resolving the relative paths into absolute paths as required,
	 *            this argument serves as an in and out argument returning the
	 *            resolved relative file identifiers back (as absolute paths on
	 *            the remote system)
	 * @param monitor
	 *            the progress monitor to use for reporting and canceling
	 * @throws IOException,
	 *             ServerNotAvailableException
	 */
	public void putFile(FileIdentifierList localIdentifiers,
			FileIdentifierList remoteIdentifiers, IProgressMonitor monitor)
			throws IOException, ServerNotAvailableException;

	/**
	 * Resets the file manager, clearing any cached state, should have almost
	 * the same consequences as constructing a new file manager, as if for the
	 * first time for this client. Typically used in cases where the file
	 * manager factory is using multiple file managers to determine what version
	 * and protocol the server supports.
	 */
	public void reset();

}
