/**********************************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ExecutionComponentImpl.java,v 1.4 2008/03/20 18:49:53 dmorris Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.execution.core.impl;

import java.util.ArrayList;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.Vector;

import org.eclipse.hyades.execution.core.ExecutionComponentStateChangeEvent;
import org.eclipse.hyades.execution.core.IControlMessage;
import org.eclipse.hyades.execution.core.IExecutionComponent;
import org.eclipse.hyades.execution.core.IExecutionComponentStateChangeListener;
import org.eclipse.hyades.execution.core.INode;
import org.eclipse.hyades.execution.core.MessageDeliveryException;
import org.eclipse.hyades.execution.core.MessageProcessingException;
import org.eclipse.hyades.execution.core.internal.resources.CoreResourceBundle;

/**
 * @author rduggan
 *
 * To change the template for this generated type comment go to
 * Window>Preferences>Java>Code Generation>Code and Comments
 */
public abstract class ExecutionComponentImpl implements IExecutionComponent {

	private static final String VERSION="1.0";

	protected String name;
	protected String id;
	protected String type;
	protected String version;
	protected int state;
	protected INode node;

	protected Vector stateChangeListeners = new Vector();

	protected ArrayList children=new ArrayList();
	protected IExecutionComponent parent;

	public ExecutionComponentImpl() {
		
	}
	
	/**
	 * @see org.eclipse.hyades.execution.core.IExecutionComponent#init()
	 */
	public void init() {
		version=VERSION;
		state=NOT_CONFIGURED;
	}

	/**
	 * @see org.eclipse.hyades.execution.core.IExecutionComponent#getName()
	 */
	public String getName() {
		return name;
	}

	/**
	 * @see org.eclipse.hyades.execution.core.IExecutionComponent#getId()
	 */
	public String getId() {
		return id;
	}

	/**
	 * @see org.eclipse.hyades.execution.core.IExecutionComponent#getType()
	 */
	public String getType() {
		return type;
	}

	/**
	 * @see org.eclipse.hyades.execution.core.IExecutionComponent#getVersion()
	 */
	public String getVersion() {
		return version;
	}

	/**
	 * @see org.eclipse.hyades.execution.core.IExecutionComponent#getParent()
	 */
	public IExecutionComponent getParent() {
		return parent;
	}
	
	/**
	 * @see org.eclipse.hyades.execution.core.IExecutionComponent#setParent(org.eclipse.hyades.execution.core.IExecutionComponent)
	 */
	public void setParent(IExecutionComponent parent) {
		this.parent=parent;
	}

	/**
	 * @see org.eclipse.hyades.execution.core.IExecutionComponent#getChildren()
	 */
	public IExecutionComponent[] getChildren() {
		synchronized(children) {
			IExecutionComponent[] array=new IExecutionComponent[children.size()];
			Iterator iter=children.iterator();
			for(int i=0; iter.hasNext(); i++) {
				array[i]=(IExecutionComponent)iter.next();
			}
			return array;
		}
	}

	/**
	 * @see org.eclipse.hyades.execution.core.IExecutionComponent#getChildById(java.lang.String)
	 */
	public IExecutionComponent getChildById(String id) {
		synchronized(children) {
			Iterator iter=children.iterator();
			while(iter.hasNext()) {
				IExecutionComponent current=((IExecutionComponent)iter.next());
				if(current.getId().equals(id)) {
					return current;
				}
			}
		}
		return null;
	}

	/**
	 * @see org.eclipse.hyades.execution.core.IExecutionComponent#getChildByName(java.lang.String)
	 */
	public IExecutionComponent getChildByName(String name) {
		synchronized(children) {
			Iterator iter=children.iterator();
			while(iter.hasNext()) {
				IExecutionComponent current=((IExecutionComponent)iter.next());
				if(current.getName().equals(name)) {
					return current;
				}
			}
		}
		return null;
	}

	/**
	 * @see org.eclipse.hyades.execution.core.IExecutionComponent#addChild(java.lang.String, java.lang.String, java.lang.String)
	 */
	public void addChild(IExecutionComponent child) {
		children.add(child);
		child.setParent(this);
	}

	/**
	 * @see org.eclipse.hyades.execution.core.IExecutionComponent#getNode()
	 */
	public INode getNode() {
		return node;
	}

	/**
	 * @see org.eclipse.hyades.execution.core.IExecutionComponent#getState()
	 */
	public int getState() {
		return state;
	}

	/**
	 * @see org.eclipse.hyades.execution.core.IExecutionComponent#addExecutionComponentStateChangeListener(org.eclipse.hyades.execution.core.IExecutionComponentStateChangeListener)
	 */
	public void addExecutionComponentStateChangeListener(IExecutionComponentStateChangeListener listener) {
		synchronized(stateChangeListeners) {
			stateChangeListeners.add(listener);
		}
	}

	/**
	 * @see org.eclipse.hyades.execution.core.IExecutionComponent#removeExecutionComponentStateChangeListener(org.eclipse.hyades.execution.core.IExecutionComponentStateChangeListener)
	 */
	public void removeExecutionComponentStateChangeListener(IExecutionComponentStateChangeListener listener) {
		synchronized(stateChangeListeners) {
			stateChangeListeners.remove(listener);
		}
	}

	/**
	 * @see org.eclipse.hyades.execution.core.IExecutionComponent#fireStateChangeEvent(org.eclipse.hyades.execution.core.ExecutionComponentStateChangeEvent)
	 */
	public void fireStateChangeEvent(ExecutionComponentStateChangeEvent event) {
		synchronized(stateChangeListeners) {
			state=event.getState();
			Enumeration e=stateChangeListeners.elements();
			while(e.hasMoreElements()) {
				IExecutionComponentStateChangeListener listener=(IExecutionComponentStateChangeListener)e.nextElement();
				listener.stateChanged(event);
			}
			// If the component has transitioned to the DEAD state, by definition
			// it can never change state again.  Therefore, we will free up any
			// objects held in memory by listeners that reference them.
			if (state == IExecutionComponent.DEAD) {
				stateChangeListeners.clear();
			}
		}
	}
	
	/**
	 * This is a default impementation which does not handle any messages directly and
	 * will always throw a MessageProcessingException.  Subclasses should provide their
	 * own implementations and if they do not know how to deal with the message should
	 * delegate the call to the superclass.
	 * 
	 * @see org.eclipse.hyades.execution.core.IExecutionComponent#handleMessage(org.eclipse.hyades.execution.core.IControlMessage)
	 */
	public void handleMessage(IControlMessage message) throws MessageProcessingException {
		throw new MessageProcessingException(CoreResourceBundle.ExecutionComponentImpl_NO_HANDLER_EXCEPTION_);
	}
	/**
	 * @see org.eclipse.hyades.execution.core.IExecutionComponent#sendMessage(org.eclipse.hyades.execution.core.IControlMessage)
	 */
	public void sendMessage(IControlMessage message) throws MessageDeliveryException {
		throw new MessageDeliveryException(CoreResourceBundle.ExecutioncomponentImpl_CLASS_SEND_MESSAGE_EXCEPTION_);
	}

	/**
	 * @see org.eclipse.hyades.execution.core.IExecutionComponent#removeChild(org.eclipse.hyades.execution.core.IExecutionComponent)
	 */
	public void removeChild(IExecutionComponent child) {
		children.remove(child);
	}

}
