/**********************************************************************
Copyright (c) 2005, 2010 IBM Corporation and others.
All rights reserved. This program and the accompanying materials
are made available under the terms of the Eclipse Public License v1.0
which accompanies this distribution, and is available at
http://www.eclipse.org/legal/epl-v10.html
$Id: ExecutionEnvironmentImpl.java,v 1.8 2010/05/07 16:18:58 jwest Exp $

Contributors:
 IBM Corporation - initial implementation
**********************************************************************/

package org.eclipse.hyades.execution.core.impl;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.Map;

import org.eclipse.hyades.execution.core.ExecutionComponentStateException;
import org.eclipse.hyades.execution.core.IExecutionEnvironment;
import org.eclipse.hyades.execution.core.IOrderedProperty;

/**
 * @author IBM
 *
 * This is an extension of the ExecutableComponentImpl class and implements the IExecutableEnvironment interface
 */
public class ExecutionEnvironmentImpl extends ExecutionComponentImpl implements IExecutionEnvironment {

	/* This is loaded at when the first instance of this class is instantiated.  This MUST not
	 * be touched by any extendors of this class.
	 */
	
	/* The local copy of the modified environment */ 
	protected static ArrayList originalProcessEnvironment = new ArrayList(); 
	protected ArrayList processEnvironment=new ArrayList();
	
	/* Whether the native impls are available; the value of this is fully
	 * known at class initialization time and is set by a static initializer 
	 * in this class.  Is immutable once set in the static initializer to
	 * avoid potential threading issues across instances of this class.
	 */
    private static final boolean nativesAvailable;
	
    /* Try and load the native library.  If this fails we will ignore the error;
     * first assign to local variable to determine if natives are loadable and then
     * assign blank final natives available static variable.
     */
    static {
        boolean nativesLoadable = false;
        try {
            System.loadLibrary("hcclco");//$NON-NLS-1$
            System.loadLibrary("hclaunch");//$NON-NLS-1$
            nativesLoadable = true;
        }
        catch (Throwable e) {
        }
        nativesAvailable = nativesLoadable;
    }

	/**
	 * Initialization routine
	 */
	public void init() {
		IOrderedProperty[] tmpEnv; /* This will receive a fixed-sized block from JNI */

		/* If this is the first time we are being created then we need to query the environment
		 * of this process
		 */
		if(originalProcessEnvironment.size()==0) {

			/* Get the properties from the system and count how many are actually there */
			tmpEnv = getNativeSystemEnvironment();
			
			/* Load the initialProcessEnvironment and processEnvironment array */
			for(int i = 0; i < tmpEnv.length; i++) {
				try {
					if(tmpEnv[i]!=null) {
						originalProcessEnvironment.add((IOrderedProperty)((OrderedPropertyImpl)tmpEnv[i]).clone());
					}
					else {
						break;
					}
				} catch (CloneNotSupportedException e1) {
					
				}
			}
		}
		
		for(int j=0; j<originalProcessEnvironment.size(); j++) {
			try {
				processEnvironment.add((IOrderedProperty)(((OrderedPropertyImpl)originalProcessEnvironment.get(j)).clone()));
			}
			catch (CloneNotSupportedException e1) {		
			}
		}
    }

	/**
	 * @see org.eclipse.hyades.execution.core.IExecutionEnvironment#getEnv()
	 */
	public IOrderedProperty[] getEnv() throws ExecutionComponentStateException {
		synchronized(processEnvironment) {
			IOrderedProperty[] validEnv=new IOrderedProperty[processEnvironment.size()];
			for(int i = 0; i < validEnv.length; i++) {
				try {
					validEnv[i] = (IOrderedProperty)((OrderedPropertyImpl)processEnvironment.get(i)).clone();
					//System.out.println(((OrderedPropertyImpl)processEnvironment.get(i)).getName()+((OrderedPropertyImpl)processEnvironment.get(i)).getValues()[0]);
				} catch (CloneNotSupportedException e) {
				}
			}
			return validEnv; /* Return only non-null properties */
		}
	}
	
	/**
	 * @see org.eclipse.hyades.execution.core.IExecutionEnvironment#getEnvByName(java.lang.String)
	 */
	public IOrderedProperty getEnvByName(String name) throws ExecutionComponentStateException {
		synchronized(processEnvironment) {
			Iterator iter=processEnvironment.iterator();
			for(int offset=0; iter.hasNext(); offset++) {
				IOrderedProperty current=(IOrderedProperty)iter.next();
				if(current.getName().equalsIgnoreCase(name)) {
					return current;
				}
			}	
		}
		return null;
	}

	/**
	 * @see org.eclipse.hyades.execution.core.IExecutionEnvironment#setEnv(org.eclipse.hyades.execution.core.IOrderedProperty)
	 */
	public void setEnv(IOrderedProperty[] env) throws ExecutionComponentStateException {
		processEnvironment.clear();
		for(int i=0;i<env.length; i++){
			processEnvironment.add(env[i]);
		}		
	}

	/**
	 * @see org.eclipse.hyades.execution.core.IExecutionEnvironment#addEnv(org.eclipse.hyades.execution.core.IOrderedProperty)
	 */
	public void addEnv(IOrderedProperty env) throws ExecutionComponentStateException {
		processEnvironment.add(env);
	}

	/**
	 * @see org.eclipse.hyades.execution.core.IExecutionEnvironment#removeProperty(java.lang.String)
	 */
	public IOrderedProperty removeProperty(String name) throws ExecutionComponentStateException {
		synchronized(processEnvironment) {
			Iterator iter=processEnvironment.iterator();
			for(int offset=0; iter.hasNext(); offset++) {
				IOrderedProperty current=(IOrderedProperty)iter.next();
				if(current.getName().equalsIgnoreCase(name)) {
					processEnvironment.remove(offset);
					return current;
				}
			}
			return null;
		}
	}

	/**
	 * @see org.eclipse.hyades.execution.core.IExecutionEnvironment#addEnv(org.eclipse.hyades.execution.core.IOrderedProperty)
	 */
	public void setEnv(IOrderedProperty env) throws ExecutionComponentStateException {
		/* Search for named property and replace it */
		synchronized(processEnvironment) {
			Iterator iter=processEnvironment.iterator();
			for(int offset=0; iter.hasNext(); offset++) {
				IOrderedProperty current=(IOrderedProperty)iter.next();
				if(current.getName().equalsIgnoreCase(env.getName())) {
					processEnvironment.remove(offset);
					processEnvironment.add(offset, env);
					return;
				}
			}
		}
		addEnv(env);
	}
	
	private IOrderedProperty[] getNativeSystemEnvironment() {
		
		Map map = System.getenv();
		ArrayList l = new ArrayList();
		
		// For each environment variable in the System env var list...
		for(Iterator it = map.entrySet().iterator(); it.hasNext();) {
			Map.Entry e = (Map.Entry)it.next();

			// Extract the Name/Value from the env var and store in an OrderedPropertyImpl
			OrderedPropertyImpl opi = new OrderedPropertyImpl();
			opi.setName((String)e.getKey());
			opi.appendValue((String)e.getValue());
			l.add(opi);
		}
		
		// Convert the result into an array
		OrderedPropertyImpl result[] = new OrderedPropertyImpl[l.size()];
		int x = 0;
		
		for(Iterator i = l.iterator(); i.hasNext(); ) {
			OrderedPropertyImpl o = (OrderedPropertyImpl)i.next();
			result[x] = o;
			x++;
		}
		
		return result;

//		if(nativesAvailable) {
//			return (IOrderedProperty[])getNativeSystemEnvironment0();
//		}
//		return new IOrderedProperty[0];
	}
	
	// ********** NATIVE METHODS  **********
	
	private native Object[] getNativeSystemEnvironment0();
}
