/********************************************************************** 
 * Copyright (c) 2005, 2008 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: ScopedChannelClassLoader.java,v 1.4 2008/03/20 18:49:52 dmorris Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/

package org.eclipse.hyades.execution.core.loader;

import java.io.IOException;
import java.io.InputStream;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.nio.ByteBuffer;
import java.nio.channels.Channels;
import java.nio.channels.ReadableByteChannel;
import java.nio.channels.WritableByteChannel;
import java.util.HashMap;
import java.util.Observable;
import java.util.regex.Pattern;

import org.eclipse.hyades.internal.execution.core.file.communicator.ChannelCommunicatorFactory;
import org.eclipse.hyades.internal.execution.core.file.communicator.IChannelCommunicator;

/**
 * This custom scoped class loader loads only classes for a given scope -- for
 * all other elements, they are delegated oto the parent class loader (which
 * means they are out-of-scope and not this classloader's responsibility). The
 * elements in scope will be loaded using a simple protocol over the readable
 * and writable byte channels.
 * 
 * The scope is defined by a regular expression that will be used to match the
 * elemnents that this classloader is responsible for loading -- unlike typical
 * classloader delegation, this classloader will take precedence over the parent
 * classloader and inverts the typical delegation scheme.
 * 
 * This class loader inner classes consumer and provider are observables and
 * observers will be notified for each class that is acted upon (in the case of
 * consumer, this means each class that is downloaded by the server from the
 * client, in the case of the provider, this means each class that was uploaded
 * to the server from the client) -- the context class loaders being cached
 * reducing uploading and downloading over the time of the client context.
 * 
 * @author Scott E. Schneider
 */
public class ScopedChannelClassLoader extends Observable {

	/**
	 * Abstract class that all control directives extend from
	 * 
	 * @author Scott E. Schneider
	 */
	private static abstract class AbstractControlDirective {
		/**
		 * The begin control directive, supports the visitor pattern for type
		 * differentiating behavior
		 * 
		 * @author Scott E. Schneider
		 */
		public static class BeginControlDirective extends AbstractControlDirective {
			/*
			 * (non-Javadoc)
			 * 
			 * @see org.eclipse.ohyades.execution.core.cl.ScopedChannelClassLoader.AbstractControlDirective
			 *      #accept(org.eclipse.hyades.execution.core.cl.ScopedChannelClassLoader.ControlDirectiveVisitor)
			 */
			public boolean accept(ControlDirectiveVisitor visitor) throws IOException {
				return visitor.visit(this);
			}
		}

		/**
		 * The control directive visitor abstract class defining an interface
		 * that all visitors will implement
		 * 
		 * @author Scott E. Schneider
		 */
		private abstract static class ControlDirectiveVisitor {
			/**
			 * Visit the begin directive
			 * 
			 * @param directive
			 *            the begin directive
			 * @return a boolean to pass back, allows the visitor controller to
			 *         interpret
			 */
			abstract boolean visit(BeginControlDirective directive) throws IOException;

			/**
			 * Visit the end directive
			 * 
			 * @param directive
			 *            the begin directive
			 * @return a boolean to pass back, allows the visitor controller to
			 *         interpret
			 */
			abstract boolean visit(EndControlDirective directive) throws IOException;

			/**
			 * Visit the identify directive
			 * 
			 * @param directive
			 *            the identify directive
			 * @return a boolean to pass baqck, allows the visitor controller to
			 *         interpret
			 * @throws IOException
			 */
			abstract boolean visit(IdentifyControlDirective directive) throws IOException;

			/**
			 * Visit the load directive
			 * 
			 * @param directive
			 *            the begin directive
			 * @return a boolean to pass back, allows the visitor controller to
			 *         interpret
			 */
			abstract boolean visit(LoadControlDirective directive) throws IOException;

		}

		/**
		 * The end control directive, supports the visitor pattern for type
		 * differentiating behavior
		 * 
		 * @author Scott E. Schneider
		 */
		public static class EndControlDirective extends AbstractControlDirective {
			/*
			 * (non-Javadoc)
			 * 
			 * @see org.eclipse.hyades.execution.core.cl.ScopedChannelClassLoader.AbstractControlDirective
			 *      #accept(org.eclipse.hyades.execution.core.cl.ScopedChannelClassLoader.ControlDirectiveVisitor)
			 */
			boolean accept(ControlDirectiveVisitor visitor) throws IOException {
				return visitor.visit(this);
			}
		}

		/**
		 * The identify control directive, supports the visitor pattern for type
		 * differentiating behavior
		 * 
		 * @author Scott E. Schneider
		 */
		public static class IdentifyControlDirective extends AbstractControlDirective {

			public boolean accept(ControlDirectiveVisitor visitor) throws IOException {
				return visitor.visit(this);
			}
		}

		/**
		 * The load control directive, supports the visitor pattern for type
		 * differentiating behavior
		 * 
		 * @author Scott E. Schneider
		 */
		private static class LoadControlDirective extends AbstractControlDirective {
			LoadControlDirective() {
				super();
			};

			/*
			 * (non-Javadoc)
			 * 
			 * @see org.eclipse.hyades.execution.core.cl.ScopedChannelClassLoader.AbstractControlDirective
			 *      #accept(org.eclipse.hyades.execution.core.cl.ScopedChannelClassLoader.ControlDirectiveVisitor)
			 */
			boolean accept(ControlDirectiveVisitor visitor) throws IOException {
				return visitor.visit(this);
			}
		}

		public AbstractControlDirective() {
		}

		/**
		 * Accept the visitor and callback on it via double-dispatch
		 * 
		 * @param visitor
		 *            the visitor to be called back on (call dispatched per
		 *            type)
		 * @return a boolean interpreted by the visitor instantiating context
		 */
		abstract boolean accept(ControlDirectiveVisitor visitor) throws IOException;

	}

	/**
	 * The scoped channel class loader consumer class is used by the consumer of
	 * the classes that will be served up through the channel from the providing
	 * instance of the scoped channel class loader
	 * 
	 * @author Scott E. Schneider
	 */
	public static class Consumer extends Observable {

		/**
		 * Context class loader is a class loader in the context of the consumer
		 * provider relationship (for each unique provider, there will be a
		 * separate context class loader instance in the provider cache, created
		 * and stored on demand as required)
		 * 
		 * The class loader keeps an internal cache of classes it has loaded to
		 * avoid loading it more than once via the load class method -- this is
		 * the state that is necessary to cache per unique provider. This
		 * context class loader should not be confused with the context class
		 * loader associated with a thread's context.
		 * 
		 * @author Scott E. Schneider
		 */
		private static class ContextClassLoader extends ClassLoader {
			/**
			 * Thread local wrapping a communicator value per thread (each
			 * isolated)
			 */
			private final InheritableThreadLocal communicator = new InheritableThreadLocal();

			/**
			 * Thread local wrapping a consumer instance per thread (each
			 * isolated)
			 */
			private final InheritableThreadLocal consumer = new InheritableThreadLocal();

			/**
			 * Thread local wrapping an isEnabled value per thread (each
			 * isolated)
			 */
			private final InheritableThreadLocal isEnabled = new InheritableThreadLocal();

			/**
			 * Thread local wrapping a scopingPattern value per thread (each
			 * isolated)
			 */
			private final InheritableThreadLocal scopingPattern = new InheritableThreadLocal();

			/**
			 * Consume resources from the provider on the other end of the
			 * readable channel, using the writable channel for control
			 * 
			 * @param name
			 *            the name of the class/resource to consume
			 * @return the consumed class in bytes or null if unable to retrieve
			 *         the bytes
			 */
			private byte[] consume(String name) {

				// Consume the bytes that form a class required
				try {

					// Send load control directive identity
					this.getCommunicator().send(AbstractControlDirective.LoadControlDirective.class.getName());

					// Send the name of the class to load
					this.getCommunicator().send(name);

					// Read the size of the resource byte response that is sent
					int classDataLength = this.getCommunicator().receiveInt();

					// Read byte response from channel and load into byte array
					byte[] data = this.getCommunicator().receiveBytes(classDataLength);

					/*
					 * Indicate class downloaded from providing client and flip
					 * changed state to prep for notify observer invocation
					 */
					this.getConsumer().setChanged();
					this.getConsumer().notifyObservers(name);

					// Returns the class data as bytes
					return data;

				} catch (Throwable t) {

					// Return null on exceptions
					return null;

				}

			}

			/**
			 * Retrieves the communicator to use by the class loader for the
			 * calling thread
			 * 
			 * @return the communicator to use for channel interaction
			 */
			private IChannelCommunicator getCommunicator() {
				return (IChannelCommunicator) this.communicator.get();
			}

			/**
			 * Returns the consummer appropriate for this context
			 */
			private ScopedChannelClassLoader.Consumer getConsumer() {
				return (ScopedChannelClassLoader.Consumer) this.consumer.get();
			}

			/**
			 * Retrieves the scoping pattern for the calling thread
			 * 
			 * @return the pre-compiled regular expression pattern
			 */
			private Pattern getScopingPattern() {
				return (Pattern) this.scopingPattern.get();
			}

			/**
			 * Determines if the class loader is enabled for the calling thread
			 * 
			 * @return true if the class loader is enabled
			 */
			private boolean isEnabled() {
				Boolean isEnabled = (Boolean) this.isEnabled.get();
				return isEnabled == null ? false : isEnabled.booleanValue();
			}

			/*
			 * (non-Javadoc)
			 * 
			 * @see java.lang.ClassLoader#loadClass(java.lang.String, boolean)
			 */
			public Class loadClass(String name, boolean resolve) throws ClassNotFoundException {
				if (this.responsibleFor(name)) {
					Class loadedClass = this.findLoadedClass(name);
					if (loadedClass == null) {
						byte[] bytes = this.loadClassData(name);
						loadedClass = defineClass(name, bytes, 0, bytes.length);
					}
					if (resolve == true) {
						this.resolveClass(loadedClass);
					}
					return loadedClass;
				} else {
					return super.loadClass(name, resolve);
				}
			}

			/**
			 * Provides the ability to load the class with the specified name,
			 * returning a byte array of the class contents
			 * 
			 * @param name
			 *            the name of the class to load
			 * @return the byte array representing the bytecode of the class
			 */
			private byte[] loadClassData(String name) {
				return this.consume(name);
			}

			/**
			 * Determines if this class loader is responsible for the following
			 * element
			 * 
			 * @param element
			 *            the element that is being requested for handling
			 * @return true if the element should be handled by this classloader
			 *         instance, returns false always if disabled
			 * 
			 * @see #setEnabled(boolean)
			 */
			private boolean responsibleFor(String element) {
				return (this.isEnabled() ? this.getScopingPattern().matcher(element).lookingAt() : false);
			}

			/**
			 * Set the state for the thread local storage that the class loader
			 * operates on, this allows the same class loader to be shared on
			 * different threads, separating out the cached class loader from
			 * the actual communicator to use it -- each class loader method
			 * that is private or called from another class loader method cannot
			 * be passed a callback from the consumer class, therefore plugging
			 * the actual communicator in and out as needed is difficult. The
			 * solution implemented here is to use the thread local storage,
			 * with each thread having its own communicator and any other
			 * required state.
			 * 
			 * @param communicator
			 *            the communicator to use for the current thread, the
			 *            thread used to invoke the set state method
			 * @param scopingPattern
			 *            pattern to use, determines if class should be loaded
			 *            by specific class loader
			 * @param isEnabled
			 *            indicates if the class loader is enabled or not,
			 *            usually the class loader is enabled and then
			 *            subsequently disabled after use
			 */
			void setState(ScopedChannelClassLoader.Consumer consumer, IChannelCommunicator communicator,
					Pattern scopingPattern, boolean isEnabled) {
				this.consumer.set(consumer);
				this.communicator.set(communicator);
				this.scopingPattern.set(scopingPattern);
				this.isEnabled.set(new Boolean(isEnabled));
			}

			/**
			 * Unset state to avoid potential memory leaks
			 */
			void unsetState() {
				this.consumer.set(null);
				this.communicator.set(null);
				this.scopingPattern.set(null);
				this.isEnabled.set(null);
			}

		}

		/**
		 * A cache of classloaders used to load classes in a particular context
		 * (unique client) where classes can be consumed from the cache and do
		 * not have to interact with the provider (therefore bypassing having to
		 * download the classes each time)
		 */
		private static final HashMap classLoaders = new HashMap();

		/**
		 * Channel communicator to send and receive control and data
		 */
		private final IChannelCommunicator communicator;

		/**
		 * Identity of context this class loader is used from (the file server
		 * command factory that initiated the entire flow) -- used as primary
		 * key in the class cache (no need to consume classes over the channel
		 * if they have been loaded previously with the same context)
		 */
		private final String context;

		/**
		 * The class loader currently serving the consumer, pulled and used from
		 * the cache if already existing per context or created and stored in
		 * the classLoaders map if not already present
		 */
		private final ContextClassLoader contextClassLoader;

		/**
		 * Host of the class loader provider
		 */
		private final String host;

		/**
		 * Indicates if this classloader is enabled, if disabled, all calls will
		 * be delegated to the parent classloader
		 */
		private boolean isEnabled = false;

		/**
		 * The regular expression that scopes the responsibility of this
		 * classloader, for all other out-of-scope requests, these are delegated
		 * to the typical classloader
		 */
		private String scopingExpression;

		/**
		 * The compiled scoping pattern using the scoping expression
		 */
		private Pattern scopingPattern;

		/**
		 * User of the class loader provider
		 */
		private final String user;

		/**
		 * Constructs a scoped class loader, a custom class loader that is
		 * interested in being responsible for only the elements/classes that
		 * match the scoping expression criteria -- if the regular expression
		 * matches, the element will be handled by this classloader, if not it
		 * will be deemed out-of-scope and delegated
		 * 
		 * @param scopingExpression
		 *            the regular expression that defines the scope
		 * @param readableChannel
		 *            the channel that supplies the data
		 * @param writableChannel
		 *            the channel that controls the data, commands the readable
		 *            channel source
		 */
		public Consumer(String scopingExpression, ReadableByteChannel readableChannel, WritableByteChannel writableChannel)
				throws IOException {

			// Assign instance state
			this.scopingExpression = scopingExpression;
			this.communicator = ChannelCommunicatorFactory.getInstance().create(readableChannel, writableChannel);
			this.scopingPattern = Pattern.compile(this.scopingExpression);

			// Send IDENTIFY command directive
			this.communicator.send(AbstractControlDirective.IdentifyControlDirective.class.getName());
			this.context = this.communicator.receiveString();
			this.host = this.communicator.receiveString();
			this.user = this.communicator.receiveString();

			// Retrieve context classloader or create and store
			synchronized (Provider.class) {
				ContextClassLoader classLoader = (ContextClassLoader) Consumer.classLoaders.get(this.context);
				if (classLoader == null) {
					classLoader = new ContextClassLoader();
					Consumer.classLoaders.put(this.context, classLoader);
				}
				this.contextClassLoader = classLoader;
			}

		}

		/**
		 * The class loader associated with this instance of consumer, typically
		 * the consumer is instanitated, the provider is instantiated and then
		 * the consumer caller would invoke this method to retrieve the class
		 * loader to use.
		 * 
		 * @return the class loader is exposed for use by caller
		 */
		public ClassLoader getContextClassLoader() {
			return this.contextClassLoader;
		}

		/**
		 * Enables and disables this classloader's responsibility to its scope,
		 * if disabled all calls are delegated to the parent classloader, if
		 * enabled, the regular scoped functionality is enabled
		 * 
		 * @param isEnabled
		 *            false turns off this classloader scoped responsibilities,
		 *            typically the provider provider caller will block until
		 *            the consumer instance is disabled -- there is some
		 *            handshaking to signal to the provider that it can continue
		 *            and that all needed classes have been delivered and that
		 *            the consumer is done requesting resources
		 */
		public void setEnabled(boolean isEnabled) throws IOException {

			// Disable classloader
			Consumer.this.isEnabled = isEnabled;

			// Send appropriate command directive
			if (Consumer.this.isEnabled == true) {

				// Associate state of this instance with class loader
				this.contextClassLoader.setState(this, this.communicator, this.scopingPattern, this.isEnabled);

				// Send BEGIN command directive
				Consumer.this.communicator.send(AbstractControlDirective.BeginControlDirective.class.getName());

			} else {

				// Send END command directive
				Consumer.this.communicator.send(AbstractControlDirective.EndControlDirective.class.getName());

				// Disassociate state of this instance with class loader
				this.contextClassLoader.unsetState();

			}

		}

		/**
		 * Output textual representation of instance
		 */
		public String toString() {
			return this.getClass().getName() + "[context=" + Consumer.this.context + ", host=" + Consumer.this.host
					+ ", user=" + Consumer.this.user + ", contextClassLoader=" + Consumer.this.contextClassLoader + "]";
		}

	}

	/**
	 * The scoped channel class loader provider class is used by the provider of
	 * the classes that will be served up through the channel to the consuming
	 * instance of the scoped channel class loader
	 * 
	 * @author Scott E. Schneider
	 */
	public static class Provider extends Observable {

		/**
		 * Channel communicator used to send and receive control and data
		 */
		private final IChannelCommunicator communicator;

		/**
		 * Identity of context this class loader is used from (the file server
		 * command factory that initiated the entire flow) -- used as primary
		 * key in the class cache (no need to consume classes over the channel
		 * if they have been loaded previously with the same context
		 */
		private final String context;

		/**
		 * Host name from provider client
		 */
		private String host;

		/**
		 * The maximum class size that can be consumer, in bytes (the input
		 * maximum class size is multiplied by 1024 and then stored)
		 */
		private final int maximumClassSize;

		/**
		 * The classloader used to retrieve resource streams and subsequently
		 * provide them to the consumer (the provider is on the client side and
		 * the consumer is on the server side)
		 */
		private ClassLoader sourceClassLoader;

		/**
		 * User name from provider client
		 */
		private final String user;

		/**
		 * Constructs a scoped channel class loader provider, used to serve up
		 * resources/classes to the consuming instance
		 * 
		 * @param context
		 *            the context this command is executing within
		 * @param sourceClassLoader
		 *            the classloader used to resolve resources to be sent over
		 *            the channel
		 * @param readableChannel
		 *            the channel that offers control to this instance
		 * @param writableChannel
		 *            the channel that provides data and minimal control to the
		 *            consuming instance
		 * @param maximumClassSize
		 *            used to optimize buffers, all class sizes must be less
		 *            than this number in kilobytes, for example, 64 = (64 *
		 *            1024) bytes maximum class size
		 */
		public Provider(String context, ClassLoader sourceClassLoader, ReadableByteChannel readableChannel,
				WritableByteChannel writableChannel, int maximumClassSize) {

			// Assign instance state
			this.context = context;
			this.sourceClassLoader = sourceClassLoader;
			this.communicator = ChannelCommunicatorFactory.getInstance().create(readableChannel, writableChannel);
			try {
				this.host = InetAddress.getLocalHost().toString();
			} catch (UnknownHostException e) {
				//
			}
			this.user = System.getProperty("user.name");//$NON-NLS-1$
			this.maximumClassSize = maximumClassSize * 1024;

		}

		/**
		 * Provides the resources requested by the remote end of the channel, an
		 * end diretive received will cause the provide method to return to the
		 * caller and therefore unblock the caller
		 */
		public void provide() throws IOException {

			// Provide until the END directive is received
			boolean provide = true;

			do {

				// Receive control directive name
				String directiveName = this.communicator.receiveString();

				// Handle based on command identity
				try {
					Class directiveClass = Class.forName(directiveName);

					// If control directive, handle appropriately
					if (AbstractControlDirective.class.isAssignableFrom(directiveClass)) {

						// Instantiate the directive object
						Object directiveObject = directiveClass.newInstance();
						AbstractControlDirective directive = (AbstractControlDirective) directiveObject;

						// Use double-dispatch to handle types
						provide = directive.accept(new AbstractControlDirective.ControlDirectiveVisitor() {
							/*
							 * (non-Javadoc)
							 * 
							 * @see org.eclipse.hyades.execution.core.cl.ScopedChannelClassLoader.ControlDirectiveVisitor
							 *      #visit(org.eclipse.hyades.execution.core.cl.ScopedChannelClassLoader.BeginControlDirective)
							 */
							public boolean visit(AbstractControlDirective.BeginControlDirective directive) {
								return true;
							}

							/*
							 * (non-Javadoc)
							 * 
							 * @see org.eclipse.hyades.execution.core.cl.ScopedChannelClassLoader.ControlDirectiveVisitor
							 *      #visit(org.eclipse.hyades.execution.core.cl.ScopedChannelClassLoader.EndControlDirective)
							 */
							public boolean visit(AbstractControlDirective.EndControlDirective directive) {
								return false;

							}

							/*
							 * (non-Javadoc)
							 * 
							 * @see org.eclipse.hyades.execution.core.loader.ScopedChannelClassLoader.AbstractControlDirective.ControlDirectiveVisitor#visit(org.eclipse.hyades.execution.core.loader.ScopedChannelClassLoader.AbstractControlDirective.IdentifyControlDirective)
							 */
							public boolean visit(AbstractControlDirective.IdentifyControlDirective directive)
									throws IOException {

								// Send identity information to consumer
								Provider.this.communicator.send(Provider.this.context);
								Provider.this.communicator.send(Provider.this.host);
								Provider.this.communicator.send(Provider.this.user);

								// Continue to allow receipt of commands
								return true;

							}

							/*
							 * (non-Javadoc)
							 * 
							 * @see org.eclipse.hyades.execution.core.cl.ScopedChannelClassLoader.ControlDirectiveVisitor
							 *      #visit(org.eclipse.hyades.execution.core.cl.ScopedChannelClassLoader.LoadControlDirective)
							 */
							public boolean visit(AbstractControlDirective.LoadControlDirective directive)
									throws IOException {

								// Receive class name to provide to consumer
								String className = Provider.this.communicator.receiveString();

								// Set-up for reading in class data
								className = className.replace('.', '/') + ".class";//$NON-NLS-1$

								// Retrieve class as readable byte channel
								InputStream classInputStream = sourceClassLoader.getResourceAsStream(className);
								ReadableByteChannel classReadableChannel = Channels.newChannel(classInputStream);

								// Allocate buffer for class data and read it
								ByteBuffer classData = ByteBuffer.allocate(Provider.this.maximumClassSize);
								int classDataLength = classReadableChannel.read(classData);
								classData.flip();

								// Send length of class byte data
								Provider.this.communicator.send(classDataLength);

								// Send resource in bytes to consumer
								byte[] classDataBytes = new byte[classDataLength];
								classData.get(classDataBytes);
								Provider.this.communicator.send(classDataBytes);

								/*
								 * Indicate class uploaded from consumer client
								 * and flip changed state to prep for notify
								 * observer invocation
								 */
								Provider.this.setChanged();
								Provider.this.notifyObservers(className);

								// Indicates to continue
								return true;

							}

						});
					}
				} catch (Throwable t) {
					//
				}

				// Continue until provision is not required
			} while (provide == true);

		}

		/**
		 * Output textual representation of provider instance
		 */
		public String toString() {
			return this.getClass().getName() + "[context=" + this.context + ", host=" + this.host + ", user=" + this.user
					+ "]";
		}

	}

}
