/********************************************************************** 
 * Copyright (c) 2005, 2008 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: PerformanceTimer.java,v 1.6 2008/03/20 18:49:56 dmorris Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/

package org.eclipse.hyades.execution.core.timer;

import java.io.PrintStream;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.Stack;

/**
 * A lightweight performance timer to be used for quick situations where
 * approximate timing of a given algorithm or code is required. This class keeps
 * a pool of identified timers, retrieve timers based on identity using the
 * getInstance(...) method.
 * 
 * @author Scott E. Schneider
 */
public class PerformanceTimer implements IPerformanceTimer {

	/**
	 * The current timers in use, keyed on performance timer name
	 */
	private static final HashMap timers;

	/**
	 * String used for unnamed performance timers
	 */
	private static final String UNNAMED = "Unnamed";//$NON-NLS-1$

	/**
	 * Initialize a static-level timers hashmap used for tracking timers in use
	 */
	static {
		timers = new HashMap(5);
	}

	/**
	 * Retrieve the specified named instance of the performance timer
	 * 
	 * @param name
	 *            the name of the instance to retrieve
	 * @return returns the named performance timer instance, created if it
	 *         doesn't current exist
	 */
	public synchronized static IPerformanceTimer getInstance(String name) {
		return PerformanceTimer.getInstance(name, System.out);
	}

	/**
	 * Retrieve the specified name instance of the performance timer using the
	 * specified stream as the output destination
	 * 
	 * @param name
	 *            the name of the instance to retrieve
	 * @param stream
	 *            the stream to output to
	 * @return returns the appropriate performance timer, created if it doesn't
	 *         current exist
	 */
	public synchronized static IPerformanceTimer getInstance(String name, PrintStream stream) {
		PerformanceTimer timer = (PerformanceTimer) PerformanceTimer.timers.get(name);
		if (timer == null) {
			timer = new PerformanceTimer(name, stream);
			PerformanceTimer.timers.put(name, timer);
		}
		return timer;
	}

	/**
	 * Pads the specified number to the following width in string format
	 * 
	 * @param number
	 *            the number to pad in the width specified string
	 * @param width
	 *            the width to pad the string to
	 * @return the padded string that indicates the number padded
	 */
	private static String pad(long number, int width) {
		String value = String.valueOf(number);
		StringBuffer buffer = new StringBuffer(value);
		for (int i = 0; i < width - value.length(); i++) {
			buffer.insert(0, ' ');
		}
		return buffer.toString();
	}

	/**
	 * The stack of counts, used for performance timer task group average
	 * calculation
	 */
	private Stack counts;

	/**
	 * The identity of the performance timer, used to uniquely identify a
	 * performance timer
	 */
	private final String identity;

	/**
	 * The stack of performance timer task names
	 */
	private Stack names;

	/**
	 * Previous size of the nested group of performance timer tasks
	 */
	private long previousSize;

	/**
	 * Sequence number for easier reference of line items
	 */
	private long sequenceNumber = 0;

	/**
	 * Size of the nested group of performance timer tasks
	 */
	private long size;

	/**
	 * The stack of performance timer task sizes
	 */
	private Stack sizes;

	/**
	 * The stack of performance timer task start times
	 */
	private Stack starts;

	/**
	 * Linked list (playing the role of a queue), the queue of performance timer
	 * task stop times
	 */
	private LinkedList stops;

	/**
	 * The stream to use for this performance timer's output
	 */
	private final PrintStream stream;

	/**
	 * Constructs a performance timer with the given identity using the
	 * following stream
	 * 
	 * @param identity
	 *            the name of the performance timer
	 * @param stream
	 *            the stream to use for output
	 */
	private PerformanceTimer(String identity, PrintStream stream) {
		this.identity = identity;
		this.stream = stream;
		this.reset();
	}

	/**
	 * Consume count in order from the stack, FILO approach
	 * 
	 * @return the count on top of the stack, popped and returned
	 */
	private long consumeCount() {
		return ((Long) this.counts.pop()).longValue();
	}

	/**
	 * Consume derived elapsed times in order using queue of stops and stack of
	 * starts
	 * 
	 * @return the elapsed time for the nested group of performance timer tasks
	 */
	private long consumeElapsed() {
		long stop = ((Long) this.stops.removeFirst()).longValue();
		long start = ((Long) this.starts.pop()).longValue();
		return stop - start;
	}

	/**
	 * Consume the names in order from the stack, popping the top value off and
	 * returning it
	 * 
	 * @return the top value, name on the stack
	 */
	private String consumeName() {
		return (String) this.names.pop();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.execution.timer.IPerformanceTimer#elapsed()
	 */
	public synchronized void elapsed() {
		String name = this.consumeName();
		long elapsed = this.consumeElapsed();
		long count = this.consumeCount();
		this.stream.print("\r\n" + PerformanceTimer.pad(++this.sequenceNumber, 15) + "\t" + Thread.currentThread()
				+ " in performance timer " + this.identity + "\r\n\t\t" + name + "\r\n\t\texecution is measured at "
				+ elapsed + "ms ");
		if (count > 0) {
			long average = elapsed / count;
			this.stream.print("(" + average + "ms on average)");
		}
		this.stream.println();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.execution.core.timer.IPerformanceTimer#elapsed(int)
	 */
	public synchronized void elapsed(int size) {
		String name = this.consumeName();
		long elapsed = this.consumeElapsed();
		long count = this.consumeCount();
		this.stream.print("\r\n" + PerformanceTimer.pad(++this.sequenceNumber, 15) + "\t" + Thread.currentThread()
				+ " in performance timer " + this.identity + "\r\n\t\t" + name + "\r\n\t\texecution is measured at "
				+ elapsed + "ms (" + elapsed / size + "ms per work unit)");
		if (count > 0) {
			long average = elapsed / count;
			this.stream.print(" (" + average + "ms on average; " + average / size + "ms per work unit on average)");
		}
		this.stream.println();
	}

	/**
	 * Get the timestamp, the current time in ms
	 * 
	 * @return the current time in milliseconds, source of all times that are
	 *         not derived
	 */
	private Long getTimestamp() {
		return new Long(System.currentTimeMillis());
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.execution.timer.IPerformanceTimer#reset()
	 */
	public synchronized void reset() {
		this.names = new Stack();
		this.starts = new Stack();
		this.stops = new LinkedList();
		this.sizes = new Stack();
		this.counts = new Stack();
		this.size = 0;
		this.previousSize = 0;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.execution.timer.IPerformanceTimer#start()
	 */
	public synchronized void start() {
		this.start(PerformanceTimer.UNNAMED);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.execution.timer.IPerformanceTimer#start(java.lang.String)
	 */
	public synchronized void start(String name) {
		this.starts.push(this.getTimestamp());
		this.names.push((this.names.size() > 0 ? (String) this.names.peek() + "::" : "") + name);
		this.sizes.push(new Long(this.size));
		this.size = 0;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.execution.timer.IPerformanceTimer#stop()
	 */
	public synchronized void stop() {
		this.stops.addLast(this.getTimestamp());
		this.previousSize = size;
		this.size = ((Long) this.sizes.pop()).longValue() + 1;
		this.counts.push(new Long(this.previousSize));
	}

}
