/**********************************************************************  
 * Copyright (c) 2005, 2009 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: ConnectionSpecifier.java,v 1.5 2009/09/28 19:17:47 paules Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/

package org.eclipse.hyades.execution.core.util;

import java.util.Iterator;
import java.util.Properties;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.eclipse.core.runtime.Assert;

/**
 * Specifies a connection to an agent controller instance -- embodies the state
 * that uniquely identifies an agent controller instance. Example connect
 * stirngs follow in the theme of an informal URI, similar to JDBC connection
 * strings.
 * 
 * <code>
 * 
 * <specification>:<class>://<host>:<port>/<instance>?user=<user>&password=<password>
 * |------- Scheme -------|-- Authority --|-- Path --|---------- Query -------------|
 *         
 * </code>
 * 
 * Default specification is tptp, default class is RAC, default host is
 * localhost, default port is 10002, default instance is default and default
 * user and password is null. For example, if you instantiate a specifier you
 * can invoke the toString() method and retrieve a valid connection string.
 * 
 * Other custom attributes can be included in the query portion of the URI
 * separated by ampersands and these will be parsed and stored in the attributes
 * properties instance variable.
 * 
 * <code>
 * 
 * 		tptp:rac://localhost:10002/default
 * 		tptp:rac://bigbox.rtp.ibm.com:10002/default?user=babel&password=fish
 * 		tptp:iac://localhost/default
 * 		
 * 		Note that 'iac' alone will be interpreted as a host name of iac and not
 * 		the agent controller server class 'iac' (something like this would get
 * 		the behavior as desired 'iac://' -- although it is recommended that
 * 		the connection strings be written in the canonical form whenever possible
 *		in code for easier reading by developers.
 * 
 * </code>
 * 
 * More examples of supported specifiers can be found in the static main method,
 * used as a simple test of the parsing capabilities of this class.
 * 
 * @see #main(String[])
 * 
 * @author Scott E. Schneider
 */
public final class ConnectionSpecifier {

	/**
	 * Precompiled regular expression that parses out the authority part of the
	 * connection specifier.
	 */
	private static final Pattern AUTHORITY_REGULAR_EXPRESSION_PATTERN = Pattern
			.compile(ConnectionSpecifier.AUTHORITY_REGULAR_EXPRESSION_STRING);

	/**
	 * Regular expression that parses out the authority part of the connection
	 * specifier.
	 */
	private static final String AUTHORITY_REGULAR_EXPRESSION_STRING = "(?<=//|^)([A-Za-z0-9.]+)(?:\\:([1-9][0-9]*))?(?=/|$)";//$NON-NLS-1$

	/**
	 * The default host if not specified
	 */
	private static final String DEFAULT_HOST = "localhost";//$NON-NLS-1$

	/**
	 * The default port if not specified
	 */
	private static final int DEFAULT_PORT = 10002;

	/**
	 * The default server class if not specified
	 */
	private static final String DEFAULT_SERVER_CLASS = "rac";//$NON-NLS-1$

	/**
	 * The default server instance for this specifier
	 */
	private static final String DEFAULT_SERVER_INSTANCE = "/default";//$NON-NLS-1$

	/**
	 * Default specification for this specifier
	 */
	private static final String DEFAULT_SPECIFICATION = "tptp";//$NON-NLS-1$

	/**
	 * Precompiled regular expression that parses out the path part of the
	 * connection specifier.
	 */
	private static final Pattern PATH_REGULAR_EXPRESSION_PATTERN = Pattern
			.compile(ConnectionSpecifier.PATH_REGULAR_EXPRESSION_STRING);

	/**
	 * Regular expression that parses out the path part of the connection
	 * specifier.
	 */
	private static final String PATH_REGULAR_EXPRESSION_STRING = "(?<!/)(?:/[A-Za-z0-9]+)+";//$NON-NLS-1$

	/**
	 * Precompiled regular expression that parses out the query part of the
	 * connection specifier.
	 */
	private static final Pattern QUERY_REGULAR_EXPRESSION_PATTERN = Pattern
			.compile(ConnectionSpecifier.QUERY_REGULAR_EXPRESSION_STRING);

	/**
	 * Regular expression that parses out the query part of the connection
	 * specifier.
	 */
	private static final String QUERY_REGULAR_EXPRESSION_STRING = "(?:\\?|&)([A-Za-z]+)=([^&]+)";//$NON-NLS-1$

	/**
	 * Precompiled regular expression that parses out the scheme part of the
	 * connection specifier.
	 */
	private static final Pattern SCHEME_REGULAR_EXPRESSION_PATTERN = Pattern
			.compile(ConnectionSpecifier.SCHEME_REGULAR_EXPRESSION_STRING);

	/**
	 * Regular expression that parses out the scheme part of the connection
	 * specifier.
	 */
	private static final String SCHEME_REGULAR_EXPRESSION_STRING = "(^[A-Za-z0-9]+)\\:(?:([A-Za-z]+)|(?:\\:|/))";//$NON-NLS-1$

	/**
	 * Constructs a connection specifier for test purposes
	 * 
	 * @param args
	 *            not used
	 */
	public static void main(String[] args) {
		ConnectionSpecifier.test(new ConnectionSpecifier(""));//$NON-NLS-1$
		ConnectionSpecifier.test(new ConnectionSpecifier("tptp2:dce"));//$NON-NLS-1$
		ConnectionSpecifier.test(new ConnectionSpecifier("tptp:rac://myriad:10002/default"));//$NON-NLS-1$
		ConnectionSpecifier.test(new ConnectionSpecifier("iac://fiend:10003/default"));//$NON-NLS-1$
		ConnectionSpecifier.test(new ConnectionSpecifier("neutron"));//$NON-NLS-1$
		ConnectionSpecifier.test(new ConnectionSpecifier("pulsar:10004"));//$NON-NLS-1$
		ConnectionSpecifier.test(new ConnectionSpecifier("//dragon/default"));//$NON-NLS-1$
		ConnectionSpecifier.test(new ConnectionSpecifier("//chain:5150/default"));//$NON-NLS-1$
		ConnectionSpecifier.test(new ConnectionSpecifier("//fenix/"));//$NON-NLS-1$
		ConnectionSpecifier.test(new ConnectionSpecifier("//atari/default?user=scotts"));//$NON-NLS-1$
		ConnectionSpecifier.test(new ConnectionSpecifier("//bigbox.rtp.raleigh.ibm.com/default?user=scotts"));//$NON-NLS-1$
		ConnectionSpecifier.test(new ConnectionSpecifier("sextet:9999/auxiliary?user=babel&password=fish"));//$NON-NLS-1$
		ConnectionSpecifier.test(new ConnectionSpecifier("illnote/default?user=babel&password=fish&other=42"));//$NON-NLS-1$
	}

	/**
	 * Tests a given connection string but rendering it to the console,
	 * generating the canonical form and making sure the canonical form can be
	 * constructed into the same connection specifier again, this method asserts
	 * if there is a problem found with the test.
	 * 
	 * @param specifier
	 *            the specifier to exercise
	 */
	private static void test(ConnectionSpecifier specifier) {
		String canonicalString = specifier.getCanonicalString();
		System.out.println("\r\n" + specifier + "\r\n\t\t  " + canonicalString);
		Assert.isTrue(new ConnectionSpecifier(canonicalString).equals(specifier));
		System.out.println();
	}

	/**
	 * Holds custom user specified and connection interpreter defined
	 * attributes, these are specified between ampersands after the question
	 * mark in the same fashion as user and password (which are the only preset
	 * named attributes at this time) -- does not include the standard user and
	 * password attributes as these are stored in typed instance variables.
	 */
	private final Properties attributes = new Properties();

	/**
	 * The connection string used to construct this specifier instance
	 */
	private String connectionString;

	/**
	 * The host identified by this specifier
	 */
	private String host;

	/**
	 * The password, present only if a secure connection is being specified
	 */
	private String password;

	/**
	 * The port used to establish a connection
	 */
	private int port;

	/**
	 * The class of server, such as IAC, RAC, perhaps DCE for the new agent
	 * controller, server classes
	 */
	private String serverClass;

	/**
	 * The named instance on the server class on the given host, unique only per
	 * host and then per server class
	 */
	private String serverInstance;

	/**
	 * The specification for this specifier, TPTP is the only one supported
	 * currently
	 */
	private String specification;

	/**
	 * The user id, present only if a secure connection is being specified
	 */
	private String user;

	/**
	 * Constructs a connection specifier by parsing the given connection string
	 * and setting the internal state of the connection specifier. Once
	 * constructed, various methods can be queried on this instance.
	 * 
	 * @param connectionString
	 *            the connection string that uniquely identifies an agent
	 *            controller
	 */
	public ConnectionSpecifier(String connectionString) {
		this.connectionString = connectionString.trim().length() > 0 ? connectionString : null;
		this.parse();
	}

	/**
	 * Constructs a connection specifier using a host and port, defaults are
	 * assumed for all other parameters
	 * 
	 * @param host
	 *            the host identified for this specifier
	 * @param port
	 *            the port identified for this specifier
	 */
	public ConnectionSpecifier(String host, int port) {
		this(ConnectionSpecifier.DEFAULT_SERVER_CLASS, host, port);
	}

	/**
	 * The most frequent specified combination of arguments
	 * 
	 * @param serverClass
	 *            IAC, RAC, etc to use for the connection
	 * @param host
	 *            the host to connect to
	 * @param port
	 *            the port that the controller listens on
	 */
	public ConnectionSpecifier(String serverClass, String host, int port) {
		this(ConnectionSpecifier.DEFAULT_SPECIFICATION, serverClass, host, port,
				ConnectionSpecifier.DEFAULT_SERVER_INSTANCE, null, null);
	}

	/**
	 * Constructs a specifier instance, given full state to be applied
	 * 
	 * @param specification
	 *            the specification identified (TPTP is the only supported one)
	 * @param serverClass
	 *            the server class (interpreted as type or protocol perhaps)
	 * @param host
	 *            the host where the controller resides (locahost, 192.168.1.2,
	 *            etc)
	 * @param port
	 *            the port the agent controller listens for connections on
	 * @param serverInstance
	 *            the named server instance (currently, only default supported)
	 * @param user
	 *            the user specified for secure connections
	 * @param password
	 *            the password for secure connections, associated with paried
	 *            user id
	 */
	public ConnectionSpecifier(String specification, String serverClass, String host, int port, String serverInstance,
			String user, String password) {
		this.specification = specification != null ? specification : ConnectionSpecifier.DEFAULT_SPECIFICATION;
		this.serverClass = serverClass != null ? serverClass : ConnectionSpecifier.DEFAULT_SERVER_CLASS;
		this.host = host != null ? host : ConnectionSpecifier.DEFAULT_HOST;
		this.port = port;
		this.serverInstance = serverInstance != null ? serverInstance : ConnectionSpecifier.DEFAULT_SERVER_INSTANCE;
		this.user = user;
		this.password = password;
		this.generate();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	public boolean equals(Object object) {
		return (this == object ? true : (object instanceof ConnectionSpecifier ? this.getCanonicalString().equals(
				((ConnectionSpecifier) object).getCanonicalString()) : false));
	}

	/**
	 * Generate connection string from internal state, the opposite of the parse
	 * method.
	 * 
	 * @see #parse()
	 */
	private void generate() {
		this.connectionString = this.getCanonicalString();
	}

	/**
	 * The URI authority is the host appended with the port for a connection
	 * specifier
	 * 
	 * @return the host followed by the port, delimited by a colon
	 */
	public String getAuthority() {
		return this.host + ":" + this.port;
	}

	/**
	 * Returns the canonical connection string, the connection string that is
	 * the exemplary connection string for this unique connection specifier
	 * (this might be different than the connection string used to construct
	 * this instance but it will identify the same connection specifier).
	 * 
	 * @return the canonical connection string
	 */
	public String getCanonicalString() {
		String query = this.getQuery();
		return this.getScheme() + "://" + this.getAuthority() + this.getPath() + (query.length() > 0 ? "?" + query : "");
	}

	/**
	 * Returns either the connection string as this instance was constructed
	 * with or the canonical form if the specifier was constructed using a
	 * non-connection string seeded constructed.
	 * 
	 * @return the connection string
	 */
	public String getConnectionString() {
		return this.connectionString;
	}

	/**
	 * Returns the host for this connection specifier
	 * 
	 * @return returns the host
	 */
	public String getHost() {
		return host;
	}

	/**
	 * Returns the password for this connection specifier
	 * 
	 * @return returns the password or null if not set
	 */
	public String getPassword() {
		return password;
	}

	/**
	 * For a connection specifier the path is currently the server instance
	 * which can be qualified in a hierarchy with slashes as desired
	 * 
	 * @return the server instance path
	 */
	public String getPath() {
		return this.serverInstance != null ? serverInstance : "";
	}

	/**
	 * Returns the port as specified by the connection specifier
	 * 
	 * @return returns the port
	 */
	public int getPort() {
		return port;
	}

	/**
	 * Return the query part of the connection specifier which is defined as the
	 * attributes such as user, password and any other custom attributes as
	 * assigned by this connection specifier instantiator
	 * 
	 * @return the query part of the connection specifier
	 */
	public String getQuery() {
		StringBuffer stringBuffer = new StringBuffer();
		if (this.user != null) {
			stringBuffer.append("user=");//$NON-NLS-1$
			stringBuffer.append(this.user);
			stringBuffer.append("&");
		}
		if (this.password != null) {
			stringBuffer.append("password=");//$NON-NLS-1$
			stringBuffer.append(this.password);
			stringBuffer.append("&");
		}
		for (Iterator pairs = this.attributes.keySet().iterator(); pairs.hasNext();) {
			String name = (String) pairs.next();
			stringBuffer.append(name);
			stringBuffer.append("=");
			stringBuffer.append(this.attributes.get(name));
			stringBuffer.append("&");
		}
		int length = stringBuffer.length();
		if (length > 0 && stringBuffer.charAt(length - 1) == '&') {
			stringBuffer.deleteCharAt(length - 1);
		}
		return stringBuffer.toString();
	}

	/**
	 * Return the scheme for this connection specifier, the scheme is defined as
	 * containing the specification followed by a colon followed by the server
	 * class or agent controller type
	 * 
	 * @return the scheme for this connection specifier is returned
	 */
	public String getScheme() {
		return this.specification + ":" + this.serverClass;
	}

	/**
	 * Returns the server class for this connection specifier
	 * 
	 * @return returns the server class also known as the agent controller type
	 */
	public String getServerClass() {
		return serverClass;
	}

	/**
	 * Retrieves the server instance for this connection specifier, this is the
	 * instance of the agent controller as specified
	 * 
	 * @return returns the server instance
	 */
	public String getServerInstance() {
		return serverInstance;
	}

	/**
	 * Returns the specification which is defined as the optional prefix to the
	 * connection string that uniquely identifies the specification of the
	 * connection specification (the parsing rules in use, version or any other
	 * use desired)
	 * 
	 * @return returns the specification
	 */
	public String getSpecification() {
		return specification;
	}

	/**
	 * The user value is returned or null if not set in the connection
	 * specifier's state
	 * 
	 * @return returns the user or null if not specified
	 */
	public String getUser() {
		return user;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.lang.Object#hashCode()
	 */
	public int hashCode() {
		return this.getCanonicalString().hashCode();
	}

	/**
	 * Initialize state of connection specifier by parsing the connection string
	 * passed in to the specifier constructor, the opposite of the generate
	 * method
	 * 
	 * @see #generate()
	 */
	private void parse() {

		/*
		 * If specified, parse the URI named sections and the parts that compose
		 * these named sections
		 */
		if (this.connectionString != null) {
			this.parseScheme();
			this.parseAuthority();
			this.parsePath();
			this.parseQuery();
		} else {
			this.useDefaults();
		}

	}

	/**
	 * Parse the authority portion of the URI, this is the host and the port
	 * separated by a colon, if the authority is specified then the host is
	 * required but the port is optional.
	 */
	private void parseAuthority() {

		// Construct a matcher using the precompiled regular expression
		Matcher matcher = ConnectionSpecifier.AUTHORITY_REGULAR_EXPRESSION_PATTERN.matcher(this.connectionString);

		// Find and store the authority parsed
		if (matcher.find()) {
			this.host = matcher.group(1);
			String port = matcher.group(2);
			if (port != null) {
				this.port = Integer.parseInt(port);
			} else {
				this.port = ConnectionSpecifier.DEFAULT_PORT;
			}

			// Not matched, use defaults
		} else {
			this.host = ConnectionSpecifier.DEFAULT_HOST;
			this.port = ConnectionSpecifier.DEFAULT_PORT;
		}

	}

	/**
	 * Parse the path section of the URI, this is the server instance qualifier,
	 * that qualifies the server instance further qualifying the host, port and
	 * server class. This is optional although required if a query is specified.
	 */
	private void parsePath() {

		// Construct a matcher using the precompiled regular expression
		Matcher matcher = ConnectionSpecifier.PATH_REGULAR_EXPRESSION_PATTERN.matcher(this.connectionString);

		// If specified store the parsed path (contains server instance)
		if (matcher.find()) {
			this.serverInstance = matcher.group();

			// Not matched, use defaults
		} else {
			this.serverInstance = ConnectionSpecifier.DEFAULT_SERVER_INSTANCE;
		}

	}

	/**
	 * Parse the query portion of the connection string, the query portion is
	 * optional and specifies user and password as well as any other server
	 * class attributes that might be supported via custom attributes.
	 */
	private void parseQuery() {

		// Construct a matcher using the precompiled regular expression
		Matcher matcher = ConnectionSpecifier.QUERY_REGULAR_EXPRESSION_PATTERN.matcher(this.connectionString);

		// Reset current attributes and clear out
		this.user = null;
		this.password = null;
		this.attributes.clear();

		// Continue to find zero to many attributes
		while (matcher.find()) {

			// Retrieve parsed name value pair
			String name = matcher.group(1);
			String value = matcher.group(2);

			/*
			 * For each standard more strictly typed attribute, stored in
			 * instance variable, for others, store in attributes property
			 * object
			 */
			if ("user".equalsIgnoreCase(name)) {//$NON-NLS-1$
				this.user = value;
			} else {
				if ("password".equalsIgnoreCase(name)) {//$NON-NLS-1$
					this.password = value;
				} else {
					this.attributes.put(name, value);
				}
			}
		}

	}

	/**
	 * Parse the scheme, the scheme is optional with defaults, but if specified
	 * with just one part, it will be interpreted as the server class -- if it
	 * is specified with two parts they will be taken as the specification
	 * followed by a colon followed by the server class individually.
	 */
	private void parseScheme() {

		// Construct a matcher using the precompiled regular expression
		Matcher matcher = ConnectionSpecifier.SCHEME_REGULAR_EXPRESSION_PATTERN.matcher(this.connectionString);

		// Find and store the scheme parsed
		if (matcher.find()) {
			String group1 = matcher.group(1);
			String group2 = matcher.group(2);
			if (group2 == null) {
				this.specification = ConnectionSpecifier.DEFAULT_SPECIFICATION;
				this.serverClass = group1;
			} else {
				this.specification = group1;
				this.serverClass = group2;
			}

			// Not matched, use defaults
		} else {
			this.specification = ConnectionSpecifier.DEFAULT_SPECIFICATION;
			this.serverClass = ConnectionSpecifier.DEFAULT_SERVER_CLASS;
		}

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.lang.Object#toString()
	 */
	public String toString() {
		StringBuffer stringBuffer = new StringBuffer("{connectionString=" + this.connectionString + ", specification="
				+ this.specification + ", serverClass=" + this.serverClass + ", host=" + this.host + ", port=" + this.port
				+ ", serverInstance=" + this.serverInstance + ", user=" + this.user + ", password=" + this.password);
		for (Iterator pairs = this.attributes.keySet().iterator(); pairs.hasNext();) {
			String name = (String) pairs.next();
			stringBuffer.append(", ");
			stringBuffer.append(name);
			stringBuffer.append("=");
			stringBuffer.append(this.attributes.get(name));
		}
		stringBuffer.append("}");
		return stringBuffer.toString();
	}

	/**
	 * Initialize connection specifier's state to the defaults
	 */
	private void useDefaults() {
		this.specification = ConnectionSpecifier.DEFAULT_SPECIFICATION;
		this.serverClass = ConnectionSpecifier.DEFAULT_SERVER_CLASS;
		this.host = ConnectionSpecifier.DEFAULT_HOST;
		this.port = ConnectionSpecifier.DEFAULT_PORT;
		this.serverInstance = ConnectionSpecifier.DEFAULT_SERVER_INSTANCE;
		this.user = null;
		this.password = null;
		this.attributes.clear();
	}

}
