/********************************************************************** 
 * Copyright (c) 2005, 2008 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: AbstractFileServerCommand.java,v 1.14 2008/03/20 18:49:50 dmorris Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/

package org.eclipse.hyades.internal.execution.core.file;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.BufferUnderflowException;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.hyades.execution.core.file.IFileManagerExtended.Cookie;
import org.eclipse.hyades.execution.core.file.IFileManagerExtended.Option;
import org.eclipse.hyades.execution.core.internal.resources.CoreResourceBundle;
import org.eclipse.hyades.internal.execution.core.file.socket.ISocketChannel;
import org.eclipse.osgi.util.NLS;

/**
 * The abstract base class for all file server commands, there are other more
 * specific abstract classes that further extend this one, a concrete file
 * server command will likely extend those downstream subclasses and not this
 * base class directly.
 * 
 * @author Scott E. Schneider
 */
public abstract class AbstractFileServerCommand implements IFileServerCommand {

	/**
	 * The pre-defined base class states are defined here, one for the client
	 * and one for the server. File server commands are expected to subclass one
	 * of these states when defining the state command that gets passed into the
	 * super constructor calls. It is not necessary to extend these subclasses,
	 * but without these, explicit connection and handling must be put all in
	 * the state command passed in, no connection code is inherited fo free. The
	 * client takes care of the system code necessary to establish the command
	 * on the server to communicate with, commands should definitely call the
	 * superclass constructor so they don't have to write this code themselves,
	 * also if the protocol or headers changes, calling the super will help
	 * existing commands be maintained going further
	 * 
	 * @author Scott E. Schneider
	 */
	abstract class Client extends State {

		/**
		 * Creates a client state personality
		 * 
		 * @param channel
		 *            the server channel to communicate with
		 */
		Client(ISocketChannel channel) {
			super(channel);
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.hyades.internal.execution.core.file.ICommand#execute()
		 */
		public void execute() throws IOException {

			// Send the command identity to the server-side to load command
			this.send(AbstractFileServerCommand.this.identity.getName());

		}

	}

	/**
	 * The server state base class is fairly straightforward and reserved for
	 * future use, subclasses should be sure to call super.execute() at the top
	 * of their execute methods in order to get future protocol and header
	 * changes without having to change command code
	 * 
	 * @author Scott E. Schneider
	 */
	abstract class Server extends State {

		/**
		 * The server side personality base class
		 * 
		 * @param client
		 *            the client channel to communicate with
		 */
		Server(ISocketChannel client) {
			super(client);
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.hyades.internal.execution.core.file.ICommand#execute()
		 */
		public void execute() throws IOException {
		}

	}

	/**
	 * The base class for the client and server state personalities
	 * 
	 * @author Scott E. Schneider
	 */
	abstract class State implements IFileServerCommand {

		/**
		 * The channel to communicate with the client or server depending on how
		 * the specializing state objects interpret it
		 */
		private final ISocketChannel channel;

		/**
		 * Constructs a state object, not intended to be directly subclassed by
		 * concrete commands, although it is not prevented
		 * 
		 * @param channel
		 *            the channel
		 */
		State(ISocketChannel channel) {
			this.channel = channel;
		}

		/**
		 * Dispose any resources held by the state, see outer class
		 * documentation for more information
		 * 
		 * @see AbstractFileServerCommand#dispose()
		 */
		public void dispose() {

			// If the channel exists and is not open, close it
			try {
				if (this.channel != null) {
					if (this.channel.isOpen()) {
						this.channel.close();
					}
				}
			} catch (IOException e) {
				// No need to handle this exception
			}

		}

		/**
		 * This is the get socket channel method, the name has been shortened
		 * since it is used quite often in practice in the command client and
		 * server subclasses
		 * 
		 * @return the socket channel, to communicate with the client if the
		 *         command is in the server state and to communicate with the
		 *         server if the command is in the client state
		 */
		ISocketChannel channel() {
			return this.channel;
		}

		/**
		 * Determines if the channel is open, if the channel is null or the
		 * channel is not open, false is returned
		 * 
		 * @return indicates if the connection is connected or not
		 */
		boolean isOpen() {
			return (this.channel != null ? this.channel.isOpen() : false);
		}

		/**
		 * Receive file from sender
		 * 
		 * @param file
		 *            file identifying the file to store the incoming file in
		 */
		void receive(File file) throws IOException {
			long length = 0;
			try {
				length = this.receiveLong();
			} catch (IOException e) {
				// The remote server did not send us the length of the file,
				// so we infer the the file does not exist.
				throw new FileNotFoundException();
			}
			try {
				FileOutputStream outputStream = new FileOutputStream(file);
				FileChannel outputChannel = outputStream.getChannel();
				outputChannel.transferFrom(this.channel, 0, length);
				outputChannel.close();
				outputStream.close();
			} finally {
				long receivedLength = -1;
				if (file.exists()) {
					receivedLength = file.length();
				}
				this.send(receivedLength);
				if (receivedLength != length) {
					throw new IOException();
					// TODO: externalize a string to explain the failure.
				}
			}
		}

		/**
		 * Receives a fixed length int from the sender
		 * 
		 * @return the int received
		 * @throws IOException
		 */
		int receiveInt() throws IOException {
			try {
				ByteBuffer buffer = ByteBuffer.allocate(4);
				this.channel.read(buffer);
				buffer.flip();
				return buffer.getInt();
			} catch (BufferUnderflowException e) {
				throw new IOException();
			}
		}

		/**
		 * Receive a fixed length long from the sender
		 * 
		 * @return the long received
		 * @throws IOException
		 */
		long receiveLong() throws IOException {
			try {
				ByteBuffer buffer = ByteBuffer.allocate(8);
				this.channel.read(buffer);
				buffer.flip();
				return buffer.getLong();
			} catch (BufferUnderflowException e) {
				throw new IOException();
			}
		}

		/**
		 * Receive variable length data from the server, using a simple protocol
		 * where the length in bytes is first sent, followed by the data itself,
		 * using the UTF-8 encoding.
		 * 
		 * @return the string received
		 * @throws IOException
		 */
		String receiveString() throws IOException {
			int length = this.receiveInt();
			byte[] bytes = this.receiveBytes(length);
			String string = new String(bytes, "UTF-8"); //$NON-NLS-1$
			return string;
		}

		/**
		 * Receive a fixed number of bytes from the server. It is expected that
		 * the caller has determined the number of bytes to be read, either
		 * through advance knowledge or through their own protocol.
		 * 
		 * @param length
		 *            the number of bytes sent, this must match the number of
		 *            bytes sent by the client
		 * @return the byte array received
		 * @throws IOException
		 */
		byte[] receiveBytes(int length) throws IOException {

			try {

				// Allocate a buffer to receive the string
				ByteBuffer buffer = ByteBuffer.allocate(length);

				// Read contents into buffer
				this.channel.read(buffer);
				buffer.flip();

				// Return the array of received bytes
				return buffer.array();

			} catch (BufferUnderflowException e) {
				throw new IOException();
			}

		}

		/**
		 * Receive strings being sent from the client
		 * 
		 * @return the received string array
		 * @throws IOException
		 */
		String[] receiveStrings() throws IOException {
			int length = this.receiveInt();
			String[] strings = new String[length];
			for (int i = 0; i < length; i++) {
				strings[i] = this.receiveString();
			}
			return strings;
		}

		/**
		 * Send file to receiver
		 * 
		 * @param file
		 *            file identifying the file to send
		 * @param length
		 *            the file length (or the amount to send)
		 */
		void send(File file) throws IOException, FileNotFoundException {
			FileInputStream inputStream = new FileInputStream(file);
			FileChannel inputChannel = inputStream.getChannel();
			long length = file.length();
			this.send(length);
			inputChannel.transferTo(0, length, this.channel);
			inputChannel.close();
			inputStream.close();
			long receivedLength = this.receiveLong();
			if (receivedLength != length) {
				throw new IOException();
				// TODO: externalize string to describe the failure
			}
		}

		/**
		 * Send a fixed length int to the channel
		 * 
		 * @param data
		 *            the data to send
		 * @throws IOException
		 */
		void send(int data) throws IOException {
			ByteBuffer buffer = ByteBuffer.allocate(4);
			buffer.putInt(data);
			buffer.flip();
			this.channel.write(buffer);
		}

		/**
		 * Send a fixed length long to the channel
		 * 
		 * @param data
		 *            the data to send
		 * @throws IOException
		 */
		void send(long data) throws IOException {
			ByteBuffer buffer = ByteBuffer.allocate(8);
			buffer.putLong(data);
			buffer.flip();
			this.channel.write(buffer);
		}

		/**
		 * Send variable length data to the server, this takes a few more server
		 * calls but is more flexible and potential takes less space since only
		 * the data needed is sent (just enough and not more) -- follows a
		 * simple protocol where the length of the data is first sent to the
		 * server is a fixed length message and then the data follows. The
		 * implementation is just a call to send the length as a long and then a
		 * call to send the string as a fixed length.
		 * 
		 * @param data
		 *            the variable length message to send
		 * @throws IOException
		 */
		void send(String data) throws IOException {
			byte[] bytes = data.getBytes("UTF-8"); //$NON-NLS-1$
			this.send(bytes.length);
			this.send(bytes);
		}

		/**
		 * Send a fixed length message of type byte[]. This is the quickest way
		 * to send and have a message received because it does not involve more
		 * than one write to the server. However, it requires advance knowledge
		 * about the number of bytes being sent and received (on both the client
		 * and server sides.)
		 * 
		 * Note that the client must know the number of bytes to consume on the
		 * receiving end of this call
		 * 
		 * @param bytes
		 *            fixed length message to send
		 * @throws IOException
		 */
		void send(byte[] bytes) throws IOException {
			int length = bytes.length;
			ByteBuffer buffer = ByteBuffer.allocate(length);
			buffer.put(bytes);
			buffer.flip();
			this.channel.write(buffer);
		}

		/**
		 * Sends an array of variable string with any optimizations on the
		 * transfer put in this reusable method
		 * 
		 * @param data
		 *            the strings to send
		 * @throws IOException
		 */
		void send(String[] data) throws IOException {
			this.send(data.length);
			for (int i = 0; i < data.length; i++) {
				this.send(data[i]);
			}
		}

	}

	/**
	 * Stores the cookie for this particular command instance pairing (between
	 * client and server)
	 */
	private Cookie cookie;

	/**
	 * Reference cookie until cookies are implemented, placeholder
	 */
	{
		if (this.cookie != null)
			;
	}

	/**
	 * The identity of this particular command, used to match up the client and
	 * server implementations at the code-level, whereas the cookie matches up
	 * identities at the instance-level across VMs between the client and the
	 * server
	 */
	private Class identity;

	/**
	 * The progress monitor to use for client-side status inquiry and canceling
	 * of task at hand, this is not supported for all commands and is optionally
	 * supported as commands need it
	 */
	private IProgressMonitor monitor;

	/**
	 * Reference monitor until monitor support is implemented, placeholder
	 */
	{
		if (this.monitor != null);
	}

	/**
	 * Store the options for this particular command, it is assumed that any
	 * options will be transmitted and initialized into the server matching
	 * instance as needed
	 */
	private Option[] options;

	/**
	 * Using the state pattern, basically a command once in a different state
	 * takes a new personality and behaves differently like it changes its class
	 * at run-time, there are two conceptual states, client and server. If the
	 * command is on the client it behaves one way (such as opening a connection
	 * to the server and sending commands) whereas if the command is in the
	 * server state it accepts an already connected socket to the client where a
	 * response will then be calculated and sent. In one state the command is
	 * the client and in the other state the command is the server. This allows
	 * the code to be encapsulated per given command yet allows the code to be
	 * written specifically for client or server based on the state that is
	 * passed in to the constructor.
	 */
	private IFileServerCommand state;

	/**
	 * Creates an abstract file server command given the identity, cookie,
	 * options and progress monitor
	 * 
	 * @param identity
	 *            the identity of the command being constructed
	 * @param cookie
	 *            the cookie identifying this instance of the command (same
	 *            cookie across VMs between client and server)
	 * @param options
	 *            the options to tweak the behavior of this command with
	 * @param monitor
	 *            the progress monitor to use
	 */
	AbstractFileServerCommand(Class identity, Cookie cookie, Option[] options, IProgressMonitor monitor) {
		this.identity = identity;
		this.cookie = cookie;
		this.options = options;
		this.monitor = monitor;
	}

	/**
	 * A convenience constructor for creating a server purposed command
	 * 
	 * @param identity
	 *            the command identity
	 */
	AbstractFileServerCommand(Class identity) {
		this(identity, Cookie.NONE, Option.NONE, new NullProgressMonitor());
	}

	/**
	 * Execute the command, instance acts polymorphically based on the way the
	 * command was constructed (either client-side or server-side)
	 */
	public final void execute() throws IOException {
		if (this.state != null) {
			this.state.execute();
		} else {
			System.out.println(NLS.bind(CoreResourceBundle.AbstractFileServerCommand_NO_BEHAVIOUR_DEFINED_, this));
		}
	}

	/**
	 * The options set for this particular command
	 * 
	 * @return the options, could be null or Option.NONE if not set yet
	 */
	Option[] getOptions() {
		return this.options;
	}

	/**
	 * Sets the state of this particular command, there are two conceptual
	 * states, server and client -- a command acts with a client personality or
	 * server personality based on the state set here, there might be other uses
	 * for this with other personality states, but right now client and server
	 * make sense (for example, maybe there is a debug server state and a debug
	 * client state in the future)
	 * 
	 * @param state
	 *            the state to set this command as, before this state is set the
	 *            command does nothing upon execute
	 */
	void setState(IFileServerCommand state) {
		this.state = state;
	}

	/**
	 * Dispose the command, will release any resources such as network
	 * connections, if not called, resources and connections will build up and
	 * likely cause un- desirable results
	 */
	public void dispose() {
		this.state.dispose();
	}

}