/********************************************************************** 
 * Copyright (c) 2005, 2008 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: FileSystemServices.java,v 1.9 2008/03/20 18:49:50 dmorris Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/

package org.eclipse.hyades.internal.execution.core.file;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.util.Properties;

/**
 * Runtime class used by file server for setting dynamic options such as debug
 * and timing.
 * 
 * @author Scott E. Schneider
 */
public final class FileSystemServices {

	/**
	 * File server options that can be enabled or disabled via java system
	 * properties.
	 * 
	 * @author Scott E. Schneider
	 */
	public static final class Options {

		/**
		 * Store all environment variables if available, properties are empty if
		 * they cannot be loaded, all or none
		 */
		private static Properties environmentVariables = new Properties();

		/**
		 * Initialize environment variables, one time at class loading time,
		 * only UNIX-like and Windows OSs are supported, all others must set
		 * their options strictly using the Java system property -D method
		 */
		static {

			// Retrieve OS name and Java runtime instance
			String os = System.getProperty("os.name").toLowerCase();//$NON-NLS-1$
			Runtime runtime = Runtime.getRuntime();
			String command = null;

			// Different command to execute based on OS name
			try {

				// Use command.com for all Windows, env for all Unix-likes
				final String TEMPORARY_FILE_NAME_PREFIX = "env";//$NON-NLS-1$
				final String TEMPORARY_FILE_NAME_EXTENSION = ".tmp";//$NON-NLS-1$

				// Create temporary file and retrieve absolute name
				File file = File.createTempFile(TEMPORARY_FILE_NAME_PREFIX, TEMPORARY_FILE_NAME_EXTENSION);
				file.deleteOnExit();
				final String fileName = file.getAbsolutePath();

				// Different logic based on the platform running on
				if (os.indexOf("windows") > -1 || os.indexOf("nt") > -1) {//$NON-NLS-1$
					command = "cmd.exe /c set";//$NON-NLS-1$
					command += " > " + fileName;
				} else {
					// Unix not currently supported
				}

				if (command != null) {

					// Execute the process and receive process object
					Process process = runtime.exec(command);

					// If process available and exited cleanly
					if (process != null) {

						// Wait for process to complete
						int exitValue = process.waitFor();

						// If successful read environment variables
						if (exitValue == 0) {
							BufferedInputStream inputStream = new BufferedInputStream(new FileInputStream(file));
							FileSystemServices.Options.environmentVariables.load(inputStream);
						}

						// Cleanup afterwards
						process.destroy();
						file.delete();

					}
				}

			} catch (Throwable t) {

				// Leave properties empty if exception occurs (all or none)
				if (FileSystemServices.Options.environmentVariables != null) {
					FileSystemServices.Options.environmentVariables.clear();
				}

			}

		}

		/**
		 * Enables verbose debug output to the console (can be combined with
		 * other options)
		 */
		public static final Options SHOW_DEBUG = new Options("SHOW_DEBUG", false);//$NON-NLS-1$

		/**
		 * Enables class loading output to the console (can be combined with
		 * other options)
		 */
		public static final Options SHOW_LOADING = new Options("SHOW_LOADING", false);//$NON-NLS-1$

		/**
		 * Enables performance timing output to the console (can be combined
		 * with other options)
		 */
		public static final Options SHOW_TIMING = new Options("SHOW_TIMING", false);//$NON-NLS-1$

		/**
		 * Indicates if the particular option is enabled or disabled
		 */
		private boolean isEnabled = false;

		/**
		 * The name of the runtime option, the name must match the name of the
		 * system property to be used to change the value of the option, to
		 * enable file server debug console output, set -DSHOW_DEBUG=TRUE for
		 * example.
		 */
		private String name;

		/**
		 * Type safe enumeration of options for the file server, sets the
		 * option's enablement attempting Java system property first, then
		 * environment variable and then default hard-coded value
		 */
		private Options(String name, boolean defaultEnablement) {

			// Set the name and value of this option
			this.name = name;
			String value = System.getProperty(this.name);

			// If system property is not set try environment variable
			if (value == null) {
				value = FileSystemServices.Options.environmentVariables.getProperty(this.name);
			}

			/*
			 * If the system property key is not found, use the default, if the
			 * value is set, use the specified value
			 */
			this.isEnabled = (value != null ? Boolean.valueOf(value).booleanValue() : defaultEnablement);

			// Display settings to output appropriately
			FileSystemServices
					.println("File server option " + this.name + " is " + (this.isEnabled ? "" : "not ")
							+ "enabled (use -D" + this.name + " = { TRUE | FALSE } to specify enablement setting)",
							this.isEnabled);

		}

		/**
		 * Retrieve the name of the option
		 * 
		 * @return the option name
		 */
		public String getName() {
			return this.name;
		}

		/**
		 * Indicates if this option is enabled or not
		 * 
		 * @return true if the option is enabled
		 */
		public boolean isEnabled() {
			return this.isEnabled;
		}

		/**
		 * Set enabled as true or false, specifying the enablement setting
		 * 
		 * @param isEnabled
		 *            set the enablement to true or false
		 */
		public void setEnabled(boolean isEnabled) {
			this.isEnabled = isEnabled;
		}
	}

	/**
	 * Sequence number for debug output console
	 */
	private static long sequenceNumber = 0;

	/**
	 * Pads the specified number to the following width in string format
	 * 
	 * @param number
	 *            the number to pad in the width specified string
	 * @param width
	 *            the width to pad the string to
	 * @return the padded string that indicates the number padded
	 */
	private static String pad(long number, int width) {
		String value = String.valueOf(number);
		StringBuffer buffer = new StringBuffer(value);
		for (int i = 0, j = value.length(); i < width - j; i++) {
			buffer.insert(0, ' ');
		}
		return buffer.toString();
	}

	/**
	 * Prints the specified text to the appropriate debug output, only displayed
	 * if SHOW_DEBUG is enabled via file server options.
	 * 
	 * @param text
	 *            the message to display, does not need to be externalized, only
	 *            used for debug purposes
	 */
	public static void println(String text) {
		FileSystemServices.println(text, FileSystemServices.Options.SHOW_DEBUG.isEnabled());
	}

	/**
	 * Outputs the specified text conditionally based on the value of the is
	 * enabled argument
	 * 
	 * @param text
	 *            the text to display
	 * @param isEnabled
	 *            indicates if the text should be truly displayed
	 */
	public static void println(String text, boolean isEnabled) {
		FileSystemServices.println(text, isEnabled, FileSystemServices.class);
	}

	/**
	 * Outputs the specified text conditionally based on the value of the is
	 * enabled argument and using the caller object for additional context to
	 * output if appropriate
	 * 
	 * @param text
	 *            the text to display
	 * @param isEnabled
	 *            indicates if the text should be suppressed
	 * @param caller
	 *            the calling output
	 */
	public synchronized static void println(String text, boolean isEnabled, Object caller) {
		if (isEnabled) {
			System.out.println("\r\n" + FileSystemServices.pad(++FileSystemServices.sequenceNumber, 15) + "\t"
					+ Thread.currentThread() + " in " + caller + "\r\n\t\t" + text);
		}
	}

	/**
	 * Prints the specified text based on the value of the is enabled argument
	 * 
	 * @param text
	 *            the text to display
	 * @param caller
	 *            the caller, used for contextual information
	 */
	public static void println(String text, Object caller) {
		FileSystemServices.println(text, FileSystemServices.Options.SHOW_DEBUG.isEnabled(), caller);
	}

	/**
	 * Intended to be use for the static options and methods exclusively
	 */
	private FileSystemServices() {
	}

}